<?php
/**
 * Wishlist Template
 */

use Cartt\Services\WishlistService;
use Cartt\Services\CurrencyService;
use Cartt\Services\CartService;

// Get wishlist
$userId = get_current_user_id();
$sessionId = !$userId ? (isset($_COOKIE['cartt_session']) ? sanitize_text_field($_COOKIE['cartt_session']) : null) : null;

if (!$userId && !$sessionId) {
    $sessionId = wp_generate_password(32, false);
    setcookie('cartt_session', $sessionId, time() + (86400 * 30), '/');
}

$wishlist = WishlistService::getWishlist($userId, $sessionId);
$items = WishlistService::getItems($wishlist->id);
$shareUrl = $wishlist->share_key && $wishlist->is_public ? home_url('/wishlist/' . $wishlist->share_key) : null;
?>

<div class="cartt-wishlist">
    <div class="cartt-wishlist-header">
        <h2>My Wishlist</h2>
        <div class="cartt-wishlist-actions">
            <?php if (!empty($items)): ?>
            <button type="button" class="cartt-btn cartt-btn-secondary cartt-share-wishlist-btn" data-public="<?php echo $wishlist->is_public ? '1' : '0'; ?>">
                <?php echo $wishlist->is_public ? 'Make Private' : 'Share Wishlist'; ?>
            </button>
            <button type="button" class="cartt-btn cartt-btn-primary cartt-add-all-to-cart">Add All to Cart</button>
            <?php endif; ?>
        </div>
    </div>

    <?php if ($shareUrl): ?>
    <div class="cartt-wishlist-share-url">
        <span>Share link:</span>
        <input type="text" readonly value="<?php echo esc_url($shareUrl); ?>" onclick="this.select();">
        <button type="button" class="cartt-copy-link" data-url="<?php echo esc_url($shareUrl); ?>">Copy</button>
    </div>
    <?php endif; ?>

    <?php if (empty($items)): ?>
    <div class="cartt-wishlist-empty">
        <p>Your wishlist is empty.</p>
        <a href="<?php echo home_url('/shop'); ?>" class="cartt-btn cartt-btn-primary">Continue Shopping</a>
    </div>
    <?php else: ?>
    <div class="cartt-wishlist-items">
        <?php foreach ($items as $item): ?>
        <div class="cartt-wishlist-item" data-product-id="<?php echo esc_attr($item->product_id); ?>" data-variation-id="<?php echo esc_attr($item->variation_id ?? ''); ?>">
            <div class="cartt-wishlist-item-image">
                <?php if ($item->image_url): ?>
                <img src="<?php echo esc_url($item->image_url); ?>" alt="<?php echo esc_attr($item->name); ?>">
                <?php else: ?>
                <div class="cartt-placeholder-image"></div>
                <?php endif; ?>
            </div>
            <div class="cartt-wishlist-item-details">
                <h3><a href="<?php echo home_url('/product/' . $item->slug); ?>"><?php echo esc_html($item->name); ?></a></h3>
                <div class="cartt-wishlist-item-price">
                    <?php if ($item->sale_price && $item->sale_price < $item->price): ?>
                    <span class="cartt-sale-price"><?php echo esc_html(CurrencyService::formatPrice(CurrencyService::convertPrice($item->sale_price))); ?></span>
                    <span class="cartt-original-price"><?php echo esc_html(CurrencyService::formatPrice(CurrencyService::convertPrice($item->price))); ?></span>
                    <?php else: ?>
                    <span><?php echo esc_html(CurrencyService::formatPrice(CurrencyService::convertPrice($item->price))); ?></span>
                    <?php endif; ?>
                </div>
                <div class="cartt-wishlist-item-stock <?php echo $item->in_stock ? 'in-stock' : 'out-of-stock'; ?>">
                    <?php if ($item->in_stock): ?>
                    <span class="cartt-stock-badge cartt-in-stock">In Stock</span>
                    <?php else: ?>
                    <span class="cartt-stock-badge cartt-out-of-stock">Out of Stock</span>
                    <label class="cartt-notify-stock">
                        <input type="checkbox" class="cartt-notify-stock-checkbox" <?php checked($item->notify_on_stock); ?>>
                        Notify me when back in stock
                    </label>
                    <?php endif; ?>
                </div>
            </div>
            <div class="cartt-wishlist-item-actions">
                <?php if ($item->in_stock): ?>
                <button type="button" class="cartt-btn cartt-btn-primary cartt-add-to-cart-btn" data-product-id="<?php echo esc_attr($item->product_id); ?>">
                    Add to Cart
                </button>
                <?php endif; ?>
                <button type="button" class="cartt-btn cartt-btn-text cartt-remove-wishlist" title="Remove">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>
                </button>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
    <?php endif; ?>
</div>

<style>
.cartt-wishlist { max-width: 900px; margin: 0 auto; padding: 20px; }
.cartt-wishlist-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
.cartt-wishlist-header h2 { margin: 0; }
.cartt-wishlist-actions { display: flex; gap: 10px; }
.cartt-wishlist-share-url { display: flex; align-items: center; gap: 10px; background: #f5f5f7; padding: 10px 15px; border-radius: 8px; margin-bottom: 20px; }
.cartt-wishlist-share-url input { flex: 1; border: 1px solid #ddd; padding: 8px; border-radius: 4px; }
.cartt-wishlist-empty { text-align: center; padding: 60px 20px; background: #f9f9f9; border-radius: 8px; }
.cartt-wishlist-items { display: flex; flex-direction: column; gap: 15px; }
.cartt-wishlist-item { display: flex; gap: 20px; padding: 20px; background: #fff; border: 1px solid #eee; border-radius: 8px; align-items: center; }
.cartt-wishlist-item-image { width: 100px; height: 100px; flex-shrink: 0; }
.cartt-wishlist-item-image img { width: 100%; height: 100%; object-fit: cover; border-radius: 4px; }
.cartt-placeholder-image { width: 100%; height: 100%; background: #f0f0f0; border-radius: 4px; }
.cartt-wishlist-item-details { flex: 1; }
.cartt-wishlist-item-details h3 { margin: 0 0 8px; font-size: 16px; }
.cartt-wishlist-item-details h3 a { color: inherit; text-decoration: none; }
.cartt-wishlist-item-details h3 a:hover { text-decoration: underline; }
.cartt-wishlist-item-price { margin-bottom: 8px; }
.cartt-sale-price { font-weight: bold; color: #e53935; }
.cartt-original-price { text-decoration: line-through; color: #999; margin-left: 8px; }
.cartt-stock-badge { display: inline-block; padding: 3px 8px; border-radius: 4px; font-size: 12px; }
.cartt-in-stock { background: #d4edda; color: #155724; }
.cartt-out-of-stock { background: #f8d7da; color: #721c24; }
.cartt-notify-stock { display: block; margin-top: 8px; font-size: 13px; color: #666; }
.cartt-wishlist-item-actions { display: flex; flex-direction: column; gap: 10px; align-items: flex-end; }
.cartt-btn { padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; transition: all 0.2s; }
.cartt-btn-primary { background: #1d1d1f; color: #fff; }
.cartt-btn-primary:hover { background: #333; }
.cartt-btn-secondary { background: #f5f5f7; color: #1d1d1f; border: 1px solid #ddd; }
.cartt-btn-secondary:hover { background: #eee; }
.cartt-btn-text { background: none; color: #666; padding: 5px; }
.cartt-btn-text:hover { color: #e53935; }
.cartt-copy-link { padding: 8px 16px; background: #1d1d1f; color: #fff; border: none; border-radius: 4px; cursor: pointer; }

@media (max-width: 600px) {
    .cartt-wishlist-item { flex-wrap: wrap; }
    .cartt-wishlist-item-image { width: 80px; height: 80px; }
    .cartt-wishlist-item-actions { flex-direction: row; width: 100%; justify-content: space-between; margin-top: 10px; }
}
</style>

<script>
(function() {
    const nonce = '<?php echo wp_create_nonce('cartt_wishlist'); ?>';
    
    // Remove from wishlist
    document.querySelectorAll('.cartt-remove-wishlist').forEach(btn => {
        btn.addEventListener('click', function() {
            const item = this.closest('.cartt-wishlist-item');
            const productId = item.dataset.productId;
            const variationId = item.dataset.variationId || '';
            
            fetch(carttFrontend.ajaxUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=cartt_remove_from_wishlist&nonce=${nonce}&product_id=${productId}&variation_id=${variationId}`
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    item.remove();
                    if (!document.querySelectorAll('.cartt-wishlist-item').length) {
                        location.reload();
                    }
                }
            });
        });
    });

    // Add to cart
    document.querySelectorAll('.cartt-add-to-cart-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const productId = this.dataset.productId;
            
            fetch(carttFrontend.ajaxUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=cartt_add_to_cart&nonce=${carttFrontend.nonce}&product_id=${productId}&quantity=1`
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    this.textContent = 'Added!';
                    setTimeout(() => this.textContent = 'Add to Cart', 2000);
                }
            });
        });
    });

    // Add all to cart
    document.querySelector('.cartt-add-all-to-cart')?.addEventListener('click', function() {
        fetch(carttFrontend.ajaxUrl, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: `action=cartt_wishlist_add_all_to_cart&nonce=${nonce}`
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                alert(`Added ${data.data.added} items to cart!`);
                location.reload();
            }
        });
    });

    // Toggle public/private
    document.querySelector('.cartt-share-wishlist-btn')?.addEventListener('click', function() {
        const isPublic = this.dataset.public === '1';
        
        fetch(carttFrontend.ajaxUrl, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: `action=cartt_toggle_wishlist_public&nonce=${nonce}&is_public=${isPublic ? 0 : 1}`
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                location.reload();
            }
        });
    });

    // Copy link
    document.querySelector('.cartt-copy-link')?.addEventListener('click', function() {
        navigator.clipboard.writeText(this.dataset.url);
        this.textContent = 'Copied!';
        setTimeout(() => this.textContent = 'Copy', 2000);
    });

    // Notify on stock
    document.querySelectorAll('.cartt-notify-stock-checkbox').forEach(cb => {
        cb.addEventListener('change', function() {
            const item = this.closest('.cartt-wishlist-item');
            const productId = item.dataset.productId;
            
            fetch(carttFrontend.ajaxUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=cartt_wishlist_notify_stock&nonce=${nonce}&product_id=${productId}&notify=${this.checked ? 1 : 0}`
            });
        });
    });
})();
</script>
