<?php
/**
 * Vendor Store Template
 * 
 * @package Cartt
 * @since 1.4.0
 * 
 * Variables available:
 * @var array $vendor - Vendor data
 * @var array $products - Vendor products
 * @var array $atts - Shortcode attributes
 */

defined('ABSPATH') || exit;
?>

<div class="cartt-vendor-store" data-vendor-id="<?php echo esc_attr($vendor['id']); ?>">
    
    <?php if ($atts['show_header'] === 'yes'): ?>
    <div class="cartt-vendor-header">
        <?php if (!empty($vendor['banner_url'])): ?>
        <div class="cartt-vendor-banner">
            <img src="<?php echo esc_url($vendor['banner_url']); ?>" alt="<?php echo esc_attr($vendor['store_name']); ?>">
        </div>
        <?php endif; ?>
        
        <div class="cartt-vendor-info">
            <?php if (!empty($vendor['logo_url'])): ?>
            <div class="cartt-vendor-logo">
                <img src="<?php echo esc_url($vendor['logo_url']); ?>" alt="<?php echo esc_attr($vendor['store_name']); ?>">
            </div>
            <?php endif; ?>
            
            <div class="cartt-vendor-details">
                <h1 class="cartt-vendor-name">
                    <?php echo esc_html($vendor['store_name']); ?>
                    <?php if (!empty($vendor['verified'])): ?>
                    <span class="cartt-verified-badge" title="Verified Vendor">&#10003;</span>
                    <?php endif; ?>
                </h1>
                
                <?php if (!empty($vendor['store_description'])): ?>
                <div class="cartt-vendor-description">
                    <?php echo wp_kses_post($vendor['store_description']); ?>
                </div>
                <?php endif; ?>
                
                <div class="cartt-vendor-meta">
                    <?php if (!empty($vendor['rating'])): ?>
                    <span class="cartt-vendor-rating">
                        <?php echo str_repeat('&#9733;', round($vendor['rating'])); ?>
                        <?php echo str_repeat('&#9734;', 5 - round($vendor['rating'])); ?>
                        <span class="cartt-rating-count">(<?php echo esc_html($vendor['rating_count'] ?? 0); ?> reviews)</span>
                    </span>
                    <?php endif; ?>
                    
                    <span class="cartt-vendor-since">
                        Selling since <?php echo esc_html(date('F Y', strtotime($vendor['created_at']))); ?>
                    </span>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <div class="cartt-vendor-products">
        <h2 class="cartt-section-title">Products from <?php echo esc_html($vendor['store_name']); ?></h2>
        
        <?php if (empty($products)): ?>
        <p class="cartt-no-products">This vendor hasn't listed any products yet.</p>
        <?php else: ?>
        <div class="cartt-products-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px;">
            <?php foreach ($products as $product): ?>
            <div class="cartt-product-card">
                <a href="<?php echo esc_url(get_permalink($product['id'])); ?>" class="cartt-product-link">
                    <?php if (!empty($product['image_url'])): ?>
                    <img src="<?php echo esc_url($product['image_url']); ?>" alt="<?php echo esc_attr($product['name']); ?>" class="cartt-product-image">
                    <?php else: ?>
                    <div class="cartt-product-placeholder"></div>
                    <?php endif; ?>
                    
                    <h3 class="cartt-product-name"><?php echo esc_html($product['name']); ?></h3>
                    
                    <div class="cartt-product-price">
                        <?php echo \Cartt\Services\CurrencyService::formatPrice($product['price']); ?>
                    </div>
                </a>
                
                <button type="button" class="cartt-add-to-cart" data-product-id="<?php echo esc_attr($product['id']); ?>">
                    Add to Cart
                </button>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</div>

<style>
.cartt-vendor-store { max-width: 1200px; margin: 0 auto; }
.cartt-vendor-banner { width: 100%; height: 200px; overflow: hidden; border-radius: 8px 8px 0 0; }
.cartt-vendor-banner img { width: 100%; height: 100%; object-fit: cover; }
.cartt-vendor-info { display: flex; gap: 20px; padding: 20px; background: #f9f9f9; border-radius: 0 0 8px 8px; margin-bottom: 30px; }
.cartt-vendor-logo { width: 100px; height: 100px; border-radius: 50%; overflow: hidden; border: 4px solid #fff; margin-top: -60px; background: #fff; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
.cartt-vendor-logo img { width: 100%; height: 100%; object-fit: cover; }
.cartt-vendor-name { margin: 0 0 10px; font-size: 24px; }
.cartt-verified-badge { color: #00a32a; font-size: 18px; vertical-align: middle; }
.cartt-vendor-description { color: #666; margin-bottom: 10px; }
.cartt-vendor-meta { font-size: 14px; color: #999; }
.cartt-vendor-rating { color: #f5a623; margin-right: 15px; }
.cartt-product-card { background: #fff; border: 1px solid #eee; border-radius: 8px; padding: 15px; }
.cartt-product-image { width: 100%; aspect-ratio: 1; object-fit: cover; border-radius: 4px; }
.cartt-product-placeholder { width: 100%; aspect-ratio: 1; background: #f0f0f0; border-radius: 4px; }
.cartt-product-name { font-size: 14px; margin: 10px 0 5px; }
.cartt-product-price { font-weight: 600; color: #333; margin-bottom: 10px; }
.cartt-add-to-cart { width: 100%; padding: 10px; background: #2271b1; color: #fff; border: none; border-radius: 4px; cursor: pointer; }
.cartt-add-to-cart:hover { background: #135e96; }
</style>
