<?php
/**
 * Vendor Dashboard Template
 * 
 * @package Cartt
 * @since 1.4.0
 * 
 * Variables available:
 * @var array $data - Dashboard data including vendor, products, orders, earnings
 */

defined('ABSPATH') || exit;

$vendor = $data['vendor'];
$products = $data['products'];
$orders = $data['orders'];
$earnings = $data['earnings'];
$tab = $data['tab'];
?>

<div class="cartt-vendor-dashboard">
    <div class="cartt-dashboard-header">
        <h1>Vendor Dashboard</h1>
        <div class="cartt-vendor-status">
            Status: <span class="status-badge status-<?php echo esc_attr($vendor['status']); ?>">
                <?php echo esc_html(ucfirst($vendor['status'])); ?>
            </span>
        </div>
    </div>

    <nav class="cartt-dashboard-nav">
        <a href="?tab=overview" class="<?php echo $tab === 'overview' ? 'active' : ''; ?>">Overview</a>
        <a href="?tab=products" class="<?php echo $tab === 'products' ? 'active' : ''; ?>">Products</a>
        <a href="?tab=orders" class="<?php echo $tab === 'orders' ? 'active' : ''; ?>">Orders</a>
        <a href="?tab=earnings" class="<?php echo $tab === 'earnings' ? 'active' : ''; ?>">Earnings</a>
        <a href="?tab=settings" class="<?php echo $tab === 'settings' ? 'active' : ''; ?>">Store Settings</a>
    </nav>

    <div class="cartt-dashboard-content">
        <?php if ($tab === 'overview'): ?>
        <div class="cartt-stats-grid">
            <div class="cartt-stat-card">
                <div class="stat-label">Total Products</div>
                <div class="stat-value"><?php echo count($products); ?></div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-label">Total Orders</div>
                <div class="stat-value"><?php echo esc_html($vendor['total_orders'] ?? 0); ?></div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-label">Available Earnings</div>
                <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings['available'] ?? 0); ?></div>
            </div>
            <div class="cartt-stat-card">
                <div class="stat-label">Lifetime Earnings</div>
                <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings['total_earned'] ?? 0); ?></div>
            </div>
        </div>

        <div class="cartt-recent-orders">
            <h3>Recent Orders</h3>
            <?php if (empty($orders)): ?>
            <p>No orders yet.</p>
            <?php else: ?>
            <table class="cartt-table">
                <thead>
                    <tr>
                        <th>Order</th>
                        <th>Date</th>
                        <th>Customer</th>
                        <th>Total</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_slice($orders, 0, 5) as $order): ?>
                    <tr>
                        <td>#<?php echo esc_html($order['id']); ?></td>
                        <td><?php echo esc_html(date('M j, Y', strtotime($order['created_at']))); ?></td>
                        <td><?php echo esc_html($order['customer_name'] ?? 'Guest'); ?></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($order['vendor_total'] ?? $order['total']); ?></td>
                        <td><span class="status-badge status-<?php echo esc_attr($order['status']); ?>"><?php echo esc_html(ucfirst($order['status'])); ?></span></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <?php elseif ($tab === 'products'): ?>
        <div class="cartt-products-section">
            <div class="section-header">
                <h3>Your Products</h3>
                <a href="#" class="cartt-button cartt-button-primary" id="add-product-btn">Add New Product</a>
            </div>
            
            <?php if (empty($products)): ?>
            <p>You haven't added any products yet.</p>
            <?php else: ?>
            <table class="cartt-table">
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>SKU</th>
                        <th>Price</th>
                        <th>Stock</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($products as $product): ?>
                    <tr>
                        <td>
                            <div class="product-cell">
                                <?php if (!empty($product['image_url'])): ?>
                                <img src="<?php echo esc_url($product['image_url']); ?>" width="40" height="40">
                                <?php endif; ?>
                                <span><?php echo esc_html($product['name']); ?></span>
                            </div>
                        </td>
                        <td><?php echo esc_html($product['sku'] ?? '-'); ?></td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($product['price']); ?></td>
                        <td><?php echo esc_html($product['stock_quantity'] ?? 'N/A'); ?></td>
                        <td><span class="status-badge status-<?php echo esc_attr($product['status']); ?>"><?php echo esc_html(ucfirst($product['status'])); ?></span></td>
                        <td>
                            <a href="#" class="edit-product" data-id="<?php echo esc_attr($product['id']); ?>">Edit</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <?php elseif ($tab === 'orders'): ?>
        <div class="cartt-orders-section">
            <h3>Your Orders</h3>
            <?php if (empty($orders)): ?>
            <p>No orders yet.</p>
            <?php else: ?>
            <table class="cartt-table">
                <thead>
                    <tr>
                        <th>Order</th>
                        <th>Date</th>
                        <th>Customer</th>
                        <th>Items</th>
                        <th>Your Earnings</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($orders as $order): ?>
                    <tr>
                        <td>#<?php echo esc_html($order['id']); ?></td>
                        <td><?php echo esc_html(date('M j, Y', strtotime($order['created_at']))); ?></td>
                        <td><?php echo esc_html($order['customer_name'] ?? 'Guest'); ?></td>
                        <td><?php echo esc_html($order['item_count'] ?? 1); ?> item(s)</td>
                        <td><?php echo \Cartt\Services\CurrencyService::formatPrice($order['vendor_earnings'] ?? 0); ?></td>
                        <td><span class="status-badge status-<?php echo esc_attr($order['status']); ?>"><?php echo esc_html(ucfirst($order['status'])); ?></span></td>
                        <td><a href="#" class="view-order" data-id="<?php echo esc_attr($order['id']); ?>">View</a></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <?php elseif ($tab === 'earnings'): ?>
        <div class="cartt-earnings-section">
            <div class="cartt-stats-grid">
                <div class="cartt-stat-card">
                    <div class="stat-label">Available for Withdrawal</div>
                    <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings['available'] ?? 0); ?></div>
                </div>
                <div class="cartt-stat-card">
                    <div class="stat-label">Pending (On Hold)</div>
                    <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings['pending'] ?? 0); ?></div>
                </div>
                <div class="cartt-stat-card">
                    <div class="stat-label">Total Withdrawn</div>
                    <div class="stat-value"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings['withdrawn'] ?? 0); ?></div>
                </div>
            </div>

            <?php if (($earnings['available'] ?? 0) >= 50): ?>
            <div class="cartt-withdrawal-section">
                <h3>Request Withdrawal</h3>
                <form method="post" class="cartt-withdrawal-form">
                    <?php wp_nonce_field('cartt_vendor_withdrawal'); ?>
                    <div class="form-row">
                        <label>Amount to Withdraw</label>
                        <input type="number" name="amount" min="50" max="<?php echo esc_attr($earnings['available']); ?>" step="0.01" required>
                    </div>
                    <div class="form-row">
                        <label>Payment Method</label>
                        <select name="payment_method">
                            <option value="paypal">PayPal</option>
                            <option value="bank">Bank Transfer</option>
                        </select>
                    </div>
                    <button type="submit" name="request_withdrawal" class="cartt-button cartt-button-primary">Request Withdrawal</button>
                </form>
            </div>
            <?php else: ?>
            <p class="cartt-notice">Minimum withdrawal amount is $50.00. Keep selling to reach the threshold!</p>
            <?php endif; ?>
        </div>

        <?php elseif ($tab === 'settings'): ?>
        <div class="cartt-settings-section">
            <h3>Store Settings</h3>
            <form method="post" class="cartt-vendor-settings-form" enctype="multipart/form-data">
                <?php wp_nonce_field('cartt_vendor_settings'); ?>
                
                <div class="form-row">
                    <label>Store Name</label>
                    <input type="text" name="store_name" value="<?php echo esc_attr($vendor['store_name']); ?>" required>
                </div>
                
                <div class="form-row">
                    <label>Store Description</label>
                    <textarea name="store_description" rows="4"><?php echo esc_textarea($vendor['store_description'] ?? ''); ?></textarea>
                </div>
                
                <div class="form-row">
                    <label>Contact Email</label>
                    <input type="email" name="email" value="<?php echo esc_attr($vendor['email'] ?? ''); ?>">
                </div>
                
                <div class="form-row">
                    <label>Phone</label>
                    <input type="text" name="phone" value="<?php echo esc_attr($vendor['phone'] ?? ''); ?>">
                </div>
                
                <div class="form-row">
                    <label>Store Logo</label>
                    <input type="file" name="logo" accept="image/*">
                    <?php if (!empty($vendor['logo_url'])): ?>
                    <img src="<?php echo esc_url($vendor['logo_url']); ?>" width="80" class="current-logo">
                    <?php endif; ?>
                </div>
                
                <div class="form-row">
                    <label>Store Banner</label>
                    <input type="file" name="banner" accept="image/*">
                </div>
                
                <button type="submit" name="update_settings" class="cartt-button cartt-button-primary">Save Settings</button>
            </form>
        </div>
        <?php endif; ?>
    </div>
</div>

<style>
.cartt-vendor-dashboard { max-width: 1000px; margin: 0 auto; }
.cartt-dashboard-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
.cartt-dashboard-nav { display: flex; gap: 5px; border-bottom: 2px solid #eee; margin-bottom: 25px; }
.cartt-dashboard-nav a { padding: 12px 20px; text-decoration: none; color: #666; border-bottom: 2px solid transparent; margin-bottom: -2px; }
.cartt-dashboard-nav a.active, .cartt-dashboard-nav a:hover { color: #2271b1; border-bottom-color: #2271b1; }
.cartt-stats-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; margin-bottom: 30px; }
.cartt-stat-card { background: #f9f9f9; padding: 20px; border-radius: 8px; text-align: center; }
.stat-label { color: #666; font-size: 13px; margin-bottom: 5px; }
.stat-value { font-size: 24px; font-weight: 600; }
.cartt-table { width: 100%; border-collapse: collapse; }
.cartt-table th, .cartt-table td { padding: 12px; text-align: left; border-bottom: 1px solid #eee; }
.cartt-table th { background: #f9f9f9; font-weight: 600; }
.status-badge { padding: 4px 10px; border-radius: 12px; font-size: 12px; }
.status-active, .status-completed { background: #d4edda; color: #155724; }
.status-pending { background: #fff3cd; color: #856404; }
.status-processing { background: #cce5ff; color: #004085; }
.form-row { margin-bottom: 20px; }
.form-row label { display: block; margin-bottom: 5px; font-weight: 500; }
.form-row input, .form-row select, .form-row textarea { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; }
.cartt-button { padding: 12px 24px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; }
.cartt-button-primary { background: #2271b1; color: #fff; }
.product-cell { display: flex; align-items: center; gap: 10px; }
.product-cell img { border-radius: 4px; }
.section-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
</style>
