<?php
/**
 * Product Search & Filter Template
 */

use Cartt\Services\ProductSearchService;
use Cartt\Services\CurrencyService;

$filterOptions = ProductSearchService::getFilterOptions();
?>

<div class="cartt-search-filter">
    <!-- Search Bar -->
    <div class="cartt-search-bar">
        <div class="cartt-search-input-wrapper">
            <svg class="cartt-search-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="11" cy="11" r="8"></circle>
                <line x1="21" y1="21" x2="16.65" y2="16.65"></line>
            </svg>
            <input type="text" id="cartt-search-input" placeholder="Search products..." autocomplete="off">
            <button type="button" id="cartt-search-clear" style="display: none;">&times;</button>
        </div>
        <div id="cartt-autocomplete" class="cartt-autocomplete" style="display: none;"></div>
    </div>

    <div class="cartt-filter-layout">
        <!-- Filters Sidebar -->
        <aside class="cartt-filters-sidebar">
            <div class="cartt-filter-header">
                <h3>Filters</h3>
                <button type="button" id="cartt-clear-filters">Clear All</button>
            </div>

            <!-- Categories -->
            <?php if (!empty($filterOptions['categories'])): ?>
            <div class="cartt-filter-group">
                <h4>Categories</h4>
                <ul class="cartt-filter-list">
                    <?php foreach ($filterOptions['categories'] as $cat): ?>
                    <li>
                        <label>
                            <input type="checkbox" name="category" value="<?php echo esc_attr($cat->id); ?>">
                            <?php echo esc_html($cat->name); ?> <span class="count">(<?php echo $cat->count; ?>)</span>
                        </label>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>

            <!-- Price Range -->
            <div class="cartt-filter-group">
                <h4>Price</h4>
                <div class="cartt-price-range">
                    <input type="number" id="cartt-min-price" placeholder="Min" min="0" step="1">
                    <span>—</span>
                    <input type="number" id="cartt-max-price" placeholder="Max" min="0" step="1">
                </div>
                <div class="cartt-price-slider">
                    <input type="range" id="cartt-price-slider" min="<?php echo intval($filterOptions['price_range']['min']); ?>" max="<?php echo intval($filterOptions['price_range']['max']); ?>" value="<?php echo intval($filterOptions['price_range']['max']); ?>">
                </div>
            </div>

            <!-- Rating -->
            <div class="cartt-filter-group">
                <h4>Rating</h4>
                <ul class="cartt-filter-list cartt-rating-filter">
                    <?php for ($i = 4; $i >= 1; $i--): ?>
                    <li>
                        <label>
                            <input type="radio" name="min_rating" value="<?php echo $i; ?>">
                            <?php echo str_repeat('★', $i) . str_repeat('☆', 5 - $i); ?> & Up
                        </label>
                    </li>
                    <?php endfor; ?>
                </ul>
            </div>

            <!-- Stock -->
            <div class="cartt-filter-group">
                <h4>Availability</h4>
                <label>
                    <input type="checkbox" id="cartt-in-stock">
                    In Stock Only
                </label>
            </div>

            <!-- On Sale -->
            <div class="cartt-filter-group">
                <label>
                    <input type="checkbox" id="cartt-on-sale">
                    On Sale
                </label>
            </div>

            <!-- Dynamic Attributes -->
            <?php foreach ($filterOptions['attributes'] as $attrName => $values): ?>
            <div class="cartt-filter-group">
                <h4><?php echo esc_html(ucfirst($attrName)); ?></h4>
                <ul class="cartt-filter-list">
                    <?php foreach ($values as $value): ?>
                    <li>
                        <label>
                            <input type="checkbox" name="attr_<?php echo esc_attr($attrName); ?>" value="<?php echo esc_attr($value); ?>">
                            <?php echo esc_html($value); ?>
                        </label>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endforeach; ?>
        </aside>

        <!-- Products Grid -->
        <main class="cartt-products-main">
            <div class="cartt-products-header">
                <div class="cartt-results-count">
                    <span id="cartt-results-count">0</span> products
                </div>
                <div class="cartt-sort-wrapper">
                    <label for="cartt-sort">Sort by:</label>
                    <select id="cartt-sort">
                        <?php foreach ($filterOptions['sort_options'] as $key => $label): ?>
                        <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($label); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div id="cartt-products-grid" class="cartt-products-grid">
                <!-- Products loaded via AJAX -->
            </div>

            <div id="cartt-loading" class="cartt-loading" style="display: none;">
                <div class="cartt-spinner"></div>
                Loading...
            </div>

            <div id="cartt-pagination" class="cartt-pagination"></div>
        </main>
    </div>
</div>

<style>
.cartt-search-filter { max-width: 1200px; margin: 0 auto; padding: 20px; }
.cartt-search-bar { position: relative; margin-bottom: 30px; }
.cartt-search-input-wrapper { position: relative; }
.cartt-search-icon { position: absolute; left: 15px; top: 50%; transform: translateY(-50%); color: #999; }
#cartt-search-input { width: 100%; padding: 15px 45px 15px 45px; font-size: 16px; border: 2px solid #eee; border-radius: 8px; transition: border-color 0.2s; }
#cartt-search-input:focus { outline: none; border-color: #1d1d1f; }
#cartt-search-clear { position: absolute; right: 15px; top: 50%; transform: translateY(-50%); background: none; border: none; font-size: 24px; cursor: pointer; color: #999; }
.cartt-autocomplete { position: absolute; top: 100%; left: 0; right: 0; background: #fff; border: 1px solid #eee; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.1); z-index: 100; max-height: 400px; overflow-y: auto; }
.cartt-autocomplete-item { display: flex; gap: 15px; padding: 12px 15px; cursor: pointer; border-bottom: 1px solid #f5f5f5; }
.cartt-autocomplete-item:hover { background: #f9f9f9; }
.cartt-autocomplete-item img { width: 50px; height: 50px; object-fit: cover; border-radius: 4px; }
.cartt-autocomplete-item-info { flex: 1; }
.cartt-autocomplete-item-name { font-weight: 500; }
.cartt-autocomplete-item-price { color: #666; font-size: 14px; }

.cartt-filter-layout { display: grid; grid-template-columns: 250px 1fr; gap: 30px; }
.cartt-filters-sidebar { background: #fff; padding: 20px; border-radius: 8px; border: 1px solid #eee; height: fit-content; position: sticky; top: 20px; }
.cartt-filter-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
.cartt-filter-header h3 { margin: 0; }
#cartt-clear-filters { background: none; border: none; color: #666; cursor: pointer; font-size: 13px; }
#cartt-clear-filters:hover { color: #e53935; }
.cartt-filter-group { margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #eee; }
.cartt-filter-group:last-child { border-bottom: none; margin-bottom: 0; padding-bottom: 0; }
.cartt-filter-group h4 { margin: 0 0 12px; font-size: 14px; font-weight: 600; }
.cartt-filter-list { list-style: none; margin: 0; padding: 0; }
.cartt-filter-list li { margin-bottom: 8px; }
.cartt-filter-list label { display: flex; align-items: center; gap: 8px; cursor: pointer; font-size: 14px; }
.cartt-filter-list .count { color: #999; font-size: 12px; }
.cartt-price-range { display: flex; gap: 10px; align-items: center; }
.cartt-price-range input { width: 80px; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
.cartt-price-slider { margin-top: 15px; }
.cartt-price-slider input { width: 100%; }
.cartt-rating-filter label { color: #f5a623; }

.cartt-products-main { }
.cartt-products-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; }
.cartt-sort-wrapper { display: flex; align-items: center; gap: 10px; }
#cartt-sort { padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; }

.cartt-products-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; }
.cartt-product-card { background: #fff; border: 1px solid #eee; border-radius: 8px; overflow: hidden; transition: box-shadow 0.2s; }
.cartt-product-card:hover { box-shadow: 0 4px 20px rgba(0,0,0,0.1); }
.cartt-product-card-image { aspect-ratio: 1; background: #f9f9f9; position: relative; }
.cartt-product-card-image img { width: 100%; height: 100%; object-fit: cover; }
.cartt-product-card-badge { position: absolute; top: 10px; left: 10px; background: #e53935; color: #fff; padding: 4px 8px; border-radius: 4px; font-size: 12px; }
.cartt-product-card-wishlist { position: absolute; top: 10px; right: 10px; background: #fff; border: none; width: 36px; height: 36px; border-radius: 50%; cursor: pointer; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
.cartt-product-card-content { padding: 15px; }
.cartt-product-card-title { margin: 0 0 8px; font-size: 15px; font-weight: 500; }
.cartt-product-card-title a { color: inherit; text-decoration: none; }
.cartt-product-card-price { font-size: 16px; font-weight: 600; }
.cartt-product-card-price .sale { color: #e53935; }
.cartt-product-card-price .original { text-decoration: line-through; color: #999; margin-left: 8px; font-weight: normal; }
.cartt-product-card-rating { margin-top: 8px; color: #f5a623; font-size: 13px; }

.cartt-loading { text-align: center; padding: 40px; color: #666; }
.cartt-spinner { width: 30px; height: 30px; border: 3px solid #eee; border-top-color: #1d1d1f; border-radius: 50%; animation: spin 0.8s linear infinite; margin: 0 auto 10px; }
@keyframes spin { to { transform: rotate(360deg); } }

.cartt-pagination { display: flex; justify-content: center; gap: 5px; margin-top: 30px; }
.cartt-pagination button { padding: 10px 15px; border: 1px solid #ddd; background: #fff; cursor: pointer; border-radius: 4px; }
.cartt-pagination button.active { background: #1d1d1f; color: #fff; border-color: #1d1d1f; }
.cartt-pagination button:disabled { opacity: 0.5; cursor: not-allowed; }

.cartt-no-results { text-align: center; padding: 60px 20px; color: #666; }

@media (max-width: 900px) {
    .cartt-filter-layout { grid-template-columns: 1fr; }
    .cartt-filters-sidebar { position: static; }
    .cartt-products-grid { grid-template-columns: repeat(2, 1fr); }
}
@media (max-width: 500px) {
    .cartt-products-grid { grid-template-columns: 1fr; }
}
</style>

<script>
(function() {
    const nonce = '<?php echo wp_create_nonce('cartt_search'); ?>';
    let currentPage = 1;
    let searchTimeout;
    let currentFilters = {};

    // Initialize
    loadProducts();

    // Search input
    const searchInput = document.getElementById('cartt-search-input');
    const autocomplete = document.getElementById('cartt-autocomplete');
    const clearBtn = document.getElementById('cartt-search-clear');

    searchInput.addEventListener('input', function() {
        clearBtn.style.display = this.value ? 'block' : 'none';
        
        clearTimeout(searchTimeout);
        
        if (this.value.length >= 2) {
            searchTimeout = setTimeout(() => showAutocomplete(this.value), 200);
        } else {
            autocomplete.style.display = 'none';
        }
    });

    searchInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter') {
            autocomplete.style.display = 'none';
            currentPage = 1;
            loadProducts();
        }
    });

    clearBtn.addEventListener('click', function() {
        searchInput.value = '';
        clearBtn.style.display = 'none';
        autocomplete.style.display = 'none';
        currentPage = 1;
        loadProducts();
    });

    // Autocomplete
    function showAutocomplete(query) {
        fetch(carttFrontend.ajaxUrl, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: `action=cartt_autocomplete&nonce=${nonce}&query=${encodeURIComponent(query)}`
        })
        .then(r => r.json())
        .then(data => {
            if (data.success && data.data.length) {
                autocomplete.innerHTML = data.data.map(p => `
                    <a href="${p.url}" class="cartt-autocomplete-item">
                        <img src="${p.image_url || ''}" alt="">
                        <div class="cartt-autocomplete-item-info">
                            <div class="cartt-autocomplete-item-name">${p.name}</div>
                            <div class="cartt-autocomplete-item-price">${p.formatted_price}</div>
                        </div>
                    </a>
                `).join('');
                autocomplete.style.display = 'block';
            } else {
                autocomplete.style.display = 'none';
            }
        });
    }

    // Close autocomplete on click outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.cartt-search-bar')) {
            autocomplete.style.display = 'none';
        }
    });

    // Filter changes
    document.querySelectorAll('.cartt-filters-sidebar input, .cartt-filters-sidebar select').forEach(el => {
        el.addEventListener('change', () => {
            currentPage = 1;
            loadProducts();
        });
    });

    // Price inputs
    ['cartt-min-price', 'cartt-max-price'].forEach(id => {
        document.getElementById(id)?.addEventListener('change', () => {
            currentPage = 1;
            loadProducts();
        });
    });

    // Sort
    document.getElementById('cartt-sort').addEventListener('change', () => {
        currentPage = 1;
        loadProducts();
    });

    // Clear filters
    document.getElementById('cartt-clear-filters').addEventListener('click', () => {
        document.querySelectorAll('.cartt-filters-sidebar input[type="checkbox"]').forEach(cb => cb.checked = false);
        document.querySelectorAll('.cartt-filters-sidebar input[type="radio"]').forEach(r => r.checked = false);
        document.getElementById('cartt-min-price').value = '';
        document.getElementById('cartt-max-price').value = '';
        document.getElementById('cartt-sort').value = 'newest';
        searchInput.value = '';
        currentPage = 1;
        loadProducts();
    });

    // Load products
    function loadProducts() {
        const filters = {
            query: searchInput.value,
            category: [...document.querySelectorAll('input[name="category"]:checked')].map(cb => cb.value).join(','),
            min_price: document.getElementById('cartt-min-price').value,
            max_price: document.getElementById('cartt-max-price').value,
            min_rating: document.querySelector('input[name="min_rating"]:checked')?.value || '',
            in_stock: document.getElementById('cartt-in-stock').checked ? 1 : 0,
            on_sale: document.getElementById('cartt-on-sale').checked ? 1 : 0,
            sort: document.getElementById('cartt-sort').value,
            page: currentPage
        };

        document.getElementById('cartt-loading').style.display = 'block';
        document.getElementById('cartt-products-grid').style.opacity = '0.5';

        const params = new URLSearchParams({ action: 'cartt_search_products', nonce, ...filters });

        fetch(carttFrontend.ajaxUrl, {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: params.toString()
        })
        .then(r => r.json())
        .then(data => {
            document.getElementById('cartt-loading').style.display = 'none';
            document.getElementById('cartt-products-grid').style.opacity = '1';

            if (data.success) {
                renderProducts(data.data.products);
                renderPagination(data.data.pages, data.data.total);
                document.getElementById('cartt-results-count').textContent = data.data.total;
            }
        });
    }

    // Render products
    function renderProducts(products) {
        const grid = document.getElementById('cartt-products-grid');
        
        if (!products.length) {
            grid.innerHTML = '<div class="cartt-no-results">No products found matching your criteria.</div>';
            return;
        }

        grid.innerHTML = products.map(p => `
            <div class="cartt-product-card">
                <div class="cartt-product-card-image">
                    ${p.image_url ? `<img src="${p.image_url}" alt="${p.name}">` : ''}
                    ${p.sale_price ? '<span class="cartt-product-card-badge">Sale</span>' : ''}
                    <button class="cartt-product-card-wishlist" data-product-id="${p.id}">♡</button>
                </div>
                <div class="cartt-product-card-content">
                    <h3 class="cartt-product-card-title"><a href="/product/${p.slug}">${p.name}</a></h3>
                    <div class="cartt-product-card-price">
                        ${p.sale_price ? `<span class="sale">${formatPrice(p.sale_price)}</span><span class="original">${formatPrice(p.price)}</span>` : formatPrice(p.price)}
                    </div>
                    ${p.avg_rating > 0 ? `<div class="cartt-product-card-rating">${'★'.repeat(Math.round(p.avg_rating))}${'☆'.repeat(5-Math.round(p.avg_rating))} (${p.avg_rating.toFixed(1)})</div>` : ''}
                </div>
            </div>
        `).join('');

        // Wishlist buttons
        grid.querySelectorAll('.cartt-product-card-wishlist').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                const productId = this.dataset.productId;
                // Add to wishlist AJAX call
                this.textContent = '♥';
            });
        });
    }

    // Render pagination
    function renderPagination(totalPages, total) {
        const pagination = document.getElementById('cartt-pagination');
        
        if (totalPages <= 1) {
            pagination.innerHTML = '';
            return;
        }

        let html = '';
        
        html += `<button ${currentPage === 1 ? 'disabled' : ''} data-page="${currentPage - 1}">←</button>`;
        
        for (let i = 1; i <= Math.min(totalPages, 5); i++) {
            html += `<button class="${i === currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`;
        }
        
        if (totalPages > 5) {
            html += '<span>...</span>';
            html += `<button data-page="${totalPages}">${totalPages}</button>`;
        }
        
        html += `<button ${currentPage === totalPages ? 'disabled' : ''} data-page="${currentPage + 1}">→</button>`;

        pagination.innerHTML = html;

        pagination.querySelectorAll('button').forEach(btn => {
            btn.addEventListener('click', function() {
                currentPage = parseInt(this.dataset.page);
                loadProducts();
                window.scrollTo({ top: 0, behavior: 'smooth' });
            });
        });
    }

    // Format price helper
    function formatPrice(price) {
        return '$' + parseFloat(price).toFixed(2);
    }
})();
</script>
