<?php
/**
 * Recommendations Template
 * 
 * @package Cartt
 * @since 1.4.0
 * 
 * Variables available:
 * @var array $products - Recommended products
 * @var string $title - Section title
 * @var array $atts - Shortcode attributes
 */

defined('ABSPATH') || exit;

$columns = isset($atts['columns']) ? (int)$atts['columns'] : 4;
?>

<div class="cartt-recommendations">
    <?php if (!empty($title)): ?>
    <h3 class="cartt-recommendations-title"><?php echo esc_html($title); ?></h3>
    <?php endif; ?>
    
    <div class="cartt-recommendations-grid" style="grid-template-columns: repeat(<?php echo $columns; ?>, 1fr);">
        <?php foreach ($products as $product): ?>
        <div class="cartt-recommendation-card">
            <a href="<?php echo esc_url(get_permalink($product['id'])); ?>" class="card-link">
                <?php if (!empty($product['image_url'])): ?>
                <div class="card-image">
                    <img src="<?php echo esc_url($product['image_url']); ?>" alt="<?php echo esc_attr($product['name']); ?>">
                    <?php if (!empty($product['sale_price']) && $product['sale_price'] < $product['price']): ?>
                    <span class="sale-badge">Sale</span>
                    <?php endif; ?>
                </div>
                <?php else: ?>
                <div class="card-image card-image-placeholder"></div>
                <?php endif; ?>
                
                <div class="card-content">
                    <h4 class="card-title"><?php echo esc_html($product['name']); ?></h4>
                    
                    <div class="card-price">
                        <?php if (!empty($product['sale_price']) && $product['sale_price'] < $product['price']): ?>
                        <span class="original-price"><?php echo \Cartt\Services\CurrencyService::formatPrice($product['price']); ?></span>
                        <span class="sale-price"><?php echo \Cartt\Services\CurrencyService::formatPrice($product['sale_price']); ?></span>
                        <?php else: ?>
                        <?php echo \Cartt\Services\CurrencyService::formatPrice($product['price']); ?>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (!empty($product['rating'])): ?>
                    <div class="card-rating">
                        <?php 
                        $rating = round($product['rating']);
                        echo str_repeat('&#9733;', $rating);
                        echo str_repeat('&#9734;', 5 - $rating);
                        if (!empty($product['review_count'])): ?>
                        <span class="rating-count">(<?php echo esc_html($product['review_count']); ?>)</span>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </a>
            
            <button type="button" class="cartt-add-to-cart-btn" data-product-id="<?php echo esc_attr($product['id']); ?>">
                Add to Cart
            </button>
        </div>
        <?php endforeach; ?>
    </div>
</div>

<style>
.cartt-recommendations { margin: 30px 0; }
.cartt-recommendations-title { font-size: 20px; margin-bottom: 20px; }
.cartt-recommendations-grid { display: grid; gap: 20px; }
.cartt-recommendation-card { background: #fff; border: 1px solid #eee; border-radius: 8px; overflow: hidden; transition: box-shadow 0.2s; }
.cartt-recommendation-card:hover { box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
.card-link { text-decoration: none; color: inherit; display: block; }
.card-image { position: relative; aspect-ratio: 1; overflow: hidden; }
.card-image img { width: 100%; height: 100%; object-fit: cover; transition: transform 0.3s; }
.cartt-recommendation-card:hover .card-image img { transform: scale(1.05); }
.card-image-placeholder { background: linear-gradient(135deg, #f0f0f0 0%, #e0e0e0 100%); }
.sale-badge { position: absolute; top: 10px; left: 10px; background: #d63638; color: #fff; padding: 4px 10px; border-radius: 4px; font-size: 12px; font-weight: 600; }
.card-content { padding: 15px; }
.card-title { font-size: 14px; margin: 0 0 8px; line-height: 1.4; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden; }
.card-price { font-weight: 600; color: #1d2327; }
.original-price { text-decoration: line-through; color: #999; font-weight: 400; margin-right: 8px; }
.sale-price { color: #d63638; }
.card-rating { color: #f5a623; font-size: 12px; margin-top: 5px; }
.rating-count { color: #999; }
.cartt-add-to-cart-btn { width: calc(100% - 30px); margin: 0 15px 15px; padding: 10px; background: #2271b1; color: #fff; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; transition: background 0.2s; }
.cartt-add-to-cart-btn:hover { background: #135e96; }
@media (max-width: 768px) {
    .cartt-recommendations-grid { grid-template-columns: repeat(2, 1fr) !important; }
}
@media (max-width: 480px) {
    .cartt-recommendations-grid { grid-template-columns: 1fr !important; }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('.cartt-add-to-cart-btn').forEach(function(btn) {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            const productId = this.dataset.productId;
            const originalText = this.textContent;
            
            this.disabled = true;
            this.textContent = 'Adding...';
            
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=cartt_add_to_cart&product_id=' + productId + '&quantity=1'
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    this.textContent = 'Added!';
                    setTimeout(() => {
                        this.textContent = originalText;
                        this.disabled = false;
                    }, 2000);
                } else {
                    this.textContent = 'Error';
                    this.disabled = false;
                }
            })
            .catch(() => {
                this.textContent = originalText;
                this.disabled = false;
            });
        });
    });
});
</script>
