<?php
/**
 * Quote Request Template
 * 
 * @package Cartt
 * @since 1.4.0
 */

defined('ABSPATH') || exit;
?>

<div class="cartt-quote-request">
    <?php if ($submitted): ?>
    <div class="cartt-notice cartt-notice-success">
        <h3>Quote Request Submitted!</h3>
        <p>We've received your quote request. Our team will prepare a custom quote and send it to your email within 1-2 business days.</p>
    </div>
    <?php else: ?>
    
    <h2>Request a Quote</h2>
    <p>Need a custom quote for bulk orders or special requirements? Fill out the form below and we'll get back to you with pricing.</p>

    <?php if ($error): ?>
    <div class="cartt-notice cartt-notice-error"><?php echo esc_html($error); ?></div>
    <?php endif; ?>

    <form method="post" class="cartt-quote-form" id="quote-request-form">
        <?php wp_nonce_field('cartt_quote_request'); ?>
        
        <div class="form-section">
            <h3>Contact Information</h3>
            
            <div class="form-row-group">
                <div class="form-row">
                    <label for="contact_name">Your Name <span class="required">*</span></label>
                    <input type="text" id="contact_name" name="contact_name" required 
                           value="<?php echo esc_attr($_POST['contact_name'] ?? ''); ?>">
                </div>
                <div class="form-row">
                    <label for="company">Company</label>
                    <input type="text" id="company" name="company" 
                           value="<?php echo esc_attr($_POST['company'] ?? ''); ?>">
                </div>
            </div>
            
            <div class="form-row-group">
                <div class="form-row">
                    <label for="email">Email <span class="required">*</span></label>
                    <input type="email" id="email" name="email" required 
                           value="<?php echo esc_attr($_POST['email'] ?? ''); ?>">
                </div>
                <div class="form-row">
                    <label for="phone">Phone</label>
                    <input type="tel" id="phone" name="phone" 
                           value="<?php echo esc_attr($_POST['phone'] ?? ''); ?>">
                </div>
            </div>
        </div>

        <div class="form-section">
            <h3>Products</h3>
            <p class="form-note">Add the products you'd like to include in your quote request.</p>
            
            <div id="quote-items">
                <div class="quote-item" data-index="0">
                    <div class="item-row">
                        <div class="item-field product-field">
                            <label>Product <span class="required">*</span></label>
                            <select name="items[0][product_id]" class="product-select" required>
                                <option value="">Select a product...</option>
                                <?php foreach ($products as $product): ?>
                                <option value="<?php echo esc_attr($product['id']); ?>" data-price="<?php echo esc_attr($product['price']); ?>">
                                    <?php echo esc_html($product['name']); ?> 
                                    (<?php echo esc_html($product['sku'] ?? 'No SKU'); ?>) - 
                                    <?php echo \Cartt\Services\CurrencyService::formatPrice($product['price']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="item-field qty-field">
                            <label>Quantity <span class="required">*</span></label>
                            <input type="number" name="items[0][quantity]" min="1" value="1" required>
                        </div>
                        <div class="item-field notes-field">
                            <label>Notes</label>
                            <input type="text" name="items[0][notes]" placeholder="Special requirements...">
                        </div>
                        <button type="button" class="remove-item-btn" style="display: none;">&times;</button>
                    </div>
                </div>
            </div>
            
            <button type="button" class="cartt-button cartt-button-secondary" id="add-quote-item">
                + Add Another Product
            </button>
        </div>

        <div class="form-section">
            <h3>Additional Notes</h3>
            <div class="form-row">
                <textarea id="notes" name="notes" rows="4" 
                          placeholder="Any special requirements, delivery preferences, or questions?"><?php echo esc_textarea($_POST['notes'] ?? ''); ?></textarea>
            </div>
        </div>

        <button type="submit" name="cartt_request_quote" class="cartt-button cartt-button-primary">
            Submit Quote Request
        </button>
    </form>
    <?php endif; ?>
</div>

<style>
.cartt-quote-request { max-width: 800px; margin: 0 auto; }
.cartt-quote-request h2 { margin-bottom: 10px; }
.cartt-quote-request > p { color: #666; margin-bottom: 30px; }
.form-section { margin-bottom: 30px; padding-bottom: 20px; border-bottom: 1px solid #eee; }
.form-section h3 { margin-bottom: 15px; font-size: 18px; }
.form-row { margin-bottom: 20px; }
.form-row-group { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px; }
.form-row label { display: block; margin-bottom: 5px; font-weight: 500; }
.form-row input, .form-row select, .form-row textarea { width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px; }
.required { color: #d63638; }
.form-note { color: #666; font-size: 13px; margin-bottom: 15px; }
.quote-item { background: #f9f9f9; padding: 15px; border-radius: 6px; margin-bottom: 15px; }
.item-row { display: grid; grid-template-columns: 2fr 1fr 1.5fr auto; gap: 15px; align-items: end; }
.item-field label { display: block; margin-bottom: 5px; font-size: 13px; font-weight: 500; }
.item-field input, .item-field select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; }
.remove-item-btn { background: #d63638; color: #fff; border: none; width: 32px; height: 38px; border-radius: 4px; cursor: pointer; font-size: 18px; }
.cartt-button { padding: 12px 24px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; }
.cartt-button-primary { background: #2271b1; color: #fff; }
.cartt-button-secondary { background: #f0f0f0; color: #333; }
.cartt-notice { padding: 20px; border-radius: 8px; margin-bottom: 20px; }
.cartt-notice-success { background: #d4edda; color: #155724; }
.cartt-notice-error { background: #f8d7da; color: #721c24; }
@media (max-width: 600px) {
    .form-row-group { grid-template-columns: 1fr; }
    .item-row { grid-template-columns: 1fr; }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    let itemIndex = 1;
    const itemsContainer = document.getElementById('quote-items');
    const addBtn = document.getElementById('add-quote-item');
    
    addBtn.addEventListener('click', function() {
        const template = itemsContainer.querySelector('.quote-item').cloneNode(true);
        template.dataset.index = itemIndex;
        
        // Update field names
        template.querySelectorAll('[name]').forEach(field => {
            field.name = field.name.replace(/\[\d+\]/, '[' + itemIndex + ']');
            if (field.tagName === 'SELECT') field.selectedIndex = 0;
            else field.value = field.type === 'number' ? 1 : '';
        });
        
        // Show remove button
        template.querySelector('.remove-item-btn').style.display = 'block';
        
        itemsContainer.appendChild(template);
        itemIndex++;
        
        updateRemoveButtons();
    });
    
    itemsContainer.addEventListener('click', function(e) {
        if (e.target.classList.contains('remove-item-btn')) {
            e.target.closest('.quote-item').remove();
            updateRemoveButtons();
        }
    });
    
    function updateRemoveButtons() {
        const items = itemsContainer.querySelectorAll('.quote-item');
        items.forEach((item, index) => {
            item.querySelector('.remove-item-btn').style.display = items.length > 1 ? 'block' : 'none';
        });
    }
});
</script>
