<?php
/**
 * Products Grid Template
 * 
 * Available variables: $atts (shortcode attributes)
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

// Load card design settings
$design = get_option('cartt_card_design', []);
$defaults = [
    'preset' => 'modern',
    'elements' => ['image', 'title', 'price', 'rating', 'add_to_cart'],
    'card' => ['background' => '#ffffff', 'border_color' => '#e0e0e0', 'border_width' => 1, 'border_radius' => 8, 'shadow' => 'small', 'hover_effect' => 'lift', 'padding' => 0, 'hover_shadow_size' => 15],
    'image' => ['aspect_ratio' => 'square', 'border_radius' => 8, 'hover_effect' => 'zoom'],
    'title' => ['font_size' => 16, 'font_weight' => '600', 'color' => '#1d2327', 'margin_top' => 12, 'lines' => 2],
    'price' => ['font_size' => 18, 'font_weight' => '700', 'color' => '#1d2327', 'sale_color' => '#dc2626', 'margin_top' => 8],
    'rating' => ['star_color' => '#fbbf24', 'show_count' => true, 'margin_top' => 8],
    'add_to_cart' => ['style' => 'button', 'bg_color' => '#1d2327', 'text_color' => '#ffffff', 'border_radius' => 6, 'full_width' => true, 'margin_top' => 12, 'button_text' => 'Add to Cart'],
    'sale_badge' => ['enabled' => true, 'bg_color' => '#dc2626', 'text_color' => '#ffffff', 'text' => 'Sale'],
    'wishlist' => ['enabled' => false, 'position' => 'top-right'],
    'grid' => ['columns' => 4, 'gap' => 24, 'columns_tablet' => 3, 'columns_mobile' => 2]
];
$design = wp_parse_args($design, $defaults);
foreach ($defaults as $key => $val) {
    if (is_array($val) && isset($design[$key])) {
        $design[$key] = wp_parse_args($design[$key], $val);
    }
}

$table = $wpdb->prefix . 'cartt_products';
$limit = intval($atts['limit']);
$columns = !empty($atts['columns']) ? intval($atts['columns']) : $design['grid']['columns'];

// Get current category filter
$currentCategory = isset($_GET['category']) ? sanitize_text_field($_GET['category']) : '';

// Get all categories with product counts
$categories = $wpdb->get_results(
    "SELECT c.*, COUNT(pc.product_id) as product_count 
     FROM {$wpdb->prefix}cartt_categories c
     LEFT JOIN {$wpdb->prefix}cartt_product_categories pc ON c.id = pc.category_id
     LEFT JOIN {$wpdb->prefix}cartt_products p ON pc.product_id = p.id AND p.status = 'publish'
     GROUP BY c.id
     HAVING product_count > 0
     ORDER BY c.name ASC"
);

// Build query based on category filter
if ($currentCategory) {
    $category = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM {$wpdb->prefix}cartt_categories WHERE slug = %s",
        $currentCategory
    ));
    
    if ($category) {
        $products = $wpdb->get_results($wpdb->prepare(
            "SELECT p.* FROM $table p
             JOIN {$wpdb->prefix}cartt_product_categories pc ON p.id = pc.product_id
             WHERE p.status = 'publish' AND pc.category_id = %d
             ORDER BY p.created_at DESC LIMIT %d",
            $category->id,
            $limit
        ));
    } else {
        $products = [];
    }
} else {
    $products = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $table WHERE status = 'publish' ORDER BY created_at DESC LIMIT %d",
        $limit
    ));
}

$currency_symbol = '$';
$api_url = rest_url('cartt/v1/');
$nonce = wp_create_nonce('wp_rest');

// Get current currency for price formatting
use Cartt\Services\CurrencyService;
$currentCurrency = CurrencyService::getCurrentCurrency();
$shopUrl = get_permalink();

// Generate shadow CSS
$shadowMap = [
    'none' => 'none',
    'small' => '0 1px 3px rgba(0,0,0,0.1)',
    'medium' => '0 4px 6px rgba(0,0,0,0.1)',
    'large' => '0 10px 25px rgba(0,0,0,0.15)'
];
$cardShadow = $shadowMap[$design['card']['shadow']] ?? 'none';

// Image aspect ratio
$aspectMap = ['square' => '100%', 'portrait' => '133%', 'landscape' => '75%'];
$aspectRatio = $aspectMap[$design['image']['aspect_ratio']] ?? '100%';
?>

<?php if (!empty($categories)): ?>
<div class="cartt-category-filter">
    <a href="<?php echo esc_url($shopUrl); ?>" class="cartt-category-link <?php echo !$currentCategory ? 'active' : ''; ?>">
        <?php esc_html_e('All', 'cartt'); ?>
    </a>
    <?php foreach ($categories as $cat): ?>
    <a href="<?php echo esc_url(add_query_arg('category', $cat->slug, $shopUrl)); ?>" 
       class="cartt-category-link <?php echo $currentCategory === $cat->slug ? 'active' : ''; ?>">
        <?php echo esc_html($cat->name); ?>
        <span class="cartt-category-count"><?php echo intval($cat->product_count); ?></span>
    </a>
    <?php endforeach; ?>
</div>
<?php endif; ?>

<?php if (empty($products)): ?>
    <p class="cartt-no-products"><?php esc_html_e('No products found.', 'cartt'); ?></p>
<?php else: ?>
<div class="cartt-products cartt-products-designed" style="--cartt-columns: <?php echo esc_attr($columns); ?>; --cartt-gap: <?php echo esc_attr($design['grid']['gap']); ?>px; --cartt-columns-tablet: <?php echo esc_attr($design['grid']['columns_tablet']); ?>; --cartt-columns-mobile: <?php echo esc_attr($design['grid']['columns_mobile']); ?>;">
    <?php foreach ($products as $product): 
        $image_url = $product->featured_image ? wp_get_attachment_image_url($product->featured_image, 'medium') : '';
        $price = floatval($product->price);
        $sale_price = $product->sale_price ? floatval($product->sale_price) : null;
        $has_sale = $sale_price && $sale_price < $price;
        
        // Get rating
        $avgRating = \Cartt\Services\ReviewService::getProductAverageRating($product->id);
        $reviewCount = \Cartt\Services\ReviewService::getProductReviewCount($product->id);
    ?>
        <div class="cartt-product-card cartt-card-designed">
            <?php foreach ($design['elements'] as $element): ?>
                <?php if ($element === 'image'): ?>
                <div class="cartt-product-image">
                    <?php if ($has_sale && in_array('sale_badge', $design['elements']) && $design['sale_badge']['enabled']): ?>
                    <span class="cartt-sale-badge"><?php echo esc_html($design['sale_badge']['text']); ?></span>
                    <?php endif; ?>
                    
                    <?php if (in_array('wishlist', $design['elements']) && $design['wishlist']['enabled']): ?>
                    <button type="button" class="cartt-wishlist-btn" data-product-id="<?php echo esc_attr($product->id); ?>">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M20.84 4.61a5.5 5.5 0 0 0-7.78 0L12 5.67l-1.06-1.06a5.5 5.5 0 0 0-7.78 7.78l1.06 1.06L12 21.23l7.78-7.78 1.06-1.06a5.5 5.5 0 0 0 0-7.78z"></path></svg>
                    </button>
                    <?php endif; ?>
                    
                    <?php if ($image_url): ?>
                        <a href="<?php echo esc_url(home_url('/product/' . $product->slug)); ?>">
                            <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($product->name); ?>">
                        </a>
                    <?php else: ?>
                        <a href="<?php echo esc_url(home_url('/product/' . $product->slug)); ?>" class="cartt-no-image">
                            <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#ccc" stroke-width="1"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect><circle cx="8.5" cy="8.5" r="1.5"></circle><polyline points="21 15 16 10 5 21"></polyline></svg>
                        </a>
                    <?php endif; ?>
                </div>
                
                <?php elseif ($element === 'title'): ?>
                <h3 class="cartt-product-title">
                    <a href="<?php echo esc_url(home_url('/product/' . $product->slug)); ?>">
                        <?php echo esc_html($product->name); ?>
                    </a>
                </h3>
                
                <?php elseif ($element === 'price'): ?>
                <div class="cartt-product-price">
                    <?php 
                    $displayPrice = CurrencyService::convertPrice($price);
                    $displaySalePrice = $sale_price ? CurrencyService::convertPrice($sale_price) : 0;
                    
                    if ($displaySalePrice && $displaySalePrice < $displayPrice): ?>
                        <span class="cartt-price-sale"><?php echo esc_html(CurrencyService::formatPrice($displaySalePrice)); ?></span>
                        <span class="cartt-price-original"><?php echo esc_html(CurrencyService::formatPrice($displayPrice)); ?></span>
                    <?php else: ?>
                        <span class="cartt-price-regular"><?php echo esc_html(CurrencyService::formatPrice($displayPrice)); ?></span>
                    <?php endif; ?>
                </div>
                
                <?php elseif ($element === 'rating' && $reviewCount > 0): ?>
                <div class="cartt-product-rating">
                    <span class="cartt-stars"><?php echo str_repeat('★', round($avgRating)) . str_repeat('☆', 5 - round($avgRating)); ?></span>
                    <?php if ($design['rating']['show_count']): ?>
                    <span class="cartt-review-count">(<?php echo $reviewCount; ?>)</span>
                    <?php endif; ?>
                </div>
                
                <?php elseif ($element === 'add_to_cart'): ?>
                <div class="cartt-product-actions">
                    <button type="button" 
                            class="cartt-add-to-cart-btn"
                            data-product-id="<?php echo esc_attr($product->id); ?>">
                        <?php echo esc_html($design['add_to_cart']['button_text'] ?? __('Add to Cart', 'cartt')); ?>
                    </button>
                </div>
                
                <?php elseif ($element === 'description' && !empty($product->short_description)): ?>
                <p class="cartt-product-desc"><?php echo esc_html(wp_trim_words($product->short_description, 15)); ?></p>
                
                <?php elseif ($element === 'sku' && !empty($product->sku)): ?>
                <p class="cartt-product-sku">SKU: <?php echo esc_html($product->sku); ?></p>
                
                <?php elseif ($element === 'stock'): ?>
                <p class="cartt-product-stock <?php echo $product->stock_quantity > 0 ? 'in-stock' : 'out-of-stock'; ?>">
                    <?php echo $product->stock_quantity > 0 ? esc_html__('In Stock', 'cartt') : esc_html__('Out of Stock', 'cartt'); ?>
                </p>
                <?php endif; ?>
            <?php endforeach; ?>
        </div>
    <?php endforeach; ?>
</div>
<?php endif; ?>

<style>
/* Category Filter */
.cartt-category-filter { display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 24px; }
.cartt-category-link { display: inline-flex; align-items: center; gap: 6px; padding: 8px 16px; background: #f5f5f7; border-radius: 20px; text-decoration: none; color: #1d1d1f; font-size: 14px; transition: all 0.2s; }
.cartt-category-link:hover { background: #e8e8ed; }
.cartt-category-link.active { background: #1d1d1f; color: #fff; }
.cartt-category-count { font-size: 12px; opacity: 0.7; }

/* Products Grid - Card Designer Styles */
.cartt-products-designed {
    display: grid;
    grid-template-columns: repeat(var(--cartt-columns, 4), 1fr);
    gap: var(--cartt-gap, 24px);
    grid-auto-rows: 1fr;
}

.cartt-card-designed {
    background: <?php echo esc_attr($design['card']['background']); ?>;
    border: <?php echo esc_attr($design['card']['border_width']); ?>px solid <?php echo esc_attr($design['card']['border_color']); ?>;
    border-radius: <?php echo esc_attr($design['card']['border_radius']); ?>px;
    box-shadow: <?php echo esc_attr($cardShadow); ?>;
    overflow: hidden;
    transition: all 0.3s ease;
    display: flex;
    flex-direction: column;
    height: 100%;
}

<?php 
$hoverShadow = $design['card']['hover_shadow_size'] ?? 15;
$hoverShadowY = round($hoverShadow * 0.5);
?>
<?php if ($design['card']['hover_effect'] === 'lift'): ?>
.cartt-card-designed:hover { transform: translateY(-4px); box-shadow: 0 <?php echo esc_attr($hoverShadowY); ?>px <?php echo esc_attr($hoverShadow); ?>px rgba(0,0,0,0.15); }
<?php elseif ($design['card']['hover_effect'] === 'shadow'): ?>
.cartt-card-designed:hover { box-shadow: 0 <?php echo esc_attr($hoverShadowY); ?>px <?php echo esc_attr($hoverShadow); ?>px rgba(0,0,0,0.15); }
<?php elseif ($design['card']['hover_effect'] === 'border'): ?>
.cartt-card-designed:hover { border-color: <?php echo esc_attr($design['add_to_cart']['bg_color']); ?>; }
<?php endif; ?>

.cartt-card-designed .cartt-product-image {
    position: relative;
    padding-bottom: <?php echo esc_attr($aspectRatio); ?>;
    overflow: hidden;
    border-radius: <?php echo esc_attr($design['image']['border_radius']); ?>px <?php echo esc_attr($design['image']['border_radius']); ?>px 0 0;
    flex-shrink: 0;
}

.cartt-card-designed .cartt-product-image img {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.3s ease;
}

<?php if ($design['image']['hover_effect'] === 'zoom'): ?>
.cartt-card-designed:hover .cartt-product-image img { transform: scale(1.05); }
<?php endif; ?>

.cartt-card-designed .cartt-no-image {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #f5f5f5;
}

.cartt-card-designed .cartt-sale-badge {
    position: absolute;
    top: 12px;
    left: 12px;
    background: <?php echo esc_attr($design['sale_badge']['bg_color']); ?>;
    color: <?php echo esc_attr($design['sale_badge']['text_color']); ?>;
    font-size: 11px;
    font-weight: 600;
    padding: 4px 10px;
    border-radius: 4px;
    z-index: 2;
}

.cartt-card-designed .cartt-wishlist-btn {
    position: absolute;
    top: 12px;
    <?php echo $design['wishlist']['position'] === 'top-left' ? 'left' : 'right'; ?>: 12px;
    width: 36px;
    height: 36px;
    background: #fff;
    border: none;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    z-index: 2;
    transition: all 0.2s;
}

.cartt-card-designed .cartt-wishlist-btn:hover { background: #f5f5f5; }
.cartt-card-designed .cartt-wishlist-btn svg { color: #666; }

.cartt-card-designed .cartt-product-title {
    padding: 0 16px;
    margin: <?php echo esc_attr($design['title']['margin_top']); ?>px 0 0;
    font-size: <?php echo esc_attr($design['title']['font_size']); ?>px;
    font-weight: <?php echo esc_attr($design['title']['font_weight']); ?>;
    line-height: 1.4;
    overflow: hidden;
    display: -webkit-box;
    -webkit-line-clamp: <?php echo esc_attr($design['title']['lines']); ?>;
    -webkit-box-orient: vertical;
}

.cartt-card-designed .cartt-product-title a {
    color: <?php echo esc_attr($design['title']['color']); ?>;
    text-decoration: none;
}

.cartt-card-designed .cartt-product-title a:hover { text-decoration: underline; }

.cartt-card-designed .cartt-product-price {
    padding: 0 16px;
    margin-top: <?php echo esc_attr($design['price']['margin_top']); ?>px;
    font-size: <?php echo esc_attr($design['price']['font_size']); ?>px;
    font-weight: <?php echo esc_attr($design['price']['font_weight']); ?>;
    color: <?php echo esc_attr($design['price']['color']); ?>;
}

.cartt-card-designed .cartt-price-sale {
    color: <?php echo esc_attr($design['price']['sale_color']); ?>;
}

.cartt-card-designed .cartt-price-original {
    text-decoration: line-through;
    color: #999;
    font-size: 0.8em;
    font-weight: 400;
    margin-left: 8px;
}

.cartt-card-designed .cartt-product-rating {
    padding: 0 16px;
    margin-top: <?php echo esc_attr($design['rating']['margin_top']); ?>px;
    display: flex;
    align-items: center;
    gap: 4px;
}

.cartt-card-designed .cartt-stars {
    color: <?php echo esc_attr($design['rating']['star_color']); ?>;
    font-size: 14px;
    letter-spacing: 1px;
}

.cartt-card-designed .cartt-review-count {
    color: #666;
    font-size: 12px;
}

.cartt-card-designed .cartt-product-actions {
    padding: 16px;
    margin-top: auto;
}

.cartt-card-designed .cartt-add-to-cart-btn {
    <?php if ($design['add_to_cart']['full_width']): ?>width: 100%;<?php endif; ?>
    padding: 10px 20px;
    background: <?php echo esc_attr($design['add_to_cart']['bg_color']); ?>;
    color: <?php echo esc_attr($design['add_to_cart']['text_color']); ?>;
    border: none;
    border-radius: <?php echo esc_attr($design['add_to_cart']['border_radius']); ?>px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s;
}

.cartt-card-designed .cartt-add-to-cart-btn:hover {
    opacity: 0.9;
}

.cartt-card-designed .cartt-product-desc,
.cartt-card-designed .cartt-product-sku {
    padding: 0 16px;
    margin: 8px 0 0;
    font-size: 13px;
    color: #666;
}

.cartt-card-designed .cartt-product-stock {
    padding: 0 16px;
    margin: 8px 0 0;
    font-size: 12px;
    font-weight: 500;
}

.cartt-card-designed .cartt-product-stock.in-stock { color: #16a34a; }
.cartt-card-designed .cartt-product-stock.out-of-stock { color: #dc2626; }

/* Responsive */
@media (max-width: 1024px) {
    .cartt-products-designed {
        grid-template-columns: repeat(var(--cartt-columns-tablet, 3), 1fr);
    }
}

@media (max-width: 640px) {
    .cartt-products-designed {
        grid-template-columns: repeat(var(--cartt-columns-mobile, 2), 1fr);
        gap: 16px;
    }
    
    .cartt-card-designed .cartt-product-title,
    .cartt-card-designed .cartt-product-price,
    .cartt-card-designed .cartt-product-rating,
    .cartt-card-designed .cartt-product-actions {
        padding-left: 12px;
        padding-right: 12px;
    }
}
</style>

<script>
(function() {
    const apiUrl = '<?php echo esc_js($api_url); ?>';
    const nonce = '<?php echo esc_js($nonce); ?>';

    document.querySelectorAll('.cartt-add-to-cart-btn').forEach(btn => {
        btn.addEventListener('click', async function() {
            const productId = this.dataset.productId;
            const originalText = this.textContent;
            
            this.disabled = true;
            this.textContent = 'Adding...';

            try {
                const response = await fetch(apiUrl + 'cart/add', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': nonce
                    },
                    body: JSON.stringify({ product_id: parseInt(productId), quantity: 1 })
                });

                const data = await response.json();
                
                if (data.success) {
                    this.textContent = 'Added!';
                    setTimeout(() => {
                        this.textContent = originalText;
                        this.disabled = false;
                    }, 1500);
                } else {
                    this.textContent = data.error || 'Error';
                    setTimeout(() => {
                        this.textContent = originalText;
                        this.disabled = false;
                    }, 2000);
                }
            } catch (error) {
                this.textContent = 'Error';
                setTimeout(() => {
                    this.textContent = originalText;
                    this.disabled = false;
                }, 2000);
            }
        });
    });
})();
</script>
