<?php
/**
 * Single Product Template
 * 
 * Available variables: $atts (shortcode attributes with id or sku)
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

$table = $wpdb->prefix . 'cartt_products';
$product = null;

if (!empty($atts['id'])) {
    $product = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table WHERE id = %d AND status = 'publish'",
        intval($atts['id'])
    ));
} elseif (!empty($atts['sku'])) {
    $product = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table WHERE sku = %s AND status = 'publish'",
        sanitize_text_field($atts['sku'])
    ));
}

if (!$product) {
    echo '<p class="cartt-no-products">' . esc_html__('Product not found.', 'cartt') . '</p>';
    return;
}

$image_url = $product->featured_image ? wp_get_attachment_image_url($product->featured_image, 'large') : '';
$gallery_images = !empty($product->gallery_images) ? json_decode($product->gallery_images, true) : [];
$price = floatval($product->price);
$sale_price = $product->sale_price ? floatval($product->sale_price) : null;
$display_price = ($sale_price && $sale_price < $price) ? $sale_price : $price;
$currency_symbol = '$';
$api_url = rest_url('cartt/v1/');
$nonce = wp_create_nonce('wp_rest');

// Currency service for price formatting
use Cartt\Services\CurrencyService;
$currentCurrency = CurrencyService::getCurrentCurrency();

// Reviews data
$reviews = \Cartt\Services\ReviewService::getProductReviews($product->id);
$avgRating = \Cartt\Services\ReviewService::getProductAverageRating($product->id);
$reviewCount = \Cartt\Services\ReviewService::getProductReviewCount($product->id);
$ratingBreakdown = \Cartt\Services\ReviewService::getProductRatingBreakdown($product->id);

// Handle review submission
$reviewMessage = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_review_nonce']) && wp_verify_nonce($_POST['cartt_review_nonce'], 'cartt_submit_review')) {
    $reviewData = [
        'product_id' => $product->id,
        'customer_name' => sanitize_text_field($_POST['review_name'] ?? ''),
        'customer_email' => sanitize_email($_POST['review_email'] ?? ''),
        'rating' => intval($_POST['review_rating'] ?? 5),
        'title' => sanitize_text_field($_POST['review_title'] ?? ''),
        'content' => sanitize_textarea_field($_POST['review_content'] ?? ''),
    ];
    
    if (is_user_logged_in()) {
        $user = wp_get_current_user();
        $reviewData['customer_id'] = $user->ID;
        $reviewData['customer_name'] = $user->display_name;
        $reviewData['customer_email'] = $user->user_email;
    }
    
    if (!empty($reviewData['customer_name']) && !empty($reviewData['customer_email']) && !empty($reviewData['content'])) {
        \Cartt\Services\ReviewService::createReview($reviewData);
        $reviewMessage = __('Thank you! Your review has been submitted and is pending approval.', 'cartt');
    }
}

// Get categories
$categories = \Cartt\Services\CategoryService::getProductCategories($product->id);

// Get variations if product is variable
$variations = [];
$hasVariations = false;
$priceRange = null;
$attributes = [];

if ($product->product_type === 'variable') {
    $variations = \Cartt\Services\VariationService::getProductVariations($product->id);
    $hasVariations = !empty($variations);
    
    if ($hasVariations) {
        // Get price range
        $priceRange = \Cartt\Services\VariationService::getPriceRange($product->id);
        
        // Get all unique attributes
        foreach ($variations as $var) {
            $varAttrs = json_decode($var->attributes, true) ?: [];
            foreach ($varAttrs as $name => $value) {
                if (!isset($attributes[$name])) {
                    $attributes[$name] = [];
                }
                if (!in_array($value, $attributes[$name])) {
                    $attributes[$name][] = $value;
                }
            }
        }
    }
}

// Get related products
$relatedProducts = \Cartt\Services\ProductRelationshipService::getRelated($product->id, 'related', 4);
if (empty($relatedProducts)) {
    $relatedProducts = \Cartt\Services\ProductRelationshipService::getAutoRelated($product->id, 4);
}
?>

<div class="cartt-single-product">
    <div class="cartt-product-gallery">
        <div class="cartt-main-image" id="main-product-image">
            <?php if ($image_url): ?>
                <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($product->name); ?>" id="main-img">
            <?php else: ?>
                <div class="cartt-no-image"><?php esc_html_e('No image available', 'cartt'); ?></div>
            <?php endif; ?>
        </div>
        
        <?php if (!empty($gallery_images) || $product->featured_image): ?>
        <div class="cartt-gallery-thumbs">
            <?php if ($product->featured_image): 
                $thumbUrl = wp_get_attachment_image_url($product->featured_image, 'thumbnail');
            ?>
            <div class="cartt-gallery-thumb active" data-full="<?php echo esc_url($image_url); ?>">
                <img src="<?php echo esc_url($thumbUrl); ?>" alt="">
            </div>
            <?php endif; ?>
            <?php foreach ($gallery_images as $imgId): 
                $thumbUrl = wp_get_attachment_image_url($imgId, 'thumbnail');
                $fullUrl = wp_get_attachment_image_url($imgId, 'large');
                if (!$thumbUrl) continue;
            ?>
            <div class="cartt-gallery-thumb" data-full="<?php echo esc_url($fullUrl); ?>">
                <img src="<?php echo esc_url($thumbUrl); ?>" alt="">
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>

    <div class="cartt-product-details">
        <?php if (!empty($categories)): ?>
        <div class="cartt-product-cats">
            <?php foreach ($categories as $cat): ?>
            <a href="<?php echo esc_url(add_query_arg('category', $cat->slug, get_permalink(get_option('cartt_shop_page_id')))); ?>"><?php echo esc_html($cat->name); ?></a>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        
        <h1><?php echo esc_html($product->name); ?></h1>
        
        <?php if ($reviewCount > 0): ?>
        <div class="cartt-product-rating-summary">
            <span class="cartt-stars"><?php echo str_repeat('★', round($avgRating)) . str_repeat('☆', 5 - round($avgRating)); ?></span>
            <span class="cartt-rating-text"><?php echo number_format($avgRating, 1); ?> out of 5</span>
            <a href="#reviews" class="cartt-review-link"><?php printf(_n('%d review', '%d reviews', $reviewCount, 'cartt'), $reviewCount); ?></a>
        </div>
        <?php endif; ?>
        
        <div class="cartt-product-price" id="cartt-product-price">
            <?php if ($hasVariations && $priceRange): ?>
                <?php 
                $convertedMin = CurrencyService::convertPrice($priceRange['min']);
                $convertedMax = CurrencyService::convertPrice($priceRange['max']);
                if ($priceRange['min'] === $priceRange['max']): ?>
                    <?php echo esc_html(CurrencyService::formatPrice($convertedMin)); ?>
                <?php else: ?>
                    <span class="cartt-price-range"><?php echo esc_html(CurrencyService::formatPrice($convertedMin)); ?> - <?php echo esc_html(CurrencyService::formatPrice($convertedMax)); ?></span>
                <?php endif; ?>
            <?php elseif ($sale_price && $sale_price < $price): ?>
                <span class="cartt-sale-price"><?php echo esc_html(CurrencyService::formatPrice(CurrencyService::convertPrice($sale_price))); ?></span>
                <span class="cartt-original-price"><?php echo esc_html(CurrencyService::formatPrice(CurrencyService::convertPrice($price))); ?></span>
                <span class="cartt-save-badge"><?php printf(__('Save %s%%', 'cartt'), round((($price - $sale_price) / $price) * 100)); ?></span>
            <?php else: ?>
                <?php echo esc_html(CurrencyService::formatPrice(CurrencyService::convertPrice($price))); ?>
            <?php endif; ?>
        </div>

        <?php if ($product->short_description): ?>
            <div class="cartt-product-description">
                <?php echo wp_kses_post($product->short_description); ?>
            </div>
        <?php endif; ?>

        <?php if ($hasVariations): ?>
        <div class="cartt-variation-selector" id="variation-selector">
            <?php foreach ($attributes as $attrName => $values): ?>
            <div class="cartt-variation-attribute">
                <label><?php echo esc_html(ucfirst($attrName)); ?></label>
                <select class="cartt-variation-select" data-attribute="<?php echo esc_attr($attrName); ?>">
                    <option value=""><?php printf(__('Choose %s', 'cartt'), ucfirst($attrName)); ?></option>
                    <?php foreach ($values as $value): ?>
                    <option value="<?php echo esc_attr($value); ?>"><?php echo esc_html($value); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endforeach; ?>
            <div class="cartt-variation-status" id="variation-status"></div>
        </div>
        <?php endif; ?>

        <div class="cartt-add-to-cart-form">
            <input type="number" class="cartt-quantity-input" id="cartt-qty" value="1" min="1" max="99">
            <button type="button" class="cartt-add-to-cart-btn" id="cartt-add-btn" 
                    data-product-id="<?php echo esc_attr($product->id); ?>"
                    <?php echo $hasVariations ? 'disabled' : ''; ?>>
                <?php echo $hasVariations ? esc_html__('Select Options', 'cartt') : esc_html__('Add to Cart', 'cartt'); ?>
            </button>
        </div>

        <?php if ($product->sku): ?>
            <p class="cartt-product-sku">
                <strong><?php esc_html_e('SKU:', 'cartt'); ?></strong> <?php echo esc_html($product->sku); ?>
            </p>
        <?php endif; ?>

        <?php if ($product->description): ?>
            <div class="cartt-product-full-description">
                <h3><?php esc_html_e('Description', 'cartt'); ?></h3>
                <?php echo wp_kses_post($product->description); ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Reviews Section -->
<div class="cartt-reviews-section" id="reviews">
    <h2><?php esc_html_e('Customer Reviews', 'cartt'); ?></h2>
    
    <?php if ($reviewCount > 0): ?>
    <div class="cartt-reviews-summary">
        <div class="cartt-rating-big">
            <span class="cartt-rating-number"><?php echo number_format($avgRating, 1); ?></span>
            <span class="cartt-stars-big"><?php echo str_repeat('★', round($avgRating)) . str_repeat('☆', 5 - round($avgRating)); ?></span>
            <span class="cartt-total-reviews"><?php printf(_n('%d review', '%d reviews', $reviewCount, 'cartt'), $reviewCount); ?></span>
        </div>
        <div class="cartt-rating-bars">
            <?php for ($i = 5; $i >= 1; $i--): 
                $count = $ratingBreakdown[$i] ?? 0;
                $percent = $reviewCount > 0 ? ($count / $reviewCount) * 100 : 0;
            ?>
            <div class="cartt-rating-bar-row">
                <span><?php echo $i; ?> ★</span>
                <div class="cartt-rating-bar"><div class="cartt-rating-bar-fill" style="width: <?php echo $percent; ?>%"></div></div>
                <span><?php echo $count; ?></span>
            </div>
            <?php endfor; ?>
        </div>
    </div>
    <?php endif; ?>
    
    <?php if ($reviewMessage): ?>
    <div class="cartt-review-message"><?php echo esc_html($reviewMessage); ?></div>
    <?php endif; ?>
    
    <!-- Review Form -->
    <div class="cartt-review-form-wrap">
        <h3><?php esc_html_e('Write a Review', 'cartt'); ?></h3>
        <form method="post" class="cartt-review-form">
            <?php wp_nonce_field('cartt_submit_review', 'cartt_review_nonce'); ?>
            
            <?php if (!is_user_logged_in()): ?>
            <div class="cartt-form-row cartt-row-half">
                <div>
                    <label><?php esc_html_e('Name', 'cartt'); ?> *</label>
                    <input type="text" name="review_name" required>
                </div>
                <div>
                    <label><?php esc_html_e('Email', 'cartt'); ?> *</label>
                    <input type="email" name="review_email" required>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="cartt-form-row">
                <label><?php esc_html_e('Rating', 'cartt'); ?> *</label>
                <div class="cartt-star-rating">
                    <?php for ($i = 5; $i >= 1; $i--): ?>
                    <input type="radio" name="review_rating" value="<?php echo $i; ?>" id="star<?php echo $i; ?>" <?php checked($i, 5); ?>>
                    <label for="star<?php echo $i; ?>">★</label>
                    <?php endfor; ?>
                </div>
            </div>
            
            <div class="cartt-form-row">
                <label><?php esc_html_e('Review Title', 'cartt'); ?></label>
                <input type="text" name="review_title" placeholder="<?php esc_attr_e('Sum up your review in a few words', 'cartt'); ?>">
            </div>
            
            <div class="cartt-form-row">
                <label><?php esc_html_e('Your Review', 'cartt'); ?> *</label>
                <textarea name="review_content" rows="4" required placeholder="<?php esc_attr_e('What did you like or dislike?', 'cartt'); ?>"></textarea>
            </div>
            
            <button type="submit" class="cartt-btn"><?php esc_html_e('Submit Review', 'cartt'); ?></button>
        </form>
    </div>
    
    <!-- Reviews List -->
    <?php if (!empty($reviews)): ?>
    <div class="cartt-reviews-list">
        <?php foreach ($reviews as $review): ?>
        <div class="cartt-review">
            <div class="cartt-review-header">
                <div class="cartt-review-stars"><?php echo str_repeat('★', $review->rating) . str_repeat('☆', 5 - $review->rating); ?></div>
                <?php if ($review->verified_purchase): ?>
                <span class="cartt-verified-badge"><?php esc_html_e('Verified Purchase', 'cartt'); ?></span>
                <?php endif; ?>
            </div>
            <?php if ($review->title): ?>
            <h4 class="cartt-review-title"><?php echo esc_html($review->title); ?></h4>
            <?php endif; ?>
            <p class="cartt-review-content"><?php echo esc_html($review->content); ?></p>
            <div class="cartt-review-meta">
                <span class="cartt-review-author"><?php echo esc_html($review->customer_name); ?></span>
                <span class="cartt-review-date"><?php echo esc_html(date_i18n(get_option('date_format'), strtotime($review->created_at))); ?></span>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
    <?php elseif (!$reviewMessage): ?>
    <p class="cartt-no-reviews"><?php esc_html_e('No reviews yet. Be the first to write a review!', 'cartt'); ?></p>
    <?php endif; ?>
</div>

<style>
.cartt-gallery-thumbs { display: flex; gap: 8px; margin-top: 12px; }
.cartt-gallery-thumb { width: 60px; height: 60px; border: 2px solid transparent; border-radius: 6px; overflow: hidden; cursor: pointer; opacity: 0.7; transition: all 0.2s; }
.cartt-gallery-thumb:hover, .cartt-gallery-thumb.active { border-color: #1d1d1f; opacity: 1; }
.cartt-gallery-thumb img { width: 100%; height: 100%; object-fit: cover; }

.cartt-product-cats { margin-bottom: 8px; }
.cartt-product-cats a { color: #86868b; font-size: 12px; text-decoration: none; text-transform: uppercase; letter-spacing: 0.5px; }
.cartt-product-cats a:hover { color: #1d1d1f; }

.cartt-product-rating-summary { display: flex; align-items: center; gap: 8px; margin: 8px 0 16px; }
.cartt-stars { color: #f5a623; }
.cartt-rating-text { font-size: 14px; color: #86868b; }
.cartt-review-link { font-size: 14px; color: #0066cc; text-decoration: none; }

.cartt-save-badge { background: #e8f5e9; color: #2e7d32; font-size: 12px; padding: 4px 8px; border-radius: 4px; margin-left: 8px; }

.cartt-reviews-section { margin-top: 60px; padding-top: 40px; border-top: 1px solid #e8e8ed; }
.cartt-reviews-section h2 { margin-bottom: 24px; }

.cartt-reviews-summary { display: flex; gap: 40px; margin-bottom: 40px; padding: 24px; background: #f5f5f7; border-radius: 12px; }
.cartt-rating-big { text-align: center; }
.cartt-rating-number { display: block; font-size: 48px; font-weight: 600; line-height: 1; }
.cartt-stars-big { display: block; color: #f5a623; font-size: 20px; margin: 8px 0; }
.cartt-total-reviews { color: #86868b; font-size: 14px; }

.cartt-rating-bars { flex: 1; }
.cartt-rating-bar-row { display: flex; align-items: center; gap: 8px; margin-bottom: 6px; font-size: 13px; }
.cartt-rating-bar { flex: 1; height: 8px; background: #e8e8ed; border-radius: 4px; overflow: hidden; }
.cartt-rating-bar-fill { height: 100%; background: #f5a623; }

.cartt-review-form-wrap { background: #f5f5f7; border-radius: 12px; padding: 24px; margin-bottom: 30px; }
.cartt-review-form-wrap h3 { margin: 0 0 16px; }
.cartt-review-form .cartt-form-row { margin-bottom: 16px; }
.cartt-review-form label { display: block; font-size: 13px; font-weight: 500; margin-bottom: 6px; }
.cartt-review-form input, .cartt-review-form textarea { width: 100%; padding: 10px 12px; border: 1px solid #d2d2d7; border-radius: 6px; font-size: 14px; }
.cartt-row-half { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; }

.cartt-star-rating { display: flex; flex-direction: row-reverse; justify-content: flex-end; }
.cartt-star-rating input { display: none; }
.cartt-star-rating label { font-size: 24px; color: #d2d2d7; cursor: pointer; padding: 0 2px; }
.cartt-star-rating label:hover, .cartt-star-rating label:hover ~ label,
.cartt-star-rating input:checked ~ label { color: #f5a623; }

.cartt-btn { background: #1d1d1f; color: #fff; border: none; padding: 12px 24px; border-radius: 6px; font-size: 14px; cursor: pointer; }

.cartt-reviews-list { display: flex; flex-direction: column; gap: 16px; }
.cartt-review { background: #fff; border: 1px solid #e8e8ed; border-radius: 8px; padding: 20px; }
.cartt-review-header { display: flex; align-items: center; gap: 12px; margin-bottom: 8px; }
.cartt-review-stars { color: #f5a623; }
.cartt-verified-badge { background: #e8f5e9; color: #2e7d32; font-size: 11px; padding: 3px 8px; border-radius: 4px; }
.cartt-review-title { margin: 0 0 8px; font-size: 15px; }
.cartt-review-content { margin: 0 0 12px; font-size: 14px; line-height: 1.6; color: #444; }
.cartt-review-meta { font-size: 12px; color: #86868b; }
.cartt-review-author { font-weight: 500; margin-right: 12px; }

.cartt-review-message { background: #e8f5e9; color: #2e7d32; padding: 12px 16px; border-radius: 6px; margin-bottom: 20px; }
.cartt-no-reviews { color: #86868b; font-style: italic; }

@media (max-width: 768px) {
    .cartt-reviews-summary { flex-direction: column; gap: 20px; }
    .cartt-row-half { grid-template-columns: 1fr; }
}
</style>

<script>
(function() {
    const apiUrl = '<?php echo esc_js($api_url); ?>';
    const nonce = '<?php echo esc_js($nonce); ?>';

    // Gallery thumbnails
    document.querySelectorAll('.cartt-gallery-thumb').forEach(thumb => {
        thumb.addEventListener('click', function() {
            document.querySelectorAll('.cartt-gallery-thumb').forEach(t => t.classList.remove('active'));
            this.classList.add('active');
            document.getElementById('main-img').src = this.dataset.full;
        });
    });

    // Add to cart
    document.getElementById('cartt-add-btn').addEventListener('click', async function() {
        const productId = this.dataset.productId;
        const quantity = parseInt(document.getElementById('cartt-qty').value) || 1;
        const originalText = this.textContent;
        
        this.disabled = true;
        this.textContent = 'Adding...';

        try {
            const response = await fetch(apiUrl + 'cart/add', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ product_id: parseInt(productId), quantity: quantity })
            });

            const data = await response.json();
            
            if (data.success) {
                this.textContent = 'Added!';
                setTimeout(() => {
                    this.textContent = originalText;
                    this.disabled = false;
                }, 1500);
            } else {
                this.textContent = data.error || 'Error';
                setTimeout(() => {
                    this.textContent = originalText;
                    this.disabled = false;
                }, 2000);
            }
        } catch (error) {
            this.textContent = 'Error';
            setTimeout(() => {
                this.textContent = originalText;
                this.disabled = false;
            }, 2000);
        }
    });
})();
</script>
