<?php
/**
 * Dynamic Checkout Template
 * Renders based on CheckoutConfig settings
 */

if (!defined('ABSPATH')) {
    exit;
}

$config = \Cartt\Services\CheckoutConfig::get();
$fields = \Cartt\Services\CheckoutConfig::getFields();
$layout = $config['layout'] ?? 'one-page';
$style = $config['style'] ?? 'modern';
$user = wp_get_current_user();

// Get current currency
$currencyData = \Cartt\Services\CurrencyService::getCurrentCurrency();

// Custom colors
$colors = $config['colors'] ?? [];
$btnColor = $colors['button'] ?? '#1d1d1f';
$btnTextColor = $colors['button_text'] ?? '#ffffff';
$accentColor = $colors['accent'] ?? '#1d1d1f';
$borderColor = $colors['border'] ?? '#d2d2d7';

// Payment gateways
$stripe_key = get_option('cartt_stripe_publishable_key', '');
$paypal_enabled = get_option('cartt_paypal_enabled', false);
$paypal_client_id = get_option('cartt_paypal_client_id', '');
$paypal_sandbox = get_option('cartt_paypal_sandbox', true);
$currency = strtoupper(get_option('cartt_currency', 'USD'));

$has_stripe = !empty($stripe_key);
$has_paypal = $paypal_enabled && !empty($paypal_client_id);
$has_payment = $has_stripe || $has_paypal;

// Express payments
$express_enabled = $config['express_payments']['enabled'] ?? false;
$apple_pay = $config['express_payments']['apple_pay'] ?? false;
$google_pay = $config['express_payments']['google_pay'] ?? false;

// Address autocomplete
$autocomplete_enabled = $config['address_autocomplete'] ?? false;
$google_places_key = $config['google_places_key'] ?? '';

// Check for cart recovery
$abandonedService = new \Cartt\Services\AbandonedCartService();
if (!empty($_GET['recover'])) {
    $abandonedService->recoverCart(sanitize_text_field($_GET['recover']));
}

// Group fields by section
$sections = [];
foreach ($fields as $id => $field) {
    $section = $field['section'] ?? 'billing';
    if (!isset($sections[$section])) {
        $sections[$section] = [];
    }
    $sections[$section][$id] = $field;
}

$section_titles = [
    'contact' => __('Contact', 'cartt'),
    'billing' => __('Billing Address', 'cartt'),
    'shipping' => __('Shipping Address', 'cartt'),
    'additional' => __('Additional Information', 'cartt'),
];
?>

<div class="cartt-checkout cartt-checkout-<?php echo esc_attr($layout); ?> cartt-style-<?php echo esc_attr($style); ?>">
    <?php if ($layout === 'multi-step'): ?>
    <div class="cartt-checkout-steps">
        <div class="cartt-step active" data-step="1"><span>1</span> <?php esc_html_e('Contact', 'cartt'); ?></div>
        <div class="cartt-step" data-step="2"><span>2</span> <?php esc_html_e('Shipping', 'cartt'); ?></div>
        <div class="cartt-step" data-step="3"><span>3</span> <?php esc_html_e('Payment', 'cartt'); ?></div>
    </div>
    <?php endif; ?>

    <div id="cartt-checkout-app">
        <form id="cartt-checkout-form" class="cartt-checkout-form">
            <div class="cartt-checkout-grid">
                <div class="cartt-checkout-main">
                    
                    <?php if ($express_enabled && $has_stripe && ($apple_pay || $google_pay)): ?>
                    <!-- Express Checkout -->
                    <div class="cartt-checkout-section cartt-express-section">
                        <h2><?php esc_html_e('Express Checkout', 'cartt'); ?></h2>
                        <div id="cartt-payment-request-button"></div>
                        <div class="cartt-express-divider">
                            <span><?php esc_html_e('or continue below', 'cartt'); ?></span>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php foreach ($sections as $sectionId => $sectionFields): ?>
                    <div class="cartt-checkout-section cartt-section-<?php echo esc_attr($sectionId); ?>" 
                         <?php if ($layout === 'multi-step'): ?>data-step="<?php echo $sectionId === 'contact' ? '1' : ($sectionId === 'billing' || $sectionId === 'shipping' ? '2' : '3'); ?>"<?php endif; ?>>
                        <h2><?php echo esc_html($section_titles[$sectionId] ?? ucfirst($sectionId)); ?></h2>
                        
                        <div class="cartt-fields-grid">
                            <?php foreach ($sectionFields as $fieldId => $field): 
                                $widthClass = 'cartt-field-' . ($field['width'] ?? 'full');
                                $inputType = $field['type'] ?? 'text';
                                $required = $field['required'] ?? false;
                                $placeholder = $field['placeholder'] ?? '';
                                $autocomplete = $field['autocomplete'] ?? '';
                                
                                // Pre-fill from user
                                $value = '';
                                if ($fieldId === 'email' && $user->user_email) $value = $user->user_email;
                                if ($fieldId === 'first_name' && $user->first_name) $value = $user->first_name;
                                if ($fieldId === 'last_name' && $user->last_name) $value = $user->last_name;
                            ?>
                            <div class="cartt-form-row <?php echo esc_attr($widthClass); ?>">
                                <label for="field-<?php echo esc_attr($fieldId); ?>">
                                    <?php echo esc_html($field['label']); ?>
                                    <?php if ($required): ?><span class="required">*</span><?php endif; ?>
                                </label>
                                
                                <?php if ($inputType === 'textarea'): ?>
                                    <textarea id="field-<?php echo esc_attr($fieldId); ?>" 
                                              name="<?php echo esc_attr($fieldId); ?>"
                                              placeholder="<?php echo esc_attr($placeholder); ?>"
                                              <?php echo $required ? 'required' : ''; ?>
                                              rows="3"></textarea>
                                
                                <?php elseif ($inputType === 'country'): ?>
                                    <select id="field-<?php echo esc_attr($fieldId); ?>" 
                                            name="<?php echo esc_attr($fieldId); ?>"
                                            <?php echo $required ? 'required' : ''; ?>>
                                        <option value="US">United States</option>
                                        <option value="CA">Canada</option>
                                        <option value="GB">United Kingdom</option>
                                        <option value="AU">Australia</option>
                                        <option value="DE">Germany</option>
                                        <option value="FR">France</option>
                                        <option value="ES">Spain</option>
                                        <option value="IT">Italy</option>
                                        <option value="NL">Netherlands</option>
                                        <option value="JP">Japan</option>
                                    </select>
                                
                                <?php else: ?>
                                    <input type="<?php echo esc_attr($inputType); ?>" 
                                           id="field-<?php echo esc_attr($fieldId); ?>" 
                                           name="<?php echo esc_attr($fieldId); ?>"
                                           value="<?php echo esc_attr($value); ?>"
                                           placeholder="<?php echo esc_attr($placeholder); ?>"
                                           <?php echo $required ? 'required' : ''; ?>
                                           <?php if ($autocomplete === 'address' && $autocomplete_enabled): ?>
                                           data-autocomplete="address"
                                           <?php endif; ?>>
                                <?php endif; ?>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>

                    <!-- Payment Section -->
                    <div class="cartt-checkout-section cartt-section-payment" <?php if ($layout === 'multi-step'): ?>data-step="3"<?php endif; ?>>
                        <h2><?php esc_html_e('Payment', 'cartt'); ?></h2>
                        
                        <?php if ($has_payment): ?>
                            <?php if ($has_stripe && $has_paypal): ?>
                            <div class="cartt-payment-methods">
                                <label class="cartt-payment-method">
                                    <input type="radio" name="payment_method" value="stripe" checked>
                                    <span><?php esc_html_e('Credit Card', 'cartt'); ?></span>
                                </label>
                                <label class="cartt-payment-method">
                                    <input type="radio" name="payment_method" value="paypal">
                                    <span><?php esc_html_e('PayPal', 'cartt'); ?></span>
                                </label>
                            </div>
                            <?php elseif ($has_stripe): ?>
                                <input type="hidden" name="payment_method" value="stripe">
                            <?php else: ?>
                                <input type="hidden" name="payment_method" value="paypal">
                            <?php endif; ?>

                            <?php if ($has_stripe): ?>
                            <div id="cartt-stripe-container" class="cartt-payment-container">
                                <div id="cartt-card-element"></div>
                                <div id="cartt-card-errors" class="cartt-error-message"></div>
                            </div>
                            <?php endif; ?>

                            <?php if ($has_paypal): ?>
                            <div id="cartt-paypal-container" class="cartt-payment-container" style="<?php echo $has_stripe ? 'display:none;' : ''; ?>">
                                <div id="paypal-button-container"></div>
                            </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="cartt-demo-payment">
                                <p><?php esc_html_e('Demo mode - no payment will be charged.', 'cartt'); ?></p>
                                <input type="hidden" name="payment_method" value="demo">
                            </div>
                        <?php endif; ?>
                    </div>

                    <?php if ($layout === 'multi-step'): ?>
                    <div class="cartt-step-nav">
                        <button type="button" class="cartt-btn cartt-btn-secondary" id="cartt-prev-step" style="display:none;">
                            <?php esc_html_e('Back', 'cartt'); ?>
                        </button>
                        <button type="button" class="cartt-btn cartt-btn-primary" id="cartt-next-step">
                            <?php esc_html_e('Continue', 'cartt'); ?>
                        </button>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Order Summary Sidebar -->
                <div class="cartt-checkout-sidebar">
                    <div class="cartt-order-summary">
                        <h2><?php esc_html_e('Order Summary', 'cartt'); ?></h2>
                        
                        <div id="cartt-order-items" class="cartt-order-items">
                            <div class="cartt-loading"><div class="cartt-spinner"></div></div>
                        </div>

                        <!-- Order Bumps -->
                        <div id="cartt-order-bumps" class="cartt-order-bumps"></div>

                        <!-- Coupon Code -->
                        <div class="cartt-coupon-section">
                            <div id="cartt-coupon-form" class="cartt-coupon-form">
                                <input type="text" id="cartt-coupon-code" placeholder="<?php esc_attr_e('Coupon code', 'cartt'); ?>">
                                <button type="button" id="cartt-apply-coupon"><?php esc_html_e('Apply', 'cartt'); ?></button>
                            </div>
                            <div id="cartt-coupon-applied" class="cartt-coupon-applied" style="display:none;">
                                <span id="cartt-coupon-code-display"></span>
                                <button type="button" id="cartt-remove-coupon">&times;</button>
                            </div>
                            <div id="cartt-coupon-error" class="cartt-coupon-error" style="display:none;"></div>
                        </div>

                        <!-- Gift Card -->
                        <?php if (get_option('cartt_gift_cards_enabled', true)): ?>
                        <div class="cartt-gift-card-section">
                            <div id="cartt-gift-card-form" class="cartt-gift-card-form">
                                <input type="text" id="cartt-gift-card-code" placeholder="<?php esc_attr_e('Gift card code', 'cartt'); ?>">
                                <button type="button" id="cartt-apply-gift-card"><?php esc_html_e('Apply', 'cartt'); ?></button>
                            </div>
                            <div id="cartt-gift-card-applied" class="cartt-gift-card-applied" style="display:none;">
                                <span id="cartt-gift-card-display"></span>
                                <button type="button" id="cartt-remove-gift-card">&times;</button>
                            </div>
                            <div id="cartt-gift-card-error" class="cartt-gift-card-error" style="display:none;"></div>
                        </div>
                        <?php endif; ?>

                        <!-- Loyalty Points -->
                        <?php 
                        $loyaltyEnabled = get_option('cartt_loyalty_enabled', false);
                        $customerId = null;
                        $customerPoints = 0;
                        if ($loyaltyEnabled && is_user_logged_in()) {
                            global $wpdb;
                            $customerId = $wpdb->get_var($wpdb->prepare(
                                "SELECT id FROM {$wpdb->prefix}cartt_customers WHERE user_id = %d",
                                get_current_user_id()
                            ));
                            if ($customerId) {
                                $pointsData = \Cartt\Services\LoyaltyService::getCustomerPoints($customerId);
                                $customerPoints = $pointsData->points ?? 0;
                            }
                        }
                        if ($loyaltyEnabled && $customerPoints > 0): 
                            $minRedeem = intval(get_option('cartt_loyalty_min_redemption', 100));
                            $pointsValue = floatval(get_option('cartt_loyalty_points_value', 0.01));
                            $maxValue = $customerPoints * $pointsValue;
                        ?>
                        <div class="cartt-loyalty-section">
                            <div class="cartt-loyalty-balance">
                                <span><?php esc_html_e('Your Points:', 'cartt'); ?></span>
                                <strong id="cartt-loyalty-balance"><?php echo number_format($customerPoints); ?></strong>
                                <small>(<?php echo esc_html(\Cartt\Services\CurrencyService::formatPrice($maxValue)); ?> value)</small>
                            </div>
                            <?php if ($customerPoints >= $minRedeem): ?>
                            <div id="cartt-loyalty-form" class="cartt-loyalty-form">
                                <input type="number" id="cartt-loyalty-points" min="<?php echo $minRedeem; ?>" max="<?php echo $customerPoints; ?>" step="1" placeholder="<?php echo sprintf(__('Min %d points', 'cartt'), $minRedeem); ?>">
                                <button type="button" id="cartt-apply-loyalty"><?php esc_html_e('Redeem', 'cartt'); ?></button>
                            </div>
                            <div id="cartt-loyalty-applied" class="cartt-loyalty-applied" style="display:none;">
                                <span id="cartt-loyalty-display"></span>
                                <button type="button" id="cartt-remove-loyalty">&times;</button>
                            </div>
                            <div id="cartt-loyalty-error" class="cartt-loyalty-error" style="display:none;"></div>
                            <?php else: ?>
                            <small class="cartt-loyalty-min-notice"><?php echo sprintf(__('Minimum %d points required to redeem', 'cartt'), $minRedeem); ?></small>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>

                        <div class="cartt-order-totals">
                            <div class="cartt-totals-row">
                                <span><?php esc_html_e('Subtotal', 'cartt'); ?></span>
                                <span id="cartt-subtotal">-</span>
                            </div>
                            <div class="cartt-totals-row cartt-discount-row" id="cartt-discount-row" style="display:none;">
                                <span><?php esc_html_e('Discount', 'cartt'); ?></span>
                                <span id="cartt-discount">-</span>
                            </div>
                            <div class="cartt-totals-row cartt-gift-card-row" id="cartt-gift-card-row" style="display:none;">
                                <span><?php esc_html_e('Gift Card', 'cartt'); ?></span>
                                <span id="cartt-gift-card-amount">-</span>
                            </div>
                            <div class="cartt-totals-row cartt-loyalty-row" id="cartt-loyalty-row" style="display:none;">
                                <span><?php esc_html_e('Points Discount', 'cartt'); ?></span>
                                <span id="cartt-loyalty-amount">-</span>
                            </div>
                            <div class="cartt-totals-row cartt-shipping-row" id="cartt-shipping-row">
                                <span><?php esc_html_e('Shipping', 'cartt'); ?></span>
                                <span id="cartt-shipping">-</span>
                            </div>
                            <div class="cartt-totals-row">
                                <span><?php esc_html_e('Tax', 'cartt'); ?></span>
                                <span id="cartt-tax">-</span>
                            </div>
                            <div class="cartt-totals-row cartt-total">
                                <span><?php esc_html_e('Total', 'cartt'); ?></span>
                                <span id="cartt-total">-</span>
                            </div>
                        </div>

                        <!-- Shipping Methods -->
                        <div id="cartt-shipping-methods" class="cartt-shipping-methods" style="display:none;"></div>

                        <button type="submit" id="cartt-submit-btn" class="cartt-place-order-btn" disabled>
                            <?php esc_html_e('Pay now', 'cartt'); ?>
                        </button>
                        
                        <div id="cartt-checkout-error" class="cartt-error-message" style="display: none;"></div>

                        <div class="cartt-secure-badge">
                            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <rect x="3" y="11" width="18" height="11" rx="2" ry="2"/>
                                <path d="M7 11V7a5 5 0 0110 0v4"/>
                            </svg>
                            <?php esc_html_e('Secure checkout', 'cartt'); ?>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<?php if ($has_stripe): ?>
<script src="https://js.stripe.com/v3/"></script>
<?php endif; ?>

<?php if ($has_paypal): ?>
<script src="https://www.paypal.com/sdk/js?client-id=<?php echo esc_attr($paypal_client_id); ?>&currency=<?php echo esc_attr($currency); ?>"></script>
<?php endif; ?>

<?php if ($autocomplete_enabled): ?>
<script src="https://maps.googleapis.com/maps/api/js?key=AIzaSyBcartt-placeholder-key&libraries=places&callback=initCarttAutocomplete" async defer></script>
<?php endif; ?>

<script>
// Autocomplete init function (called by Google Maps callback)
function initCarttAutocomplete() {
    window.carttMapsLoaded = true;
    if (typeof window.initAddressAutocomplete === 'function') {
        window.initAddressAutocomplete();
    }
}

(function() {
    const apiUrl = '<?php echo esc_js(rest_url('cartt/v1/')); ?>';
    const nonce = '<?php echo esc_js(wp_create_nonce('wp_rest')); ?>';
    const hasStripe = <?php echo $has_stripe ? 'true' : 'false'; ?>;
    const hasPaypal = <?php echo $has_paypal ? 'true' : 'false'; ?>;
    const stripeKey = '<?php echo esc_js($stripe_key); ?>';
    const layout = '<?php echo esc_js($layout); ?>';
    const expressEnabled = <?php echo ($express_enabled && $has_stripe) ? 'true' : 'false'; ?>;
    const autocompleteEnabled = <?php echo $autocomplete_enabled ? 'true' : 'false'; ?>;
    const emailFirst = <?php echo ($config['email_first'] ?? true) ? 'true' : 'false'; ?>;
    const orderBumps = <?php echo json_encode(\Cartt\Services\CheckoutConfig::getOrderBumps()); ?>;
    
    // Currency settings
    const currencySymbol = '<?php echo esc_js($currencyData->symbol ?? '$'); ?>';
    const currencyPosition = '<?php echo esc_js($currencyData->symbol_position ?? 'left'); ?>';
    const exchangeRate = <?php echo floatval($currencyData->exchange_rate ?? 1); ?>;
    const currencyDecimals = <?php echo intval($currencyData->decimals ?? 2); ?>;
    const thousandSep = '<?php echo esc_js($currencyData->thousand_separator ?? ','); ?>';
    const decimalSep = '<?php echo esc_js($currencyData->decimal_separator ?? '.'); ?>';
    
    let stripe, elements, cardElement, paymentRequest;
    let cart = null;
    let currentStep = 1;

    function formatCurrency(amount) {
        // Convert using exchange rate
        let converted = parseFloat(amount) * exchangeRate;
        
        // Format number
        let formatted = converted.toFixed(currencyDecimals);
        let parts = formatted.split('.');
        parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);
        if (parts[1]) {
            formatted = parts[0] + decimalSep + parts[1];
        } else {
            formatted = parts[0];
        }
        
        // Apply symbol position
        switch (currencyPosition) {
            case 'right': return formatted + currencySymbol;
            case 'left_space': return currencySymbol + ' ' + formatted;
            case 'right_space': return formatted + ' ' + currencySymbol;
            default: return currencySymbol + formatted;
        }
    }

    function collectFormData() {
        const data = {};
        document.querySelectorAll('#cartt-checkout-form input, #cartt-checkout-form select, #cartt-checkout-form textarea').forEach(el => {
            if (el.name && el.type !== 'radio' && el.type !== 'checkbox') {
                data[el.name] = el.value;
            } else if (el.type === 'radio' && el.checked) {
                data[el.name] = el.value;
            } else if (el.type === 'checkbox' && el.checked) {
                data[el.name] = el.value || true;
            }
        });
        return data;
    }

    function getSelectedPaymentMethod() {
        const radio = document.querySelector('input[name="payment_method"]:checked');
        return radio ? radio.value : 'demo';
    }

    // Capture email for abandoned cart
    function captureEmail() {
        if (!emailFirst) return;
        const emailField = document.getElementById('field-email');
        if (emailField && emailField.value) {
            fetch(apiUrl + 'cart/capture-email', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ 
                    email: emailField.value,
                    billing: collectFormData()
                })
            }).catch(() => {});
        }
    }

    async function loadCart() {
        try {
            const response = await fetch(apiUrl + 'cart', {
                headers: { 'X-WP-Nonce': nonce }
            });
            const data = await response.json();
            cart = data.data;
            renderOrderSummary();
        } catch (error) {
            console.error('Failed to load cart:', error);
        }
    }

    function renderOrderSummary() {
        const itemsContainer = document.getElementById('cartt-order-items');
        const submitBtn = document.getElementById('cartt-submit-btn');
        
        if (!cart || Object.keys(cart.items).length === 0) {
            itemsContainer.innerHTML = '<p class="cartt-empty-cart">Your cart is empty.</p>';
            submitBtn.disabled = true;
            return;
        }

        let html = '';
        for (const key in cart.items) {
            const item = cart.items[key];
            html += `
                <div class="cartt-order-item">
                    ${item.image ? `<img src="${item.image}" class="cartt-order-item-image">` : ''}
                    <div class="cartt-order-item-details">
                        <span class="cartt-order-item-name">${item.name}</span>
                        <span class="cartt-order-item-qty">Qty: ${item.quantity}</span>
                    </div>
                    <span class="cartt-order-item-price">${formatCurrency(item.price * item.quantity)}</span>
                </div>
            `;
        }
        itemsContainer.innerHTML = html;

        document.getElementById('cartt-subtotal').textContent = formatCurrency(cart.totals.subtotal);
        document.getElementById('cartt-tax').textContent = formatCurrency(cart.totals.tax);
        document.getElementById('cartt-total').textContent = formatCurrency(cart.totals.total);
        
        // Show/hide discount row
        const discountRow = document.getElementById('cartt-discount-row');
        if (cart.totals.discount > 0) {
            discountRow.style.display = 'flex';
            document.getElementById('cartt-discount').textContent = '-' + formatCurrency(cart.totals.discount);
        } else {
            discountRow.style.display = 'none';
        }

        // Show/hide gift card row
        const giftCardRow = document.getElementById('cartt-gift-card-row');
        if (giftCardRow && cart.totals.gift_card_amount > 0) {
            giftCardRow.style.display = 'flex';
            document.getElementById('cartt-gift-card-amount').textContent = '-' + formatCurrency(cart.totals.gift_card_amount);
        } else if (giftCardRow) {
            giftCardRow.style.display = 'none';
        }

        // Show/hide loyalty points row
        const loyaltyRow = document.getElementById('cartt-loyalty-row');
        if (loyaltyRow && cart.totals.loyalty_discount > 0) {
            loyaltyRow.style.display = 'flex';
            document.getElementById('cartt-loyalty-amount').textContent = '-' + formatCurrency(cart.totals.loyalty_discount);
        } else if (loyaltyRow) {
            loyaltyRow.style.display = 'none';
        }
        
        // Show shipping
        const shippingEl = document.getElementById('cartt-shipping');
        if (cart.totals.shipping > 0) {
            shippingEl.textContent = formatCurrency(cart.totals.shipping);
        } else if (cart.shipping_method) {
            shippingEl.textContent = 'Free';
        } else {
            shippingEl.textContent = 'Calculated at next step';
        }
        
        // Show applied coupon
        const couponForm = document.getElementById('cartt-coupon-form');
        const couponApplied = document.getElementById('cartt-coupon-applied');
        if (cart.coupon) {
            couponForm.style.display = 'none';
            couponApplied.style.display = 'flex';
            document.getElementById('cartt-coupon-code-display').textContent = cart.coupon.code;
        } else {
            couponForm.style.display = 'flex';
            couponApplied.style.display = 'none';
        }

        // Show applied gift card
        const gcForm = document.getElementById('cartt-gift-card-form');
        const gcApplied = document.getElementById('cartt-gift-card-applied');
        if (gcForm && gcApplied) {
            if (cart.gift_card) {
                gcForm.style.display = 'none';
                gcApplied.style.display = 'flex';
                document.getElementById('cartt-gift-card-display').textContent = 
                    cart.gift_card.code + ' (-' + formatCurrency(cart.totals.gift_card_amount) + ')';
            } else {
                gcForm.style.display = 'flex';
                gcApplied.style.display = 'none';
            }
        }

        // Show applied loyalty points
        const loyaltyForm = document.getElementById('cartt-loyalty-form');
        const loyaltyApplied = document.getElementById('cartt-loyalty-applied');
        if (loyaltyForm && loyaltyApplied) {
            if (cart.loyalty_points > 0) {
                loyaltyForm.style.display = 'none';
                loyaltyApplied.style.display = 'flex';
                document.getElementById('cartt-loyalty-display').textContent = 
                    cart.loyalty_points.toLocaleString() + ' points (-' + formatCurrency(cart.totals.loyalty_discount) + ')';
            } else {
                loyaltyForm.style.display = 'flex';
                loyaltyApplied.style.display = 'none';
            }
        }
        
        if (layout !== 'multi-step') {
            submitBtn.disabled = false;
        }
        submitBtn.textContent = 'Pay ' + formatCurrency(cart.totals.total);
        
        // Render order bumps
        renderOrderBumps();
    }

    function renderOrderBumps() {
        const container = document.getElementById('cartt-order-bumps');
        if (!container || !orderBumps || Object.keys(orderBumps).length === 0) {
            return;
        }
        
        let html = '';
        for (const bumpId in orderBumps) {
            const bump = orderBumps[bumpId];
            if (!bump.enabled) continue;
            
            const isAdded = cart.bumps && cart.bumps[bumpId];
            const addedData = isAdded ? cart.bumps[bumpId] : null;
            
            html += `
                <div class="cartt-order-bump ${isAdded ? 'selected' : ''}" data-bump-id="${bumpId}">
                    <div class="cartt-bump-checkbox">
                        <input type="checkbox" ${isAdded ? 'checked' : ''}>
                    </div>
                    <div class="cartt-bump-content">
                        <div class="cartt-bump-title">${bump.title || 'Add this to your order!'}</div>
                        <div class="cartt-bump-description">${bump.description || ''}</div>
                    </div>
                    <div class="cartt-bump-price">
                        ${addedData ? formatCurrency(addedData.price) : ''}
                    </div>
                </div>
            `;
        }
        
        container.innerHTML = html;
        
        // Add click handlers
        container.querySelectorAll('.cartt-order-bump').forEach(el => {
            el.addEventListener('click', async function(e) {
                if (e.target.type === 'checkbox') return;
                
                const bumpId = this.dataset.bumpId;
                const checkbox = this.querySelector('input[type="checkbox"]');
                const isChecked = checkbox.checked;
                
                // Toggle
                if (isChecked) {
                    await removeBump(bumpId);
                } else {
                    await addBump(bumpId);
                }
            });
            
            el.querySelector('input[type="checkbox"]').addEventListener('change', async function() {
                const bumpId = this.closest('.cartt-order-bump').dataset.bumpId;
                if (this.checked) {
                    await addBump(bumpId);
                } else {
                    await removeBump(bumpId);
                }
            });
        });
    }
    
    async function addBump(bumpId) {
        try {
            const response = await fetch(apiUrl + 'cart/add-bump', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-WP-Nonce': nonce },
                body: JSON.stringify({ bump_id: bumpId })
            });
            const result = await response.json();
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
            }
        } catch (error) {
            console.error('Failed to add bump');
        }
    }
    
    async function removeBump(bumpId) {
        try {
            const response = await fetch(apiUrl + 'cart/remove-bump', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json', 'X-WP-Nonce': nonce },
                body: JSON.stringify({ bump_id: bumpId })
            });
            const result = await response.json();
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
            }
        } catch (error) {
            console.error('Failed to remove bump');
        }
    }

    // Coupon handlers
    async function applyCoupon() {
        const codeInput = document.getElementById('cartt-coupon-code');
        const errorEl = document.getElementById('cartt-coupon-error');
        const code = codeInput.value.trim();
        
        if (!code) return;
        
        errorEl.style.display = 'none';
        
        try {
            const response = await fetch(apiUrl + 'coupon/apply', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ 
                    code: code,
                    email: document.getElementById('field-email')?.value || ''
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
                codeInput.value = '';
            } else {
                errorEl.textContent = result.error || 'Invalid coupon';
                errorEl.style.display = 'block';
            }
        } catch (error) {
            errorEl.textContent = 'Failed to apply coupon';
            errorEl.style.display = 'block';
        }
    }
    
    async function removeCoupon() {
        try {
            const response = await fetch(apiUrl + 'coupon/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
            }
        } catch (error) {
            console.error('Failed to remove coupon');
        }
    }

    // Gift Card handlers
    async function applyGiftCard() {
        const codeInput = document.getElementById('cartt-gift-card-code');
        const errorEl = document.getElementById('cartt-gift-card-error');
        const code = codeInput.value.trim().toUpperCase();
        
        if (!code) {
            errorEl.textContent = 'Please enter a gift card code';
            errorEl.style.display = 'block';
            return;
        }
        
        errorEl.style.display = 'none';
        
        try {
            const response = await fetch(apiUrl + 'gift-card/apply', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ code })
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
                codeInput.value = '';
            } else {
                errorEl.textContent = result.error || 'Invalid gift card';
                errorEl.style.display = 'block';
            }
        } catch (error) {
            errorEl.textContent = 'Failed to apply gift card';
            errorEl.style.display = 'block';
        }
    }
    
    async function removeGiftCard() {
        try {
            const response = await fetch(apiUrl + 'gift-card/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
            }
        } catch (error) {
            console.error('Failed to remove gift card');
        }
    }

    // Loyalty Points handlers
    async function applyLoyaltyPoints() {
        const pointsInput = document.getElementById('cartt-loyalty-points');
        const errorEl = document.getElementById('cartt-loyalty-error');
        const points = parseInt(pointsInput.value) || 0;
        
        if (points < 1) {
            errorEl.textContent = 'Please enter points to redeem';
            errorEl.style.display = 'block';
            return;
        }
        
        errorEl.style.display = 'none';
        
        try {
            const response = await fetch(apiUrl + 'loyalty/apply', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ points })
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
                pointsInput.value = '';
                // Update balance display
                const balanceEl = document.getElementById('cartt-loyalty-balance');
                if (balanceEl && result.remaining_points !== undefined) {
                    balanceEl.textContent = result.remaining_points.toLocaleString();
                }
            } else {
                errorEl.textContent = result.error || 'Failed to redeem points';
                errorEl.style.display = 'block';
            }
        } catch (error) {
            errorEl.textContent = 'Failed to apply points';
            errorEl.style.display = 'block';
        }
    }
    
    async function removeLoyaltyPoints() {
        try {
            const response = await fetch(apiUrl + 'loyalty/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderOrderSummary();
                // Restore balance display
                const balanceEl = document.getElementById('cartt-loyalty-balance');
                if (balanceEl && result.remaining_points !== undefined) {
                    balanceEl.textContent = result.remaining_points.toLocaleString();
                }
            }
        } catch (error) {
            console.error('Failed to remove loyalty points');
        }
    }

    // Shipping handlers
    let shippingMethods = [];
    
    async function calculateShipping() {
        const country = document.getElementById('field-country')?.value || '';
        const state = document.getElementById('field-state')?.value || '';
        const postcode = document.getElementById('field-postcode')?.value || '';
        const city = document.getElementById('field-city')?.value || '';
        
        if (!country) return;
        
        try {
            const response = await fetch(apiUrl + 'shipping/calculate', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ country, state, postcode, city })
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                shippingMethods = result.shipping.methods || [];
                renderShippingMethods();
                renderOrderSummary();
            }
        } catch (error) {
            console.error('Failed to calculate shipping');
        }
    }
    
    function renderShippingMethods() {
        const container = document.getElementById('cartt-shipping-methods');
        if (!container) return;
        
        if (shippingMethods.length === 0) {
            container.style.display = 'none';
            return;
        }
        
        if (shippingMethods.length === 1) {
            container.style.display = 'none';
            return;
        }
        
        let html = '<div class="cartt-shipping-title">Shipping Method</div>';
        shippingMethods.forEach(method => {
            const isSelected = cart.shipping_method && cart.shipping_method.id === method.id;
            const costText = method.cost > 0 ? formatCurrency(method.cost) : 'Free';
            html += `
                <label class="cartt-shipping-option ${isSelected ? 'selected' : ''}">
                    <input type="radio" name="shipping_method" value="${method.id}" ${isSelected ? 'checked' : ''}>
                    <span class="cartt-shipping-name">${method.name}</span>
                    <span class="cartt-shipping-cost">${costText}</span>
                </label>
            `;
        });
        
        container.innerHTML = html;
        container.style.display = 'block';
        
        // Add listeners
        container.querySelectorAll('input[name="shipping_method"]').forEach(input => {
            input.addEventListener('change', () => selectShippingMethod(parseInt(input.value)));
        });
    }
    
    async function selectShippingMethod(methodId) {
        try {
            const response = await fetch(apiUrl + 'shipping/select', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ method_id: methodId })
            });
            
            const result = await response.json();
            
            if (result.success) {
                cart = result.cart;
                renderShippingMethods();
                renderOrderSummary();
            }
        } catch (error) {
            console.error('Failed to select shipping method');
        }
    }

    function initStripe() {
        if (!hasStripe) return;
        
        stripe = Stripe(stripeKey);
        elements = stripe.elements();
        
        cardElement = elements.create('card', {
            style: {
                base: {
                    fontSize: '15px',
                    fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
                    color: '#1d1d1f',
                    '::placeholder': { color: '#86868b' }
                }
            }
        });
        
        const cardContainer = document.getElementById('cartt-card-element');
        if (cardContainer) {
            cardElement.mount('#cartt-card-element');
            
            cardElement.on('change', function(event) {
                const errorEl = document.getElementById('cartt-card-errors');
                if (errorEl) {
                    errorEl.textContent = event.error ? event.error.message : '';
                }
            });
        }

        // Express payments (Apple Pay, Google Pay)
        if (expressEnabled && cart) {
            initExpressPayments();
        }
    }

    function initExpressPayments() {
        if (!cart || !cart.totals) return;

        paymentRequest = stripe.paymentRequest({
            country: 'US',
            currency: 'usd',
            total: {
                label: 'Total',
                amount: Math.round(cart.totals.total * 100),
            },
            requestPayerName: true,
            requestPayerEmail: true,
        });

        const prButton = elements.create('paymentRequestButton', {
            paymentRequest: paymentRequest,
        });

        paymentRequest.canMakePayment().then(function(result) {
            if (result) {
                prButton.mount('#cartt-payment-request-button');
            } else {
                document.querySelector('.cartt-express-section')?.remove();
            }
        });

        paymentRequest.on('paymentmethod', async function(ev) {
            // Create order and confirm payment
            const response = await fetch(apiUrl + 'checkout', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({
                    billing: {
                        email: ev.payerEmail,
                        first_name: ev.payerName?.split(' ')[0] || '',
                        last_name: ev.payerName?.split(' ').slice(1).join(' ') || '',
                    },
                    payment_method: 'stripe'
                })
            });
            const result = await response.json();

            if (result.error) {
                ev.complete('fail');
                return;
            }

            const { error, paymentIntent } = await stripe.confirmCardPayment(
                result.client_secret,
                { payment_method: ev.paymentMethod.id },
                { handleActions: false }
            );

            if (error) {
                ev.complete('fail');
            } else {
                ev.complete('success');
                window.location.href = result.redirect || '/?order=' + result.order_id;
            }
        });
    }

    function initPayPal() {
        if (!hasPaypal) return;
        const container = document.getElementById('paypal-button-container');
        if (!container) return;
        
        paypal.Buttons({
            createOrder: async function() {
                const response = await fetch(apiUrl + 'paypal/create-order', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': nonce
                    }
                });
                const data = await response.json();
                if (!data.success) {
                    throw new Error(data.error || 'Failed to create PayPal order');
                }
                return data.order_id;
            },
            onApprove: async function(data) {
                const submitBtn = document.getElementById('cartt-submit-btn');
                submitBtn.disabled = true;
                submitBtn.textContent = 'Processing...';

                try {
                    const formData = collectFormData();
                    const response = await fetch(apiUrl + 'paypal/capture', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-WP-Nonce': nonce
                        },
                        body: JSON.stringify({
                            paypal_order_id: data.orderID,
                            billing: formData
                        })
                    });
                    const result = await response.json();
                    
                    if (result.success) {
                        window.location.href = result.redirect;
                    } else {
                        throw new Error(result.error || 'Payment failed');
                    }
                } catch (error) {
                    document.getElementById('cartt-checkout-error').textContent = error.message;
                    document.getElementById('cartt-checkout-error').style.display = 'block';
                    submitBtn.disabled = false;
                    submitBtn.textContent = 'Pay ' + formatCurrency(cart.totals.total);
                }
            }
        }).render('#paypal-button-container');
    }

    function initPaymentMethodToggle() {
        const radios = document.querySelectorAll('input[name="payment_method"]');
        const stripeContainer = document.getElementById('cartt-stripe-container');
        const paypalContainer = document.getElementById('cartt-paypal-container');
        const submitBtn = document.getElementById('cartt-submit-btn');

        radios.forEach(radio => {
            radio.addEventListener('change', function() {
                if (this.value === 'stripe') {
                    if (stripeContainer) stripeContainer.style.display = 'block';
                    if (paypalContainer) paypalContainer.style.display = 'none';
                    if (submitBtn) submitBtn.style.display = 'block';
                } else if (this.value === 'paypal') {
                    if (stripeContainer) stripeContainer.style.display = 'none';
                    if (paypalContainer) paypalContainer.style.display = 'block';
                    if (submitBtn) submitBtn.style.display = 'none';
                }
            });
        });
    }

    function initAddressAutocomplete() {
        if (!autocompleteEnabled) return;
        
        const addressField = document.querySelector('[data-autocomplete="address"]');
        if (!addressField || typeof google === 'undefined') return;

        const autocomplete = new google.maps.places.Autocomplete(addressField, {
            types: ['address'],
            fields: ['address_components', 'formatted_address']
        });

        autocomplete.addListener('place_changed', function() {
            const place = autocomplete.getPlace();
            if (!place.address_components) return;

            const components = {};
            place.address_components.forEach(c => {
                components[c.types[0]] = c.long_name;
                components[c.types[0] + '_short'] = c.short_name;
            });

            // Fill in fields
            const setField = (id, value) => {
                const el = document.getElementById('field-' + id);
                if (el && value) el.value = value;
            };

            setField('address_1', (components.street_number || '') + ' ' + (components.route || ''));
            setField('city', components.locality || components.sublocality_level_1 || '');
            setField('state', components.administrative_area_level_1_short || '');
            setField('postcode', components.postal_code || '');
            setField('country', components.country_short || '');
        });
    }

    function initMultiStep() {
        if (layout !== 'multi-step') return;

        const steps = document.querySelectorAll('.cartt-step');
        const sections = document.querySelectorAll('[data-step]');
        const prevBtn = document.getElementById('cartt-prev-step');
        const nextBtn = document.getElementById('cartt-next-step');
        const submitBtn = document.getElementById('cartt-submit-btn');

        function showStep(step) {
            currentStep = step;
            
            steps.forEach((s, i) => {
                s.classList.toggle('active', i + 1 <= step);
                s.classList.toggle('current', i + 1 === step);
            });

            sections.forEach(s => {
                s.style.display = parseInt(s.dataset.step) === step ? 'block' : 'none';
            });

            prevBtn.style.display = step > 1 ? 'inline-flex' : 'none';
            
            if (step === 3) {
                nextBtn.style.display = 'none';
                submitBtn.style.display = 'block';
                submitBtn.disabled = false;
            } else {
                nextBtn.style.display = 'inline-flex';
                submitBtn.style.display = 'none';
            }
        }

        prevBtn.addEventListener('click', () => showStep(currentStep - 1));
        nextBtn.addEventListener('click', () => {
            // Validate current step
            const currentSection = document.querySelector(`[data-step="${currentStep}"]`);
            const inputs = currentSection.querySelectorAll('[required]');
            let valid = true;
            inputs.forEach(input => {
                if (!input.value) {
                    input.classList.add('error');
                    valid = false;
                } else {
                    input.classList.remove('error');
                }
            });

            if (valid) {
                if (currentStep === 1) captureEmail();
                showStep(currentStep + 1);
            }
        });

        showStep(1);
    }

    async function handleSubmit(e) {
        e.preventDefault();
        
        const paymentMethod = getSelectedPaymentMethod();
        if (paymentMethod === 'paypal') return;
        
        const submitBtn = document.getElementById('cartt-submit-btn');
        const errorEl = document.getElementById('cartt-checkout-error');
        
        submitBtn.disabled = true;
        submitBtn.textContent = 'Processing...';
        errorEl.style.display = 'none';

        const formData = collectFormData();

        try {
            const response = await fetch(apiUrl + 'checkout', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({
                    billing: formData,
                    payment_method: paymentMethod
                })
            });

            const result = await response.json();

            if (result.error) {
                throw new Error(result.error);
            }

            if (!hasStripe || !result.requires_action) {
                window.location.href = result.redirect || '/?order=' + result.order_id;
                return;
            }

            const { error, paymentIntent } = await stripe.confirmCardPayment(result.client_secret, {
                payment_method: {
                    card: cardElement,
                    billing_details: {
                        name: (formData.first_name || '') + ' ' + (formData.last_name || ''),
                        email: formData.email,
                        address: {
                            line1: formData.address_1,
                            city: formData.city,
                            state: formData.state,
                            postal_code: formData.postcode,
                            country: formData.country,
                        }
                    }
                }
            });

            if (error) {
                throw new Error(error.message);
            }

            const confirmResponse = await fetch(apiUrl + 'checkout/confirm', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({
                    order_id: result.order_id,
                    payment_intent_id: paymentIntent.id
                })
            });

            const confirmResult = await confirmResponse.json();

            if (confirmResult.error) {
                throw new Error(confirmResult.error);
            }

            window.location.href = result.redirect || '/?order=' + result.order_id;

        } catch (error) {
            errorEl.textContent = error.message;
            errorEl.style.display = 'block';
            submitBtn.disabled = false;
            submitBtn.textContent = 'Pay ' + formatCurrency(cart.totals.total);
        }
    }

    // Email blur handler for abandoned cart
    document.addEventListener('DOMContentLoaded', function() {
        const emailField = document.getElementById('field-email');
        if (emailField) {
            emailField.addEventListener('blur', captureEmail);
        }
    });

    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
        loadCart().then(() => {
            initStripe();
            initPayPal();
            initPaymentMethodToggle();
            initAddressAutocomplete();
            initMultiStep();
        });
        
        document.getElementById('cartt-checkout-form').addEventListener('submit', handleSubmit);
        
        // Coupon handlers
        document.getElementById('cartt-apply-coupon').addEventListener('click', applyCoupon);
        document.getElementById('cartt-coupon-code').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                applyCoupon();
            }
        });
        document.getElementById('cartt-remove-coupon').addEventListener('click', removeCoupon);

        // Gift Card handlers
        const applyGcBtn = document.getElementById('cartt-apply-gift-card');
        if (applyGcBtn) {
            applyGcBtn.addEventListener('click', applyGiftCard);
            document.getElementById('cartt-gift-card-code').addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    applyGiftCard();
                }
            });
            document.getElementById('cartt-remove-gift-card').addEventListener('click', removeGiftCard);
        }

        // Loyalty Points handlers
        const applyLoyaltyBtn = document.getElementById('cartt-apply-loyalty');
        if (applyLoyaltyBtn) {
            applyLoyaltyBtn.addEventListener('click', applyLoyaltyPoints);
            document.getElementById('cartt-loyalty-points').addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    applyLoyaltyPoints();
                }
            });
            document.getElementById('cartt-remove-loyalty').addEventListener('click', removeLoyaltyPoints);
        }
        
        // Shipping calculation on address change
        ['field-country', 'field-state', 'field-postcode'].forEach(id => {
            const el = document.getElementById(id);
            if (el) {
                el.addEventListener('change', calculateShipping);
                if (id === 'field-postcode') {
                    el.addEventListener('blur', calculateShipping);
                }
            }
        });
    });
})();
</script>

<style>
.cartt-checkout { max-width: 1000px; margin: 0 auto; padding: 32px 20px; }
.cartt-checkout h1 { font-size: 24px; font-weight: 600; margin: 0 0 24px; }

/* Multi-step indicators */
.cartt-checkout-steps { display: flex; justify-content: center; gap: 40px; margin-bottom: 32px; }
.cartt-step { display: flex; align-items: center; gap: 8px; color: #86868b; font-size: 13px; }
.cartt-step span { display: flex; align-items: center; justify-content: center; width: 24px; height: 24px; border: 2px solid #d2d2d7; border-radius: 50%; font-weight: 600; font-size: 12px; }
.cartt-step.active { color: #1d1d1f; }
.cartt-step.active span { border-color: #1d1d1f; background: #1d1d1f; color: #fff; }
.cartt-step.current span { background: #fff; color: #1d1d1f; }

.cartt-checkout-grid { display: grid; grid-template-columns: 1fr 380px; gap: 40px; }
@media (max-width: 900px) { .cartt-checkout-grid { grid-template-columns: 1fr; } }

.cartt-checkout-section { background: #fff; border: 1px solid #e8e8ed; border-radius: 12px; padding: 24px; margin-bottom: 16px; }
.cartt-checkout-section h2 { font-size: 15px; font-weight: 600; margin: 0 0 20px; color: #1d1d1f; }

/* Fields Grid */
.cartt-fields-grid { display: flex; flex-wrap: wrap; gap: 12px; }
.cartt-form-row { margin-bottom: 0; }
.cartt-form-row.cartt-field-full { width: 100%; }
.cartt-form-row.cartt-field-half { width: calc(50% - 6px); }
.cartt-form-row.cartt-field-third { width: calc(33.333% - 8px); }
.cartt-form-row.cartt-field-quarter { width: calc(25% - 9px); }
@media (max-width: 600px) {
    .cartt-form-row.cartt-field-half,
    .cartt-form-row.cartt-field-third,
    .cartt-form-row.cartt-field-quarter { width: 100%; }
}

.cartt-form-row label { display: block; font-size: 13px; font-weight: 500; margin-bottom: 6px; color: #1d1d1f; }
.cartt-form-row label .required { color: #c62828; }
.cartt-form-row input,
.cartt-form-row select,
.cartt-form-row textarea { width: 100%; padding: 12px 14px; font-size: 15px; border: 1px solid #d2d2d7; border-radius: 8px; transition: border-color 0.15s; }
.cartt-form-row input:focus,
.cartt-form-row select:focus,
.cartt-form-row textarea:focus { border-color: #1d1d1f; outline: none; }
.cartt-form-row input.error { border-color: #c62828; }

/* Express Checkout */
.cartt-express-section { text-align: center; }
.cartt-express-divider { display: flex; align-items: center; margin-top: 20px; color: #86868b; font-size: 13px; }
.cartt-express-divider::before,
.cartt-express-divider::after { content: ''; flex: 1; height: 1px; background: #e8e8ed; }
.cartt-express-divider span { padding: 0 16px; }

/* Payment Methods */
.cartt-payment-methods { display: flex; gap: 12px; margin-bottom: 20px; }
.cartt-payment-method { flex: 1; cursor: pointer; }
.cartt-payment-method input { display: none; }
.cartt-payment-method span { display: block; padding: 14px 20px; border: 2px solid #e8e8ed; border-radius: 8px; text-align: center; font-size: 14px; font-weight: 500; transition: all 0.15s; }
.cartt-payment-method input:checked + span { border-color: #1d1d1f; background: #fafafa; }
.cartt-payment-method span:hover { border-color: #1d1d1f; }

#cartt-card-element { padding: 14px; border: 1px solid #d2d2d7; border-radius: 8px; background: #fff; }
.cartt-error-message { color: #c62828; font-size: 13px; margin-top: 8px; }
.cartt-demo-payment { padding: 20px; background: #f5f5f7; border-radius: 8px; text-align: center; }
.cartt-demo-payment p { margin: 0; color: #86868b; font-size: 14px; }

/* Step Navigation */
.cartt-step-nav { display: flex; justify-content: space-between; margin-top: 16px; }
.cartt-btn { display: inline-flex; align-items: center; justify-content: center; height: 48px; padding: 0 32px; font-size: 15px; font-weight: 600; border-radius: 8px; cursor: pointer; transition: all 0.15s; }
.cartt-btn-primary { background: #1d1d1f; color: #fff; border: none; }
.cartt-btn-primary:hover { background: #000; }
.cartt-btn-secondary { background: #fff; color: #1d1d1f; border: 1px solid #d2d2d7; }
.cartt-btn-secondary:hover { border-color: #1d1d1f; }

/* Order Summary */
.cartt-order-summary { background: #f5f5f7; border-radius: 12px; padding: 24px; position: sticky; top: 20px; }
.cartt-order-summary h2 { font-size: 15px; font-weight: 600; margin: 0 0 20px; }

.cartt-order-item { display: flex; align-items: center; gap: 12px; padding: 12px 0; border-bottom: 1px solid #e8e8ed; }
.cartt-order-item:last-child { border-bottom: none; }
.cartt-order-item-image { width: 48px; height: 48px; border-radius: 6px; object-fit: cover; background: #fff; }
.cartt-order-item-details { flex: 1; min-width: 0; }
.cartt-order-item-name { display: block; font-size: 14px; font-weight: 500; }
.cartt-order-item-qty { display: block; font-size: 12px; color: #86868b; }
.cartt-order-item-price { font-size: 14px; font-weight: 500; }

.cartt-order-bumps { margin-top: 16px; }
.cartt-order-bump { display: flex; align-items: flex-start; gap: 10px; background: #fff; border: 2px dashed #d2d2d7; border-radius: 8px; padding: 12px; margin-bottom: 8px; cursor: pointer; transition: all 0.15s; }
.cartt-order-bump:hover { border-color: #1d1d1f; }
.cartt-order-bump.selected { border-style: solid; border-color: #2e7d32; background: #e8f5e9; }
.cartt-bump-checkbox { flex-shrink: 0; padding-top: 2px; }
.cartt-bump-checkbox input { width: 16px; height: 16px; margin: 0; cursor: pointer; }
.cartt-bump-content { flex: 1; min-width: 0; }
.cartt-bump-title { font-size: 13px; font-weight: 600; color: #1d1d1f; }
.cartt-bump-description { font-size: 12px; color: #86868b; margin-top: 2px; }
.cartt-bump-price { font-size: 14px; font-weight: 600; color: #2e7d32; white-space: nowrap; }

/* Coupon */
.cartt-coupon-section { margin-top: 16px; padding-top: 16px; border-top: 1px solid #e8e8ed; }
.cartt-coupon-form { display: flex; gap: 8px; }
.cartt-coupon-form input { flex: 1; padding: 10px 12px; font-size: 13px; border: 1px solid #d2d2d7; border-radius: 6px; text-transform: uppercase; }
.cartt-coupon-form input:focus { border-color: #1d1d1f; outline: none; }
.cartt-coupon-form button { padding: 10px 16px; font-size: 13px; font-weight: 500; background: #fff; border: 1px solid #d2d2d7; border-radius: 6px; cursor: pointer; white-space: nowrap; }
.cartt-coupon-form button:hover { border-color: #1d1d1f; }
.cartt-coupon-applied { display: flex; align-items: center; justify-content: space-between; background: #e8f5e9; padding: 10px 12px; border-radius: 6px; }
.cartt-coupon-applied span { font-family: monospace; font-size: 13px; font-weight: 600; color: #2e7d32; }
.cartt-coupon-applied button { background: none; border: none; font-size: 18px; color: #2e7d32; cursor: pointer; padding: 0 4px; }
.cartt-coupon-applied button:hover { color: #1b5e20; }
.cartt-coupon-error { color: #c62828; font-size: 12px; margin-top: 6px; }
.cartt-discount-row { color: #2e7d32; }

/* Gift Card */
.cartt-gift-card-section { margin-top: 12px; }
.cartt-gift-card-form { display: flex; gap: 8px; }
.cartt-gift-card-form input { flex: 1; padding: 10px 12px; font-size: 13px; border: 1px solid #d2d2d7; border-radius: 6px; text-transform: uppercase; }
.cartt-gift-card-form input:focus { border-color: #1d1d1f; outline: none; }
.cartt-gift-card-form button { padding: 10px 16px; font-size: 13px; font-weight: 500; background: #fff; border: 1px solid #d2d2d7; border-radius: 6px; cursor: pointer; white-space: nowrap; }
.cartt-gift-card-form button:hover { border-color: #1d1d1f; }
.cartt-gift-card-applied { display: flex; align-items: center; justify-content: space-between; background: #e3f2fd; padding: 10px 12px; border-radius: 6px; }
.cartt-gift-card-applied span { font-family: monospace; font-size: 13px; font-weight: 600; color: #1565c0; }
.cartt-gift-card-applied button { background: none; border: none; font-size: 18px; color: #1565c0; cursor: pointer; padding: 0 4px; }
.cartt-gift-card-applied button:hover { color: #0d47a1; }
.cartt-gift-card-error { color: #c62828; font-size: 12px; margin-top: 6px; }
.cartt-gift-card-row { color: #1565c0; }

/* Loyalty Points */
.cartt-loyalty-section { margin-top: 12px; padding: 12px; background: linear-gradient(135deg, #ffd700 0%, #ffb300 100%); border-radius: 8px; }
.cartt-loyalty-balance { display: flex; align-items: center; gap: 8px; font-size: 13px; color: #333; margin-bottom: 8px; }
.cartt-loyalty-balance strong { font-size: 16px; }
.cartt-loyalty-balance small { opacity: 0.7; }
.cartt-loyalty-form { display: flex; gap: 8px; }
.cartt-loyalty-form input { flex: 1; padding: 10px 12px; font-size: 13px; border: 1px solid rgba(0,0,0,0.2); border-radius: 6px; background: rgba(255,255,255,0.9); }
.cartt-loyalty-form input:focus { border-color: rgba(0,0,0,0.4); outline: none; }
.cartt-loyalty-form button { padding: 10px 16px; font-size: 13px; font-weight: 600; background: #333; color: #fff; border: none; border-radius: 6px; cursor: pointer; white-space: nowrap; }
.cartt-loyalty-form button:hover { background: #000; }
.cartt-loyalty-applied { display: flex; align-items: center; justify-content: space-between; background: rgba(255,255,255,0.9); padding: 10px 12px; border-radius: 6px; }
.cartt-loyalty-applied span { font-size: 13px; font-weight: 600; color: #333; }
.cartt-loyalty-applied button { background: none; border: none; font-size: 18px; color: #333; cursor: pointer; padding: 0 4px; }
.cartt-loyalty-error { color: #c62828; font-size: 12px; margin-top: 6px; background: #fff; padding: 4px 8px; border-radius: 4px; }
.cartt-loyalty-min-notice { display: block; font-size: 11px; opacity: 0.7; margin-top: 4px; }
.cartt-loyalty-row { color: #f57c00; }

.cartt-order-totals { border-top: 1px solid #d2d2d7; margin-top: 20px; padding-top: 20px; }
.cartt-totals-row { display: flex; justify-content: space-between; padding: 6px 0; font-size: 14px; }
.cartt-totals-row.cartt-total { font-size: 17px; font-weight: 600; margin-top: 12px; padding-top: 12px; border-top: 1px solid #d2d2d7; }

.cartt-place-order-btn { width: 100%; padding: 16px; font-size: 16px; font-weight: 600; color: #fff; background: #1d1d1f; border: none; border-radius: 8px; cursor: pointer; margin-top: 20px; transition: background 0.15s; }
.cartt-place-order-btn:hover { background: #000; }
.cartt-place-order-btn:disabled { background: #d2d2d7; cursor: not-allowed; }

.cartt-secure-badge { display: flex; align-items: center; justify-content: center; gap: 6px; margin-top: 16px; font-size: 12px; color: #86868b; }

.cartt-loading { text-align: center; padding: 24px; }
.cartt-spinner { display: inline-block; width: 24px; height: 24px; border: 2px solid #e8e8ed; border-top-color: #1d1d1f; border-radius: 50%; animation: spin 0.6s linear infinite; }
@keyframes spin { to { transform: rotate(360deg); } }

.cartt-empty-cart { color: #86868b; font-size: 14px; text-align: center; padding: 20px; }

/* Shipping Methods */
.cartt-shipping-methods { margin-top: 16px; padding-top: 16px; border-top: 1px solid #e8e8ed; }
.cartt-shipping-title { font-size: 12px; font-weight: 600; color: #86868b; text-transform: uppercase; margin-bottom: 8px; }
.cartt-shipping-option { display: flex; align-items: center; gap: 8px; padding: 10px 12px; background: #f5f5f7; border-radius: 6px; margin-bottom: 6px; cursor: pointer; }
.cartt-shipping-option:hover { background: #e8e8ed; }
.cartt-shipping-option.selected { background: #e8f5e9; }
.cartt-shipping-option input { margin: 0; }
.cartt-shipping-name { flex: 1; font-size: 13px; }
.cartt-shipping-cost { font-size: 13px; font-weight: 500; }
.cartt-shipping-row span:last-child { font-size: 12px; color: #86868b; }

/* Style: Minimal */
.cartt-style-minimal .cartt-checkout-section { border: none; border-radius: 0; padding: 24px 0; border-bottom: 1px solid #e8e8ed; }
.cartt-style-minimal .cartt-form-row input,
.cartt-style-minimal .cartt-form-row select,
.cartt-style-minimal .cartt-form-row textarea { border: none; border-bottom: 1px solid #d2d2d7; border-radius: 0; padding: 12px 0; }
.cartt-style-minimal .cartt-form-row input:focus,
.cartt-style-minimal .cartt-form-row select:focus { border-bottom-color: #1d1d1f; }
.cartt-style-minimal .cartt-order-summary { background: transparent; border: 1px solid #e8e8ed; border-radius: 0; }
.cartt-style-minimal #cartt-card-element { border: none; border-bottom: 1px solid #d2d2d7; border-radius: 0; }

/* Style: Classic */
.cartt-style-classic .cartt-checkout-section { border-radius: 0; }
.cartt-style-classic .cartt-form-row input,
.cartt-style-classic .cartt-form-row select,
.cartt-style-classic .cartt-form-row textarea { border-radius: 0; }
.cartt-style-classic .cartt-place-order-btn { border-radius: 0; }
.cartt-style-classic .cartt-order-summary { border-radius: 0; }
.cartt-style-classic .cartt-payment-method span { border-radius: 0; }
.cartt-style-classic #cartt-card-element { border-radius: 0; }
.cartt-style-classic .cartt-order-item-image { border-radius: 0; }

/* Style: Bold */
.cartt-style-bold .cartt-checkout-section { border-radius: 16px; border-width: 2px; }
.cartt-style-bold .cartt-form-row input,
.cartt-style-bold .cartt-form-row select,
.cartt-style-bold .cartt-form-row textarea { border-radius: 12px; background: #f5f5f7; border: none; padding: 14px 16px; font-size: 16px; }
.cartt-style-bold .cartt-form-row input:focus,
.cartt-style-bold .cartt-form-row select:focus { background: #fff; box-shadow: 0 0 0 2px #1d1d1f; }
.cartt-style-bold .cartt-place-order-btn { border-radius: 12px; padding: 18px; font-size: 17px; }
.cartt-style-bold .cartt-order-summary { border-radius: 16px; }
.cartt-style-bold .cartt-payment-method span { border-radius: 12px; border-width: 2px; padding: 16px; }
.cartt-style-bold #cartt-card-element { border-radius: 12px; background: #f5f5f7; border: none; padding: 16px; }
.cartt-style-bold .cartt-order-item-image { border-radius: 10px; }
.cartt-style-bold .cartt-checkout-section h2 { font-size: 17px; }

/* Custom Colors */
.cartt-checkout .cartt-place-order-btn { 
    background: <?php echo esc_attr($btnColor); ?>; 
    color: <?php echo esc_attr($btnTextColor); ?>; 
}
.cartt-checkout .cartt-place-order-btn:hover { 
    background: <?php echo esc_attr($btnColor); ?>; 
    filter: brightness(0.9);
}
.cartt-checkout .cartt-form-row input,
.cartt-checkout .cartt-form-row select,
.cartt-checkout .cartt-form-row textarea { 
    border-color: <?php echo esc_attr($borderColor); ?>; 
}
.cartt-checkout .cartt-form-row input:focus,
.cartt-checkout .cartt-form-row select:focus { 
    border-color: <?php echo esc_attr($accentColor); ?>; 
}
.cartt-checkout .cartt-checkout-section { 
    border-color: <?php echo esc_attr($borderColor); ?>; 
}
.cartt-checkout .cartt-payment-method input:checked + span { 
    border-color: <?php echo esc_attr($accentColor); ?>; 
}
.cartt-style-minimal .cartt-place-order-btn { 
    background: transparent; 
    color: <?php echo esc_attr($btnColor); ?>; 
    border: 2px solid <?php echo esc_attr($btnColor); ?>; 
}
.cartt-style-minimal .cartt-place-order-btn:hover { 
    background: <?php echo esc_attr($btnColor); ?>; 
    color: <?php echo esc_attr($btnTextColor); ?>; 
    filter: none;
}
</style>
