<?php
/**
 * Cart Template - Server-side cart
 */

if (!defined('ABSPATH')) {
    exit;
}

use Cartt\Services\CurrencyService;

$checkout_page_id = get_option('cartt_checkout_page_id');
$shop_page_id = get_option('cartt_shop_page_id');
$checkout_url = $checkout_page_id ? get_permalink($checkout_page_id) : home_url('/checkout/');
$shop_url = $shop_page_id ? get_permalink($shop_page_id) : home_url('/shop/');

// Get current currency
$currency = CurrencyService::getCurrentCurrency();
?>

<div class="cartt-cart">
    <h1><?php esc_html_e('Cart', 'cartt'); ?></h1>
    
    <div id="cartt-cart-app">
        <div class="cartt-loading"><div class="cartt-spinner"></div></div>
    </div>
</div>

<script>
(function() {
    const apiUrl = '<?php echo esc_js(rest_url('cartt/v1/')); ?>';
    const nonce = '<?php echo esc_js(wp_create_nonce('wp_rest')); ?>';
    const checkoutUrl = '<?php echo esc_js($checkout_url); ?>';
    const shopUrl = '<?php echo esc_js($shop_url); ?>';
    
    // Currency settings
    const currencySymbol = '<?php echo esc_js($currency->symbol ?? '$'); ?>';
    const currencyPosition = '<?php echo esc_js($currency->symbol_position ?? 'left'); ?>';
    const exchangeRate = <?php echo floatval($currency->exchange_rate ?? 1); ?>;
    const currencyDecimals = <?php echo intval($currency->decimals ?? 2); ?>;
    const thousandSep = '<?php echo esc_js($currency->thousand_separator ?? ','); ?>';
    const decimalSep = '<?php echo esc_js($currency->decimal_separator ?? '.'); ?>';

    function formatCurrency(amount) {
        // Convert using exchange rate
        let converted = parseFloat(amount) * exchangeRate;
        
        // Format number
        let formatted = converted.toFixed(currencyDecimals);
        let parts = formatted.split('.');
        parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousandSep);
        if (parts[1]) {
            formatted = parts[0] + decimalSep + parts[1];
        } else {
            formatted = parts[0];
        }
        
        // Apply symbol position
        switch (currencyPosition) {
            case 'right': return formatted + currencySymbol;
            case 'left_space': return currencySymbol + ' ' + formatted;
            case 'right_space': return formatted + ' ' + currencySymbol;
            default: return currencySymbol + formatted;
        }
    }

    async function loadCart() {
        try {
            const response = await fetch(apiUrl + 'cart', {
                headers: { 'X-WP-Nonce': nonce }
            });
            const data = await response.json();
            renderCart(data.data);
        } catch (error) {
            console.error('Failed to load cart:', error);
            document.getElementById('cartt-cart-app').innerHTML = '<p>Failed to load cart.</p>';
        }
    }

    function renderCart(cart) {
        const container = document.getElementById('cartt-cart-app');
        
        if (!cart || Object.keys(cart.items).length === 0) {
            container.innerHTML = `
                <div class="cartt-cart-empty">
                    <div class="cartt-empty-icon">
                        <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                            <circle cx="9" cy="21" r="1"/><circle cx="20" cy="21" r="1"/>
                            <path d="M1 1h4l2.68 13.39a2 2 0 002 1.61h9.72a2 2 0 002-1.61L23 6H6"/>
                        </svg>
                    </div>
                    <h2>Your cart is empty</h2>
                    <p>Add some products to get started.</p>
                    <a href="${shopUrl}" class="cartt-btn cartt-btn-primary">Continue Shopping</a>
                </div>
            `;
            return;
        }

        let itemsHtml = '';
        for (const key in cart.items) {
            const item = cart.items[key];
            itemsHtml += `
                <div class="cartt-cart-item" data-key="${key}">
                    <div class="cartt-cart-item-image">
                        ${item.image ? `<img src="${item.image}" alt="">` : ''}
                    </div>
                    <div class="cartt-cart-item-details">
                        <div class="cartt-cart-item-name">${item.name}</div>
                        <div class="cartt-cart-item-price">${formatCurrency(item.price)}</div>
                    </div>
                    <div class="cartt-cart-item-quantity">
                        <button type="button" class="cartt-qty-btn" data-action="decrease" data-key="${key}">-</button>
                        <span class="cartt-qty-value">${item.quantity}</span>
                        <button type="button" class="cartt-qty-btn" data-action="increase" data-key="${key}">+</button>
                    </div>
                    <div class="cartt-cart-item-total">
                        ${formatCurrency(item.price * item.quantity)}
                    </div>
                    <button type="button" class="cartt-cart-item-remove" data-key="${key}">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/>
                        </svg>
                    </button>
                </div>
            `;
        }

        container.innerHTML = `
            <div class="cartt-cart-items">${itemsHtml}</div>
            <div class="cartt-cart-footer">
                <div class="cartt-cart-totals">
                    <div class="cartt-totals-row">
                        <span>Subtotal</span>
                        <span>${formatCurrency(cart.totals.subtotal)}</span>
                    </div>
                    ${cart.totals.tax > 0 ? `
                    <div class="cartt-totals-row">
                        <span>Tax</span>
                        <span>${formatCurrency(cart.totals.tax)}</span>
                    </div>
                    ` : ''}
                    <div class="cartt-totals-row cartt-total">
                        <span>Total</span>
                        <span>${formatCurrency(cart.totals.total)}</span>
                    </div>
                </div>
                <div class="cartt-cart-actions">
                    <a href="${shopUrl}" class="cartt-btn cartt-btn-secondary">Continue Shopping</a>
                    <a href="${checkoutUrl}" class="cartt-btn cartt-btn-primary">Checkout</a>
                </div>
            </div>
        `;

        bindEvents();
    }

    async function updateQuantity(key, change) {
        const item = document.querySelector(`.cartt-cart-item[data-key="${key}"]`);
        const qtySpan = item.querySelector('.cartt-qty-value');
        let newQty = parseInt(qtySpan.textContent) + change;
        
        if (newQty < 1) {
            await removeItem(key);
            return;
        }

        try {
            const response = await fetch(apiUrl + 'cart/update', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ item_key: key, quantity: newQty })
            });
            const data = await response.json();
            renderCart(data.cart);
        } catch (error) {
            console.error('Failed to update cart:', error);
        }
    }

    async function removeItem(key) {
        try {
            const response = await fetch(apiUrl + 'cart/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': nonce
                },
                body: JSON.stringify({ item_key: key })
            });
            const data = await response.json();
            renderCart(data.cart);
        } catch (error) {
            console.error('Failed to remove item:', error);
        }
    }

    function bindEvents() {
        document.querySelectorAll('.cartt-qty-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const key = this.dataset.key;
                const change = this.dataset.action === 'increase' ? 1 : -1;
                updateQuantity(key, change);
            });
        });

        document.querySelectorAll('.cartt-cart-item-remove').forEach(btn => {
            btn.addEventListener('click', function() {
                removeItem(this.dataset.key);
            });
        });
    }

    document.addEventListener('DOMContentLoaded', loadCart);
})();
</script>

<style>
.cartt-cart { max-width: 800px; margin: 0 auto; padding: 32px 20px; }
.cartt-cart h1 { font-size: 24px; font-weight: 600; margin: 0 0 24px; }

.cartt-cart-empty { text-align: center; padding: 60px 20px; }
.cartt-empty-icon { color: #d2d2d7; margin-bottom: 16px; }
.cartt-cart-empty h2 { font-size: 18px; font-weight: 600; margin: 0 0 8px; }
.cartt-cart-empty p { color: #86868b; margin: 0 0 20px; }

.cartt-cart-item { display: grid; grid-template-columns: 60px 1fr auto auto 32px; gap: 16px; align-items: center; padding: 16px 0; border-bottom: 1px solid #e8e8ed; }
.cartt-cart-item-image { width: 60px; height: 60px; background: #f5f5f7; border-radius: 6px; overflow: hidden; }
.cartt-cart-item-image img { width: 100%; height: 100%; object-fit: cover; }
.cartt-cart-item-name { font-weight: 500; font-size: 14px; }
.cartt-cart-item-price { font-size: 13px; color: #86868b; margin-top: 2px; }

.cartt-cart-item-quantity { display: flex; align-items: center; gap: 8px; }
.cartt-qty-btn { width: 28px; height: 28px; border: 1px solid #d2d2d7; border-radius: 4px; background: #fff; cursor: pointer; font-size: 16px; }
.cartt-qty-btn:hover { border-color: #1d1d1f; }
.cartt-qty-value { width: 24px; text-align: center; font-size: 14px; }

.cartt-cart-item-total { font-weight: 600; font-size: 14px; min-width: 70px; text-align: right; }
.cartt-cart-item-remove { background: none; border: none; cursor: pointer; color: #86868b; padding: 4px; }
.cartt-cart-item-remove:hover { color: #c62828; }

.cartt-cart-footer { margin-top: 24px; }
.cartt-cart-totals { background: #f5f5f7; border-radius: 8px; padding: 20px; margin-bottom: 20px; }
.cartt-totals-row { display: flex; justify-content: space-between; padding: 6px 0; font-size: 14px; }
.cartt-totals-row.cartt-total { font-size: 16px; font-weight: 600; border-top: 1px solid #d2d2d7; margin-top: 8px; padding-top: 12px; }

.cartt-cart-actions { display: flex; justify-content: space-between; gap: 12px; }
.cartt-btn { display: inline-flex; align-items: center; justify-content: center; height: 44px; padding: 0 24px; font-size: 14px; font-weight: 500; border-radius: 6px; text-decoration: none; cursor: pointer; }
.cartt-btn-primary { background: #1d1d1f; color: #fff; border: none; }
.cartt-btn-primary:hover { background: #000; color: #fff; }
.cartt-btn-secondary { background: #fff; color: #1d1d1f; border: 1px solid #d2d2d7; }
.cartt-btn-secondary:hover { border-color: #1d1d1f; color: #1d1d1f; }

.cartt-loading { text-align: center; padding: 40px; }
.cartt-spinner { display: inline-block; width: 24px; height: 24px; border: 2px solid #e8e8ed; border-top-color: #1d1d1f; border-radius: 50%; animation: spin 0.6s linear infinite; }
@keyframes spin { to { transform: rotate(360deg); } }

@media (max-width: 600px) {
    .cartt-cart-item { grid-template-columns: 50px 1fr; gap: 12px; }
    .cartt-cart-item-quantity { grid-column: 2; }
    .cartt-cart-item-total { grid-column: 2; }
    .cartt-cart-item-remove { position: absolute; right: 0; top: 16px; }
    .cartt-cart-actions { flex-direction: column; }
    .cartt-btn { width: 100%; }
}
</style>
