<?php
/**
 * Bulk Order Form Template
 * 
 * @package Cartt
 * @since 1.4.0
 */

defined('ABSPATH') || exit;
?>

<div class="cartt-bulk-order-form">
    <h2>Bulk Order Form</h2>
    
    <?php if ($isWholesale): ?>
    <div class="cartt-notice cartt-notice-info">
        <strong>Wholesale Pricing Active</strong> - You're seeing <?php echo esc_html(ucfirst($tier)); ?> tier pricing.
    </div>
    <?php else: ?>
    <div class="cartt-notice cartt-notice-warning">
        <strong>Retail Pricing</strong> - <a href="<?php echo esc_url(get_permalink(get_option('cartt_wholesale_application_page'))); ?>">Apply for wholesale</a> to access bulk discounts.
    </div>
    <?php endif; ?>

    <form method="post" class="cartt-bulk-form" id="bulk-order-form">
        <?php wp_nonce_field('cartt_bulk_order'); ?>
        
        <table class="cartt-bulk-table">
            <thead>
                <tr>
                    <th style="width: 40%;">Product</th>
                    <th style="width: 15%;">SKU</th>
                    <th style="width: 15%;">Unit Price</th>
                    <?php if ($atts['show_quantity_breaks'] === 'yes'): ?>
                    <th style="width: 15%;">Qty Breaks</th>
                    <?php endif; ?>
                    <th style="width: 10%;">Quantity</th>
                    <th style="width: 10%;">Subtotal</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($products as $product): ?>
                <tr class="product-row" data-product-id="<?php echo esc_attr($product['id']); ?>" data-base-price="<?php echo esc_attr($product['price']); ?>">
                    <td class="product-name">
                        <?php if (!empty($product['image_url'])): ?>
                        <img src="<?php echo esc_url($product['image_url']); ?>" width="40" height="40" style="vertical-align: middle; margin-right: 10px; border-radius: 4px;">
                        <?php endif; ?>
                        <?php echo esc_html($product['name']); ?>
                    </td>
                    <td class="product-sku"><?php echo esc_html($product['sku'] ?? '-'); ?></td>
                    <td class="product-price" data-price="<?php echo esc_attr($product['price']); ?>">
                        <?php echo \Cartt\Services\CurrencyService::formatPrice($product['price']); ?>
                    </td>
                    <?php if ($atts['show_quantity_breaks'] === 'yes'): ?>
                    <td class="quantity-breaks">
                        <?php if (!empty($product['quantity_pricing'])): ?>
                        <button type="button" class="view-breaks-btn" data-product="<?php echo esc_attr($product['id']); ?>">
                            View Breaks
                        </button>
                        <div class="breaks-popup" id="breaks-<?php echo esc_attr($product['id']); ?>" style="display: none;">
                            <table class="breaks-table">
                                <tr><th>Qty</th><th>Price</th></tr>
                                <?php foreach ($product['quantity_pricing'] as $qp): ?>
                                <tr>
                                    <td><?php echo esc_html($qp['min_quantity']); ?>+</td>
                                    <td><?php echo \Cartt\Services\CurrencyService::formatPrice($qp['price']); ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </table>
                        </div>
                        <?php else: ?>
                        <span class="no-breaks">-</span>
                        <?php endif; ?>
                    </td>
                    <?php endif; ?>
                    <td class="product-quantity">
                        <input type="number" name="items[<?php echo esc_attr($product['id']); ?>]" min="0" value="0" 
                               class="qty-input" data-pricing='<?php echo json_encode($product['quantity_pricing'] ?? []); ?>'>
                    </td>
                    <td class="product-subtotal">
                        <span class="subtotal-value">$0.00</span>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr class="order-total-row">
                    <td colspan="<?php echo $atts['show_quantity_breaks'] === 'yes' ? 5 : 4; ?>" style="text-align: right; font-weight: 600;">
                        Order Total:
                    </td>
                    <td class="order-total">
                        <strong id="order-total">$0.00</strong>
                    </td>
                </tr>
            </tfoot>
        </table>

        <div class="bulk-order-actions">
            <button type="button" class="cartt-button cartt-button-secondary" id="clear-quantities">Clear All</button>
            <button type="submit" class="cartt-button cartt-button-primary" id="add-bulk-to-cart">Add All to Cart</button>
        </div>
    </form>
</div>

<style>
.cartt-bulk-order-form { max-width: 1000px; margin: 0 auto; }
.cartt-bulk-table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
.cartt-bulk-table th { background: #f6f7f7; padding: 12px; text-align: left; font-weight: 600; border-bottom: 2px solid #ddd; }
.cartt-bulk-table td { padding: 12px; border-bottom: 1px solid #eee; vertical-align: middle; }
.cartt-bulk-table tbody tr:hover { background: #f9f9f9; }
.qty-input { width: 80px; padding: 8px; border: 1px solid #ddd; border-radius: 4px; text-align: center; }
.product-subtotal { font-weight: 500; }
.order-total-row { background: #f6f7f7; }
.order-total { font-size: 18px; }
.bulk-order-actions { display: flex; gap: 15px; justify-content: flex-end; }
.cartt-button { padding: 12px 24px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px; }
.cartt-button-primary { background: #2271b1; color: #fff; }
.cartt-button-secondary { background: #f0f0f0; color: #333; }
.view-breaks-btn { background: none; border: 1px solid #2271b1; color: #2271b1; padding: 4px 8px; border-radius: 4px; cursor: pointer; font-size: 12px; }
.breaks-popup { position: absolute; background: #fff; border: 1px solid #ddd; border-radius: 4px; padding: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); z-index: 100; }
.breaks-table { font-size: 12px; }
.breaks-table th, .breaks-table td { padding: 4px 8px; }
.quantity-breaks { position: relative; }
.cartt-notice { padding: 15px; border-radius: 6px; margin-bottom: 20px; }
.cartt-notice-info { background: #cce5ff; color: #004085; }
.cartt-notice-warning { background: #fff3cd; color: #856404; }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('bulk-order-form');
    const totalEl = document.getElementById('order-total');
    
    function calculateSubtotal(row) {
        const qty = parseInt(row.querySelector('.qty-input').value) || 0;
        const basePrice = parseFloat(row.dataset.basePrice);
        const pricing = JSON.parse(row.querySelector('.qty-input').dataset.pricing || '[]');
        
        let price = basePrice;
        for (const tier of pricing) {
            if (qty >= tier.min_quantity) {
                price = parseFloat(tier.price);
            }
        }
        
        const subtotal = qty * price;
        row.querySelector('.subtotal-value').textContent = '$' + subtotal.toFixed(2);
        return subtotal;
    }
    
    function updateTotal() {
        let total = 0;
        document.querySelectorAll('.product-row').forEach(row => {
            total += calculateSubtotal(row);
        });
        totalEl.textContent = '$' + total.toFixed(2);
    }
    
    document.querySelectorAll('.qty-input').forEach(input => {
        input.addEventListener('input', updateTotal);
    });
    
    document.querySelectorAll('.view-breaks-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const popup = document.getElementById('breaks-' + this.dataset.product);
            popup.style.display = popup.style.display === 'none' ? 'block' : 'none';
        });
    });
    
    document.getElementById('clear-quantities').addEventListener('click', function() {
        document.querySelectorAll('.qty-input').forEach(input => input.value = 0);
        updateTotal();
    });
    
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        // AJAX add to cart logic would go here
        alert('Items added to cart!');
    });
});
</script>
