<?php
/**
 * Customer Account Template
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!is_user_logged_in()) {
    ?>
    <div class="cartt-account-login">
        <h2><?php esc_html_e('Login', 'cartt'); ?></h2>
        <p><?php esc_html_e('Please log in to view your account.', 'cartt'); ?></p>
        <?php wp_login_form(['redirect' => get_permalink(), 'form_id' => 'cartt-login-form']); ?>
    </div>
    <?php
    return;
}

$user = wp_get_current_user();
$activeTab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'dashboard';

global $wpdb;

// Get or create customer record
$customer = $wpdb->get_row($wpdb->prepare(
    "SELECT * FROM {$wpdb->prefix}cartt_customers WHERE user_id = %d OR email = %s",
    $user->ID, $user->user_email
));

if (!$customer) {
    $wpdb->insert($wpdb->prefix . 'cartt_customers', [
        'user_id' => $user->ID,
        'email' => $user->user_email,
        'first_name' => $user->first_name,
        'last_name' => $user->last_name,
    ]);
    $customer = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}cartt_customers WHERE id = %d", $wpdb->insert_id));
}

// Get orders
$orders = $wpdb->get_results($wpdb->prepare(
    "SELECT * FROM {$wpdb->prefix}cartt_orders WHERE customer_id = %d OR customer_email = %s ORDER BY created_at DESC",
    $customer->id ?? 0, $user->user_email
));

// Get downloads
$downloads = $wpdb->get_results($wpdb->prepare(
    "SELECT dp.*, d.name, d.file_url, d.download_limit, o.order_number
     FROM {$wpdb->prefix}cartt_download_permissions dp
     JOIN {$wpdb->prefix}cartt_downloads d ON dp.download_id = d.id
     JOIN {$wpdb->prefix}cartt_orders o ON dp.order_id = o.id
     WHERE dp.customer_id = %d ORDER BY dp.access_granted DESC",
    $customer->id ?? 0
));

$billing = !empty($customer->billing_address) ? json_decode($customer->billing_address, true) : [];
$shipping = !empty($customer->shipping_address) ? json_decode($customer->shipping_address, true) : [];

// Handle address save
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_account_nonce']) && wp_verify_nonce($_POST['cartt_account_nonce'], 'cartt_save_account')) {
    if (isset($_POST['save_billing'])) {
        $billing = array_map('sanitize_text_field', [
            'first_name' => $_POST['billing_first_name'] ?? '',
            'last_name' => $_POST['billing_last_name'] ?? '',
            'address_1' => $_POST['billing_address_1'] ?? '',
            'city' => $_POST['billing_city'] ?? '',
            'state' => $_POST['billing_state'] ?? '',
            'postcode' => $_POST['billing_postcode'] ?? '',
            'country' => $_POST['billing_country'] ?? '',
            'phone' => $_POST['billing_phone'] ?? '',
        ]);
        $wpdb->update($wpdb->prefix . 'cartt_customers', ['billing_address' => json_encode($billing)], ['id' => $customer->id]);
        $message = __('Billing address saved.', 'cartt');
    }
    if (isset($_POST['save_shipping'])) {
        $shipping = array_map('sanitize_text_field', [
            'first_name' => $_POST['shipping_first_name'] ?? '',
            'last_name' => $_POST['shipping_last_name'] ?? '',
            'address_1' => $_POST['shipping_address_1'] ?? '',
            'city' => $_POST['shipping_city'] ?? '',
            'state' => $_POST['shipping_state'] ?? '',
            'postcode' => $_POST['shipping_postcode'] ?? '',
            'country' => $_POST['shipping_country'] ?? '',
        ]);
        $wpdb->update($wpdb->prefix . 'cartt_customers', ['shipping_address' => json_encode($shipping)], ['id' => $customer->id]);
        $message = __('Shipping address saved.', 'cartt');
    }
}

$accountUrl = get_permalink();
?>

<div class="cartt-account">
    <h1><?php esc_html_e('My Account', 'cartt'); ?></h1>
    
    <?php if ($message): ?>
    <div class="cartt-notice"><?php echo esc_html($message); ?></div>
    <?php endif; ?>
    
    <div class="cartt-account-grid">
        <nav class="cartt-account-nav">
            <a href="<?php echo esc_url(add_query_arg('tab', 'dashboard', $accountUrl)); ?>" class="<?php echo $activeTab === 'dashboard' ? 'active' : ''; ?>"><?php esc_html_e('Dashboard', 'cartt'); ?></a>
            <a href="<?php echo esc_url(add_query_arg('tab', 'orders', $accountUrl)); ?>" class="<?php echo $activeTab === 'orders' ? 'active' : ''; ?>"><?php esc_html_e('Orders', 'cartt'); ?> <span><?php echo count($orders); ?></span></a>
            <a href="<?php echo esc_url(add_query_arg('tab', 'downloads', $accountUrl)); ?>" class="<?php echo $activeTab === 'downloads' ? 'active' : ''; ?>"><?php esc_html_e('Downloads', 'cartt'); ?> <span><?php echo count($downloads); ?></span></a>
            <a href="<?php echo esc_url(add_query_arg('tab', 'addresses', $accountUrl)); ?>" class="<?php echo $activeTab === 'addresses' ? 'active' : ''; ?>"><?php esc_html_e('Addresses', 'cartt'); ?></a>
            <a href="<?php echo esc_url(wp_logout_url(home_url())); ?>"><?php esc_html_e('Logout', 'cartt'); ?></a>
        </nav>

        <div class="cartt-account-content">
            <?php if ($activeTab === 'dashboard'): ?>
            <p><?php printf(esc_html__('Hello %s! From your dashboard you can view orders, downloads, and manage addresses.', 'cartt'), '<strong>' . esc_html($user->display_name) . '</strong>'); ?></p>
            <div class="cartt-stats-row">
                <div class="cartt-stat-box"><span class="num"><?php echo count($orders); ?></span><span class="label"><?php esc_html_e('Orders', 'cartt'); ?></span></div>
                <div class="cartt-stat-box"><span class="num">$<?php echo number_format($customer->total_spent ?? 0, 2); ?></span><span class="label"><?php esc_html_e('Spent', 'cartt'); ?></span></div>
                <div class="cartt-stat-box"><span class="num"><?php echo count($downloads); ?></span><span class="label"><?php esc_html_e('Downloads', 'cartt'); ?></span></div>
            </div>

            <?php elseif ($activeTab === 'orders'): 
                $viewOrder = isset($_GET['view']) ? intval($_GET['view']) : 0;
                if ($viewOrder):
                    $order = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}cartt_orders WHERE id = %d AND (customer_id = %d OR customer_email = %s)", $viewOrder, $customer->id ?? 0, $user->user_email));
                    if ($order):
                        $items = $wpdb->get_results($wpdb->prepare("SELECT * FROM {$wpdb->prefix}cartt_order_items WHERE order_id = %d", $order->id));
            ?>
            <p><a href="<?php echo esc_url(add_query_arg('tab', 'orders', $accountUrl)); ?>">&larr; <?php esc_html_e('Back', 'cartt'); ?></a></p>
            <h2><?php printf(esc_html__('Order #%s', 'cartt'), $order->order_number); ?></h2>
            <p><?php echo esc_html(date_i18n(get_option('date_format'), strtotime($order->created_at))); ?> &bull; <span class="cartt-status cartt-status-<?php echo esc_attr($order->status); ?>"><?php echo esc_html(ucfirst($order->status)); ?></span></p>
            <table class="cartt-table">
                <thead><tr><th><?php esc_html_e('Product', 'cartt'); ?></th><th><?php esc_html_e('Qty', 'cartt'); ?></th><th><?php esc_html_e('Total', 'cartt'); ?></th></tr></thead>
                <tbody>
                <?php foreach ($items as $item): ?>
                <tr><td><?php echo esc_html($item->product_name); ?></td><td><?php echo intval($item->quantity); ?></td><td>$<?php echo number_format($item->total, 2); ?></td></tr>
                <?php endforeach; ?>
                </tbody>
                <tfoot>
                <tr><td colspan="2"><?php esc_html_e('Subtotal', 'cartt'); ?></td><td>$<?php echo number_format($order->subtotal, 2); ?></td></tr>
                <?php if ($order->shipping_total > 0): ?><tr><td colspan="2"><?php esc_html_e('Shipping', 'cartt'); ?></td><td>$<?php echo number_format($order->shipping_total, 2); ?></td></tr><?php endif; ?>
                <?php if ($order->tax_total > 0): ?><tr><td colspan="2"><?php esc_html_e('Tax', 'cartt'); ?></td><td>$<?php echo number_format($order->tax_total, 2); ?></td></tr><?php endif; ?>
                <tr class="total"><td colspan="2"><?php esc_html_e('Total', 'cartt'); ?></td><td>$<?php echo number_format($order->total, 2); ?></td></tr>
                </tfoot>
            </table>
                    <?php endif; ?>
                <?php else: ?>
            <h2><?php esc_html_e('Orders', 'cartt'); ?></h2>
            <?php if (empty($orders)): ?><p><?php esc_html_e('No orders yet.', 'cartt'); ?></p><?php else: ?>
            <table class="cartt-table">
                <thead><tr><th><?php esc_html_e('Order', 'cartt'); ?></th><th><?php esc_html_e('Date', 'cartt'); ?></th><th><?php esc_html_e('Status', 'cartt'); ?></th><th><?php esc_html_e('Total', 'cartt'); ?></th><th></th></tr></thead>
                <tbody>
                <?php foreach ($orders as $o): ?>
                <tr>
                    <td>#<?php echo esc_html($o->order_number); ?></td>
                    <td><?php echo esc_html(date_i18n(get_option('date_format'), strtotime($o->created_at))); ?></td>
                    <td><span class="cartt-status cartt-status-<?php echo esc_attr($o->status); ?>"><?php echo esc_html(ucfirst($o->status)); ?></span></td>
                    <td>$<?php echo number_format($o->total, 2); ?></td>
                    <td><a href="<?php echo esc_url(add_query_arg(['tab' => 'orders', 'view' => $o->id], $accountUrl)); ?>"><?php esc_html_e('View', 'cartt'); ?></a></td>
                </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; endif; ?>

            <?php elseif ($activeTab === 'downloads'): ?>
            <h2><?php esc_html_e('Downloads', 'cartt'); ?></h2>
            <?php if (empty($downloads)): ?><p><?php esc_html_e('No downloads available.', 'cartt'); ?></p><?php else: ?>
            <table class="cartt-table">
                <thead><tr><th><?php esc_html_e('File', 'cartt'); ?></th><th><?php esc_html_e('Order', 'cartt'); ?></th><th><?php esc_html_e('Remaining', 'cartt'); ?></th><th></th></tr></thead>
                <tbody>
                <?php foreach ($downloads as $dl): $remaining = $dl->download_limit ? ($dl->download_limit - $dl->download_count) : '∞'; ?>
                <tr>
                    <td><?php echo esc_html($dl->name); ?></td>
                    <td>#<?php echo esc_html($dl->order_number); ?></td>
                    <td><?php echo $remaining; ?></td>
                    <td><a href="<?php echo esc_url(add_query_arg(['cartt_download' => $dl->id, '_wpnonce' => wp_create_nonce('download_' . $dl->id)], home_url())); ?>" class="cartt-btn-sm"><?php esc_html_e('Download', 'cartt'); ?></a></td>
                </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>

            <?php elseif ($activeTab === 'addresses'): ?>
            <h2><?php esc_html_e('Addresses', 'cartt'); ?></h2>
            <p><?php esc_html_e('These addresses will be used at checkout by default.', 'cartt'); ?></p>
            <div class="cartt-addresses-row">
                <div class="cartt-address-form">
                    <h3><?php esc_html_e('Billing Address', 'cartt'); ?></h3>
                    <form method="post">
                        <?php wp_nonce_field('cartt_save_account', 'cartt_account_nonce'); ?>
                        <div class="cartt-row-half"><div><label><?php esc_html_e('First Name', 'cartt'); ?></label><input type="text" name="billing_first_name" value="<?php echo esc_attr($billing['first_name'] ?? ''); ?>"></div><div><label><?php esc_html_e('Last Name', 'cartt'); ?></label><input type="text" name="billing_last_name" value="<?php echo esc_attr($billing['last_name'] ?? ''); ?>"></div></div>
                        <div><label><?php esc_html_e('Address', 'cartt'); ?></label><input type="text" name="billing_address_1" value="<?php echo esc_attr($billing['address_1'] ?? ''); ?>"></div>
                        <div class="cartt-row-half"><div><label><?php esc_html_e('City', 'cartt'); ?></label><input type="text" name="billing_city" value="<?php echo esc_attr($billing['city'] ?? ''); ?>"></div><div><label><?php esc_html_e('State', 'cartt'); ?></label><input type="text" name="billing_state" value="<?php echo esc_attr($billing['state'] ?? ''); ?>"></div></div>
                        <div class="cartt-row-half"><div><label><?php esc_html_e('Postcode', 'cartt'); ?></label><input type="text" name="billing_postcode" value="<?php echo esc_attr($billing['postcode'] ?? ''); ?>"></div><div><label><?php esc_html_e('Country', 'cartt'); ?></label><input type="text" name="billing_country" value="<?php echo esc_attr($billing['country'] ?? ''); ?>"></div></div>
                        <div><label><?php esc_html_e('Phone', 'cartt'); ?></label><input type="tel" name="billing_phone" value="<?php echo esc_attr($billing['phone'] ?? ''); ?>"></div>
                        <button type="submit" name="save_billing" class="cartt-btn"><?php esc_html_e('Save', 'cartt'); ?></button>
                    </form>
                </div>
                <div class="cartt-address-form">
                    <h3><?php esc_html_e('Shipping Address', 'cartt'); ?></h3>
                    <form method="post">
                        <?php wp_nonce_field('cartt_save_account', 'cartt_account_nonce'); ?>
                        <div class="cartt-row-half"><div><label><?php esc_html_e('First Name', 'cartt'); ?></label><input type="text" name="shipping_first_name" value="<?php echo esc_attr($shipping['first_name'] ?? ''); ?>"></div><div><label><?php esc_html_e('Last Name', 'cartt'); ?></label><input type="text" name="shipping_last_name" value="<?php echo esc_attr($shipping['last_name'] ?? ''); ?>"></div></div>
                        <div><label><?php esc_html_e('Address', 'cartt'); ?></label><input type="text" name="shipping_address_1" value="<?php echo esc_attr($shipping['address_1'] ?? ''); ?>"></div>
                        <div class="cartt-row-half"><div><label><?php esc_html_e('City', 'cartt'); ?></label><input type="text" name="shipping_city" value="<?php echo esc_attr($shipping['city'] ?? ''); ?>"></div><div><label><?php esc_html_e('State', 'cartt'); ?></label><input type="text" name="shipping_state" value="<?php echo esc_attr($shipping['state'] ?? ''); ?>"></div></div>
                        <div class="cartt-row-half"><div><label><?php esc_html_e('Postcode', 'cartt'); ?></label><input type="text" name="shipping_postcode" value="<?php echo esc_attr($shipping['postcode'] ?? ''); ?>"></div><div><label><?php esc_html_e('Country', 'cartt'); ?></label><input type="text" name="shipping_country" value="<?php echo esc_attr($shipping['country'] ?? ''); ?>"></div></div>
                        <button type="submit" name="save_shipping" class="cartt-btn"><?php esc_html_e('Save', 'cartt'); ?></button>
                    </form>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.cartt-account { max-width: 960px; margin: 0 auto; padding: 20px; }
.cartt-account h1 { font-size: 28px; margin-bottom: 30px; }
.cartt-account-grid { display: grid; grid-template-columns: 180px 1fr; gap: 40px; }
.cartt-account-nav { display: flex; flex-direction: column; gap: 4px; }
.cartt-account-nav a { display: flex; justify-content: space-between; padding: 10px 14px; border-radius: 6px; text-decoration: none; color: #1d1d1f; font-size: 14px; }
.cartt-account-nav a:hover { background: #f5f5f7; }
.cartt-account-nav a.active { background: #1d1d1f; color: #fff; }
.cartt-account-nav a span { opacity: 0.6; font-size: 12px; }
.cartt-account-content { background: #fff; border: 1px solid #e8e8ed; border-radius: 10px; padding: 24px; }
.cartt-stats-row { display: grid; grid-template-columns: repeat(3, 1fr); gap: 16px; margin: 24px 0; }
.cartt-stat-box { background: #f5f5f7; padding: 20px; border-radius: 8px; text-align: center; }
.cartt-stat-box .num { display: block; font-size: 24px; font-weight: 600; }
.cartt-stat-box .label { font-size: 12px; color: #86868b; }
.cartt-table { width: 100%; border-collapse: collapse; font-size: 14px; }
.cartt-table th { text-align: left; padding: 10px 8px; border-bottom: 1px solid #e8e8ed; color: #86868b; font-weight: 500; }
.cartt-table td { padding: 12px 8px; border-bottom: 1px solid #e8e8ed; }
.cartt-table a { color: #0066cc; text-decoration: none; }
.cartt-table tfoot td { font-weight: 500; }
.cartt-table .total td { font-weight: 600; font-size: 16px; }
.cartt-status { display: inline-block; padding: 3px 10px; border-radius: 20px; font-size: 12px; }
.cartt-status-pending { background: #fff3e0; color: #e65100; }
.cartt-status-processing { background: #e3f2fd; color: #1565c0; }
.cartt-status-completed { background: #e8f5e9; color: #2e7d32; }
.cartt-status-cancelled, .cartt-status-refunded { background: #ffebee; color: #c62828; }
.cartt-addresses-row { display: grid; grid-template-columns: 1fr 1fr; gap: 24px; }
.cartt-address-form h3 { font-size: 15px; margin: 0 0 16px; }
.cartt-address-form div { margin-bottom: 12px; }
.cartt-address-form label { display: block; font-size: 12px; font-weight: 500; margin-bottom: 4px; }
.cartt-address-form input { width: 100%; padding: 10px; border: 1px solid #d2d2d7; border-radius: 6px; font-size: 14px; box-sizing: border-box; }
.cartt-row-half { display: grid; grid-template-columns: 1fr 1fr; gap: 12px; }
.cartt-btn { background: #1d1d1f; color: #fff; border: none; padding: 12px 20px; border-radius: 6px; font-size: 14px; cursor: pointer; margin-top: 8px; }
.cartt-btn-sm { background: #1d1d1f; color: #fff; padding: 6px 12px; border-radius: 4px; font-size: 12px; text-decoration: none; }
.cartt-notice { background: #e8f5e9; color: #2e7d32; padding: 12px 16px; border-radius: 6px; margin-bottom: 20px; }
.cartt-account-login { max-width: 400px; margin: 40px auto; padding: 30px; background: #fff; border: 1px solid #e8e8ed; border-radius: 10px; }
@media (max-width: 768px) {
    .cartt-account-grid { grid-template-columns: 1fr; }
    .cartt-account-nav { flex-direction: row; flex-wrap: wrap; gap: 8px; }
    .cartt-stats-row, .cartt-addresses-row, .cartt-row-half { grid-template-columns: 1fr; }
}
</style>
