<?php
/**
 * Product Q&A Section Template
 * Include this in product.php template
 */

use Cartt\Services\ProductQAService;

if (!isset($product)) {
    return;
}

$qa = ProductQAService::getProductQA($product->id);
$questionCount = ProductQAService::getProductQuestionCount($product->id);
$isLoggedIn = is_user_logged_in();
$currentUser = wp_get_current_user();
?>

<div class="cartt-product-qa" id="product-qa">
    <h3>Questions & Answers (<?php echo $questionCount; ?>)</h3>

    <!-- Ask a Question Form -->
    <div class="cartt-qa-form-wrapper">
        <button type="button" class="cartt-btn cartt-btn-secondary" id="cartt-toggle-qa-form">
            Ask a Question
        </button>
        
        <form id="cartt-qa-form" class="cartt-qa-form" style="display: none;">
            <input type="hidden" name="product_id" value="<?php echo esc_attr($product->id); ?>">
            
            <?php if (!$isLoggedIn): ?>
            <div class="cartt-qa-form-row">
                <input type="text" name="author_name" placeholder="Your Name *" required>
                <input type="email" name="author_email" placeholder="Your Email *" required>
            </div>
            <?php else: ?>
            <input type="hidden" name="author_name" value="<?php echo esc_attr($currentUser->display_name); ?>">
            <input type="hidden" name="author_email" value="<?php echo esc_attr($currentUser->user_email); ?>">
            <?php endif; ?>
            
            <textarea name="content" placeholder="Type your question here..." rows="3" required></textarea>
            
            <div class="cartt-qa-form-actions">
                <button type="submit" class="cartt-btn cartt-btn-primary">Submit Question</button>
                <button type="button" class="cartt-btn cartt-btn-text" id="cartt-cancel-qa">Cancel</button>
            </div>
        </form>
    </div>

    <!-- Search Questions -->
    <?php if ($questionCount > 3): ?>
    <div class="cartt-qa-search">
        <input type="text" id="cartt-qa-search-input" placeholder="Search questions...">
    </div>
    <?php endif; ?>

    <!-- Questions List -->
    <div class="cartt-qa-list" id="cartt-qa-list">
        <?php if (empty($qa)): ?>
        <div class="cartt-qa-empty">
            <p>No questions yet. Be the first to ask!</p>
        </div>
        <?php else: ?>
        <?php foreach ($qa as $question): ?>
        <div class="cartt-qa-item" data-question-id="<?php echo esc_attr($question->id); ?>">
            <div class="cartt-qa-question">
                <div class="cartt-qa-icon">Q</div>
                <div class="cartt-qa-content">
                    <div class="cartt-qa-text"><?php echo esc_html($question->content); ?></div>
                    <div class="cartt-qa-meta">
                        Asked by <?php echo esc_html($question->author_name); ?> 
                        on <?php echo date('M j, Y', strtotime($question->created_at)); ?>
                    </div>
                    <div class="cartt-qa-actions">
                        <button type="button" class="cartt-qa-upvote" data-id="<?php echo $question->id; ?>">
                            <span class="upvote-icon">▲</span>
                            <span class="upvote-count"><?php echo $question->upvotes; ?></span>
                        </button>
                        <button type="button" class="cartt-qa-answer-btn" data-question-id="<?php echo $question->id; ?>">
                            Answer
                        </button>
                    </div>
                </div>
            </div>

            <!-- Answers -->
            <?php if (!empty($question->answers)): ?>
            <div class="cartt-qa-answers">
                <?php foreach ($question->answers as $answer): ?>
                <div class="cartt-qa-answer <?php echo $answer->is_vendor_answer ? 'cartt-qa-vendor-answer' : ''; ?>">
                    <div class="cartt-qa-icon">A</div>
                    <div class="cartt-qa-content">
                        <?php if ($answer->is_vendor_answer): ?>
                        <span class="cartt-qa-vendor-badge">Official Answer</span>
                        <?php endif; ?>
                        <div class="cartt-qa-text"><?php echo esc_html($answer->content); ?></div>
                        <div class="cartt-qa-meta">
                            <?php echo esc_html($answer->author_name); ?>
                            — <?php echo date('M j, Y', strtotime($answer->created_at)); ?>
                        </div>
                        <div class="cartt-qa-actions">
                            <button type="button" class="cartt-qa-upvote" data-id="<?php echo $answer->id; ?>">
                                <span class="upvote-icon">▲</span>
                                <span class="upvote-count"><?php echo $answer->upvotes; ?></span>
                            </button>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <!-- Answer Form (hidden) -->
            <div class="cartt-qa-answer-form" style="display: none;">
                <form class="cartt-answer-form">
                    <input type="hidden" name="question_id" value="<?php echo $question->id; ?>">
                    <?php if (!$isLoggedIn): ?>
                    <div class="cartt-qa-form-row">
                        <input type="text" name="author_name" placeholder="Your Name *" required>
                        <input type="email" name="author_email" placeholder="Your Email *" required>
                    </div>
                    <?php else: ?>
                    <input type="hidden" name="author_name" value="<?php echo esc_attr($currentUser->display_name); ?>">
                    <input type="hidden" name="author_email" value="<?php echo esc_attr($currentUser->user_email); ?>">
                    <?php endif; ?>
                    <textarea name="content" placeholder="Write your answer..." rows="2" required></textarea>
                    <div class="cartt-qa-form-actions">
                        <button type="submit" class="cartt-btn cartt-btn-primary cartt-btn-small">Submit Answer</button>
                        <button type="button" class="cartt-btn cartt-btn-text cartt-cancel-answer">Cancel</button>
                    </div>
                </form>
            </div>
        </div>
        <?php endforeach; ?>
        <?php endif; ?>
    </div>
</div>

<style>
.cartt-product-qa { margin-top: 40px; padding-top: 40px; border-top: 1px solid #eee; }
.cartt-product-qa h3 { margin: 0 0 20px; }
.cartt-qa-form-wrapper { margin-bottom: 25px; }
.cartt-qa-form { background: #f9f9f9; padding: 20px; border-radius: 8px; margin-top: 15px; }
.cartt-qa-form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 10px; margin-bottom: 10px; }
.cartt-qa-form input, .cartt-qa-form textarea { width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 6px; font-size: 14px; }
.cartt-qa-form textarea { resize: vertical; min-height: 80px; }
.cartt-qa-form-actions { display: flex; gap: 10px; margin-top: 15px; }
.cartt-qa-search { margin-bottom: 20px; }
.cartt-qa-search input { width: 100%; padding: 12px 15px; border: 1px solid #ddd; border-radius: 6px; }
.cartt-qa-empty { text-align: center; padding: 40px; color: #666; background: #f9f9f9; border-radius: 8px; }
.cartt-qa-list { display: flex; flex-direction: column; gap: 20px; }
.cartt-qa-item { background: #fff; border: 1px solid #eee; border-radius: 8px; padding: 20px; }
.cartt-qa-question, .cartt-qa-answer { display: flex; gap: 15px; }
.cartt-qa-icon { width: 32px; height: 32px; background: #1d1d1f; color: #fff; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: bold; font-size: 14px; flex-shrink: 0; }
.cartt-qa-answer .cartt-qa-icon { background: #28a745; }
.cartt-qa-content { flex: 1; }
.cartt-qa-text { font-size: 15px; line-height: 1.5; }
.cartt-qa-meta { font-size: 12px; color: #999; margin-top: 8px; }
.cartt-qa-actions { display: flex; gap: 15px; margin-top: 12px; }
.cartt-qa-upvote { display: flex; align-items: center; gap: 5px; background: #f5f5f7; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer; font-size: 12px; }
.cartt-qa-upvote:hover { background: #eee; }
.cartt-qa-upvote .upvote-icon { color: #666; }
.cartt-qa-upvote.upvoted { background: #e8f5e9; }
.cartt-qa-upvote.upvoted .upvote-icon { color: #28a745; }
.cartt-qa-answer-btn { background: none; border: none; color: #1d1d1f; cursor: pointer; font-size: 13px; padding: 0; }
.cartt-qa-answer-btn:hover { text-decoration: underline; }
.cartt-qa-answers { margin-top: 20px; margin-left: 47px; display: flex; flex-direction: column; gap: 15px; }
.cartt-qa-vendor-answer { background: #f0f9ff; padding: 15px; border-radius: 6px; margin: 0 -15px; }
.cartt-qa-vendor-badge { display: inline-block; background: #1d1d1f; color: #fff; font-size: 10px; padding: 2px 6px; border-radius: 3px; margin-bottom: 8px; text-transform: uppercase; }
.cartt-qa-answer-form { margin-top: 15px; margin-left: 47px; }
.cartt-answer-form { background: #f9f9f9; padding: 15px; border-radius: 6px; }
.cartt-btn { padding: 10px 20px; border: none; border-radius: 6px; cursor: pointer; font-size: 14px; }
.cartt-btn-primary { background: #1d1d1f; color: #fff; }
.cartt-btn-secondary { background: #f5f5f7; color: #1d1d1f; border: 1px solid #ddd; }
.cartt-btn-text { background: none; color: #666; }
.cartt-btn-small { padding: 8px 16px; font-size: 13px; }

@media (max-width: 600px) {
    .cartt-qa-form-row { grid-template-columns: 1fr; }
    .cartt-qa-answers { margin-left: 20px; }
}
</style>

<script>
(function() {
    const nonce = '<?php echo wp_create_nonce('cartt_qa'); ?>';
    const productId = <?php echo $product->id; ?>;

    // Toggle question form
    document.getElementById('cartt-toggle-qa-form')?.addEventListener('click', function() {
        const form = document.getElementById('cartt-qa-form');
        form.style.display = form.style.display === 'none' ? 'block' : 'none';
    });

    document.getElementById('cartt-cancel-qa')?.addEventListener('click', function() {
        document.getElementById('cartt-qa-form').style.display = 'none';
    });

    // Submit question
    document.getElementById('cartt-qa-form')?.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        formData.append('action', 'cartt_submit_question');
        formData.append('nonce', nonce);

        fetch(carttFrontend.ajaxUrl, {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                this.reset();
                this.style.display = 'none';
                alert('Your question has been submitted and will appear after review.');
            } else {
                alert('Error: ' + data.data.message);
            }
        });
    });

    // Toggle answer forms
    document.querySelectorAll('.cartt-qa-answer-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const item = this.closest('.cartt-qa-item');
            const form = item.querySelector('.cartt-qa-answer-form');
            form.style.display = form.style.display === 'none' ? 'block' : 'none';
        });
    });

    document.querySelectorAll('.cartt-cancel-answer').forEach(btn => {
        btn.addEventListener('click', function() {
            this.closest('.cartt-qa-answer-form').style.display = 'none';
        });
    });

    // Submit answers
    document.querySelectorAll('.cartt-answer-form').forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'cartt_submit_answer');
            formData.append('nonce', nonce);

            fetch(carttFrontend.ajaxUrl, {
                method: 'POST',
                body: formData
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    alert('Your answer has been submitted.');
                    location.reload();
                } else {
                    alert('Error: ' + data.data.message);
                }
            });
        });
    });

    // Upvote
    document.querySelectorAll('.cartt-qa-upvote').forEach(btn => {
        btn.addEventListener('click', function() {
            const id = this.dataset.id;
            const countEl = this.querySelector('.upvote-count');
            
            fetch(carttFrontend.ajaxUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: `action=cartt_qa_upvote&nonce=${nonce}&id=${id}`
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    countEl.textContent = parseInt(countEl.textContent) + 1;
                    this.classList.add('upvoted');
                }
            });
        });
    });

    // Search
    document.getElementById('cartt-qa-search-input')?.addEventListener('input', function() {
        const query = this.value.toLowerCase();
        document.querySelectorAll('.cartt-qa-item').forEach(item => {
            const text = item.querySelector('.cartt-qa-text').textContent.toLowerCase();
            item.style.display = text.includes(query) ? 'block' : 'none';
        });
    });
})();
</script>
