<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * WooCommerce Migration Service
 * Imports products, customers, orders from WooCommerce
 */
class WooMigrationService
{
    private array $stats = [
        'products' => 0,
        'categories' => 0,
        'customers' => 0,
        'orders' => 0,
        'errors' => [],
    ];

    private array $categoryMap = [];
    private array $productMap = [];
    private array $customerMap = [];

    public function isWooCommerceActive(): bool
    {
        return class_exists('WooCommerce') || 
               in_array('woocommerce/woocommerce.php', get_option('active_plugins', []));
    }

    public function hasWooCommerceData(): bool
    {
        global $wpdb;
        
        // Check for WooCommerce products
        $count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'product' AND post_status IN ('publish', 'draft')"
        );
        
        return (int) $count > 0;
    }

    public function getWooCommerceStats(): array
    {
        global $wpdb;

        return [
            'products' => (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'product' AND post_status IN ('publish', 'draft')"
            ),
            'categories' => (int) $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->term_taxonomy} WHERE taxonomy = %s",
                    'product_cat'
                )
            ),
            'customers' => (int) $wpdb->get_var(
                "SELECT COUNT(DISTINCT meta_value) FROM {$wpdb->postmeta} WHERE meta_key = '_billing_email'"
            ),
            'orders' => (int) $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'shop_order'"
            ),
        ];
    }

    public function migrate(array $options = []): array
    {
        $defaults = [
            'products' => true,
            'categories' => true,
            'customers' => true,
            'orders' => true,
        ];
        $options = array_merge($defaults, $options);

        // Order matters: categories first, then products, then customers, then orders
        if ($options['categories']) {
            $this->migrateCategories();
        }

        if ($options['products']) {
            $this->migrateProducts();
        }

        if ($options['customers']) {
            $this->migrateCustomers();
        }

        if ($options['orders']) {
            $this->migrateOrders();
        }

        return $this->stats;
    }

    private function migrateCategories(): void
    {
        global $wpdb;

        $categories = $wpdb->get_results("
            SELECT t.term_id, t.name, t.slug, tt.description, tt.parent
            FROM {$wpdb->terms} t
            JOIN {$wpdb->term_taxonomy} tt ON t.term_id = tt.term_id
            WHERE tt.taxonomy = 'product_cat'
            ORDER BY tt.parent ASC, t.name ASC
        ");

        $table = $wpdb->prefix . 'cartt_categories';
        $parentMap = [];

        foreach ($categories as $cat) {
            // Check if already migrated
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE slug = %s",
                $cat->slug
            ));

            if ($existing) {
                $this->categoryMap[$cat->term_id] = (int) $existing;
                continue;
            }

            // Get image
            $thumbnail_id = get_term_meta($cat->term_id, 'thumbnail_id', true);

            $wpdb->insert($table, [
                'name' => $cat->name,
                'slug' => $cat->slug,
                'description' => $cat->description,
                'parent_id' => $cat->parent ? ($this->categoryMap[$cat->parent] ?? null) : null,
                'image' => $thumbnail_id ?: null,
            ]);

            $newId = $wpdb->insert_id;
            $this->categoryMap[$cat->term_id] = $newId;
            $this->stats['categories']++;
        }
    }

    private function migrateProducts(): void
    {
        global $wpdb;

        $products = $wpdb->get_results("
            SELECT ID, post_title, post_name, post_content, post_excerpt, post_status, post_date
            FROM {$wpdb->posts}
            WHERE post_type = 'product'
            AND post_status IN ('publish', 'draft', 'private')
            ORDER BY ID ASC
        ");

        $table = $wpdb->prefix . 'cartt_products';
        $catRelTable = $wpdb->prefix . 'cartt_product_categories';

        foreach ($products as $product) {
            try {
                // Check if already migrated
                $existing = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM $table WHERE slug = %s",
                    $product->post_name
                ));

                if ($existing) {
                    $this->productMap[$product->ID] = (int) $existing;
                    continue;
                }

                // Get product meta
                $meta = $this->getProductMeta($product->ID);
                
                // Determine product type
                $wooType = $this->getWooProductType($product->ID);
                $carttType = $this->mapProductType($wooType);

                // Get featured image
                $featuredImage = get_post_thumbnail_id($product->ID);

                // Map status
                $status = $product->post_status === 'publish' ? 'publish' : 'draft';

                // Insert product
                $wpdb->insert($table, [
                    'name' => $product->post_title,
                    'slug' => $product->post_name,
                    'description' => $product->post_content,
                    'short_description' => $product->post_excerpt,
                    'product_type' => $carttType,
                    'status' => $status,
                    'price' => $meta['_regular_price'] ?? $meta['_price'] ?? null,
                    'sale_price' => $meta['_sale_price'] ?? null,
                    'sku' => $meta['_sku'] ?? null,
                    'stock_quantity' => $meta['_stock'] ?? null,
                    'stock_status' => $meta['_stock_status'] ?? 'instock',
                    'manage_stock' => ($meta['_manage_stock'] ?? 'no') === 'yes' ? 1 : 0,
                    'weight' => $meta['_weight'] ?? null,
                    'length' => $meta['_length'] ?? null,
                    'width' => $meta['_width'] ?? null,
                    'height' => $meta['_height'] ?? null,
                    'downloadable' => ($meta['_downloadable'] ?? 'no') === 'yes' ? 1 : 0,
                    'virtual' => ($meta['_virtual'] ?? 'no') === 'yes' ? 1 : 0,
                    'featured_image' => $featuredImage ?: null,
                    'gallery_images' => $meta['_product_image_gallery'] ?? null,
                    'created_at' => $product->post_date,
                ]);

                $newId = $wpdb->insert_id;
                $this->productMap[$product->ID] = $newId;

                // Migrate category relationships
                $this->migrateProductCategories($product->ID, $newId);

                // Migrate downloadable files if digital product
                if (($meta['_downloadable'] ?? 'no') === 'yes') {
                    $this->migrateDownloads($product->ID, $newId);
                }

                $this->stats['products']++;

            } catch (\Exception $e) {
                $this->stats['errors'][] = "Product {$product->ID}: " . $e->getMessage();
            }
        }
    }

    private function getProductMeta(int $productId): array
    {
        global $wpdb;

        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT meta_key, meta_value FROM {$wpdb->postmeta} WHERE post_id = %d",
            $productId
        ));

        $meta = [];
        foreach ($results as $row) {
            $meta[$row->meta_key] = $row->meta_value;
        }

        return $meta;
    }

    private function getWooProductType(int $productId): string
    {
        global $wpdb;

        $type = $wpdb->get_var($wpdb->prepare("
            SELECT t.slug
            FROM {$wpdb->terms} t
            JOIN {$wpdb->term_taxonomy} tt ON t.term_id = tt.term_id
            JOIN {$wpdb->term_relationships} tr ON tt.term_taxonomy_id = tr.term_taxonomy_id
            WHERE tr.object_id = %d AND tt.taxonomy = 'product_type'
        ", $productId));

        return $type ?: 'simple';
    }

    private function mapProductType(string $wooType): string
    {
        $map = [
            'simple' => 'physical',
            'variable' => 'physical',
            'grouped' => 'physical',
            'external' => 'physical',
            'subscription' => 'subscription',
            'variable-subscription' => 'subscription',
            'booking' => 'service',
        ];

        return $map[$wooType] ?? 'physical';
    }

    private function migrateProductCategories(int $wooProductId, int $carttProductId): void
    {
        global $wpdb;

        $categories = $wpdb->get_col($wpdb->prepare("
            SELECT tt.term_id
            FROM {$wpdb->term_relationships} tr
            JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
            WHERE tr.object_id = %d AND tt.taxonomy = 'product_cat'
        ", $wooProductId));

        $table = $wpdb->prefix . 'cartt_product_categories';

        foreach ($categories as $wooCatId) {
            if (isset($this->categoryMap[$wooCatId])) {
                $wpdb->insert($table, [
                    'product_id' => $carttProductId,
                    'category_id' => $this->categoryMap[$wooCatId],
                ]);
            }
        }
    }

    private function migrateDownloads(int $wooProductId, int $carttProductId): void
    {
        global $wpdb;

        $downloads = get_post_meta($wooProductId, '_downloadable_files', true);
        
        if (!is_array($downloads)) {
            return;
        }

        $table = $wpdb->prefix . 'cartt_downloads';
        $sort = 0;

        foreach ($downloads as $download) {
            $wpdb->insert($table, [
                'product_id' => $carttProductId,
                'name' => $download['name'] ?? 'Download',
                'file_url' => $download['file'] ?? '',
                'download_limit' => get_post_meta($wooProductId, '_download_limit', true) ?: null,
                'expiry_days' => get_post_meta($wooProductId, '_download_expiry', true) ?: null,
                'sort_order' => $sort++,
            ]);
        }
    }

    private function migrateCustomers(): void
    {
        global $wpdb;

        // Get unique customers from orders
        $customers = $wpdb->get_results("
            SELECT DISTINCT
                pm_email.meta_value as email,
                pm_first.meta_value as first_name,
                pm_last.meta_value as last_name,
                pm_phone.meta_value as phone,
                p.post_date as created_at
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm_email ON p.ID = pm_email.post_id AND pm_email.meta_key = '_billing_email'
            LEFT JOIN {$wpdb->postmeta} pm_first ON p.ID = pm_first.post_id AND pm_first.meta_key = '_billing_first_name'
            LEFT JOIN {$wpdb->postmeta} pm_last ON p.ID = pm_last.post_id AND pm_last.meta_key = '_billing_last_name'
            LEFT JOIN {$wpdb->postmeta} pm_phone ON p.ID = pm_phone.post_id AND pm_phone.meta_key = '_billing_phone'
            WHERE p.post_type = 'shop_order'
            AND pm_email.meta_value IS NOT NULL
            AND pm_email.meta_value != ''
            GROUP BY pm_email.meta_value
            ORDER BY p.post_date ASC
        ");

        $table = $wpdb->prefix . 'cartt_customers';

        foreach ($customers as $customer) {
            // Check if already exists
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE email = %s",
                $customer->email
            ));

            if ($existing) {
                $this->customerMap[$customer->email] = (int) $existing;
                continue;
            }

            // Get billing address from most recent order
            $billing = $this->getCustomerBillingAddress($customer->email);

            // Check if linked to WP user
            $user = get_user_by('email', $customer->email);

            $wpdb->insert($table, [
                'user_id' => $user ? $user->ID : null,
                'email' => $customer->email,
                'first_name' => $customer->first_name,
                'last_name' => $customer->last_name,
                'phone' => $customer->phone,
                'billing_address' => json_encode($billing),
                'created_at' => $customer->created_at,
            ]);

            $this->customerMap[$customer->email] = $wpdb->insert_id;
            $this->stats['customers']++;
        }
    }

    private function getCustomerBillingAddress(string $email): array
    {
        global $wpdb;

        $orderId = $wpdb->get_var($wpdb->prepare("
            SELECT p.ID
            FROM {$wpdb->posts} p
            JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'shop_order'
            AND pm.meta_key = '_billing_email'
            AND pm.meta_value = %s
            ORDER BY p.post_date DESC
            LIMIT 1
        ", $email));

        if (!$orderId) {
            return [];
        }

        $fields = ['first_name', 'last_name', 'company', 'address_1', 'address_2', 'city', 'state', 'postcode', 'country', 'phone', 'email'];
        $billing = [];

        foreach ($fields as $field) {
            $billing[$field] = get_post_meta($orderId, '_billing_' . $field, true);
        }

        return $billing;
    }

    private function migrateOrders(): void
    {
        global $wpdb;

        $orders = $wpdb->get_results("
            SELECT ID, post_status, post_date, post_date_gmt
            FROM {$wpdb->posts}
            WHERE post_type = 'shop_order'
            ORDER BY ID ASC
        ");

        $ordersTable = $wpdb->prefix . 'cartt_orders';
        $itemsTable = $wpdb->prefix . 'cartt_order_items';

        foreach ($orders as $order) {
            try {
                $meta = $this->getOrderMeta($order->ID);
                
                // Get or create customer
                $customerEmail = $meta['_billing_email'] ?? '';
                $customerId = $this->customerMap[$customerEmail] ?? $this->createCustomerFromOrder($meta);

                // Map order status
                $status = $this->mapOrderStatus($order->post_status);

                // Generate order number
                $orderNumber = $meta['_order_number'] ?? ('WC-' . $order->ID);

                // Check if already migrated
                $existing = $wpdb->get_var($wpdb->prepare(
                    "SELECT id FROM $ordersTable WHERE order_number = %s",
                    $orderNumber
                ));

                if ($existing) {
                    continue;
                }

                // Build addresses
                $billing = $this->buildAddressFromMeta($meta, 'billing');
                $shipping = $this->buildAddressFromMeta($meta, 'shipping');

                $wpdb->insert($ordersTable, [
                    'order_number' => $orderNumber,
                    'customer_id' => $customerId,
                    'status' => $status,
                    'currency' => $meta['_order_currency'] ?? 'USD',
                    'subtotal' => $meta['_order_total'] ?? 0,
                    'discount_total' => $meta['_cart_discount'] ?? 0,
                    'shipping_total' => $meta['_order_shipping'] ?? 0,
                    'tax_total' => $meta['_order_tax'] ?? 0,
                    'total' => $meta['_order_total'] ?? 0,
                    'payment_method' => $meta['_payment_method'] ?? null,
                    'payment_status' => in_array($status, ['completed', 'processing']) ? 'paid' : 'pending',
                    'transaction_id' => $meta['_transaction_id'] ?? null,
                    'billing_address' => json_encode($billing),
                    'shipping_address' => json_encode($shipping),
                    'customer_note' => $meta['_customer_note'] ?? null,
                    'ip_address' => $meta['_customer_ip_address'] ?? null,
                    'user_agent' => $meta['_customer_user_agent'] ?? null,
                    'created_at' => $order->post_date,
                    'completed_at' => $status === 'completed' ? ($meta['_completed_date'] ?? $order->post_date) : null,
                ]);

                $newOrderId = $wpdb->insert_id;

                // Migrate order items
                $this->migrateOrderItems($order->ID, $newOrderId);

                // Update customer totals
                if ($customerId) {
                    $wpdb->query($wpdb->prepare(
                        "UPDATE {$wpdb->prefix}cartt_customers 
                         SET total_spent = total_spent + %f, order_count = order_count + 1 
                         WHERE id = %d",
                        floatval($meta['_order_total'] ?? 0),
                        $customerId
                    ));
                }

                $this->stats['orders']++;

            } catch (\Exception $e) {
                $this->stats['errors'][] = "Order {$order->ID}: " . $e->getMessage();
            }
        }
    }

    private function getOrderMeta(int $orderId): array
    {
        global $wpdb;

        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT meta_key, meta_value FROM {$wpdb->postmeta} WHERE post_id = %d",
            $orderId
        ));

        $meta = [];
        foreach ($results as $row) {
            $meta[$row->meta_key] = $row->meta_value;
        }

        return $meta;
    }

    private function mapOrderStatus(string $wooStatus): string
    {
        $map = [
            'wc-pending' => 'pending',
            'wc-processing' => 'processing',
            'wc-on-hold' => 'pending',
            'wc-completed' => 'completed',
            'wc-cancelled' => 'cancelled',
            'wc-refunded' => 'refunded',
            'wc-failed' => 'failed',
        ];

        return $map[$wooStatus] ?? 'pending';
    }

    private function buildAddressFromMeta(array $meta, string $type): array
    {
        $fields = ['first_name', 'last_name', 'company', 'address_1', 'address_2', 'city', 'state', 'postcode', 'country', 'phone', 'email'];
        $address = [];

        foreach ($fields as $field) {
            $key = "_{$type}_{$field}";
            $address[$field] = $meta[$key] ?? '';
        }

        return $address;
    }

    private function createCustomerFromOrder(array $meta): ?int
    {
        $email = $meta['_billing_email'] ?? '';
        if (!$email) {
            return null;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customers';

        $billing = $this->buildAddressFromMeta($meta, 'billing');

        $wpdb->insert($table, [
            'email' => $email,
            'first_name' => $meta['_billing_first_name'] ?? '',
            'last_name' => $meta['_billing_last_name'] ?? '',
            'phone' => $meta['_billing_phone'] ?? '',
            'billing_address' => json_encode($billing),
        ]);

        $id = $wpdb->insert_id;
        $this->customerMap[$email] = $id;
        
        return $id;
    }

    private function migrateOrderItems(int $wooOrderId, int $carttOrderId): void
    {
        global $wpdb;

        // Get WooCommerce order items
        $items = $wpdb->get_results($wpdb->prepare("
            SELECT oi.order_item_id, oi.order_item_name, oi.order_item_type
            FROM {$wpdb->prefix}woocommerce_order_items oi
            WHERE oi.order_id = %d
            AND oi.order_item_type = 'line_item'
        ", $wooOrderId));

        $table = $wpdb->prefix . 'cartt_order_items';

        foreach ($items as $item) {
            $itemMeta = $this->getOrderItemMeta($item->order_item_id);
            
            $productId = $itemMeta['_product_id'] ?? 0;
            $carttProductId = $this->productMap[$productId] ?? null;

            $wpdb->insert($table, [
                'order_id' => $carttOrderId,
                'product_id' => $carttProductId,
                'name' => $item->order_item_name,
                'sku' => $itemMeta['_sku'] ?? null,
                'quantity' => $itemMeta['_qty'] ?? 1,
                'price' => $itemMeta['_line_subtotal'] / max(1, $itemMeta['_qty'] ?? 1),
                'subtotal' => $itemMeta['_line_subtotal'] ?? 0,
                'tax' => $itemMeta['_line_tax'] ?? 0,
                'total' => $itemMeta['_line_total'] ?? 0,
            ]);
        }
    }

    private function getOrderItemMeta(int $itemId): array
    {
        global $wpdb;

        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT meta_key, meta_value FROM {$wpdb->prefix}woocommerce_order_itemmeta WHERE order_item_id = %d",
            $itemId
        ));

        $meta = [];
        foreach ($results as $row) {
            $meta[$row->meta_key] = $row->meta_value;
        }

        return $meta;
    }
}
