<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Wishlist Service
 * Manages customer wishlists with sharing and notifications
 */
class WishlistService
{
    /**
     * Get or create wishlist for user/session
     */
    public static function getWishlist(?int $userId = null, ?string $sessionId = null): object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlists';

        if ($userId) {
            $wishlist = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table WHERE user_id = %d LIMIT 1",
                $userId
            ));
        } elseif ($sessionId) {
            $wishlist = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table WHERE session_id = %s LIMIT 1",
                $sessionId
            ));
        } else {
            return (object) ['id' => 0, 'items' => []];
        }

        if (!$wishlist) {
            $wishlist = self::createWishlist($userId, $sessionId);
        }

        return $wishlist;
    }

    /**
     * Create a new wishlist
     */
    public static function createWishlist(?int $userId = null, ?string $sessionId = null, string $name = 'My Wishlist'): object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlists';

        $shareKey = wp_generate_password(32, false);

        $wpdb->insert($table, [
            'user_id' => $userId,
            'session_id' => $sessionId,
            'share_key' => $shareKey,
            'name' => $name,
        ]);

        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $wpdb->insert_id));
    }

    /**
     * Get wishlist by share key
     */
    public static function getWishlistByShareKey(string $shareKey): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlists';

        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE share_key = %s AND is_public = 1",
            $shareKey
        ));
    }

    /**
     * Add item to wishlist
     */
    public static function addItem(int $wishlistId, int $productId, ?int $variationId = null): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlist_items';

        // Check if already exists
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE wishlist_id = %d AND product_id = %d AND (variation_id = %d OR (variation_id IS NULL AND %d IS NULL))",
            $wishlistId, $productId, $variationId, $variationId
        ));

        if ($exists) {
            return true;
        }

        return $wpdb->insert($table, [
            'wishlist_id' => $wishlistId,
            'product_id' => $productId,
            'variation_id' => $variationId,
        ]) !== false;
    }

    /**
     * Remove item from wishlist
     */
    public static function removeItem(int $wishlistId, int $productId, ?int $variationId = null): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlist_items';

        if ($variationId) {
            return $wpdb->delete($table, [
                'wishlist_id' => $wishlistId,
                'product_id' => $productId,
                'variation_id' => $variationId,
            ]) !== false;
        }

        return $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE wishlist_id = %d AND product_id = %d AND variation_id IS NULL",
            $wishlistId, $productId
        )) !== false;
    }

    /**
     * Get wishlist items with product data
     */
    public static function getItems(int $wishlistId): array
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_wishlist_items';
        $products_table = $wpdb->prefix . 'cartt_products';

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT wi.*, p.name, p.slug, p.price, p.sale_price, p.featured_image, p.stock_quantity, p.stock_status
             FROM $items_table wi
             JOIN $products_table p ON wi.product_id = p.id
             WHERE wi.wishlist_id = %d
             ORDER BY wi.added_at DESC",
            $wishlistId
        ));

        foreach ($items as &$item) {
            if ($item->featured_image) {
                $item->image_url = wp_get_attachment_image_url($item->featured_image, 'medium');
            }
            $item->in_stock = $item->stock_status === 'instock' || ($item->stock_quantity === null || $item->stock_quantity > 0);
        }

        return $items;
    }

    /**
     * Get item count
     */
    public static function getItemCount(int $wishlistId): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlist_items';

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE wishlist_id = %d",
            $wishlistId
        ));
    }

    /**
     * Check if product is in wishlist
     */
    public static function isInWishlist(int $wishlistId, int $productId, ?int $variationId = null): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlist_items';

        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE wishlist_id = %d AND product_id = %d AND (variation_id = %d OR (variation_id IS NULL AND %d IS NULL))",
            $wishlistId, $productId, $variationId, $variationId
        ));

        return $count > 0;
    }

    /**
     * Toggle wishlist public/private
     */
    public static function setPublic(int $wishlistId, bool $isPublic): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlists';

        return $wpdb->update($table, ['is_public' => $isPublic ? 1 : 0], ['id' => $wishlistId]) !== false;
    }

    /**
     * Merge session wishlist into user wishlist
     */
    public static function mergeWishlists(int $userWishlistId, int $sessionWishlistId): void
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_wishlist_items';
        $wishlists_table = $wpdb->prefix . 'cartt_wishlists';

        // Get session items
        $sessionItems = $wpdb->get_results($wpdb->prepare(
            "SELECT product_id, variation_id FROM $items_table WHERE wishlist_id = %d",
            $sessionWishlistId
        ));

        foreach ($sessionItems as $item) {
            self::addItem($userWishlistId, $item->product_id, $item->variation_id);
        }

        // Delete session wishlist
        $wpdb->delete($items_table, ['wishlist_id' => $sessionWishlistId]);
        $wpdb->delete($wishlists_table, ['id' => $sessionWishlistId]);
    }

    /**
     * Set notification preferences
     */
    public static function setNotifications(int $wishlistId, int $productId, bool $onSale = false, bool $onStock = false): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wishlist_items';

        return $wpdb->update($table, [
            'notify_on_sale' => $onSale ? 1 : 0,
            'notify_on_stock' => $onStock ? 1 : 0,
        ], [
            'wishlist_id' => $wishlistId,
            'product_id' => $productId,
        ]) !== false;
    }

    /**
     * Get items needing stock notification
     */
    public static function getStockNotificationSubscribers(int $productId): array
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_wishlist_items';
        $wishlists_table = $wpdb->prefix . 'cartt_wishlists';
        $customers_table = $wpdb->prefix . 'cartt_customers';

        return $wpdb->get_results($wpdb->prepare(
            "SELECT wi.*, w.user_id, c.email
             FROM $items_table wi
             JOIN $wishlists_table w ON wi.wishlist_id = w.id
             LEFT JOIN $customers_table c ON w.user_id = c.user_id
             WHERE wi.product_id = %d AND wi.notify_on_stock = 1",
            $productId
        ));
    }

    /**
     * Move all items to cart
     */
    public static function moveAllToCart(int $wishlistId): int
    {
        $items = self::getItems($wishlistId);
        $added = 0;

        foreach ($items as $item) {
            if ($item->in_stock) {
                CartService::addItem($item->product_id, 1, $item->variation_id ? ['variation_id' => $item->variation_id] : []);
                self::removeItem($wishlistId, $item->product_id, $item->variation_id);
                $added++;
            }
        }

        return $added;
    }
}
