<?php
/**
 * Wholesale Service (B2B)
 * 
 * Features:
 * - Tiered pricing (quantity discounts)
 * - Customer-specific pricing
 * - Wholesale customer roles
 * - Minimum order quantities
 * - Quote requests
 * - Tax exemption
 * - Hidden prices until login
 * - Bulk order forms
 * - Payment terms (Net 30, etc.)
 * 
 * @package Cartt
 * @since 1.4.0
 */

declare(strict_types=1);

namespace Cartt\Services;

class WholesaleService
{
    private static ?self $instance = null;
    private array $settings;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->settings = get_option('cartt_wholesale', $this->get_defaults());

        // Filters
        add_filter('cartt_product_price', [$this, 'apply_wholesale_price'], 10, 2);
        add_filter('cartt_cart_item_price', [$this, 'apply_tiered_price'], 10, 3);
        add_filter('cartt_can_checkout', [$this, 'check_minimum_order'], 10, 1);
        add_filter('cartt_apply_tax', [$this, 'check_tax_exempt'], 10, 2);

        // AJAX
        add_action('wp_ajax_cartt_submit_quote', [$this, 'ajax_submit_quote']);
        add_action('wp_ajax_nopriv_cartt_submit_quote', [$this, 'ajax_submit_quote']);
        add_action('wp_ajax_cartt_wholesale_apply', [$this, 'ajax_apply_wholesale']);
        add_action('wp_ajax_nopriv_cartt_wholesale_apply', [$this, 'ajax_apply_wholesale']);
        add_action('wp_ajax_cartt_admin_quote_action', [$this, 'ajax_admin_quote_action']);
    }

    /**
     * Default settings
     */
    private function get_defaults(): array
    {
        return [
            'enabled' => true,
            'hide_prices' => false,
            'require_approval' => true,
            'min_order_amount' => 0,
            'default_discount' => 10,
            'tiers' => [
                ['name' => 'Bronze', 'min_orders' => 0, 'discount' => 5],
                ['name' => 'Silver', 'min_orders' => 5, 'discount' => 10],
                ['name' => 'Gold', 'min_orders' => 20, 'discount' => 15],
                ['name' => 'Platinum', 'min_orders' => 50, 'discount' => 20]
            ],
            'payment_terms' => ['immediate', 'net_15', 'net_30', 'net_60']
        ];
    }

    /**
     * Check if user is wholesale customer
     */
    public function is_wholesale_customer(?int $user_id = null): bool
    {
        $user_id = $user_id ?: get_current_user_id();
        if (!$user_id) return false;

        $user = get_user_by('id', $user_id);
        return $user && in_array('cartt_wholesale', $user->roles);
    }

    /**
     * Get customer wholesale profile
     */
    public function get_wholesale_profile(int $customer_id): ?array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wholesale_customers';

        $profile = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE customer_id = %d",
            $customer_id
        ), ARRAY_A);

        if (!$profile) return null;

        // Add tier info
        $profile['tier'] = $this->calculate_tier($customer_id);

        return $profile;
    }

    /**
     * Calculate customer tier based on order history
     */
    public function calculate_tier(int $customer_id): array
    {
        global $wpdb;
        $orders_table = $wpdb->prefix . 'cartt_orders';

        $order_count = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) FROM {$orders_table}
            WHERE customer_id = %d
            AND status IN ('completed', 'processing')
        ", $customer_id));

        $tiers = $this->settings['tiers'];
        $current_tier = $tiers[0];

        foreach ($tiers as $tier) {
            if ($order_count >= $tier['min_orders']) {
                $current_tier = $tier;
            }
        }

        return [
            'name' => $current_tier['name'],
            'discount' => $current_tier['discount'],
            'order_count' => $order_count,
            'next_tier' => $this->get_next_tier($order_count)
        ];
    }

    /**
     * Get next tier info
     */
    private function get_next_tier(int $order_count): ?array
    {
        $tiers = $this->settings['tiers'];

        foreach ($tiers as $tier) {
            if ($tier['min_orders'] > $order_count) {
                return [
                    'name' => $tier['name'],
                    'orders_needed' => $tier['min_orders'] - $order_count,
                    'discount' => $tier['discount']
                ];
            }
        }

        return null; // Already at highest tier
    }

    /**
     * Apply wholesale price to product
     */
    public function apply_wholesale_price(float $price, int $product_id): float
    {
        if (!$this->is_wholesale_customer()) {
            return $price;
        }

        $customer_id = get_current_user_id();

        // Check for customer-specific price
        $custom_price = $this->get_customer_price($customer_id, $product_id);
        if ($custom_price !== null) {
            return $custom_price;
        }

        // Apply tier discount
        $tier = $this->calculate_tier($customer_id);
        $discount = $tier['discount'];

        return $price * (1 - $discount / 100);
    }

    /**
     * Apply tiered/quantity pricing
     */
    public function apply_tiered_price(float $price, int $product_id, int $quantity): float
    {
        // Get quantity pricing tiers for product
        $tiers = $this->get_product_quantity_tiers($product_id);

        if (empty($tiers)) {
            return $price;
        }

        foreach (array_reverse($tiers) as $tier) {
            if ($quantity >= $tier['min_qty']) {
                if (isset($tier['price'])) {
                    return (float) $tier['price'];
                } elseif (isset($tier['discount'])) {
                    return $price * (1 - $tier['discount'] / 100);
                }
            }
        }

        return $price;
    }

    /**
     * Get product quantity pricing tiers
     */
    public function get_product_quantity_tiers(int $product_id): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_quantity_pricing';

        return $wpdb->get_results($wpdb->prepare("
            SELECT min_qty, max_qty, price, discount
            FROM {$table}
            WHERE product_id = %d
            ORDER BY min_qty ASC
        ", $product_id), ARRAY_A);
    }

    /**
     * Set quantity pricing tiers for product
     */
    public function set_product_quantity_tiers(int $product_id, array $tiers): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_quantity_pricing';

        // Clear existing
        $wpdb->delete($table, ['product_id' => $product_id]);

        // Insert new
        foreach ($tiers as $tier) {
            $wpdb->insert($table, [
                'product_id' => $product_id,
                'min_qty' => (int) $tier['min_qty'],
                'max_qty' => isset($tier['max_qty']) ? (int) $tier['max_qty'] : null,
                'price' => isset($tier['price']) ? (float) $tier['price'] : null,
                'discount' => isset($tier['discount']) ? (float) $tier['discount'] : null
            ]);
        }

        return true;
    }

    /**
     * Get customer-specific price
     */
    public function get_customer_price(int $customer_id, int $product_id): ?float
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customer_pricing';

        $price = $wpdb->get_var($wpdb->prepare("
            SELECT price FROM {$table}
            WHERE customer_id = %d AND product_id = %d
        ", $customer_id, $product_id));

        return $price !== null ? (float) $price : null;
    }

    /**
     * Set customer-specific price
     */
    public function set_customer_price(int $customer_id, int $product_id, float $price): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customer_pricing';

        return $wpdb->replace($table, [
            'customer_id' => $customer_id,
            'product_id' => $product_id,
            'price' => $price
        ]) !== false;
    }

    /**
     * Check minimum order requirement
     */
    public function check_minimum_order(array $checkout_data): array
    {
        if (!$this->is_wholesale_customer()) {
            return $checkout_data;
        }

        $profile = $this->get_wholesale_profile(get_current_user_id());
        $min_order = $profile['min_order_amount'] ?? $this->settings['min_order_amount'];

        if ($min_order > 0 && $checkout_data['subtotal'] < $min_order) {
            $checkout_data['can_checkout'] = false;
            $checkout_data['error'] = sprintf(
                'Minimum order amount is %s. Your current total is %s.',
                cartt_format_price($min_order),
                cartt_format_price($checkout_data['subtotal'])
            );
        }

        return $checkout_data;
    }

    /**
     * Check tax exemption
     */
    public function check_tax_exempt(bool $apply_tax, int $customer_id): bool
    {
        if (!$this->is_wholesale_customer($customer_id)) {
            return $apply_tax;
        }

        $profile = $this->get_wholesale_profile($customer_id);
        
        if ($profile && $profile['tax_exempt'] && $profile['tax_id']) {
            return false;
        }

        return $apply_tax;
    }

    /**
     * Apply for wholesale account
     */
    public function apply_wholesale(array $data): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wholesale_applications';

        $required = ['company_name', 'contact_name', 'email', 'phone'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                return ['success' => false, 'error' => "Field '{$field}' is required"];
            }
        }

        // Check if already applied
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE email = %s AND status = 'pending'",
            $data['email']
        ));

        if ($existing) {
            return ['success' => false, 'error' => 'Application already pending'];
        }

        $wpdb->insert($table, [
            'user_id' => get_current_user_id() ?: null,
            'company_name' => sanitize_text_field($data['company_name']),
            'contact_name' => sanitize_text_field($data['contact_name']),
            'email' => sanitize_email($data['email']),
            'phone' => sanitize_text_field($data['phone']),
            'address' => sanitize_textarea_field($data['address'] ?? ''),
            'city' => sanitize_text_field($data['city'] ?? ''),
            'state' => sanitize_text_field($data['state'] ?? ''),
            'country' => sanitize_text_field($data['country'] ?? ''),
            'postal_code' => sanitize_text_field($data['postal_code'] ?? ''),
            'tax_id' => sanitize_text_field($data['tax_id'] ?? ''),
            'business_type' => sanitize_text_field($data['business_type'] ?? ''),
            'expected_volume' => sanitize_text_field($data['expected_volume'] ?? ''),
            'website' => esc_url_raw($data['website'] ?? ''),
            'notes' => sanitize_textarea_field($data['notes'] ?? ''),
            'status' => 'pending',
            'created_at' => current_time('mysql')
        ]);

        $this->notify_wholesale_application($wpdb->insert_id);

        return [
            'success' => true,
            'message' => 'Application submitted. We will review and contact you soon.'
        ];
    }

    /**
     * Approve wholesale application
     */
    public function approve_application(int $application_id, array $options = []): bool
    {
        global $wpdb;
        $apps_table = $wpdb->prefix . 'cartt_wholesale_applications';
        $wholesale_table = $wpdb->prefix . 'cartt_wholesale_customers';

        $application = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$apps_table} WHERE id = %d",
            $application_id
        ));

        if (!$application) return false;

        // Create user if needed
        $user_id = $application->user_id;
        if (!$user_id) {
            $user_id = wp_create_user(
                $application->email,
                wp_generate_password(),
                $application->email
            );

            if (is_wp_error($user_id)) {
                return false;
            }

            wp_update_user([
                'ID' => $user_id,
                'first_name' => explode(' ', $application->contact_name)[0],
                'last_name' => implode(' ', array_slice(explode(' ', $application->contact_name), 1))
            ]);
        }

        // Add wholesale role
        $user = get_user_by('id', $user_id);
        $user->add_role('cartt_wholesale');

        // Create wholesale profile
        $wpdb->insert($wholesale_table, [
            'customer_id' => $user_id,
            'company_name' => $application->company_name,
            'tax_id' => $application->tax_id,
            'tax_exempt' => !empty($application->tax_id) ? 1 : 0,
            'discount_percent' => $options['discount'] ?? $this->settings['default_discount'],
            'min_order_amount' => $options['min_order'] ?? $this->settings['min_order_amount'],
            'payment_terms' => $options['payment_terms'] ?? 'immediate',
            'credit_limit' => $options['credit_limit'] ?? 0,
            'status' => 'active',
            'created_at' => current_time('mysql')
        ]);

        // Update application
        $wpdb->update($apps_table, [
            'status' => 'approved',
            'approved_at' => current_time('mysql'),
            'approved_by' => get_current_user_id()
        ], ['id' => $application_id]);

        // Send welcome email
        $this->send_wholesale_welcome($application->email, $user_id);

        return true;
    }

    /**
     * Reject wholesale application
     */
    public function reject_application(int $application_id, string $reason = ''): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wholesale_applications';

        $application = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $application_id
        ));

        if (!$application) return false;

        $wpdb->update($table, [
            'status' => 'rejected',
            'rejection_reason' => $reason,
            'approved_at' => current_time('mysql'),
            'approved_by' => get_current_user_id()
        ], ['id' => $application_id]);

        // Notify applicant
        $this->send_rejection_email($application->email, $reason);

        return true;
    }

    /**
     * Submit quote request
     */
    public function submit_quote(array $data): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_quotes';

        $required = ['items', 'email'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                return ['success' => false, 'error' => "Field '{$field}' is required"];
            }
        }

        $quote_number = 'Q-' . strtoupper(wp_generate_password(8, false));

        $wpdb->insert($table, [
            'quote_number' => $quote_number,
            'customer_id' => get_current_user_id() ?: null,
            'customer_name' => sanitize_text_field($data['name'] ?? ''),
            'customer_email' => sanitize_email($data['email']),
            'customer_phone' => sanitize_text_field($data['phone'] ?? ''),
            'company_name' => sanitize_text_field($data['company'] ?? ''),
            'items' => json_encode($data['items']),
            'notes' => sanitize_textarea_field($data['notes'] ?? ''),
            'status' => 'pending',
            'created_at' => current_time('mysql'),
            'expires_at' => date('Y-m-d H:i:s', strtotime('+30 days'))
        ]);

        $quote_id = $wpdb->insert_id;

        // Notify admin
        $this->notify_quote_request($quote_id);

        return [
            'success' => true,
            'quote_id' => $quote_id,
            'quote_number' => $quote_number
        ];
    }

    /**
     * Get quote by ID
     */
    public function get_quote(int $quote_id): ?array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_quotes';

        $quote = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $quote_id
        ), ARRAY_A);

        if ($quote) {
            $quote['items'] = json_decode($quote['items'], true);
        }

        return $quote;
    }

    /**
     * Update quote with pricing
     */
    public function update_quote(int $quote_id, array $data): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_quotes';

        $update_data = [
            'updated_at' => current_time('mysql')
        ];

        if (isset($data['items'])) {
            $update_data['items'] = json_encode($data['items']);
        }
        if (isset($data['subtotal'])) {
            $update_data['subtotal'] = (float) $data['subtotal'];
        }
        if (isset($data['discount'])) {
            $update_data['discount'] = (float) $data['discount'];
        }
        if (isset($data['total'])) {
            $update_data['total'] = (float) $data['total'];
        }
        if (isset($data['status'])) {
            $update_data['status'] = $data['status'];
        }
        if (isset($data['admin_notes'])) {
            $update_data['admin_notes'] = $data['admin_notes'];
        }
        if (isset($data['expires_at'])) {
            $update_data['expires_at'] = $data['expires_at'];
        }

        return $wpdb->update($table, $update_data, ['id' => $quote_id]) !== false;
    }

    /**
     * Send quote to customer
     */
    public function send_quote(int $quote_id): bool
    {
        $quote = $this->get_quote($quote_id);
        if (!$quote) return false;

        $this->update_quote($quote_id, ['status' => 'sent']);

        // Send email with quote details
        $subject = "Your Quote #{$quote['quote_number']}";
        $message = sprintf(
            "Thank you for your quote request.\n\n" .
            "Quote Number: %s\n" .
            "Total: %s\n" .
            "Valid Until: %s\n\n" .
            "View your quote: %s",
            $quote['quote_number'],
            cartt_format_price($quote['total']),
            date('F j, Y', strtotime($quote['expires_at'])),
            home_url('/quote/' . $quote['quote_number'])
        );

        return wp_mail($quote['customer_email'], $subject, $message);
    }

    /**
     * Convert quote to order
     */
    public function convert_quote_to_order(int $quote_id): ?int
    {
        $quote = $this->get_quote($quote_id);
        if (!$quote || $quote['status'] !== 'sent') {
            return null;
        }

        // Check expiration
        if (strtotime($quote['expires_at']) < time()) {
            return null;
        }

        // Create order from quote items
        // This would integrate with OrderService
        // For now, just mark quote as accepted
        $this->update_quote($quote_id, ['status' => 'accepted']);

        return $quote_id; // Would return order_id in full implementation
    }

    /**
     * Get pending applications (admin)
     */
    public function get_pending_applications(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_wholesale_applications';

        return $wpdb->get_results(
            "SELECT * FROM {$table} WHERE status = 'pending' ORDER BY created_at ASC",
            ARRAY_A
        );
    }

    /**
     * Get pending quotes (admin)
     */
    public function get_pending_quotes(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_quotes';

        return $wpdb->get_results(
            "SELECT * FROM {$table} WHERE status = 'pending' ORDER BY created_at ASC",
            ARRAY_A
        );
    }

    /**
     * Render bulk order form
     */
    public function render_bulk_order_form(array $product_ids = []): string
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        if (empty($product_ids)) {
            $products = $wpdb->get_results(
                "SELECT id, name, sku, price, stock_quantity FROM {$table} 
                 WHERE status = 'published' ORDER BY name ASC",
                ARRAY_A
            );
        } else {
            $ids = implode(',', array_map('intval', $product_ids));
            $products = $wpdb->get_results(
                "SELECT id, name, sku, price, stock_quantity FROM {$table} 
                 WHERE id IN ({$ids}) ORDER BY name ASC",
                ARRAY_A
            );
        }

        ob_start();
        ?>
        <form class="cartt-bulk-order-form" method="post">
            <table class="cartt-bulk-table">
                <thead>
                    <tr>
                        <th>SKU</th>
                        <th>Product</th>
                        <th>Price</th>
                        <th>Stock</th>
                        <th>Quantity</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($products as $product): ?>
                    <tr>
                        <td><?php echo esc_html($product['sku']); ?></td>
                        <td><?php echo esc_html($product['name']); ?></td>
                        <td><?php echo cartt_format_price($this->apply_wholesale_price($product['price'], $product['id'])); ?></td>
                        <td><?php echo $product['stock_quantity'] !== null ? intval($product['stock_quantity']) : 'In Stock'; ?></td>
                        <td>
                            <input type="number" 
                                   name="items[<?php echo intval($product['id']); ?>]" 
                                   min="0" 
                                   value="0"
                                   data-product-id="<?php echo intval($product['id']); ?>">
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <div class="bulk-order-actions">
                <button type="submit" name="action" value="add_to_cart" class="btn-primary">Add to Cart</button>
                <button type="submit" name="action" value="request_quote" class="btn-secondary">Request Quote</button>
            </div>
        </form>
        <?php
        return ob_get_clean();
    }

    /**
     * AJAX: Submit quote
     */
    public function ajax_submit_quote(): void
    {
        check_ajax_referer('cartt_wholesale', 'nonce');

        $result = $this->submit_quote([
            'items' => $_POST['items'] ?? [],
            'name' => sanitize_text_field($_POST['name'] ?? ''),
            'email' => sanitize_email($_POST['email'] ?? ''),
            'phone' => sanitize_text_field($_POST['phone'] ?? ''),
            'company' => sanitize_text_field($_POST['company'] ?? ''),
            'notes' => sanitize_textarea_field($_POST['notes'] ?? '')
        ]);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * AJAX: Apply for wholesale
     */
    public function ajax_apply_wholesale(): void
    {
        check_ajax_referer('cartt_wholesale', 'nonce');

        $result = $this->apply_wholesale($_POST);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * AJAX: Admin quote action
     */
    public function ajax_admin_quote_action(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $action = sanitize_text_field($_POST['quote_action'] ?? '');
        $id = intval($_POST['id'] ?? 0);

        switch ($action) {
            case 'approve_application':
                $result = $this->approve_application($id, $_POST['options'] ?? []);
                break;
            case 'reject_application':
                $result = $this->reject_application($id, $_POST['reason'] ?? '');
                break;
            case 'update_quote':
                $result = $this->update_quote($id, $_POST);
                break;
            case 'send_quote':
                $result = $this->send_quote($id);
                break;
            default:
                wp_send_json_error('Invalid action');
        }

        wp_send_json_success(['result' => $result]);
    }

    /**
     * Notify admin of wholesale application
     */
    private function notify_wholesale_application(int $app_id): void
    {
        $admin_email = get_option('admin_email');
        $subject = 'New Wholesale Account Application';
        $message = sprintf(
            "A new wholesale account application has been submitted.\n\n" .
            "Review: %s",
            admin_url('admin.php?page=cartt-wholesale')
        );

        wp_mail($admin_email, $subject, $message);
    }

    /**
     * Notify admin of quote request
     */
    private function notify_quote_request(int $quote_id): void
    {
        $quote = $this->get_quote($quote_id);
        $admin_email = get_option('admin_email');

        $subject = "New Quote Request #{$quote['quote_number']}";
        $message = sprintf(
            "A new quote request has been submitted.\n\n" .
            "Company: %s\n" .
            "Contact: %s\n" .
            "Email: %s\n\n" .
            "Review: %s",
            $quote['company_name'],
            $quote['customer_name'],
            $quote['customer_email'],
            admin_url('admin.php?page=cartt-quotes')
        );

        wp_mail($admin_email, $subject, $message);
    }

    /**
     * Send wholesale welcome email
     */
    private function send_wholesale_welcome(string $email, int $user_id): void
    {
        $password_reset = wp_lostpassword_url();

        $subject = 'Welcome to Our Wholesale Program';
        $message = sprintf(
            "Your wholesale account has been approved!\n\n" .
            "You now have access to wholesale pricing and features.\n\n" .
            "To set your password, visit: %s\n\n" .
            "Then log in at: %s",
            $password_reset,
            wp_login_url(home_url('/wholesale'))
        );

        wp_mail($email, $subject, $message);
    }

    /**
     * Send rejection email
     */
    private function send_rejection_email(string $email, string $reason): void
    {
        $subject = 'Wholesale Application Update';
        $message = "Thank you for your interest in our wholesale program.\n\n" .
                   "Unfortunately, we are unable to approve your application at this time.\n\n" .
                   ($reason ? "Reason: {$reason}\n\n" : "") .
                   "If you have questions, please contact us.";

        wp_mail($email, $subject, $message);
    }

    /**
     * Get count of pending wholesale applications
     */
    public static function getPendingApplicationsCount(): int
    {
        global $wpdb;
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_wholesale_applications WHERE status = 'pending'"
        );
    }
}
