<?php
/**
 * Vendor Service (Multi-Vendor Marketplace)
 * 
 * Features:
 * - Vendor registration and approval workflow
 * - Vendor profiles and storefronts
 * - Product ownership and management
 * - Vendor dashboard
 * - Store settings (logo, banner, policies)
 * - Vendor ratings and reviews
 * - Seller verification badges
 * 
 * @package Cartt
 * @since 1.4.0
 */

declare(strict_types=1);

namespace Cartt\Services;

class VendorService
{
    private static ?self $instance = null;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // AJAX handlers
        add_action('wp_ajax_cartt_vendor_register', [$this, 'ajax_register']);
        add_action('wp_ajax_nopriv_cartt_vendor_register', [$this, 'ajax_register']);
        add_action('wp_ajax_cartt_vendor_update_profile', [$this, 'ajax_update_profile']);
        add_action('wp_ajax_cartt_vendor_add_product', [$this, 'ajax_add_product']);
        add_action('wp_ajax_cartt_vendor_dashboard_data', [$this, 'ajax_dashboard_data']);

        // Admin approval
        add_action('wp_ajax_cartt_admin_vendor_action', [$this, 'ajax_admin_action']);
    }

    /**
     * Register a new vendor
     */
    public function register(array $data): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        // Validate required fields
        $required = ['user_id', 'store_name', 'email'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                return ['success' => false, 'error' => "Field '{$field}' is required"];
            }
        }

        // Check if user already has a vendor account
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE user_id = %d",
            $data['user_id']
        ));

        if ($existing) {
            return ['success' => false, 'error' => 'You already have a vendor account'];
        }

        // Check store name uniqueness
        $name_exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE store_name = %s",
            $data['store_name']
        ));

        if ($name_exists) {
            return ['success' => false, 'error' => 'Store name already taken'];
        }

        // Generate store slug
        $slug = sanitize_title($data['store_name']);
        $base_slug = $slug;
        $counter = 1;
        while ($wpdb->get_var($wpdb->prepare("SELECT id FROM {$table} WHERE store_slug = %s", $slug))) {
            $slug = $base_slug . '-' . $counter++;
        }

        // Determine initial status
        $auto_approve = get_option('cartt_vendor_auto_approve', false);
        $status = $auto_approve ? 'active' : 'pending';

        $result = $wpdb->insert($table, [
            'user_id' => (int) $data['user_id'],
            'store_name' => sanitize_text_field($data['store_name']),
            'store_slug' => $slug,
            'email' => sanitize_email($data['email']),
            'phone' => sanitize_text_field($data['phone'] ?? ''),
            'description' => sanitize_textarea_field($data['description'] ?? ''),
            'logo_url' => esc_url_raw($data['logo_url'] ?? ''),
            'banner_url' => esc_url_raw($data['banner_url'] ?? ''),
            'address' => sanitize_textarea_field($data['address'] ?? ''),
            'city' => sanitize_text_field($data['city'] ?? ''),
            'state' => sanitize_text_field($data['state'] ?? ''),
            'country' => sanitize_text_field($data['country'] ?? ''),
            'postal_code' => sanitize_text_field($data['postal_code'] ?? ''),
            'commission_rate' => (float) get_option('cartt_default_commission_rate', 10),
            'status' => $status,
            'verified' => 0,
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        ]);

        if (!$result) {
            return ['success' => false, 'error' => 'Failed to create vendor account'];
        }

        $vendor_id = $wpdb->insert_id;

        // Add vendor role to user
        $user = get_user_by('id', $data['user_id']);
        if ($user) {
            $user->add_role('cartt_vendor');
        }

        // Send notification
        if ($status === 'pending') {
            $this->send_pending_notification($vendor_id);
        } else {
            $this->send_approved_notification($vendor_id);
        }

        return [
            'success' => true,
            'vendor_id' => $vendor_id,
            'status' => $status
        ];
    }

    /**
     * Get vendor by ID
     */
    public function get(int $vendor_id): ?array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $vendor = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $vendor_id
        ), ARRAY_A);

        if (!$vendor) return null;

        // Add stats
        $vendor['stats'] = $this->get_vendor_stats($vendor_id);

        return $vendor;
    }

    /**
     * Get vendor by user ID
     */
    public function get_by_user(int $user_id): ?array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $vendor = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE user_id = %d",
            $user_id
        ), ARRAY_A);

        if (!$vendor) return null;

        $vendor['stats'] = $this->get_vendor_stats($vendor['id']);

        return $vendor;
    }

    /**
     * Get vendor by store slug
     */
    public function get_by_slug(string $slug): ?array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $vendor = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE store_slug = %s AND status = 'active'",
            $slug
        ), ARRAY_A);

        if (!$vendor) return null;

        $vendor['stats'] = $this->get_vendor_stats($vendor['id']);

        return $vendor;
    }

    /**
     * Update vendor profile
     */
    public function update(int $vendor_id, array $data): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $allowed = [
            'store_name', 'description', 'logo_url', 'banner_url', 'phone',
            'address', 'city', 'state', 'country', 'postal_code',
            'payment_email', 'payment_method', 'tax_id', 'return_policy',
            'shipping_policy', 'social_facebook', 'social_twitter', 
            'social_instagram', 'vacation_mode', 'vacation_message'
        ];

        $update_data = [];
        foreach ($allowed as $field) {
            if (isset($data[$field])) {
                $update_data[$field] = $data[$field];
            }
        }

        if (empty($update_data)) return false;

        $update_data['updated_at'] = current_time('mysql');

        return $wpdb->update($table, $update_data, ['id' => $vendor_id]) !== false;
    }

    /**
     * Approve vendor
     */
    public function approve(int $vendor_id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $result = $wpdb->update(
            $table,
            ['status' => 'active', 'updated_at' => current_time('mysql')],
            ['id' => $vendor_id]
        );

        if ($result) {
            $this->send_approved_notification($vendor_id);
        }

        return $result !== false;
    }

    /**
     * Reject vendor
     */
    public function reject(int $vendor_id, string $reason = ''): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $result = $wpdb->update(
            $table,
            [
                'status' => 'rejected',
                'rejection_reason' => $reason,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $vendor_id]
        );

        if ($result) {
            $this->send_rejected_notification($vendor_id, $reason);
        }

        return $result !== false;
    }

    /**
     * Suspend vendor
     */
    public function suspend(int $vendor_id, string $reason = ''): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        // Also hide all vendor products
        $products_table = $wpdb->prefix . 'cartt_products';
        $wpdb->update(
            $products_table,
            ['status' => 'suspended'],
            ['vendor_id' => $vendor_id]
        );

        return $wpdb->update(
            $table,
            [
                'status' => 'suspended',
                'suspension_reason' => $reason,
                'updated_at' => current_time('mysql')
            ],
            ['id' => $vendor_id]
        ) !== false;
    }

    /**
     * Verify vendor
     */
    public function verify(int $vendor_id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        return $wpdb->update(
            $table,
            ['verified' => 1, 'verified_at' => current_time('mysql')],
            ['id' => $vendor_id]
        ) !== false;
    }

    /**
     * Get vendor stats
     */
    public function get_vendor_stats(int $vendor_id): array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $reviews_table = $wpdb->prefix . 'cartt_reviews';

        // Product count
        $products = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$products_table} WHERE vendor_id = %d AND status = 'published'",
            $vendor_id
        ));

        // Sales stats
        $sales = $wpdb->get_row($wpdb->prepare("
            SELECT 
                COUNT(DISTINCT o.id) as order_count,
                COALESCE(SUM(oi.subtotal), 0) as total_sales
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE p.vendor_id = %d
            AND o.status IN ('completed', 'processing')
        ", $vendor_id), ARRAY_A);

        // Average rating
        $rating = $wpdb->get_row($wpdb->prepare("
            SELECT AVG(rating) as avg_rating, COUNT(*) as review_count
            FROM {$reviews_table} r
            JOIN {$products_table} p ON p.id = r.product_id
            WHERE p.vendor_id = %d AND r.status = 'approved'
        ", $vendor_id), ARRAY_A);

        return [
            'products' => $products,
            'orders' => (int) ($sales['order_count'] ?? 0),
            'total_sales' => (float) ($sales['total_sales'] ?? 0),
            'avg_rating' => round((float) ($rating['avg_rating'] ?? 0), 1),
            'review_count' => (int) ($rating['review_count'] ?? 0)
        ];
    }

    /**
     * Get vendor products
     */
    public function get_products(int $vendor_id, array $args = []): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        $defaults = [
            'status' => null,
            'limit' => 20,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC'
        ];
        $args = wp_parse_args($args, $defaults);

        $where = $wpdb->prepare("WHERE vendor_id = %d", $vendor_id);
        if ($args['status']) {
            $where .= $wpdb->prepare(" AND status = %s", $args['status']);
        }

        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']) ?: 'created_at DESC';

        return $wpdb->get_results("
            SELECT * FROM {$table}
            {$where}
            ORDER BY {$orderby}
            LIMIT {$args['limit']} OFFSET {$args['offset']}
        ", ARRAY_A);
    }

    /**
     * Get vendor orders
     */
    public function get_orders(int $vendor_id, array $args = []): array
    {
        global $wpdb;
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $products_table = $wpdb->prefix . 'cartt_products';

        $defaults = [
            'status' => null,
            'limit' => 20,
            'offset' => 0
        ];
        $args = wp_parse_args($args, $defaults);

        $where = "";
        if ($args['status']) {
            $where = $wpdb->prepare("AND o.status = %s", $args['status']);
        }

        return $wpdb->get_results($wpdb->prepare("
            SELECT DISTINCT
                o.id,
                o.order_number,
                o.status,
                o.billing_first_name,
                o.billing_last_name,
                o.billing_email,
                o.created_at,
                (
                    SELECT SUM(oi2.subtotal)
                    FROM {$items_table} oi2
                    JOIN {$products_table} p2 ON p2.id = oi2.product_id
                    WHERE oi2.order_id = o.id AND p2.vendor_id = %d
                ) as vendor_subtotal
            FROM {$orders_table} o
            JOIN {$items_table} oi ON oi.order_id = o.id
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE p.vendor_id = %d
            {$where}
            ORDER BY o.created_at DESC
            LIMIT %d OFFSET %d
        ", $vendor_id, $vendor_id, $args['limit'], $args['offset']), ARRAY_A);
    }

    /**
     * List all vendors (admin)
     */
    public function list_vendors(array $args = []): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        $defaults = [
            'status' => null,
            'search' => '',
            'limit' => 20,
            'offset' => 0,
            'orderby' => 'created_at',
            'order' => 'DESC'
        ];
        $args = wp_parse_args($args, $defaults);

        $where = "WHERE 1=1";
        if ($args['status']) {
            $where .= $wpdb->prepare(" AND status = %s", $args['status']);
        }
        if ($args['search']) {
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $where .= $wpdb->prepare(" AND (store_name LIKE %s OR email LIKE %s)", $search, $search);
        }

        $orderby = sanitize_sql_orderby($args['orderby'] . ' ' . $args['order']) ?: 'created_at DESC';

        $vendors = $wpdb->get_results("
            SELECT * FROM {$table}
            {$where}
            ORDER BY {$orderby}
            LIMIT {$args['limit']} OFFSET {$args['offset']}
        ", ARRAY_A);

        $total = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} {$where}");

        // Add stats to each vendor
        foreach ($vendors as &$vendor) {
            $vendor['stats'] = $this->get_vendor_stats($vendor['id']);
        }

        return [
            'vendors' => $vendors,
            'total' => $total
        ];
    }

    /**
     * Get pending vendors count
     */
    public function get_pending_count(): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status = 'pending'");
    }

    /**
     * Check if user is vendor
     */
    public function is_vendor(int $user_id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';
        return (bool) $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE user_id = %d AND status = 'active'",
            $user_id
        ));
    }

    /**
     * AJAX: Register vendor
     */
    public function ajax_register(): void
    {
        check_ajax_referer('cartt_vendor', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error('Please log in to become a vendor');
        }

        $result = $this->register([
            'user_id' => get_current_user_id(),
            'store_name' => sanitize_text_field($_POST['store_name'] ?? ''),
            'email' => sanitize_email($_POST['email'] ?? ''),
            'phone' => sanitize_text_field($_POST['phone'] ?? ''),
            'description' => sanitize_textarea_field($_POST['description'] ?? '')
        ]);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * AJAX: Update profile
     */
    public function ajax_update_profile(): void
    {
        check_ajax_referer('cartt_vendor', 'nonce');

        $vendor = $this->get_by_user(get_current_user_id());
        if (!$vendor) {
            wp_send_json_error('Vendor not found');
        }

        $result = $this->update($vendor['id'], $_POST);
        wp_send_json_success(['updated' => $result]);
    }

    /**
     * AJAX: Dashboard data
     */
    public function ajax_dashboard_data(): void
    {
        check_ajax_referer('cartt_vendor', 'nonce');

        $vendor = $this->get_by_user(get_current_user_id());
        if (!$vendor) {
            wp_send_json_error('Vendor not found');
        }

        $period = sanitize_text_field($_POST['period'] ?? '30days');
        $commission_service = VendorCommissionService::instance();

        wp_send_json_success([
            'vendor' => $vendor,
            'stats' => $vendor['stats'],
            'earnings' => $commission_service->get_vendor_earnings($vendor['id'], $period),
            'recent_orders' => $this->get_orders($vendor['id'], ['limit' => 5]),
            'pending_earnings' => $commission_service->get_pending_earnings($vendor['id'])
        ]);
    }

    /**
     * AJAX: Admin vendor action
     */
    public function ajax_admin_action(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $vendor_id = intval($_POST['vendor_id'] ?? 0);
        $action = sanitize_text_field($_POST['vendor_action'] ?? '');
        $reason = sanitize_textarea_field($_POST['reason'] ?? '');

        switch ($action) {
            case 'approve':
                $result = $this->approve($vendor_id);
                break;
            case 'reject':
                $result = $this->reject($vendor_id, $reason);
                break;
            case 'suspend':
                $result = $this->suspend($vendor_id, $reason);
                break;
            case 'verify':
                $result = $this->verify($vendor_id);
                break;
            default:
                wp_send_json_error('Invalid action');
        }

        wp_send_json_success(['result' => $result]);
    }

    /**
     * Send pending notification to admin
     */
    private function send_pending_notification(int $vendor_id): void
    {
        $vendor = $this->get($vendor_id);
        if (!$vendor) return;

        $admin_email = get_option('admin_email');
        $subject = 'New Vendor Registration Pending Approval';
        $message = sprintf(
            "A new vendor has registered and is pending approval:\n\n" .
            "Store Name: %s\n" .
            "Email: %s\n\n" .
            "Review and approve: %s",
            $vendor['store_name'],
            $vendor['email'],
            admin_url('admin.php?page=cartt-vendors')
        );

        wp_mail($admin_email, $subject, $message);
    }

    /**
     * Send approval notification
     */
    private function send_approved_notification(int $vendor_id): void
    {
        $vendor = $this->get($vendor_id);
        if (!$vendor) return;

        $subject = 'Your Vendor Account Has Been Approved!';
        $message = sprintf(
            "Congratulations! Your vendor account '%s' has been approved.\n\n" .
            "You can now start adding products and selling on our marketplace.\n\n" .
            "Visit your dashboard: %s",
            $vendor['store_name'],
            home_url('/vendor/dashboard')
        );

        wp_mail($vendor['email'], $subject, $message);
    }

    /**
     * Send rejection notification
     */
    private function send_rejected_notification(int $vendor_id, string $reason): void
    {
        $vendor = $this->get($vendor_id);
        if (!$vendor) return;

        $subject = 'Vendor Application Status Update';
        $message = sprintf(
            "Thank you for your interest in becoming a vendor.\n\n" .
            "Unfortunately, your application for '%s' was not approved at this time.\n\n" .
            "%s\n\n" .
            "If you have questions, please contact support.",
            $vendor['store_name'],
            $reason ? "Reason: {$reason}" : ""
        );

        wp_mail($vendor['email'], $subject, $message);
    }

    /**
     * Render vendor storefront
     */
    public function render_storefront(string $slug): string
    {
        $vendor = $this->get_by_slug($slug);
        if (!$vendor) {
            return '<p>Store not found.</p>';
        }

        if ($vendor['vacation_mode']) {
            return $this->render_vacation_mode($vendor);
        }

        $products = $this->get_products($vendor['id'], ['status' => 'published', 'limit' => 50]);

        ob_start();
        ?>
        <div class="cartt-vendor-storefront">
            <div class="vendor-header" style="<?php echo $vendor['banner_url'] ? 'background-image:url(' . esc_url($vendor['banner_url']) . ')' : ''; ?>">
                <?php if ($vendor['logo_url']): ?>
                <img src="<?php echo esc_url($vendor['logo_url']); ?>" alt="<?php echo esc_attr($vendor['store_name']); ?>" class="vendor-logo">
                <?php endif; ?>
                <div class="vendor-info">
                    <h1>
                        <?php echo esc_html($vendor['store_name']); ?>
                        <?php if ($vendor['verified']): ?>
                        <span class="verified-badge" title="Verified Seller">✓</span>
                        <?php endif; ?>
                    </h1>
                    <?php if ($vendor['stats']['avg_rating'] > 0): ?>
                    <div class="vendor-rating">
                        <?php echo $this->render_stars($vendor['stats']['avg_rating']); ?>
                        <span>(<?php echo $vendor['stats']['review_count']; ?> reviews)</span>
                    </div>
                    <?php endif; ?>
                    <?php if ($vendor['description']): ?>
                    <p class="vendor-description"><?php echo esc_html($vendor['description']); ?></p>
                    <?php endif; ?>
                </div>
            </div>

            <div class="vendor-products">
                <h2><?php echo count($products); ?> Products</h2>
                <div class="cartt-products-grid">
                    <?php foreach ($products as $product): ?>
                    <div class="cartt-product-card">
                        <a href="<?php echo esc_url(home_url('/product/' . $product['slug'])); ?>">
                            <?php if ($product['image_url']): ?>
                            <img src="<?php echo esc_url($product['image_url']); ?>" alt="<?php echo esc_attr($product['name']); ?>">
                            <?php endif; ?>
                            <h3><?php echo esc_html($product['name']); ?></h3>
                            <span class="price"><?php echo cartt_format_price($product['sale_price'] ?: $product['price']); ?></span>
                        </a>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render vacation mode message
     */
    private function render_vacation_mode(array $vendor): string
    {
        ob_start();
        ?>
        <div class="cartt-vendor-vacation">
            <h1><?php echo esc_html($vendor['store_name']); ?></h1>
            <p class="vacation-message">
                <?php echo $vendor['vacation_message'] ?: 'This store is temporarily closed. Please check back later.'; ?>
            </p>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render star rating
     */
    private function render_stars(float $rating): string
    {
        $full = floor($rating);
        $half = ($rating - $full) >= 0.5 ? 1 : 0;
        $empty = 5 - $full - $half;

        $html = '';
        for ($i = 0; $i < $full; $i++) $html .= '★';
        if ($half) $html .= '½';
        for ($i = 0; $i < $empty; $i++) $html .= '☆';

        return '<span class="stars">' . $html . '</span>';
    }

    /**
     * Get count of pending vendor applications
     */
    public static function getPendingCount(): int
    {
        global $wpdb;
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_vendors WHERE status = 'pending'"
        );
    }
}
