<?php
/**
 * Vendor Commission Service
 * 
 * Features:
 * - Commission calculation per order
 * - Variable commission rates (per vendor, per category, per product)
 * - Payout management
 * - Payment methods (PayPal, Stripe Connect, Bank Transfer)
 * - Commission reports
 * - Automatic payout scheduling
 * - Refund handling
 * 
 * @package Cartt
 * @since 1.4.0
 */

declare(strict_types=1);

namespace Cartt\Services;

class VendorCommissionService
{
    private static ?self $instance = null;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // Calculate commission when order completed
        add_action('cartt_order_completed', [$this, 'calculate_order_commission'], 10, 1);
        add_action('cartt_order_refunded', [$this, 'handle_refund'], 10, 2);

        // Payout cron
        add_action('cartt_process_vendor_payouts', [$this, 'process_scheduled_payouts']);

        // AJAX
        add_action('wp_ajax_cartt_vendor_request_payout', [$this, 'ajax_request_payout']);
        add_action('wp_ajax_cartt_admin_process_payout', [$this, 'ajax_process_payout']);
    }

    /**
     * Calculate commission for an order
     */
    public function calculate_order_commission(int $order_id): void
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $products_table = $wpdb->prefix . 'cartt_products';
        $commissions_table = $wpdb->prefix . 'cartt_vendor_commissions';

        // Get order items with vendor info
        $items = $wpdb->get_results($wpdb->prepare("
            SELECT 
                oi.id as item_id,
                oi.product_id,
                oi.quantity,
                oi.subtotal,
                p.vendor_id
            FROM {$items_table} oi
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE oi.order_id = %d AND p.vendor_id IS NOT NULL
        ", $order_id), ARRAY_A);

        foreach ($items as $item) {
            $commission_rate = $this->get_commission_rate(
                $item['vendor_id'],
                $item['product_id']
            );

            $vendor_amount = $item['subtotal'] * (1 - $commission_rate / 100);
            $platform_fee = $item['subtotal'] * ($commission_rate / 100);

            $wpdb->insert($commissions_table, [
                'order_id' => $order_id,
                'order_item_id' => $item['item_id'],
                'vendor_id' => $item['vendor_id'],
                'product_id' => $item['product_id'],
                'order_subtotal' => $item['subtotal'],
                'commission_rate' => $commission_rate,
                'platform_fee' => $platform_fee,
                'vendor_amount' => $vendor_amount,
                'status' => 'pending',
                'created_at' => current_time('mysql')
            ]);
        }
    }

    /**
     * Get commission rate for vendor/product
     */
    public function get_commission_rate(int $vendor_id, int $product_id = null): float
    {
        global $wpdb;
        $vendors_table = $wpdb->prefix . 'cartt_vendors';
        $products_table = $wpdb->prefix . 'cartt_products';
        $categories_table = $wpdb->prefix . 'cartt_categories';

        // Check product-specific rate
        if ($product_id) {
            $product = $wpdb->get_row($wpdb->prepare(
                "SELECT commission_rate, category_id FROM {$products_table} WHERE id = %d",
                $product_id
            ));

            if ($product && $product->commission_rate !== null) {
                return (float) $product->commission_rate;
            }

            // Check category rate
            if ($product && $product->category_id) {
                $category_rate = $wpdb->get_var($wpdb->prepare(
                    "SELECT commission_rate FROM {$categories_table} WHERE id = %d",
                    $product->category_id
                ));

                if ($category_rate !== null) {
                    return (float) $category_rate;
                }
            }
        }

        // Check vendor-specific rate
        $vendor_rate = $wpdb->get_var($wpdb->prepare(
            "SELECT commission_rate FROM {$vendors_table} WHERE id = %d",
            $vendor_id
        ));

        if ($vendor_rate !== null) {
            return (float) $vendor_rate;
        }

        // Default rate
        return (float) get_option('cartt_default_commission_rate', 10);
    }

    /**
     * Set vendor commission rate
     */
    public function set_vendor_rate(int $vendor_id, float $rate): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendors';

        return $wpdb->update(
            $table,
            ['commission_rate' => $rate],
            ['id' => $vendor_id]
        ) !== false;
    }

    /**
     * Get vendor earnings for period
     */
    public function get_vendor_earnings(int $vendor_id, string $period = '30days'): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendor_commissions';

        $date_filter = $this->get_date_filter($period);

        // Total earnings
        $totals = $wpdb->get_row($wpdb->prepare("
            SELECT 
                COALESCE(SUM(vendor_amount), 0) as total_earned,
                COALESCE(SUM(CASE WHEN status = 'paid' THEN vendor_amount ELSE 0 END), 0) as total_paid,
                COALESCE(SUM(CASE WHEN status = 'pending' THEN vendor_amount ELSE 0 END), 0) as total_pending,
                COUNT(DISTINCT order_id) as order_count
            FROM {$table}
            WHERE vendor_id = %d
            AND status != 'refunded'
            {$date_filter}
        ", $vendor_id), ARRAY_A);

        // Daily breakdown
        $daily = $wpdb->get_results($wpdb->prepare("
            SELECT 
                DATE(created_at) as date,
                SUM(vendor_amount) as amount,
                COUNT(DISTINCT order_id) as orders
            FROM {$table}
            WHERE vendor_id = %d
            AND status != 'refunded'
            {$date_filter}
            GROUP BY DATE(created_at)
            ORDER BY date DESC
        ", $vendor_id), ARRAY_A);

        return [
            'total_earned' => (float) $totals['total_earned'],
            'total_paid' => (float) $totals['total_paid'],
            'total_pending' => (float) $totals['total_pending'],
            'order_count' => (int) $totals['order_count'],
            'daily' => $daily
        ];
    }

    /**
     * Get pending earnings ready for payout
     */
    public function get_pending_earnings(int $vendor_id): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendor_commissions';

        // Only include earnings from completed orders older than hold period
        $hold_days = (int) get_option('cartt_payout_hold_days', 7);
        
        $result = $wpdb->get_row($wpdb->prepare("
            SELECT 
                COALESCE(SUM(vendor_amount), 0) as amount,
                COUNT(*) as items
            FROM {$table}
            WHERE vendor_id = %d
            AND status = 'pending'
            AND created_at < DATE_SUB(NOW(), INTERVAL %d DAY)
        ", $vendor_id, $hold_days), ARRAY_A);

        return [
            'amount' => (float) $result['amount'],
            'items' => (int) $result['items'],
            'hold_days' => $hold_days
        ];
    }

    /**
     * Request payout
     */
    public function request_payout(int $vendor_id, float $amount = null): array
    {
        global $wpdb;
        $commissions_table = $wpdb->prefix . 'cartt_vendor_commissions';
        $payouts_table = $wpdb->prefix . 'cartt_vendor_payouts';
        $vendors_table = $wpdb->prefix . 'cartt_vendors';

        // Get pending earnings
        $pending = $this->get_pending_earnings($vendor_id);
        
        if ($pending['amount'] <= 0) {
            return ['success' => false, 'error' => 'No pending earnings available for payout'];
        }

        $payout_amount = $amount ? min($amount, $pending['amount']) : $pending['amount'];
        $min_payout = (float) get_option('cartt_min_payout', 50);

        if ($payout_amount < $min_payout) {
            return ['success' => false, 'error' => "Minimum payout amount is {$min_payout}"];
        }

        // Get vendor payment info
        $vendor = $wpdb->get_row($wpdb->prepare(
            "SELECT payment_method, payment_email FROM {$vendors_table} WHERE id = %d",
            $vendor_id
        ));

        if (!$vendor || !$vendor->payment_email) {
            return ['success' => false, 'error' => 'Please set up your payment information first'];
        }

        // Create payout request
        $wpdb->insert($payouts_table, [
            'vendor_id' => $vendor_id,
            'amount' => $payout_amount,
            'payment_method' => $vendor->payment_method ?: 'paypal',
            'payment_email' => $vendor->payment_email,
            'status' => 'pending',
            'requested_at' => current_time('mysql')
        ]);

        $payout_id = $wpdb->insert_id;

        // Mark commissions as processing
        $hold_days = (int) get_option('cartt_payout_hold_days', 7);
        $wpdb->query($wpdb->prepare("
            UPDATE {$commissions_table}
            SET status = 'processing', payout_id = %d
            WHERE vendor_id = %d
            AND status = 'pending'
            AND created_at < DATE_SUB(NOW(), INTERVAL %d DAY)
            ORDER BY created_at ASC
        ", $payout_id, $vendor_id, $hold_days));

        // Notify admin
        $this->notify_payout_request($vendor_id, $payout_amount);

        return [
            'success' => true,
            'payout_id' => $payout_id,
            'amount' => $payout_amount
        ];
    }

    /**
     * Process payout (admin)
     */
    public function process_payout(int $payout_id, string $transaction_id = ''): array
    {
        global $wpdb;
        $payouts_table = $wpdb->prefix . 'cartt_vendor_payouts';
        $commissions_table = $wpdb->prefix . 'cartt_vendor_commissions';

        $payout = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$payouts_table} WHERE id = %d",
            $payout_id
        ));

        if (!$payout) {
            return ['success' => false, 'error' => 'Payout not found'];
        }

        if ($payout->status !== 'pending') {
            return ['success' => false, 'error' => 'Payout already processed'];
        }

        // Update payout status
        $wpdb->update($payouts_table, [
            'status' => 'completed',
            'transaction_id' => $transaction_id,
            'processed_at' => current_time('mysql'),
            'processed_by' => get_current_user_id()
        ], ['id' => $payout_id]);

        // Update commission statuses
        $wpdb->update(
            $commissions_table,
            ['status' => 'paid'],
            ['payout_id' => $payout_id]
        );

        // Notify vendor
        $this->notify_payout_completed($payout->vendor_id, $payout->amount, $transaction_id);

        return ['success' => true];
    }

    /**
     * Reject payout (admin)
     */
    public function reject_payout(int $payout_id, string $reason = ''): array
    {
        global $wpdb;
        $payouts_table = $wpdb->prefix . 'cartt_vendor_payouts';
        $commissions_table = $wpdb->prefix . 'cartt_vendor_commissions';

        $payout = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$payouts_table} WHERE id = %d",
            $payout_id
        ));

        if (!$payout) {
            return ['success' => false, 'error' => 'Payout not found'];
        }

        // Update payout status
        $wpdb->update($payouts_table, [
            'status' => 'rejected',
            'rejection_reason' => $reason,
            'processed_at' => current_time('mysql'),
            'processed_by' => get_current_user_id()
        ], ['id' => $payout_id]);

        // Reset commission statuses back to pending
        $wpdb->update(
            $commissions_table,
            ['status' => 'pending', 'payout_id' => null],
            ['payout_id' => $payout_id]
        );

        return ['success' => true];
    }

    /**
     * Handle refund - reverse commission
     */
    public function handle_refund(int $order_id, float $refund_amount): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendor_commissions';

        // Mark commissions as refunded
        $wpdb->update(
            $table,
            ['status' => 'refunded'],
            ['order_id' => $order_id]
        );

        // If already paid out, create negative balance record
        $paid = $wpdb->get_results($wpdb->prepare("
            SELECT * FROM {$table}
            WHERE order_id = %d AND status = 'paid'
        ", $order_id), ARRAY_A);

        foreach ($paid as $commission) {
            // Create negative commission to offset
            $wpdb->insert($table, [
                'order_id' => $order_id,
                'vendor_id' => $commission['vendor_id'],
                'product_id' => $commission['product_id'],
                'order_subtotal' => -$commission['order_subtotal'],
                'commission_rate' => $commission['commission_rate'],
                'platform_fee' => -$commission['platform_fee'],
                'vendor_amount' => -$commission['vendor_amount'],
                'status' => 'pending',
                'note' => 'Refund adjustment',
                'created_at' => current_time('mysql')
            ]);
        }
    }

    /**
     * Get all pending payouts (admin)
     */
    public function get_pending_payouts(): array
    {
        global $wpdb;
        $payouts_table = $wpdb->prefix . 'cartt_vendor_payouts';
        $vendors_table = $wpdb->prefix . 'cartt_vendors';

        return $wpdb->get_results("
            SELECT 
                p.*,
                v.store_name,
                v.email as vendor_email
            FROM {$payouts_table} p
            JOIN {$vendors_table} v ON v.id = p.vendor_id
            WHERE p.status = 'pending'
            ORDER BY p.requested_at ASC
        ", ARRAY_A);
    }

    /**
     * Get payout history (vendor or admin)
     */
    public function get_payout_history(int $vendor_id = null, int $limit = 20): array
    {
        global $wpdb;
        $payouts_table = $wpdb->prefix . 'cartt_vendor_payouts';
        $vendors_table = $wpdb->prefix . 'cartt_vendors';

        $where = "";
        if ($vendor_id) {
            $where = $wpdb->prepare("WHERE p.vendor_id = %d", $vendor_id);
        }

        return $wpdb->get_results("
            SELECT 
                p.*,
                v.store_name
            FROM {$payouts_table} p
            JOIN {$vendors_table} v ON v.id = p.vendor_id
            {$where}
            ORDER BY p.requested_at DESC
            LIMIT {$limit}
        ", ARRAY_A);
    }

    /**
     * Commission report for date range
     */
    public function get_commission_report(string $start_date, string $end_date): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_vendor_commissions';
        $vendors_table = $wpdb->prefix . 'cartt_vendors';

        $summary = $wpdb->get_row($wpdb->prepare("
            SELECT 
                COUNT(DISTINCT order_id) as total_orders,
                SUM(order_subtotal) as total_sales,
                SUM(platform_fee) as total_commission,
                SUM(vendor_amount) as total_vendor_earnings
            FROM {$table}
            WHERE created_at BETWEEN %s AND %s
            AND status != 'refunded'
        ", $start_date, $end_date . ' 23:59:59'), ARRAY_A);

        $by_vendor = $wpdb->get_results($wpdb->prepare("
            SELECT 
                c.vendor_id,
                v.store_name,
                COUNT(DISTINCT c.order_id) as orders,
                SUM(c.order_subtotal) as sales,
                SUM(c.platform_fee) as commission,
                SUM(c.vendor_amount) as earnings
            FROM {$table} c
            JOIN {$vendors_table} v ON v.id = c.vendor_id
            WHERE c.created_at BETWEEN %s AND %s
            AND c.status != 'refunded'
            GROUP BY c.vendor_id
            ORDER BY sales DESC
        ", $start_date, $end_date . ' 23:59:59'), ARRAY_A);

        return [
            'summary' => $summary,
            'by_vendor' => $by_vendor
        ];
    }

    /**
     * AJAX: Request payout
     */
    public function ajax_request_payout(): void
    {
        check_ajax_referer('cartt_vendor', 'nonce');

        $vendor = VendorService::instance()->get_by_user(get_current_user_id());
        if (!$vendor) {
            wp_send_json_error('Vendor not found');
        }

        $amount = isset($_POST['amount']) ? (float) $_POST['amount'] : null;
        $result = $this->request_payout($vendor['id'], $amount);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * AJAX: Process payout (admin)
     */
    public function ajax_process_payout(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $payout_id = intval($_POST['payout_id']);
        $action = sanitize_text_field($_POST['payout_action'] ?? '');
        $transaction_id = sanitize_text_field($_POST['transaction_id'] ?? '');
        $reason = sanitize_textarea_field($_POST['reason'] ?? '');

        if ($action === 'approve') {
            $result = $this->process_payout($payout_id, $transaction_id);
        } elseif ($action === 'reject') {
            $result = $this->reject_payout($payout_id, $reason);
        } else {
            wp_send_json_error('Invalid action');
        }

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * Notify admin of payout request
     */
    private function notify_payout_request(int $vendor_id, float $amount): void
    {
        $vendor = VendorService::instance()->get($vendor_id);
        $admin_email = get_option('admin_email');

        $subject = 'Vendor Payout Request';
        $message = sprintf(
            "A vendor has requested a payout:\n\n" .
            "Store: %s\n" .
            "Amount: %s\n\n" .
            "Review: %s",
            $vendor['store_name'],
            cartt_format_price($amount),
            admin_url('admin.php?page=cartt-payouts')
        );

        wp_mail($admin_email, $subject, $message);
    }

    /**
     * Notify vendor of completed payout
     */
    private function notify_payout_completed(int $vendor_id, float $amount, string $transaction_id): void
    {
        $vendor = VendorService::instance()->get($vendor_id);

        $subject = 'Payout Completed';
        $message = sprintf(
            "Your payout has been processed!\n\n" .
            "Amount: %s\n" .
            "Transaction ID: %s\n\n" .
            "The funds should arrive in your account within 1-3 business days.",
            cartt_format_price($amount),
            $transaction_id
        );

        wp_mail($vendor['email'], $subject, $message);
    }

    /**
     * Get date filter SQL
     */
    private function get_date_filter(string $period): string
    {
        switch ($period) {
            case '7days':
                return "AND created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
            case '30days':
                return "AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
            case '90days':
                return "AND created_at >= DATE_SUB(NOW(), INTERVAL 90 DAY)";
            case 'year':
                return "AND created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)";
            case 'all':
            default:
                return "";
        }
    }

    /**
     * Get count of pending payout requests
     */
    public static function getPendingPayoutsCount(): int
    {
        global $wpdb;
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_vendor_payouts WHERE status = 'pending'"
        );
    }
}
