<?php
/**
 * Software Licensing Service
 * 
 * Handles license key generation, validation, and management for Cartt add-ons.
 * Used on cartt.app to sell licenses, and in Cartt plugin to verify them.
 * 
 * @package Cartt
 * @since 1.4.1
 */

declare(strict_types=1);

namespace Cartt\Services;

class SoftwareLicensingService
{
    private static ?self $instance = null;
    
    private string $table_licenses;
    private string $table_activations;
    private string $api_endpoint;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        global $wpdb;
        $this->table_licenses = $wpdb->prefix . 'cartt_licenses';
        $this->table_activations = $wpdb->prefix . 'cartt_license_activations';
        $this->api_endpoint = 'https://cartt.app/wp-json/cartt/v1/license';
        
        // REST API endpoints for license server
        add_action('rest_api_init', [$this, 'registerRestRoutes']);
        
        // Hook into order completion to generate licenses
        add_action('cartt_order_completed', [$this, 'generateLicenseForOrder']);
        
        // AJAX handlers
        add_action('wp_ajax_cartt_verify_license', [$this, 'ajaxVerifyLicense']);
        add_action('wp_ajax_nopriv_cartt_verify_license', [$this, 'ajaxVerifyLicense']);
    }

    /**
     * Register REST API routes for license validation
     */
    public function registerRestRoutes(): void
    {
        register_rest_route('cartt/v1', '/license/verify', [
            'methods' => 'POST',
            'callback' => [$this, 'restVerifyLicense'],
            'permission_callback' => '__return_true',
            'args' => [
                'license_key' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'addon' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ],
                'site_url' => [
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'esc_url_raw'
                ]
            ]
        ]);

        register_rest_route('cartt/v1', '/license/activate', [
            'methods' => 'POST',
            'callback' => [$this, 'restActivateLicense'],
            'permission_callback' => '__return_true',
            'args' => [
                'license_key' => [
                    'required' => true,
                    'type' => 'string'
                ],
                'addon' => [
                    'required' => true,
                    'type' => 'string'
                ],
                'site_url' => [
                    'required' => true,
                    'type' => 'string'
                ]
            ]
        ]);

        register_rest_route('cartt/v1', '/license/deactivate', [
            'methods' => 'POST',
            'callback' => [$this, 'restDeactivateLicense'],
            'permission_callback' => '__return_true',
            'args' => [
                'license_key' => [
                    'required' => true,
                    'type' => 'string'
                ],
                'site_url' => [
                    'required' => true,
                    'type' => 'string'
                ]
            ]
        ]);

        register_rest_route('cartt/v1', '/license/check', [
            'methods' => 'GET',
            'callback' => [$this, 'restCheckLicense'],
            'permission_callback' => '__return_true',
            'args' => [
                'license_key' => [
                    'required' => true,
                    'type' => 'string'
                ]
            ]
        ]);
    }

    /**
     * Generate a unique license key
     */
    public function generateLicenseKey(): string
    {
        $segments = [];
        for ($i = 0; $i < 4; $i++) {
            $segments[] = strtoupper(substr(md5(uniqid((string)mt_rand(), true)), 0, 4));
        }
        return implode('-', $segments);
    }

    /**
     * Create a new license
     */
    public function createLicense(array $data): ?int
    {
        global $wpdb;
        
        $license_key = $this->generateLicenseKey();
        
        // Ensure unique
        while ($this->getLicenseByKey($license_key)) {
            $license_key = $this->generateLicenseKey();
        }
        
        $defaults = [
            'license_key' => $license_key,
            'addon' => '',
            'customer_id' => 0,
            'customer_email' => '',
            'order_id' => 0,
            'activation_limit' => 1,
            'status' => 'active',
            'expires_at' => date('Y-m-d H:i:s', strtotime('+1 year')),
        ];
        
        $data = array_merge($defaults, $data);
        $data['license_key'] = $license_key;
        $data['created_at'] = current_time('mysql');
        
        $result = $wpdb->insert($this->table_licenses, $data);
        
        if ($result) {
            do_action('cartt_license_created', $wpdb->insert_id, $data);
            return $wpdb->insert_id;
        }
        
        return null;
    }

    /**
     * Get license by key
     */
    public function getLicenseByKey(string $key): ?object
    {
        global $wpdb;
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_licenses} WHERE license_key = %s",
            $key
        ));
    }

    /**
     * Get licenses by customer
     */
    public function getLicensesByCustomer(int $customer_id): array
    {
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_licenses} 
             WHERE customer_id = %d 
             ORDER BY created_at DESC",
            $customer_id
        ));
    }

    /**
     * Get licenses by order
     */
    public function getLicensesByOrder(int $order_id): array
    {
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_licenses} WHERE order_id = %d",
            $order_id
        ));
    }

    /**
     * Verify a license key
     */
    public function verifyLicense(string $key, string $addon, string $site_url): array
    {
        $license = $this->getLicenseByKey($key);
        
        if (!$license) {
            return [
                'valid' => false,
                'error' => 'invalid_key',
                'message' => 'License key not found'
            ];
        }
        
        // Check addon match
        if ($license->addon !== $addon && $license->addon !== 'all_access') {
            return [
                'valid' => false,
                'error' => 'addon_mismatch',
                'message' => 'License is not valid for this add-on'
            ];
        }
        
        // Check status
        if ($license->status !== 'active') {
            return [
                'valid' => false,
                'error' => 'license_' . $license->status,
                'message' => 'License is ' . $license->status
            ];
        }
        
        // Check expiration
        if ($license->expires_at && strtotime($license->expires_at) < time()) {
            return [
                'valid' => false,
                'error' => 'license_expired',
                'message' => 'License has expired',
                'expires_at' => $license->expires_at
            ];
        }
        
        // Check if site is already activated
        $activation = $this->getActivation($license->id, $site_url);
        
        if ($activation) {
            return [
                'valid' => true,
                'license_id' => $license->id,
                'addon' => $license->addon,
                'expires_at' => $license->expires_at,
                'activated' => true
            ];
        }
        
        // Check activation limit
        $activation_count = $this->getActivationCount($license->id);
        
        if ($activation_count >= $license->activation_limit) {
            return [
                'valid' => false,
                'error' => 'activation_limit',
                'message' => 'Activation limit reached (' . $license->activation_limit . ' sites)',
                'activations' => $activation_count,
                'limit' => $license->activation_limit
            ];
        }
        
        return [
            'valid' => true,
            'license_id' => $license->id,
            'addon' => $license->addon,
            'expires_at' => $license->expires_at,
            'activated' => false,
            'activations_remaining' => $license->activation_limit - $activation_count
        ];
    }

    /**
     * Activate a license on a site
     */
    public function activateLicense(string $key, string $addon, string $site_url): array
    {
        $verification = $this->verifyLicense($key, $addon, $site_url);
        
        if (!$verification['valid']) {
            return $verification;
        }
        
        if ($verification['activated'] ?? false) {
            return [
                'success' => true,
                'message' => 'License already activated on this site'
            ];
        }
        
        global $wpdb;
        
        $license = $this->getLicenseByKey($key);
        
        $result = $wpdb->insert($this->table_activations, [
            'license_id' => $license->id,
            'site_url' => $this->normalizeSiteUrl($site_url),
            'activated_at' => current_time('mysql'),
            'last_check' => current_time('mysql'),
            'status' => 'active'
        ]);
        
        if ($result) {
            do_action('cartt_license_activated', $license->id, $site_url);
            
            return [
                'success' => true,
                'message' => 'License activated successfully',
                'license_id' => $license->id,
                'expires_at' => $license->expires_at
            ];
        }
        
        return [
            'success' => false,
            'error' => 'activation_failed',
            'message' => 'Failed to activate license'
        ];
    }

    /**
     * Deactivate a license from a site
     */
    public function deactivateLicense(string $key, string $site_url): array
    {
        global $wpdb;
        
        $license = $this->getLicenseByKey($key);
        
        if (!$license) {
            return [
                'success' => false,
                'error' => 'invalid_key',
                'message' => 'License key not found'
            ];
        }
        
        $site_url = $this->normalizeSiteUrl($site_url);
        
        $result = $wpdb->delete($this->table_activations, [
            'license_id' => $license->id,
            'site_url' => $site_url
        ]);
        
        if ($result) {
            do_action('cartt_license_deactivated', $license->id, $site_url);
            
            return [
                'success' => true,
                'message' => 'License deactivated successfully'
            ];
        }
        
        return [
            'success' => false,
            'error' => 'not_activated',
            'message' => 'License was not activated on this site'
        ];
    }

    /**
     * Get activation record
     */
    public function getActivation(int $license_id, string $site_url): ?object
    {
        global $wpdb;
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_activations} 
             WHERE license_id = %d AND site_url = %s AND status = 'active'",
            $license_id,
            $this->normalizeSiteUrl($site_url)
        ));
    }

    /**
     * Get activation count for a license
     */
    public function getActivationCount(int $license_id): int
    {
        global $wpdb;
        
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_activations} 
             WHERE license_id = %d AND status = 'active'",
            $license_id
        ));
    }

    /**
     * Get all activations for a license
     */
    public function getActivations(int $license_id): array
    {
        global $wpdb;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_activations} 
             WHERE license_id = %d 
             ORDER BY activated_at DESC",
            $license_id
        ));
    }

    /**
     * Update license status
     */
    public function updateLicenseStatus(int $license_id, string $status): bool
    {
        global $wpdb;
        
        $valid_statuses = ['active', 'expired', 'suspended', 'revoked'];
        
        if (!in_array($status, $valid_statuses)) {
            return false;
        }
        
        $result = $wpdb->update(
            $this->table_licenses,
            ['status' => $status, 'updated_at' => current_time('mysql')],
            ['id' => $license_id]
        );
        
        if ($result !== false) {
            do_action('cartt_license_status_changed', $license_id, $status);
            return true;
        }
        
        return false;
    }

    /**
     * Renew a license
     */
    public function renewLicense(int $license_id, int $days = 365): bool
    {
        global $wpdb;
        
        $license = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_licenses} WHERE id = %d",
            $license_id
        ));
        
        if (!$license) {
            return false;
        }
        
        // Calculate new expiration from current expiration or now
        $base_date = $license->expires_at && strtotime($license->expires_at) > time()
            ? $license->expires_at
            : current_time('mysql');
        
        $new_expiration = date('Y-m-d H:i:s', strtotime($base_date . " +{$days} days"));
        
        $result = $wpdb->update(
            $this->table_licenses,
            [
                'expires_at' => $new_expiration,
                'status' => 'active',
                'updated_at' => current_time('mysql')
            ],
            ['id' => $license_id]
        );
        
        if ($result !== false) {
            do_action('cartt_license_renewed', $license_id, $new_expiration);
            return true;
        }
        
        return false;
    }

    /**
     * Generate license for completed order
     */
    public function generateLicenseForOrder(int $order_id): void
    {
        global $wpdb;
        
        // Get order
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $order_id
        ));
        
        if (!$order) {
            return;
        }
        
        // Get order items
        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT oi.*, p.type, p.meta 
             FROM {$wpdb->prefix}cartt_order_items oi
             JOIN {$wpdb->prefix}cartt_products p ON oi.product_id = p.id
             WHERE oi.order_id = %d",
            $order_id
        ));
        
        foreach ($items as $item) {
            // Check if this is a license product
            $meta = json_decode($item->meta, true) ?: [];
            
            if (($meta['is_license_product'] ?? false) || $item->type === 'license') {
                $addon_key = $meta['addon_key'] ?? 'all_access';
                $activation_limit = $meta['activation_limit'] ?? 1;
                $license_days = $meta['license_days'] ?? 365;
                
                // Create license for each quantity
                for ($i = 0; $i < $item->quantity; $i++) {
                    $this->createLicense([
                        'addon' => $addon_key,
                        'customer_id' => $order->customer_id,
                        'customer_email' => $order->email,
                        'order_id' => $order_id,
                        'activation_limit' => $activation_limit,
                        'expires_at' => date('Y-m-d H:i:s', strtotime("+{$license_days} days"))
                    ]);
                }
            }
        }
    }

    /**
     * Normalize site URL for comparison
     */
    private function normalizeSiteUrl(string $url): string
    {
        $url = strtolower(trim($url));
        $url = preg_replace('#^https?://#', '', $url);
        $url = rtrim($url, '/');
        return $url;
    }

    /**
     * REST: Verify license
     */
    public function restVerifyLicense(\WP_REST_Request $request): \WP_REST_Response
    {
        $result = $this->verifyLicense(
            $request->get_param('license_key'),
            $request->get_param('addon'),
            $request->get_param('site_url')
        );
        
        return new \WP_REST_Response($result);
    }

    /**
     * REST: Activate license
     */
    public function restActivateLicense(\WP_REST_Request $request): \WP_REST_Response
    {
        $result = $this->activateLicense(
            $request->get_param('license_key'),
            $request->get_param('addon'),
            $request->get_param('site_url')
        );
        
        return new \WP_REST_Response($result);
    }

    /**
     * REST: Deactivate license
     */
    public function restDeactivateLicense(\WP_REST_Request $request): \WP_REST_Response
    {
        $result = $this->deactivateLicense(
            $request->get_param('license_key'),
            $request->get_param('site_url')
        );
        
        return new \WP_REST_Response($result);
    }

    /**
     * REST: Check license status
     */
    public function restCheckLicense(\WP_REST_Request $request): \WP_REST_Response
    {
        $license = $this->getLicenseByKey($request->get_param('license_key'));
        
        if (!$license) {
            return new \WP_REST_Response([
                'valid' => false,
                'error' => 'invalid_key'
            ]);
        }
        
        $activations = $this->getActivations($license->id);
        
        return new \WP_REST_Response([
            'valid' => $license->status === 'active',
            'addon' => $license->addon,
            'status' => $license->status,
            'expires_at' => $license->expires_at,
            'activation_limit' => $license->activation_limit,
            'activations' => count($activations),
            'sites' => array_map(function($a) {
                return [
                    'url' => $a->site_url,
                    'activated_at' => $a->activated_at
                ];
            }, $activations)
        ]);
    }

    /**
     * AJAX: Verify license (for plugin-side verification)
     */
    public function ajaxVerifyLicense(): void
    {
        $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
        $addon = isset($_POST['addon']) ? sanitize_text_field($_POST['addon']) : '';
        
        if (empty($license_key) || empty($addon)) {
            wp_send_json_error('Missing license key or addon');
        }
        
        $site_url = home_url();
        
        // Call remote API to verify
        $response = wp_remote_post($this->api_endpoint . '/verify', [
            'body' => [
                'license_key' => $license_key,
                'addon' => $addon,
                'site_url' => $site_url
            ],
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error('Could not connect to license server');
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($body['valid'] ?? false) {
            wp_send_json_success($body);
        } else {
            wp_send_json_error($body['message'] ?? 'License verification failed');
        }
    }

    /**
     * Client-side: Verify and activate license (called from Cartt plugin)
     */
    public static function clientVerifyAndActivate(string $license_key, string $addon): array
    {
        $site_url = home_url();
        
        // Try relay server first
        $response = wp_remote_post('https://cartt.app/wp-content/uploads/cartt-relay/relay.php', [
            'body' => [
                'action' => 'activate_addon_license',
                'license_key' => $license_key,
                'addon' => $addon,
                'site_url' => $site_url
            ],
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) {
            return [
                'success' => false,
                'error' => 'connection_failed',
                'message' => 'Could not connect to license server'
            ];
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($body['success'] ?? false) {
            // Store license locally
            $licenses = get_option('cartt_addon_licenses', []);
            $licenses[$addon] = [
                'key' => $license_key,
                'expires_at' => $body['expires_at'] ?? null,
                'verified_at' => current_time('mysql')
            ];
            update_option('cartt_addon_licenses', $licenses);
            
            // Activate addon
            $active_addons = get_option('cartt_active_addons', []);
            if (!in_array($addon, $active_addons)) {
                $active_addons[] = $addon;
                update_option('cartt_active_addons', $active_addons);
            }
            
            return ['success' => true, 'message' => 'License activated', 'expires_at' => $body['expires_at'] ?? null];
        }
        
        return [
            'success' => false,
            'message' => $body['error'] ?? 'License verification failed'
        ];
    }

    /**
     * Client-side: Periodic license check (should be called via cron)
     */
    public static function clientPeriodicCheck(): void
    {
        $licenses = get_option('cartt_addon_licenses', []);
        
        foreach ($licenses as $addon => $license_data) {
            $response = wp_remote_post('https://cartt.app/wp-content/uploads/cartt-relay/relay.php', [
                'body' => [
                    'action' => 'verify_addon_license',
                    'license_key' => $license_data['key'],
                    'addon' => $addon,
                    'site_url' => home_url()
                ],
                'timeout' => 15
            ]);
            
            if (is_wp_error($response)) {
                continue;
            }
            
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            if (!($body['valid'] ?? false)) {
                // Deactivate addon
                $active_addons = get_option('cartt_active_addons', []);
                $active_addons = array_diff($active_addons, [$addon]);
                update_option('cartt_active_addons', array_values($active_addons));
                
                // Remove license
                unset($licenses[$addon]);
                update_option('cartt_addon_licenses', $licenses);
            } else {
                // Update expiration date
                $licenses[$addon]['expires_at'] = $body['expires_at'] ?? null;
                $licenses[$addon]['verified_at'] = current_time('mysql');
                update_option('cartt_addon_licenses', $licenses);
            }
        }
    }
}
