<?php
/**
 * Social Proof Service
 * 
 * Features:
 * - Recent purchase popups ("John from NYC just bought...")
 * - Live visitor count
 * - Stock urgency ("Only 3 left!")
 * - Cart activity ("5 people have this in cart")
 * - Reviews summary badge
 * - Sales count ("500+ sold")
 * - Trust badges
 * 
 * @package Cartt
 * @since 1.4.0
 */

declare(strict_types=1);

namespace Cartt\Services;

class SocialProofService
{
    private static ?self $instance = null;
    private array $settings;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->settings = get_option('cartt_social_proof', $this->get_defaults());
        
        add_action('wp_ajax_cartt_social_proof_data', [$this, 'ajax_get_data']);
        add_action('wp_ajax_nopriv_cartt_social_proof_data', [$this, 'ajax_get_data']);
        add_action('wp_footer', [$this, 'render_popup_container']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
    }

    /**
     * Default settings
     */
    private function get_defaults(): array
    {
        return [
            'enabled' => true,
            'purchase_popup' => [
                'enabled' => true,
                'show_real' => true,
                'show_fake' => false,
                'display_time' => 5000,
                'delay_between' => 15000,
                'max_per_session' => 10,
                'position' => 'bottom-left'
            ],
            'visitor_count' => [
                'enabled' => true,
                'show_real' => true,
                'min_display' => 5,
                'text' => '{count} people are viewing this right now'
            ],
            'stock_urgency' => [
                'enabled' => true,
                'threshold' => 10,
                'text_low' => 'Only {count} left in stock!',
                'text_selling_fast' => 'Selling fast - {count} sold in last 24 hours'
            ],
            'cart_activity' => [
                'enabled' => true,
                'text' => '{count} people have this in their cart'
            ],
            'sales_count' => [
                'enabled' => true,
                'threshold' => 10,
                'text' => '{count}+ sold'
            ],
            'trust_badges' => [
                'enabled' => true,
                'badges' => ['secure_checkout', 'money_back', 'fast_shipping']
            ]
        ];
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_scripts(): void
    {
        if (!$this->settings['enabled']) return;

        wp_enqueue_style(
            'cartt-social-proof',
            CARTT_PLUGIN_URL . 'assets/css/social-proof.css',
            [],
            CARTT_VERSION
        );

        wp_enqueue_script(
            'cartt-social-proof',
            CARTT_PLUGIN_URL . 'assets/js/social-proof.js',
            ['jquery'],
            CARTT_VERSION,
            true
        );

        wp_localize_script('cartt-social-proof', 'carttSocialProof', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cartt_social_proof'),
            'settings' => $this->get_frontend_settings()
        ]);
    }

    /**
     * Get settings for frontend
     */
    private function get_frontend_settings(): array
    {
        return [
            'purchasePopup' => $this->settings['purchase_popup'],
            'visitorCount' => $this->settings['visitor_count']['enabled'],
            'stockUrgency' => $this->settings['stock_urgency']['enabled'],
            'cartActivity' => $this->settings['cart_activity']['enabled']
        ];
    }

    /**
     * Get recent purchases for popup
     */
    public function get_recent_purchases(int $limit = 10): array
    {
        global $wpdb;
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $products_table = $wpdb->prefix . 'cartt_products';

        $purchases = $wpdb->get_results($wpdb->prepare("
            SELECT 
                o.billing_first_name as first_name,
                o.billing_city as city,
                o.billing_state as state,
                o.billing_country as country,
                p.name as product_name,
                p.image_url,
                p.slug,
                o.created_at
            FROM {$orders_table} o
            JOIN {$items_table} oi ON oi.order_id = o.id
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE o.status IN ('completed', 'processing')
            AND o.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
            ORDER BY o.created_at DESC
            LIMIT %d
        ", $limit), ARRAY_A);

        // Anonymize for privacy
        return array_map(function($purchase) {
            return [
                'name' => $this->anonymize_name($purchase['first_name']),
                'location' => $this->format_location($purchase),
                'product' => $purchase['product_name'],
                'image' => $purchase['image_url'],
                'slug' => $purchase['slug'],
                'time_ago' => $this->time_ago($purchase['created_at'])
            ];
        }, $purchases);
    }

    /**
     * Get live visitor count for a product
     */
    public function get_visitor_count(int $product_id): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_views';

        // Count unique sessions in last 5 minutes
        $count = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(DISTINCT session_id)
            FROM {$table}
            WHERE product_id = %d
            AND viewed_at >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)
        ", $product_id));

        // Apply minimum threshold
        if ($count < $this->settings['visitor_count']['min_display']) {
            return 0;
        }

        return $count;
    }

    /**
     * Get cart activity count
     */
    public function get_cart_activity(int $product_id): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_carts';

        // Count carts with this product in last hour
        $count = (int) $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(DISTINCT session_id)
            FROM {$table}
            WHERE items LIKE %s
            AND updated_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ", '%"product_id":' . $product_id . '%'));

        return $count;
    }

    /**
     * Get stock urgency data
     */
    public function get_stock_urgency(int $product_id): ?array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $orders_table = $wpdb->prefix . 'cartt_orders';

        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT stock_quantity FROM {$products_table} WHERE id = %d",
            $product_id
        ));

        if (!$product) return null;

        $stock = (int) $product->stock_quantity;
        $threshold = $this->settings['stock_urgency']['threshold'];

        $result = [
            'stock' => $stock,
            'show_low_stock' => false,
            'show_selling_fast' => false
        ];

        // Low stock warning
        if ($stock > 0 && $stock <= $threshold) {
            $result['show_low_stock'] = true;
            $result['low_stock_text'] = str_replace(
                '{count}',
                (string) $stock,
                $this->settings['stock_urgency']['text_low']
            );
        }

        // Sales velocity (sold in last 24 hours)
        $recent_sales = (int) $wpdb->get_var($wpdb->prepare("
            SELECT SUM(oi.quantity)
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            WHERE oi.product_id = %d
            AND o.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
            AND o.status IN ('completed', 'processing')
        ", $product_id));

        if ($recent_sales >= 5) {
            $result['show_selling_fast'] = true;
            $result['selling_fast_text'] = str_replace(
                '{count}',
                (string) $recent_sales,
                $this->settings['stock_urgency']['text_selling_fast']
            );
        }

        return $result;
    }

    /**
     * Get total sales count for a product
     */
    public function get_sales_count(int $product_id): int
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $orders_table = $wpdb->prefix . 'cartt_orders';

        return (int) $wpdb->get_var($wpdb->prepare("
            SELECT COALESCE(SUM(oi.quantity), 0)
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            WHERE oi.product_id = %d
            AND o.status IN ('completed', 'processing')
        ", $product_id));
    }

    /**
     * Get trust badges HTML
     */
    public function get_trust_badges(): string
    {
        if (!$this->settings['trust_badges']['enabled']) {
            return '';
        }

        $badges = $this->settings['trust_badges']['badges'];
        $badge_html = [
            'secure_checkout' => '<span class="cartt-trust-badge secure"><svg viewBox="0 0 24 24" width="20" height="20"><path fill="currentColor" d="M12,1L3,5v6c0,5.55,3.84,10.74,9,12c5.16-1.26,9-6.45,9-12V5L12,1z M10,17l-4-4l1.41-1.41L10,14.17l6.59-6.59L18,9L10,17z"/></svg> Secure Checkout</span>',
            'money_back' => '<span class="cartt-trust-badge money-back"><svg viewBox="0 0 24 24" width="20" height="20"><path fill="currentColor" d="M12.5,2C9.64,2,7.14,3.5,5.82,5.82L3,3v6h6L6.18,6.18C7.15,4.27,9.65,3,12.5,3c4.14,0,7.5,3.36,7.5,7.5S16.64,18,12.5,18c-2.9,0-5.4-1.64-6.65-4H4.18c1.35,3.5,4.73,6,8.82,6c5.24,0,9.5-4.26,9.5-9.5S17.74,2,12.5,2z"/></svg> Money-Back Guarantee</span>',
            'fast_shipping' => '<span class="cartt-trust-badge fast-shipping"><svg viewBox="0 0 24 24" width="20" height="20"><path fill="currentColor" d="M20,8h-3V4H3c-1.1,0-2,0.9-2,2v11h2c0,1.66,1.34,3,3,3s3-1.34,3-3h6c0,1.66,1.34,3,3,3s3-1.34,3-3h2v-5L20,8z M6,18.5c-0.83,0-1.5-0.67-1.5-1.5s0.67-1.5,1.5-1.5s1.5,0.67,1.5,1.5S6.83,18.5,6,18.5z M18,18.5c-0.83,0-1.5-0.67-1.5-1.5s0.67-1.5,1.5-1.5s1.5,0.67,1.5,1.5S18.83,18.5,18,18.5z M17,12V9.5h2.5l1.96,2.5H17z"/></svg> Fast Shipping</span>',
            'quality_assured' => '<span class="cartt-trust-badge quality"><svg viewBox="0 0 24 24" width="20" height="20"><path fill="currentColor" d="M12,2L4,5v6.09c0,5.05,3.41,9.76,8,10.91c4.59-1.15,8-5.86,8-10.91V5L12,2z M10.94,15.54L7.4,12l1.41-1.41l2.12,2.12l4.24-4.24l1.41,1.41L10.94,15.54z"/></svg> Quality Assured</span>',
            'free_returns' => '<span class="cartt-trust-badge returns"><svg viewBox="0 0 24 24" width="20" height="20"><path fill="currentColor" d="M19,7h-8v6h8V7z M19,3H5c-1.11,0-2,0.9-2,2v14c0,1.1,0.89,2,2,2h14c1.1,0,2-0.9,2-2V5C21,3.9,20.1,3,19,3z M19,19H5V5h14V19z"/></svg> Free Returns</span>'
        ];

        $output = '<div class="cartt-trust-badges">';
        foreach ($badges as $badge) {
            if (isset($badge_html[$badge])) {
                $output .= $badge_html[$badge];
            }
        }
        $output .= '</div>';

        return $output;
    }

    /**
     * AJAX handler
     */
    public function ajax_get_data(): void
    {
        check_ajax_referer('cartt_social_proof', 'nonce');

        $type = sanitize_text_field($_POST['type'] ?? '');
        $product_id = intval($_POST['product_id'] ?? 0);

        switch ($type) {
            case 'purchases':
                $data = $this->get_recent_purchases();
                break;
            case 'visitors':
                $data = ['count' => $this->get_visitor_count($product_id)];
                break;
            case 'cart_activity':
                $data = ['count' => $this->get_cart_activity($product_id)];
                break;
            case 'stock_urgency':
                $data = $this->get_stock_urgency($product_id);
                break;
            case 'sales_count':
                $data = ['count' => $this->get_sales_count($product_id)];
                break;
            case 'all':
                $data = [
                    'purchases' => $this->get_recent_purchases(5),
                    'visitors' => $this->get_visitor_count($product_id),
                    'cart_activity' => $this->get_cart_activity($product_id),
                    'stock' => $this->get_stock_urgency($product_id),
                    'sales' => $this->get_sales_count($product_id)
                ];
                break;
            default:
                $data = [];
        }

        wp_send_json_success($data);
    }

    /**
     * Render popup container
     */
    public function render_popup_container(): void
    {
        if (!$this->settings['enabled'] || !$this->settings['purchase_popup']['enabled']) {
            return;
        }

        $position = $this->settings['purchase_popup']['position'];
        ?>
        <div id="cartt-social-proof-popup" class="cartt-sp-popup position-<?php echo esc_attr($position); ?>" style="display:none;">
            <button type="button" class="cartt-sp-close">&times;</button>
            <div class="cartt-sp-content">
                <img class="cartt-sp-image" src="" alt="">
                <div class="cartt-sp-info">
                    <p class="cartt-sp-message"></p>
                    <p class="cartt-sp-time"></p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Anonymize name (show first name + initial)
     */
    private function anonymize_name(string $name): string
    {
        $name = trim($name);
        if (empty($name)) {
            return 'Someone';
        }
        return ucfirst(strtolower($name)) . '.';
    }

    /**
     * Format location string
     */
    private function format_location(array $data): string
    {
        $parts = array_filter([
            $data['city'] ?? '',
            $data['state'] ?? '',
            $data['country'] ?? ''
        ]);

        if (empty($parts)) {
            return '';
        }

        // Show city + state for US, city + country for others
        if (($data['country'] ?? '') === 'US' && !empty($data['city']) && !empty($data['state'])) {
            return $data['city'] . ', ' . $data['state'];
        }

        return implode(', ', array_slice($parts, 0, 2));
    }

    /**
     * Human-readable time ago
     */
    private function time_ago(string $datetime): string
    {
        $timestamp = strtotime($datetime);
        $diff = time() - $timestamp;

        if ($diff < 60) {
            return 'just now';
        } elseif ($diff < 3600) {
            $mins = round($diff / 60);
            return $mins . ' minute' . ($mins > 1 ? 's' : '') . ' ago';
        } elseif ($diff < 86400) {
            $hours = round($diff / 3600);
            return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
        } else {
            return 'today';
        }
    }

    /**
     * Render product page social proof
     */
    public function render_product_proof(int $product_id): string
    {
        if (!$this->settings['enabled']) return '';

        $output = '<div class="cartt-product-social-proof">';

        // Visitor count
        if ($this->settings['visitor_count']['enabled']) {
            $visitors = $this->get_visitor_count($product_id);
            if ($visitors > 0) {
                $text = str_replace('{count}', (string) $visitors, $this->settings['visitor_count']['text']);
                $output .= '<div class="cartt-sp-visitors"><span class="pulse"></span> ' . esc_html($text) . '</div>';
            }
        }

        // Cart activity
        if ($this->settings['cart_activity']['enabled']) {
            $carts = $this->get_cart_activity($product_id);
            if ($carts > 0) {
                $text = str_replace('{count}', (string) $carts, $this->settings['cart_activity']['text']);
                $output .= '<div class="cartt-sp-carts">' . esc_html($text) . '</div>';
            }
        }

        // Stock urgency
        if ($this->settings['stock_urgency']['enabled']) {
            $stock = $this->get_stock_urgency($product_id);
            if ($stock && $stock['show_low_stock']) {
                $output .= '<div class="cartt-sp-low-stock">' . esc_html($stock['low_stock_text']) . '</div>';
            }
            if ($stock && $stock['show_selling_fast']) {
                $output .= '<div class="cartt-sp-selling-fast">' . esc_html($stock['selling_fast_text']) . '</div>';
            }
        }

        // Sales count
        if ($this->settings['sales_count']['enabled']) {
            $sales = $this->get_sales_count($product_id);
            if ($sales >= $this->settings['sales_count']['threshold']) {
                $text = str_replace('{count}', (string) $sales, $this->settings['sales_count']['text']);
                $output .= '<div class="cartt-sp-sales">' . esc_html($text) . '</div>';
            }
        }

        $output .= '</div>';

        // Trust badges
        $output .= $this->get_trust_badges();

        return $output;
    }

    /**
     * Update settings
     */
    public function update_settings(array $settings): bool
    {
        $this->settings = wp_parse_args($settings, $this->get_defaults());
        return update_option('cartt_social_proof', $this->settings);
    }
}
