<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * SEO Service
 * Manages product SEO meta and schema markup
 */
class SEOService
{
    /**
     * Get product SEO data
     */
    public static function getProductSEO(int $productId): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_seo';

        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE product_id = %d", $productId));
    }

    /**
     * Save product SEO data
     */
    public static function saveProductSEO(int $productId, array $data): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_seo';

        $existing = self::getProductSEO($productId);

        $seoData = [
            'product_id' => $productId,
            'meta_title' => sanitize_text_field($data['meta_title'] ?? ''),
            'meta_description' => sanitize_textarea_field($data['meta_description'] ?? ''),
            'focus_keyword' => sanitize_text_field($data['focus_keyword'] ?? ''),
            'og_title' => sanitize_text_field($data['og_title'] ?? ''),
            'og_description' => sanitize_textarea_field($data['og_description'] ?? ''),
            'og_image' => !empty($data['og_image']) ? intval($data['og_image']) : null,
            'canonical_url' => esc_url_raw($data['canonical_url'] ?? ''),
            'noindex' => !empty($data['noindex']) ? 1 : 0,
        ];

        if ($existing) {
            return $wpdb->update($table, $seoData, ['id' => $existing->id]) !== false;
        } else {
            return $wpdb->insert($table, $seoData) !== false;
        }
    }

    /**
     * Generate meta title
     */
    public static function getMetaTitle(object $product): string
    {
        $seo = self::getProductSEO($product->id);
        
        if ($seo && !empty($seo->meta_title)) {
            return $seo->meta_title;
        }

        $siteName = get_bloginfo('name');
        return $product->name . ' | ' . $siteName;
    }

    /**
     * Generate meta description
     */
    public static function getMetaDescription(object $product): string
    {
        $seo = self::getProductSEO($product->id);
        
        if ($seo && !empty($seo->meta_description)) {
            return $seo->meta_description;
        }

        // Auto-generate from product description
        $description = strip_tags($product->short_description ?: $product->description);
        $description = preg_replace('/\s+/', ' ', $description);
        
        if (strlen($description) > 160) {
            $description = substr($description, 0, 157) . '...';
        }

        return $description;
    }

    /**
     * Generate product schema markup
     */
    public static function getProductSchema(object $product): array
    {
        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Product',
            'name' => $product->name,
            'description' => strip_tags($product->short_description ?: $product->description),
            'sku' => $product->sku ?: 'PROD-' . $product->id,
            'url' => home_url('/product/' . $product->slug),
        ];

        // Image
        if ($product->featured_image) {
            $imageUrl = wp_get_attachment_image_url($product->featured_image, 'large');
            if ($imageUrl) {
                $schema['image'] = $imageUrl;
            }
        }

        // Price / Offers
        $price = $product->sale_price ?: $product->price;
        $currency = get_option('cartt_currency', 'USD');
        
        $schema['offers'] = [
            '@type' => 'Offer',
            'price' => number_format($price, 2, '.', ''),
            'priceCurrency' => $currency,
            'availability' => $product->stock_status === 'instock' 
                ? 'https://schema.org/InStock' 
                : 'https://schema.org/OutOfStock',
            'url' => home_url('/product/' . $product->slug),
        ];

        // Reviews / Aggregate Rating
        $avgRating = ReviewService::getProductAverageRating($product->id);
        $reviewCount = ReviewService::getProductReviewCount($product->id);

        if ($reviewCount > 0) {
            $schema['aggregateRating'] = [
                '@type' => 'AggregateRating',
                'ratingValue' => number_format($avgRating, 1),
                'reviewCount' => $reviewCount,
                'bestRating' => '5',
                'worstRating' => '1',
            ];
        }

        // Brand (if available)
        if (!empty($product->brand)) {
            $schema['brand'] = [
                '@type' => 'Brand',
                'name' => $product->brand,
            ];
        }

        return $schema;
    }

    /**
     * Generate breadcrumb schema
     */
    public static function getBreadcrumbSchema(object $product): array
    {
        $breadcrumbs = [
            [
                '@type' => 'ListItem',
                'position' => 1,
                'name' => 'Home',
                'item' => home_url(),
            ],
            [
                '@type' => 'ListItem',
                'position' => 2,
                'name' => 'Shop',
                'item' => home_url('/shop/'),
            ],
        ];

        // Add category if available
        $categories = CategoryService::getProductCategories($product->id);
        if (!empty($categories)) {
            $category = $categories[0];
            $breadcrumbs[] = [
                '@type' => 'ListItem',
                'position' => 3,
                'name' => $category->name,
                'item' => home_url('/shop/?category=' . $category->slug),
            ];
            $breadcrumbs[] = [
                '@type' => 'ListItem',
                'position' => 4,
                'name' => $product->name,
            ];
        } else {
            $breadcrumbs[] = [
                '@type' => 'ListItem',
                'position' => 3,
                'name' => $product->name,
            ];
        }

        return [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => $breadcrumbs,
        ];
    }

    /**
     * Generate organization schema
     */
    public static function getOrganizationSchema(): array
    {
        $siteName = get_bloginfo('name');
        $siteUrl = home_url();
        $logo = get_option('cartt_logo_url', '');

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => $siteName,
            'url' => $siteUrl,
        ];

        if ($logo) {
            $schema['logo'] = $logo;
        }

        return $schema;
    }

    /**
     * Generate local business schema (if enabled)
     */
    public static function getLocalBusinessSchema(): ?array
    {
        if (!get_option('cartt_local_business_enabled')) {
            return null;
        }

        $schema = [
            '@context' => 'https://schema.org',
            '@type' => 'LocalBusiness',
            'name' => get_bloginfo('name'),
            'url' => home_url(),
        ];

        $address = get_option('cartt_business_address');
        if ($address) {
            $schema['address'] = [
                '@type' => 'PostalAddress',
                'streetAddress' => $address['street'] ?? '',
                'addressLocality' => $address['city'] ?? '',
                'addressRegion' => $address['state'] ?? '',
                'postalCode' => $address['zip'] ?? '',
                'addressCountry' => $address['country'] ?? '',
            ];
        }

        $phone = get_option('cartt_business_phone');
        if ($phone) {
            $schema['telephone'] = $phone;
        }

        return $schema;
    }

    /**
     * Output all schema for product page
     */
    public static function outputProductSchema(object $product): void
    {
        $schemas = [
            self::getProductSchema($product),
            self::getBreadcrumbSchema($product),
        ];

        foreach ($schemas as $schema) {
            echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES) . '</script>' . "\n";
        }
    }

    /**
     * Generate sitemap entries for products
     */
    public static function getSitemapEntries(): array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        $seo_table = $wpdb->prefix . 'cartt_product_seo';

        $products = $wpdb->get_results(
            "SELECT p.id, p.slug, p.updated_at, s.noindex
             FROM $products_table p
             LEFT JOIN $seo_table s ON p.id = s.product_id
             WHERE p.status = 'publish' AND (s.noindex IS NULL OR s.noindex = 0)
             ORDER BY p.updated_at DESC"
        );

        $entries = [];
        foreach ($products as $product) {
            $entries[] = [
                'loc' => home_url('/product/' . $product->slug),
                'lastmod' => date('Y-m-d', strtotime($product->updated_at)),
                'changefreq' => 'weekly',
                'priority' => '0.8',
            ];
        }

        return $entries;
    }

    /**
     * Analyze SEO score
     */
    public static function analyzeSEO(int $productId): array
    {
        global $wpdb;
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $productId
        ));

        if (!$product) {
            return ['score' => 0, 'issues' => ['Product not found']];
        }

        $seo = self::getProductSEO($productId);
        $issues = [];
        $score = 100;

        // Check meta title
        if (!$seo || empty($seo->meta_title)) {
            $issues[] = 'Missing meta title';
            $score -= 15;
        } elseif (strlen($seo->meta_title) > 60) {
            $issues[] = 'Meta title too long (max 60 characters)';
            $score -= 5;
        }

        // Check meta description
        if (!$seo || empty($seo->meta_description)) {
            $issues[] = 'Missing meta description';
            $score -= 15;
        } elseif (strlen($seo->meta_description) > 160) {
            $issues[] = 'Meta description too long (max 160 characters)';
            $score -= 5;
        } elseif (strlen($seo->meta_description) < 50) {
            $issues[] = 'Meta description too short (min 50 characters)';
            $score -= 5;
        }

        // Check focus keyword
        if (!$seo || empty($seo->focus_keyword)) {
            $issues[] = 'No focus keyword set';
            $score -= 10;
        } else {
            // Check if keyword appears in title
            if (!$seo->meta_title || stripos($seo->meta_title, $seo->focus_keyword) === false) {
                $issues[] = 'Focus keyword not in meta title';
                $score -= 5;
            }
            // Check if keyword appears in description
            if (!$seo->meta_description || stripos($seo->meta_description, $seo->focus_keyword) === false) {
                $issues[] = 'Focus keyword not in meta description';
                $score -= 5;
            }
        }

        // Check product description
        if (empty($product->description)) {
            $issues[] = 'Missing product description';
            $score -= 10;
        } elseif (str_word_count(strip_tags($product->description)) < 100) {
            $issues[] = 'Product description is thin (less than 100 words)';
            $score -= 5;
        }

        // Check image
        if (empty($product->featured_image)) {
            $issues[] = 'Missing featured image';
            $score -= 10;
        }

        return [
            'score' => max(0, $score),
            'issues' => $issues,
            'grade' => self::getGrade($score),
        ];
    }

    /**
     * Get letter grade from score
     */
    private static function getGrade(int $score): string
    {
        if ($score >= 90) return 'A';
        if ($score >= 80) return 'B';
        if ($score >= 70) return 'C';
        if ($score >= 60) return 'D';
        return 'F';
    }
}
