<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Review Service
 * Manages product reviews and ratings
 */
class ReviewService
{
    public static function getProductReviews(int $productId, string $status = 'approved'): array
    {
        global $wpdb;
        
        $where = $status === 'all' 
            ? "" 
            : $wpdb->prepare(" AND status = %s", $status);
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_product_reviews 
             WHERE product_id = %d $where
             ORDER BY created_at DESC",
            $productId
        ));
    }

    public static function getAllReviews(string $status = 'all', int $limit = 50): array
    {
        global $wpdb;
        
        $where = $status === 'all' ? "" : $wpdb->prepare("WHERE status = %s", $status);
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT r.*, p.name as product_name 
             FROM {$wpdb->prefix}cartt_product_reviews r
             LEFT JOIN {$wpdb->prefix}cartt_products p ON r.product_id = p.id
             $where
             ORDER BY r.created_at DESC
             LIMIT %d",
            $limit
        ));
    }

    public static function getReview(int $id): ?object
    {
        global $wpdb;
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_product_reviews WHERE id = %d",
            $id
        ));
    }

    public static function createReview(array $data): int|string
    {
        global $wpdb;
        
        // Check if customer has purchased product
        $verifiedPurchase = false;
        if (!empty($data['customer_email'])) {
            $verifiedPurchase = self::hasCustomerPurchasedProduct(
                $data['customer_email'],
                (int) $data['product_id']
            );
        }
        
        // If reviews are restricted to verified purchasers only
        $verifiedOnly = get_option('cartt_reviews_verified_only', true);
        if ($verifiedOnly && !$verifiedPurchase) {
            return 'not_purchased';
        }
        
        // Determine initial status
        // Auto-approve verified purchases if enabled, otherwise pending
        $autoApproveVerified = get_option('cartt_reviews_auto_approve_verified', true);
        $status = ($verifiedPurchase && $autoApproveVerified) ? 'approved' : 'pending';
        
        $wpdb->insert(
            $wpdb->prefix . 'cartt_product_reviews',
            [
                'product_id' => (int) $data['product_id'],
                'customer_id' => !empty($data['customer_id']) ? (int) $data['customer_id'] : null,
                'customer_name' => sanitize_text_field($data['customer_name']),
                'customer_email' => sanitize_email($data['customer_email']),
                'rating' => min(5, max(1, (int) $data['rating'])),
                'title' => sanitize_text_field($data['title'] ?? ''),
                'content' => sanitize_textarea_field($data['content'] ?? ''),
                'status' => $status,
                'verified_purchase' => $verifiedPurchase ? 1 : 0,
            ],
            ['%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%d']
        );
        
        return (int) $wpdb->insert_id;
    }

    public static function updateReview(int $id, array $data): bool
    {
        global $wpdb;
        
        $updateData = [];
        $formats = [];
        
        if (isset($data['status'])) {
            $updateData['status'] = sanitize_text_field($data['status']);
            $formats[] = '%s';
        }
        
        if (isset($data['rating'])) {
            $updateData['rating'] = min(5, max(1, (int) $data['rating']));
            $formats[] = '%d';
        }
        
        if (isset($data['title'])) {
            $updateData['title'] = sanitize_text_field($data['title']);
            $formats[] = '%s';
        }
        
        if (isset($data['content'])) {
            $updateData['content'] = sanitize_textarea_field($data['content']);
            $formats[] = '%s';
        }
        
        if (empty($updateData)) {
            return false;
        }
        
        return $wpdb->update(
            $wpdb->prefix . 'cartt_product_reviews',
            $updateData,
            ['id' => $id],
            $formats,
            ['%d']
        ) !== false;
    }

    public static function deleteReview(int $id): bool
    {
        global $wpdb;
        return $wpdb->delete(
            $wpdb->prefix . 'cartt_product_reviews',
            ['id' => $id],
            ['%d']
        ) !== false;
    }

    public static function approveReview(int $id): bool
    {
        return self::updateReview($id, ['status' => 'approved']);
    }

    public static function spamReview(int $id): bool
    {
        return self::updateReview($id, ['status' => 'spam']);
    }

    public static function getProductAverageRating(int $productId): float
    {
        global $wpdb;
        $avg = $wpdb->get_var($wpdb->prepare(
            "SELECT AVG(rating) FROM {$wpdb->prefix}cartt_product_reviews 
             WHERE product_id = %d AND status = 'approved'",
            $productId
        ));
        return round((float) $avg, 1);
    }

    public static function getProductReviewCount(int $productId): int
    {
        global $wpdb;
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_product_reviews 
             WHERE product_id = %d AND status = 'approved'",
            $productId
        ));
    }

    public static function getProductRatingBreakdown(int $productId): array
    {
        global $wpdb;
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT rating, COUNT(*) as count FROM {$wpdb->prefix}cartt_product_reviews 
             WHERE product_id = %d AND status = 'approved'
             GROUP BY rating",
            $productId
        ));
        
        $breakdown = [1 => 0, 2 => 0, 3 => 0, 4 => 0, 5 => 0];
        foreach ($results as $row) {
            $breakdown[(int) $row->rating] = (int) $row->count;
        }
        
        return $breakdown;
    }

    public static function getPendingReviewCount(): int
    {
        global $wpdb;
        return (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_product_reviews WHERE status = 'pending'"
        );
    }

    private static function hasCustomerPurchasedProduct(string $email, int $productId): bool
    {
        global $wpdb;
        
        return (bool) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_orders o
             JOIN {$wpdb->prefix}cartt_order_items oi ON o.id = oi.order_id
             WHERE o.customer_email = %s AND oi.product_id = %d AND o.status IN ('completed', 'processing')",
            $email,
            $productId
        ));
    }
}
