<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Refund Service
 * Manages order refunds
 */
class RefundService
{
    public static function getOrderRefunds(int $orderId): array
    {
        global $wpdb;
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_refunds WHERE order_id = %d ORDER BY created_at DESC",
            $orderId
        ));
    }

    public static function getRefund(int $id): ?object
    {
        global $wpdb;
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_refunds WHERE id = %d",
            $id
        ));
    }

    public static function createRefund(array $data): int
    {
        global $wpdb;
        
        $wpdb->insert(
            $wpdb->prefix . 'cartt_refunds',
            [
                'order_id' => (int) $data['order_id'],
                'amount' => (float) $data['amount'],
                'reason' => sanitize_textarea_field($data['reason'] ?? ''),
                'refunded_by' => get_current_user_id() ?: null,
                'payment_refund_id' => sanitize_text_field($data['payment_refund_id'] ?? ''),
                'line_items' => !empty($data['line_items']) ? json_encode($data['line_items']) : null,
                'status' => $data['status'] ?? 'completed',
            ],
            ['%d', '%f', '%s', '%d', '%s', '%s', '%s']
        );
        
        $refundId = (int) $wpdb->insert_id;
        
        if ($refundId) {
            // Update order total and status if fully refunded
            self::updateOrderAfterRefund((int) $data['order_id']);
            
            // Restore stock if requested
            if (!empty($data['restore_stock']) && !empty($data['line_items'])) {
                self::restoreStock($data['line_items']);
            }
            
            // Add order note
            OrderNoteService::addNote(
                (int) $data['order_id'],
                sprintf(
                    __('Refund of %s processed. %s', 'cartt'),
                    '$' . number_format((float) $data['amount'], 2),
                    $data['reason'] ?? ''
                ),
                false
            );
        }
        
        return $refundId;
    }

    public static function getOrderTotalRefunded(int $orderId): float
    {
        global $wpdb;
        return (float) $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(amount), 0) FROM {$wpdb->prefix}cartt_refunds 
             WHERE order_id = %d AND status = 'completed'",
            $orderId
        ));
    }

    public static function processStripeRefund(int $orderId, float $amount, string $reason = ''): array
    {
        global $wpdb;
        
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));
        
        if (!$order) {
            return ['success' => false, 'error' => 'Order not found'];
        }
        
        $paymentIntentId = $order->payment_intent_id ?? '';
        if (empty($paymentIntentId)) {
            return ['success' => false, 'error' => 'No payment intent found for this order'];
        }
        
        $secretKey = get_option('cartt_stripe_secret_key', '');
        if (empty($secretKey)) {
            return ['success' => false, 'error' => 'Stripe not configured'];
        }
        
        // Create refund via Stripe API
        $response = wp_remote_post('https://api.stripe.com/v1/refunds', [
            'headers' => [
                'Authorization' => 'Bearer ' . $secretKey,
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => [
                'payment_intent' => $paymentIntentId,
                'amount' => (int) ($amount * 100), // Convert to cents
                'reason' => 'requested_by_customer',
            ],
        ]);
        
        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($body['error'])) {
            return ['success' => false, 'error' => $body['error']['message']];
        }
        
        // Create refund record
        $refundId = self::createRefund([
            'order_id' => $orderId,
            'amount' => $amount,
            'reason' => $reason,
            'payment_refund_id' => $body['id'] ?? '',
            'status' => 'completed',
        ]);
        
        return ['success' => true, 'refund_id' => $refundId, 'stripe_refund_id' => $body['id']];
    }

    private static function updateOrderAfterRefund(int $orderId): void
    {
        global $wpdb;
        
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT total FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));
        
        if (!$order) {
            return;
        }
        
        $totalRefunded = self::getOrderTotalRefunded($orderId);
        
        // If fully refunded, update status
        if ($totalRefunded >= $order->total) {
            $wpdb->update(
                $wpdb->prefix . 'cartt_orders',
                ['status' => 'refunded'],
                ['id' => $orderId],
                ['%s'],
                ['%d']
            );
        }
    }

    private static function restoreStock(array $lineItems): void
    {
        global $wpdb;
        
        foreach ($lineItems as $item) {
            if (!empty($item['product_id']) && !empty($item['quantity'])) {
                $wpdb->query($wpdb->prepare(
                    "UPDATE {$wpdb->prefix}cartt_products 
                     SET stock_quantity = stock_quantity + %d 
                     WHERE id = %d AND manage_stock = 1",
                    (int) $item['quantity'],
                    (int) $item['product_id']
                ));
            }
        }
    }
}
