<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * RMA Service
 * Manages product returns and refund requests
 */
class RMAService
{
    private static array $statuses = [
        'pending' => 'Pending Review',
        'approved' => 'Approved',
        'rejected' => 'Rejected',
        'shipped' => 'Return Shipped',
        'received' => 'Received',
        'refunded' => 'Refunded',
        'completed' => 'Completed',
    ];

    private static array $reasons = [
        'defective' => 'Defective or Damaged',
        'wrong_item' => 'Wrong Item Received',
        'not_as_described' => 'Not as Described',
        'changed_mind' => 'Changed My Mind',
        'other' => 'Other',
    ];

    private static array $refundTypes = [
        'original' => 'Original Payment Method',
        'store_credit' => 'Store Credit',
        'exchange' => 'Exchange for Different Item',
    ];

    /**
     * Generate RMA number
     */
    public static function generateRMANumber(): string
    {
        $prefix = 'RMA';
        $number = $prefix . date('Ymd') . strtoupper(wp_generate_password(6, false, false));
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';
        
        while ($wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE rma_number = %s", $number))) {
            $number = $prefix . date('Ymd') . strtoupper(wp_generate_password(6, false, false));
        }
        
        return $number;
    }

    /**
     * Create return request
     */
    public static function createRequest(array $data): ?object
    {
        global $wpdb;
        $rma_table = $wpdb->prefix . 'cartt_rma';
        $items_table = $wpdb->prefix . 'cartt_rma_items';

        // Validate order exists and belongs to customer
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $orders_table WHERE id = %d AND customer_id = %d",
            intval($data['order_id']),
            intval($data['customer_id'])
        ));

        if (!$order) {
            return null;
        }

        // Check return window (default 30 days)
        $returnDays = intval(get_option('cartt_return_days', 30));
        $orderDate = strtotime($order->created_at);
        if (time() - $orderDate > $returnDays * 86400) {
            return null; // Past return window
        }

        $rmaNumber = self::generateRMANumber();

        $wpdb->insert($rma_table, [
            'rma_number' => $rmaNumber,
            'order_id' => intval($data['order_id']),
            'customer_id' => intval($data['customer_id']),
            'reason' => sanitize_text_field($data['reason']),
            'reason_detail' => sanitize_textarea_field($data['reason_detail'] ?? ''),
            'refund_type' => sanitize_text_field($data['refund_type'] ?? 'original'),
            'status' => 'pending',
        ]);

        $rmaId = $wpdb->insert_id;

        // Add items
        if (!empty($data['items']) && is_array($data['items'])) {
            foreach ($data['items'] as $item) {
                $wpdb->insert($items_table, [
                    'rma_id' => $rmaId,
                    'order_item_id' => intval($item['order_item_id']),
                    'product_id' => intval($item['product_id']),
                    'quantity' => intval($item['quantity']),
                ]);
            }
        }

        // Send notification email
        self::sendStatusEmail($rmaId, 'created');

        return self::getById($rmaId);
    }

    /**
     * Get RMA by ID
     */
    public static function getById(int $id): ?object
    {
        global $wpdb;
        $rma_table = $wpdb->prefix . 'cartt_rma';
        $items_table = $wpdb->prefix . 'cartt_rma_items';

        $rma = $wpdb->get_row($wpdb->prepare("SELECT * FROM $rma_table WHERE id = %d", $id));
        
        if ($rma) {
            $rma->items = $wpdb->get_results($wpdb->prepare(
                "SELECT ri.*, p.name as product_name
                 FROM $items_table ri
                 JOIN {$wpdb->prefix}cartt_products p ON ri.product_id = p.id
                 WHERE ri.rma_id = %d",
                $id
            ));
            $rma->status_label = self::$statuses[$rma->status] ?? $rma->status;
            $rma->reason_label = self::$reasons[$rma->reason] ?? $rma->reason;
        }

        return $rma;
    }

    /**
     * Get RMA by number
     */
    public static function getByNumber(string $number): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        $id = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE rma_number = %s", $number));
        
        return $id ? self::getById((int) $id) : null;
    }

    /**
     * Get customer's RMAs
     */
    public static function getCustomerRMAs(int $customerId): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        $rmas = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE customer_id = %d ORDER BY created_at DESC",
            $customerId
        ));

        foreach ($rmas as &$rma) {
            $rma->status_label = self::$statuses[$rma->status] ?? $rma->status;
        }

        return $rmas;
    }

    /**
     * Get all RMAs (admin)
     */
    public static function getAll(array $args = []): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        $where = '1=1';
        $params = [];

        if (!empty($args['status'])) {
            $where .= ' AND status = %s';
            $params[] = $args['status'];
        }

        if (!empty($args['search'])) {
            $where .= ' AND (rma_number LIKE %s)';
            $params[] = '%' . $wpdb->esc_like($args['search']) . '%';
        }

        $orderBy = 'created_at DESC';
        if (!empty($args['orderby'])) {
            $orderBy = sanitize_sql_orderby($args['orderby'] . ' ' . ($args['order'] ?? 'DESC')) ?: $orderBy;
        }

        $limit = intval($args['limit'] ?? 50);
        $offset = intval($args['offset'] ?? 0);

        $sql = "SELECT * FROM $table WHERE $where ORDER BY $orderBy LIMIT %d OFFSET %d";
        $params[] = $limit;
        $params[] = $offset;

        $rmas = $wpdb->get_results($wpdb->prepare($sql, ...$params));

        foreach ($rmas as &$rma) {
            $rma->status_label = self::$statuses[$rma->status] ?? $rma->status;
        }

        return $rmas;
    }

    /**
     * Update RMA status
     */
    public static function updateStatus(int $id, string $status, ?int $processedBy = null, ?string $notes = null): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        $update = ['status' => $status];
        
        if ($processedBy) {
            $update['processed_by'] = $processedBy;
        }
        
        if ($notes) {
            $rma = self::getById($id);
            $update['admin_notes'] = $rma->admin_notes ? $rma->admin_notes . "\n\n" . $notes : $notes;
        }

        $result = $wpdb->update($table, $update, ['id' => $id]);

        if ($result !== false) {
            self::sendStatusEmail($id, $status);
        }

        return $result !== false;
    }

    /**
     * Approve RMA
     */
    public static function approve(int $id, float $refundAmount, float $restockingFee = 0, ?string $notes = null): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        $wpdb->update($table, [
            'status' => 'approved',
            'refund_amount' => $refundAmount,
            'restocking_fee' => $restockingFee,
            'admin_notes' => $notes,
            'processed_by' => get_current_user_id(),
        ], ['id' => $id]);

        self::sendStatusEmail($id, 'approved');

        return true;
    }

    /**
     * Reject RMA
     */
    public static function reject(int $id, string $reason): bool
    {
        return self::updateStatus($id, 'rejected', get_current_user_id(), $reason);
    }

    /**
     * Add tracking number
     */
    public static function addTracking(int $id, string $trackingNumber): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        return $wpdb->update($table, [
            'tracking_number' => sanitize_text_field($trackingNumber),
            'status' => 'shipped',
        ], ['id' => $id]) !== false;
    }

    /**
     * Mark as received
     */
    public static function markReceived(int $id, array $itemConditions = []): bool
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_rma_items';

        // Update item conditions
        foreach ($itemConditions as $itemId => $condition) {
            $wpdb->update($items_table, [
                'condition_received' => sanitize_text_field($condition),
            ], ['id' => $itemId]);
        }

        return self::updateStatus($id, 'received', get_current_user_id());
    }

    /**
     * Process refund
     */
    public static function processRefund(int $id): bool
    {
        $rma = self::getById($id);
        
        if (!$rma || $rma->status !== 'received') {
            return false;
        }

        // Restock items if needed
        foreach ($rma->items as $item) {
            if ($item->restock) {
                self::restockItem($item->product_id, $item->quantity);
            }
        }

        if ($rma->refund_type === 'store_credit') {
            // Create gift card as store credit
            GiftCardService::create([
                'amount' => $rma->refund_amount,
                'recipient_email' => self::getCustomerEmail($rma->customer_id),
                'message' => 'Store credit from return ' . $rma->rma_number,
            ]);
        } elseif ($rma->refund_type === 'original') {
            // Create refund record
            RefundService::createRefund([
                'order_id' => $rma->order_id,
                'amount' => $rma->refund_amount,
                'reason' => 'RMA: ' . $rma->rma_number,
                'restore_stock' => false, // Already handled above
            ]);
        }

        return self::updateStatus($id, 'refunded', get_current_user_id());
    }

    /**
     * Complete RMA
     */
    public static function complete(int $id): bool
    {
        return self::updateStatus($id, 'completed', get_current_user_id());
    }

    /**
     * Restock item
     */
    private static function restockItem(int $productId, int $quantity): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        $wpdb->query($wpdb->prepare(
            "UPDATE $table SET stock_quantity = stock_quantity + %d WHERE id = %d",
            $quantity, $productId
        ));
    }

    /**
     * Get customer email
     */
    private static function getCustomerEmail(int $customerId): string
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customers';

        return $wpdb->get_var($wpdb->prepare("SELECT email FROM $table WHERE id = %d", $customerId)) ?? '';
    }

    /**
     * Send status email
     */
    private static function sendStatusEmail(int $rmaId, string $status): void
    {
        $rma = self::getById($rmaId);
        if (!$rma) return;

        $email = self::getCustomerEmail($rma->customer_id);
        if (!$email) return;

        $subjects = [
            'created' => 'Return Request Received - ' . $rma->rma_number,
            'approved' => 'Return Request Approved - ' . $rma->rma_number,
            'rejected' => 'Return Request Update - ' . $rma->rma_number,
            'received' => 'Return Received - ' . $rma->rma_number,
            'refunded' => 'Refund Processed - ' . $rma->rma_number,
        ];

        $subject = $subjects[$status] ?? 'Return Update - ' . $rma->rma_number;

        // Simple email template
        $message = self::getEmailTemplate($rma, $status);

        wp_mail($email, $subject, $message, ['Content-Type: text/html; charset=UTF-8']);
    }

    /**
     * Get email template
     */
    private static function getEmailTemplate(object $rma, string $status): string
    {
        $siteName = get_bloginfo('name');
        
        $statusMessages = [
            'created' => 'We have received your return request and will review it shortly.',
            'approved' => 'Your return has been approved! Please ship your items back to us.',
            'rejected' => 'Unfortunately, your return request could not be approved.',
            'received' => 'We have received your returned items and are processing your refund.',
            'refunded' => 'Your refund has been processed.',
        ];

        $message = $statusMessages[$status] ?? 'Your return request has been updated.';

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; margin: 0; padding: 20px; }
                .container { max-width: 600px; margin: 0 auto; }
                .header { background: #1d1d1f; color: #fff; padding: 20px; text-align: center; }
                .body { padding: 30px; background: #fff; }
                .footer { padding: 20px; text-align: center; font-size: 12px; color: #666; }
                .details { background: #f5f5f7; padding: 15px; border-radius: 8px; margin: 20px 0; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>Return Request Update</h1>
                </div>
                <div class="body">
                    <p><?php echo esc_html($message); ?></p>
                    
                    <div class="details">
                        <p><strong>RMA Number:</strong> <?php echo esc_html($rma->rma_number); ?></p>
                        <p><strong>Status:</strong> <?php echo esc_html($rma->status_label); ?></p>
                        <?php if ($rma->refund_amount): ?>
                        <p><strong>Refund Amount:</strong> <?php echo esc_html(CurrencyService::formatPrice($rma->refund_amount)); ?></p>
                        <?php endif; ?>
                    </div>

                    <?php if ($status === 'approved' && $rma->return_label_url): ?>
                    <p><a href="<?php echo esc_url($rma->return_label_url); ?>">Download Return Label</a></p>
                    <?php endif; ?>

                    <?php if ($rma->admin_notes && in_array($status, ['rejected', 'approved'])): ?>
                    <p><strong>Notes:</strong> <?php echo esc_html($rma->admin_notes); ?></p>
                    <?php endif; ?>
                </div>
                <div class="footer">
                    <p>&copy; <?php echo date('Y'); ?> <?php echo esc_html($siteName); ?></p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Get status options
     */
    public static function getStatuses(): array
    {
        return self::$statuses;
    }

    /**
     * Get reason options
     */
    public static function getReasons(): array
    {
        return self::$reasons;
    }

    /**
     * Get refund type options
     */
    public static function getRefundTypes(): array
    {
        return self::$refundTypes;
    }

    /**
     * Get pending RMA count
     */
    public static function getPendingCount(): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_rma';

        return (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status = 'pending'");
    }
}
