<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Product Q&A Service
 * Manages customer questions and answers on products
 */
class ProductQAService
{
    /**
     * Submit a question
     */
    public static function submitQuestion(array $data): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        $autoApprove = get_option('cartt_qa_auto_approve', false);

        $wpdb->insert($table, [
            'product_id' => intval($data['product_id']),
            'user_id' => !empty($data['user_id']) ? intval($data['user_id']) : null,
            'author_name' => sanitize_text_field($data['author_name']),
            'author_email' => sanitize_email($data['author_email']),
            'content' => sanitize_textarea_field($data['content']),
            'is_answer' => 0,
            'status' => $autoApprove ? 'approved' : 'pending',
        ]);

        $id = $wpdb->insert_id;

        // Notify admin
        if (!$autoApprove) {
            self::notifyAdmin($id);
        }

        return $id ?: null;
    }

    /**
     * Submit an answer
     */
    public static function submitAnswer(array $data): ?int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        // Get the question to get product_id
        $question = self::getById(intval($data['question_id']));
        if (!$question) {
            return null;
        }

        $isVendor = !empty($data['is_vendor']) || current_user_can('manage_options');

        $wpdb->insert($table, [
            'product_id' => $question->product_id,
            'parent_id' => $question->id,
            'user_id' => !empty($data['user_id']) ? intval($data['user_id']) : null,
            'author_name' => sanitize_text_field($data['author_name']),
            'author_email' => sanitize_email($data['author_email']),
            'content' => sanitize_textarea_field($data['content']),
            'is_answer' => 1,
            'is_vendor_answer' => $isVendor ? 1 : 0,
            'status' => $isVendor ? 'approved' : 'pending',
        ]);

        $id = $wpdb->insert_id;

        // Notify question author
        if ($id && $isVendor) {
            self::notifyQuestionAuthor($question->id, $id);
        }

        return $id ?: null;
    }

    /**
     * Get Q&A item by ID
     */
    public static function getById(int $id): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));
    }

    /**
     * Get product questions with answers
     */
    public static function getProductQA(int $productId, bool $approvedOnly = true): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        $statusClause = $approvedOnly ? "AND status = 'approved'" : '';

        // Get questions
        $questions = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table 
             WHERE product_id = %d AND is_answer = 0 $statusClause
             ORDER BY upvotes DESC, created_at DESC",
            $productId
        ));

        // Get answers for each question
        foreach ($questions as &$question) {
            $question->answers = $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM $table 
                 WHERE parent_id = %d AND is_answer = 1 $statusClause
                 ORDER BY is_vendor_answer DESC, upvotes DESC, created_at ASC",
                $question->id
            ));
            $question->answer_count = count($question->answers);
        }

        return $questions;
    }

    /**
     * Get all Q&A (admin)
     */
    public static function getAll(array $args = []): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        $where = 'is_answer = 0'; // Only get questions
        $params = [];

        if (!empty($args['status'])) {
            $where .= ' AND status = %s';
            $params[] = $args['status'];
        }

        if (!empty($args['product_id'])) {
            $where .= ' AND product_id = %d';
            $params[] = intval($args['product_id']);
        }

        $limit = intval($args['limit'] ?? 50);
        $offset = intval($args['offset'] ?? 0);

        $sql = "SELECT qa.*, p.name as product_name
                FROM $table qa
                LEFT JOIN {$wpdb->prefix}cartt_products p ON qa.product_id = p.id
                WHERE $where
                ORDER BY qa.created_at DESC
                LIMIT %d OFFSET %d";
        $params[] = $limit;
        $params[] = $offset;

        $questions = $wpdb->get_results($wpdb->prepare($sql, ...$params));

        // Get answer counts
        foreach ($questions as &$q) {
            $q->answer_count = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE parent_id = %d",
                $q->id
            ));
        }

        return $questions;
    }

    /**
     * Update status
     */
    public static function updateStatus(int $id, string $status): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        $result = $wpdb->update($table, ['status' => $status], ['id' => $id]);

        // If approving a question, notify author
        if ($result !== false && $status === 'approved') {
            $item = self::getById($id);
            if ($item && !$item->is_answer) {
                self::notifyApproval($id);
            }
        }

        return $result !== false;
    }

    /**
     * Delete Q&A item
     */
    public static function delete(int $id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        // Delete answers first if this is a question
        $wpdb->delete($table, ['parent_id' => $id]);

        return $wpdb->delete($table, ['id' => $id]) !== false;
    }

    /**
     * Upvote
     */
    public static function upvote(int $id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        return $wpdb->query($wpdb->prepare(
            "UPDATE $table SET upvotes = upvotes + 1 WHERE id = %d",
            $id
        )) !== false;
    }

    /**
     * Get pending count
     */
    public static function getPendingCount(): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        return (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status = 'pending'");
    }

    /**
     * Get question count for product
     */
    public static function getProductQuestionCount(int $productId): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE product_id = %d AND is_answer = 0 AND status = 'approved'",
            $productId
        ));
    }

    /**
     * Notify admin of new question
     */
    private static function notifyAdmin(int $questionId): void
    {
        $question = self::getById($questionId);
        if (!$question) return;

        $adminEmail = get_option('admin_email');
        $subject = 'New Product Question Pending Review';
        
        $message = sprintf(
            "A new question has been submitted and is pending review.\n\nQuestion: %s\n\nFrom: %s\n\nReview it here: %s",
            $question->content,
            $question->author_name,
            admin_url('admin.php?page=cartt-qa')
        );

        wp_mail($adminEmail, $subject, $message);
    }

    /**
     * Notify question author of answer
     */
    private static function notifyQuestionAuthor(int $questionId, int $answerId): void
    {
        $question = self::getById($questionId);
        $answer = self::getById($answerId);
        
        if (!$question || !$answer || !$question->author_email) return;

        global $wpdb;
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT name, slug FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $question->product_id
        ));

        $subject = 'Your question has been answered!';
        
        $productUrl = home_url('/product/' . $product->slug);
        
        $message = sprintf(
            "Your question about %s has received an answer!\n\nYour question: %s\n\nAnswer: %s\n\nView it here: %s",
            $product->name,
            $question->content,
            $answer->content,
            $productUrl
        );

        wp_mail($question->author_email, $subject, $message);
    }

    /**
     * Notify author of approval
     */
    private static function notifyApproval(int $questionId): void
    {
        $question = self::getById($questionId);
        if (!$question || !$question->author_email) return;

        global $wpdb;
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT name, slug FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $question->product_id
        ));

        $subject = 'Your question has been published';
        $productUrl = home_url('/product/' . $product->slug);
        
        $message = sprintf(
            "Your question about %s has been approved and is now visible.\n\nView it here: %s",
            $product->name,
            $productUrl
        );

        wp_mail($question->author_email, $subject, $message);
    }

    /**
     * Search Q&A
     */
    public static function search(int $productId, string $query): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_qa';

        $search = '%' . $wpdb->esc_like($query) . '%';

        $questions = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table 
             WHERE product_id = %d 
             AND is_answer = 0 
             AND status = 'approved'
             AND content LIKE %s
             ORDER BY upvotes DESC, created_at DESC
             LIMIT 20",
            $productId, $search
        ));

        foreach ($questions as &$question) {
            $question->answers = $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM $table 
                 WHERE parent_id = %d AND is_answer = 1 AND status = 'approved'
                 ORDER BY is_vendor_answer DESC, upvotes DESC",
                $question->id
            ));
        }

        return $questions;
    }
}
