<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Live Shipping Service
 * Real-time shipping rates from carriers
 */
class LiveShippingService
{
    private static array $carriers = [
        'usps' => [
            'name' => 'USPS',
            'api_url' => 'https://secure.shippingapis.com/ShippingAPI.dll',
        ],
        'ups' => [
            'name' => 'UPS',
            'api_url' => 'https://onlinetools.ups.com/api/rating/v1/Rate',
        ],
        'fedex' => [
            'name' => 'FedEx',
            'api_url' => 'https://apis.fedex.com/rate/v1/rates/quotes',
        ],
    ];

    /**
     * Get settings
     */
    public static function getSettings(): array
    {
        return [
            'enabled' => get_option('cartt_live_shipping_enabled', false),
            'usps_user_id' => get_option('cartt_usps_user_id', ''),
            'ups_client_id' => get_option('cartt_ups_client_id', ''),
            'ups_client_secret' => get_option('cartt_ups_client_secret', ''),
            'fedex_api_key' => get_option('cartt_fedex_api_key', ''),
            'fedex_secret_key' => get_option('cartt_fedex_secret_key', ''),
            'fedex_account_number' => get_option('cartt_fedex_account_number', ''),
            'origin_zip' => get_option('cartt_shipping_origin_zip', ''),
            'origin_country' => get_option('cartt_shipping_origin_country', 'US'),
            'handling_fee' => floatval(get_option('cartt_shipping_handling_fee', 0)),
            'free_shipping_threshold' => floatval(get_option('cartt_free_shipping_threshold', 0)),
        ];
    }

    /**
     * Get live rates for cart
     */
    public static function getRates(array $package): array
    {
        $settings = self::getSettings();
        
        if (!$settings['enabled']) {
            return [];
        }

        // Check free shipping threshold
        if ($settings['free_shipping_threshold'] > 0 && $package['total'] >= $settings['free_shipping_threshold']) {
            return [[
                'id' => 'free_shipping',
                'carrier' => 'Store',
                'service' => 'Free Shipping',
                'rate' => 0,
                'delivery_days' => null,
            ]];
        }

        $rates = [];

        // Get rates from enabled carriers
        if ($settings['usps_user_id']) {
            $uspsRates = self::getUSPSRates($package, $settings);
            $rates = array_merge($rates, $uspsRates);
        }

        if ($settings['ups_client_id'] && $settings['ups_client_secret']) {
            $upsRates = self::getUPSRates($package, $settings);
            $rates = array_merge($rates, $upsRates);
        }

        if ($settings['fedex_api_key']) {
            $fedexRates = self::getFedExRates($package, $settings);
            $rates = array_merge($rates, $fedexRates);
        }

        // Add handling fee
        if ($settings['handling_fee'] > 0) {
            foreach ($rates as &$rate) {
                $rate['rate'] += $settings['handling_fee'];
            }
        }

        // Sort by rate
        usort($rates, fn($a, $b) => $a['rate'] <=> $b['rate']);

        // Cache rates
        $cacheKey = 'cartt_shipping_rates_' . md5(json_encode($package));
        set_transient($cacheKey, $rates, 300); // 5 minutes

        return $rates;
    }

    /**
     * Get USPS rates
     */
    private static function getUSPSRates(array $package, array $settings): array
    {
        $userId = $settings['usps_user_id'];
        $originZip = $settings['origin_zip'];
        $destZip = $package['destination']['postcode'] ?? '';
        
        if (!$destZip || strlen($destZip) < 5) {
            return [];
        }

        // Calculate package weight (default to 1 lb if not set)
        $weight = $package['weight'] ?? 1;
        $pounds = floor($weight);
        $ounces = ($weight - $pounds) * 16;

        $xml = '<?xml version="1.0"?>
            <RateV4Request USERID="' . esc_attr($userId) . '">
                <Revision>2</Revision>
                <Package ID="1ST">
                    <Service>ALL</Service>
                    <ZipOrigination>' . substr($originZip, 0, 5) . '</ZipOrigination>
                    <ZipDestination>' . substr($destZip, 0, 5) . '</ZipDestination>
                    <Pounds>' . intval($pounds) . '</Pounds>
                    <Ounces>' . intval($ounces) . '</Ounces>
                    <Container>VARIABLE</Container>
                    <Width></Width>
                    <Length></Length>
                    <Height></Height>
                    <Girth></Girth>
                    <Machinable>TRUE</Machinable>
                </Package>
            </RateV4Request>';

        $response = wp_remote_get(
            self::$carriers['usps']['api_url'] . '?API=RateV4&XML=' . urlencode($xml),
            ['timeout' => 15]
        );

        if (is_wp_error($response)) {
            return [];
        }

        $body = wp_remote_retrieve_body($response);
        $xml = @simplexml_load_string($body);

        if (!$xml || isset($xml->Error)) {
            return [];
        }

        $rates = [];
        if (isset($xml->Package->Postage)) {
            foreach ($xml->Package->Postage as $postage) {
                $serviceName = (string) $postage->MailService;
                // Clean up service name
                $serviceName = preg_replace('/&lt;.*?&gt;/', '', $serviceName);
                $serviceName = html_entity_decode($serviceName);
                $serviceName = preg_replace('/\s+/', ' ', trim($serviceName));

                $rates[] = [
                    'id' => 'usps_' . sanitize_title($serviceName),
                    'carrier' => 'USPS',
                    'service' => $serviceName,
                    'rate' => floatval($postage->Rate),
                    'delivery_days' => self::parseUSPSDeliveryDays((string) ($postage->CommitmentName ?? '')),
                ];
            }
        }

        return $rates;
    }

    /**
     * Parse USPS delivery commitment
     */
    private static function parseUSPSDeliveryDays(string $commitment): ?int
    {
        if (preg_match('/(\d+)/', $commitment, $matches)) {
            return intval($matches[1]);
        }
        return null;
    }

    /**
     * Get UPS rates
     */
    private static function getUPSRates(array $package, array $settings): array
    {
        // Get OAuth token
        $token = self::getUPSToken($settings);
        if (!$token) {
            return [];
        }

        $weight = $package['weight'] ?? 1;
        $destCountry = $package['destination']['country'] ?? 'US';
        $destZip = $package['destination']['postcode'] ?? '';
        $destCity = $package['destination']['city'] ?? '';
        $destState = $package['destination']['state'] ?? '';

        $requestBody = [
            'RateRequest' => [
                'Request' => [
                    'TransactionReference' => [
                        'CustomerContext' => 'Cartt Rate Request',
                    ],
                ],
                'Shipment' => [
                    'Shipper' => [
                        'Address' => [
                            'PostalCode' => $settings['origin_zip'],
                            'CountryCode' => $settings['origin_country'],
                        ],
                    ],
                    'ShipTo' => [
                        'Address' => [
                            'City' => $destCity,
                            'StateProvinceCode' => $destState,
                            'PostalCode' => $destZip,
                            'CountryCode' => $destCountry,
                        ],
                    ],
                    'ShipFrom' => [
                        'Address' => [
                            'PostalCode' => $settings['origin_zip'],
                            'CountryCode' => $settings['origin_country'],
                        ],
                    ],
                    'Package' => [
                        'PackagingType' => [
                            'Code' => '02', // Customer Supplied Package
                        ],
                        'PackageWeight' => [
                            'UnitOfMeasurement' => [
                                'Code' => 'LBS',
                            ],
                            'Weight' => strval(max(0.1, $weight)),
                        ],
                    ],
                ],
            ],
        ];

        $response = wp_remote_post(self::$carriers['ups']['api_url'], [
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
                'transId' => uniqid(),
                'transactionSrc' => 'Cartt',
            ],
            'body' => json_encode($requestBody),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!isset($body['RateResponse']['RatedShipment'])) {
            return [];
        }

        $rates = [];
        $serviceNames = [
            '01' => 'UPS Next Day Air',
            '02' => 'UPS 2nd Day Air',
            '03' => 'UPS Ground',
            '12' => 'UPS 3 Day Select',
            '13' => 'UPS Next Day Air Saver',
            '14' => 'UPS Next Day Air Early',
            '59' => 'UPS 2nd Day Air A.M.',
        ];

        foreach ($body['RateResponse']['RatedShipment'] as $shipment) {
            $serviceCode = $shipment['Service']['Code'] ?? '';
            $serviceName = $serviceNames[$serviceCode] ?? 'UPS Service ' . $serviceCode;

            $rates[] = [
                'id' => 'ups_' . $serviceCode,
                'carrier' => 'UPS',
                'service' => $serviceName,
                'rate' => floatval($shipment['TotalCharges']['MonetaryValue'] ?? 0),
                'delivery_days' => intval($shipment['GuaranteedDelivery']['BusinessDaysInTransit'] ?? null),
            ];
        }

        return $rates;
    }

    /**
     * Get UPS OAuth token
     */
    private static function getUPSToken(array $settings): ?string
    {
        $cached = get_transient('cartt_ups_token');
        if ($cached) {
            return $cached;
        }

        $response = wp_remote_post('https://onlinetools.ups.com/security/v1/oauth/token', [
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Authorization' => 'Basic ' . base64_encode($settings['ups_client_id'] . ':' . $settings['ups_client_secret']),
            ],
            'body' => 'grant_type=client_credentials',
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['access_token'])) {
            set_transient('cartt_ups_token', $body['access_token'], 3600);
            return $body['access_token'];
        }

        return null;
    }

    /**
     * Get FedEx rates
     */
    private static function getFedExRates(array $package, array $settings): array
    {
        // Get OAuth token
        $token = self::getFedExToken($settings);
        if (!$token) {
            return [];
        }

        $weight = $package['weight'] ?? 1;
        $destCountry = $package['destination']['country'] ?? 'US';
        $destZip = $package['destination']['postcode'] ?? '';
        $destCity = $package['destination']['city'] ?? '';
        $destState = $package['destination']['state'] ?? '';

        $requestBody = [
            'accountNumber' => ['value' => $settings['fedex_account_number']],
            'requestedShipment' => [
                'shipper' => [
                    'address' => [
                        'postalCode' => $settings['origin_zip'],
                        'countryCode' => $settings['origin_country'],
                    ],
                ],
                'recipient' => [
                    'address' => [
                        'city' => $destCity,
                        'stateOrProvinceCode' => $destState,
                        'postalCode' => $destZip,
                        'countryCode' => $destCountry,
                    ],
                ],
                'pickupType' => 'DROPOFF_AT_FEDEX_LOCATION',
                'rateRequestType' => ['LIST'],
                'requestedPackageLineItems' => [[
                    'weight' => [
                        'units' => 'LB',
                        'value' => max(0.1, $weight),
                    ],
                ]],
            ],
        ];

        $response = wp_remote_post(self::$carriers['fedex']['api_url'], [
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
                'X-locale' => 'en_US',
            ],
            'body' => json_encode($requestBody),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!isset($body['output']['rateReplyDetails'])) {
            return [];
        }

        $rates = [];
        foreach ($body['output']['rateReplyDetails'] as $detail) {
            $serviceName = $detail['serviceName'] ?? $detail['serviceType'] ?? 'FedEx Service';
            $rate = $detail['ratedShipmentDetails'][0]['totalNetCharge'] ?? 0;

            $rates[] = [
                'id' => 'fedex_' . sanitize_title($detail['serviceType'] ?? ''),
                'carrier' => 'FedEx',
                'service' => $serviceName,
                'rate' => floatval($rate),
                'delivery_days' => intval($detail['commit']['transitDays'] ?? null) ?: null,
            ];
        }

        return $rates;
    }

    /**
     * Get FedEx OAuth token
     */
    private static function getFedExToken(array $settings): ?string
    {
        $cached = get_transient('cartt_fedex_token');
        if ($cached) {
            return $cached;
        }

        $response = wp_remote_post('https://apis.fedex.com/oauth/token', [
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => [
                'grant_type' => 'client_credentials',
                'client_id' => $settings['fedex_api_key'],
                'client_secret' => $settings['fedex_secret_key'],
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return null;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (isset($body['access_token'])) {
            set_transient('cartt_fedex_token', $body['access_token'], 3600);
            return $body['access_token'];
        }

        return null;
    }

    /**
     * Get cached rates
     */
    public static function getCachedRates(array $package): ?array
    {
        $cacheKey = 'cartt_shipping_rates_' . md5(json_encode($package));
        return get_transient($cacheKey) ?: null;
    }

    /**
     * Calculate package weight from cart items
     */
    public static function calculatePackageWeight(array $items): float
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        
        $weight = 0;
        foreach ($items as $item) {
            $productWeight = $wpdb->get_var($wpdb->prepare(
                "SELECT weight FROM $products_table WHERE id = %d",
                $item['product_id']
            ));
            $weight += floatval($productWeight ?: 0.5) * $item['quantity'];
        }

        return max(0.1, $weight);
    }

    /**
     * Get available carriers
     */
    public static function getCarriers(): array
    {
        return self::$carriers;
    }
}
