<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Invoice Service
 * Generates invoices and packing slips
 */
class InvoiceService
{
    /**
     * Get company info for documents
     */
    private static function getCompanyInfo(): array
    {
        return [
            'name' => get_option('cartt_company_name', get_bloginfo('name')),
            'address' => get_option('cartt_company_address', ''),
            'city' => get_option('cartt_company_city', ''),
            'state' => get_option('cartt_company_state', ''),
            'zip' => get_option('cartt_company_zip', ''),
            'country' => get_option('cartt_company_country', ''),
            'phone' => get_option('cartt_company_phone', ''),
            'email' => get_option('cartt_company_email', get_option('admin_email')),
            'logo' => get_option('cartt_company_logo', ''),
            'tax_id' => get_option('cartt_company_tax_id', ''),
        ];
    }

    /**
     * Get order data for documents
     */
    private static function getOrderData(int $orderId): ?array
    {
        global $wpdb;

        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT o.*, c.first_name, c.last_name, c.email, c.phone
             FROM {$wpdb->prefix}cartt_orders o
             LEFT JOIN {$wpdb->prefix}cartt_customers c ON o.customer_id = c.id
             WHERE o.id = %d",
            $orderId
        ));

        if (!$order) {
            return null;
        }

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_order_items WHERE order_id = %d",
            $orderId
        ));

        $billing = json_decode($order->billing_address, true) ?: [];
        $shipping = json_decode($order->shipping_address, true) ?: [];

        return [
            'order' => $order,
            'items' => $items,
            'billing' => $billing,
            'shipping' => $shipping,
        ];
    }

    /**
     * Generate invoice HTML
     */
    public static function generateInvoice(int $orderId): ?string
    {
        $data = self::getOrderData($orderId);
        if (!$data) {
            return null;
        }

        $company = self::getCompanyInfo();
        $order = $data['order'];
        $items = $data['items'];
        $billing = $data['billing'];

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Invoice #<?php echo esc_html($order->order_number); ?></title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; font-size: 14px; line-height: 1.5; color: #333; padding: 40px; }
                .invoice { max-width: 800px; margin: 0 auto; }
                .header { display: flex; justify-content: space-between; margin-bottom: 40px; }
                .logo img { max-height: 60px; }
                .company-info { text-align: right; font-size: 12px; color: #666; }
                .invoice-title { font-size: 28px; font-weight: bold; color: #1d1d1f; margin-bottom: 20px; }
                .invoice-meta { display: flex; justify-content: space-between; margin-bottom: 40px; }
                .meta-block { }
                .meta-block h3 { font-size: 12px; text-transform: uppercase; color: #666; margin-bottom: 5px; }
                .meta-block p { margin: 2px 0; }
                table { width: 100%; border-collapse: collapse; margin: 20px 0; }
                th { text-align: left; padding: 12px; background: #f5f5f7; border-bottom: 2px solid #ddd; font-size: 12px; text-transform: uppercase; color: #666; }
                td { padding: 12px; border-bottom: 1px solid #eee; }
                .text-right { text-align: right; }
                .totals { margin-top: 20px; margin-left: auto; width: 300px; }
                .totals-row { display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #eee; }
                .totals-row.total { font-size: 18px; font-weight: bold; border-top: 2px solid #333; border-bottom: none; padding-top: 12px; }
                .footer { margin-top: 60px; padding-top: 20px; border-top: 1px solid #eee; text-align: center; font-size: 12px; color: #666; }
                .status { display: inline-block; padding: 4px 12px; border-radius: 4px; font-size: 12px; font-weight: 500; }
                .status-paid { background: #d4edda; color: #155724; }
                .status-pending { background: #fff3cd; color: #856404; }
                @media print { body { padding: 20px; } .no-print { display: none; } }
            </style>
        </head>
        <body>
            <div class="invoice">
                <div class="header">
                    <div class="logo">
                        <?php if ($company['logo']): ?>
                            <img src="<?php echo esc_url($company['logo']); ?>" alt="<?php echo esc_attr($company['name']); ?>">
                        <?php else: ?>
                            <h1 style="font-size: 24px;"><?php echo esc_html($company['name']); ?></h1>
                        <?php endif; ?>
                    </div>
                    <div class="company-info">
                        <strong><?php echo esc_html($company['name']); ?></strong><br>
                        <?php if ($company['address']): ?><?php echo esc_html($company['address']); ?><br><?php endif; ?>
                        <?php if ($company['city'] || $company['state'] || $company['zip']): ?>
                            <?php echo esc_html($company['city']); ?>, <?php echo esc_html($company['state']); ?> <?php echo esc_html($company['zip']); ?><br>
                        <?php endif; ?>
                        <?php if ($company['phone']): ?><?php echo esc_html($company['phone']); ?><br><?php endif; ?>
                        <?php if ($company['email']): ?><?php echo esc_html($company['email']); ?><br><?php endif; ?>
                        <?php if ($company['tax_id']): ?>Tax ID: <?php echo esc_html($company['tax_id']); ?><?php endif; ?>
                    </div>
                </div>

                <h1 class="invoice-title">
                    Invoice #<?php echo esc_html($order->order_number); ?>
                    <span class="status status-<?php echo $order->status === 'completed' ? 'paid' : 'pending'; ?>">
                        <?php echo $order->status === 'completed' ? 'PAID' : strtoupper($order->status); ?>
                    </span>
                </h1>

                <div class="invoice-meta">
                    <div class="meta-block">
                        <h3>Bill To</h3>
                        <p><strong><?php echo esc_html(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? '')); ?></strong></p>
                        <?php if (!empty($billing['company'])): ?><p><?php echo esc_html($billing['company']); ?></p><?php endif; ?>
                        <p><?php echo esc_html($billing['address_1'] ?? ''); ?></p>
                        <?php if (!empty($billing['address_2'])): ?><p><?php echo esc_html($billing['address_2']); ?></p><?php endif; ?>
                        <p><?php echo esc_html(($billing['city'] ?? '') . ', ' . ($billing['state'] ?? '') . ' ' . ($billing['postcode'] ?? '')); ?></p>
                        <p><?php echo esc_html($billing['country'] ?? ''); ?></p>
                        <?php if (!empty($billing['email'])): ?><p><?php echo esc_html($billing['email']); ?></p><?php endif; ?>
                    </div>
                    <div class="meta-block" style="text-align: right;">
                        <h3>Invoice Details</h3>
                        <p><strong>Invoice Date:</strong> <?php echo date('F j, Y', strtotime($order->created_at)); ?></p>
                        <p><strong>Order Date:</strong> <?php echo date('F j, Y', strtotime($order->created_at)); ?></p>
                        <?php if ($order->payment_method): ?>
                        <p><strong>Payment Method:</strong> <?php echo esc_html(ucfirst($order->payment_method)); ?></p>
                        <?php endif; ?>
                    </div>
                </div>

                <table>
                    <thead>
                        <tr>
                            <th>Item</th>
                            <th>SKU</th>
                            <th class="text-right">Qty</th>
                            <th class="text-right">Unit Price</th>
                            <th class="text-right">Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($items as $item): ?>
                        <tr>
                            <td><?php echo esc_html($item->name); ?></td>
                            <td><?php echo esc_html($item->sku ?: '-'); ?></td>
                            <td class="text-right"><?php echo esc_html($item->quantity); ?></td>
                            <td class="text-right"><?php echo esc_html(CurrencyService::formatPrice($item->price)); ?></td>
                            <td class="text-right"><?php echo esc_html(CurrencyService::formatPrice($item->total)); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <div class="totals">
                    <div class="totals-row">
                        <span>Subtotal</span>
                        <span><?php echo esc_html(CurrencyService::formatPrice($order->subtotal)); ?></span>
                    </div>
                    <?php if ($order->discount_total > 0): ?>
                    <div class="totals-row">
                        <span>Discount</span>
                        <span>-<?php echo esc_html(CurrencyService::formatPrice($order->discount_total)); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order->shipping_total > 0): ?>
                    <div class="totals-row">
                        <span>Shipping</span>
                        <span><?php echo esc_html(CurrencyService::formatPrice($order->shipping_total)); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($order->tax_total > 0): ?>
                    <div class="totals-row">
                        <span>Tax</span>
                        <span><?php echo esc_html(CurrencyService::formatPrice($order->tax_total)); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="totals-row total">
                        <span>Total</span>
                        <span><?php echo esc_html(CurrencyService::formatPrice($order->total)); ?></span>
                    </div>
                </div>

                <div class="footer">
                    <p>Thank you for your business!</p>
                    <p class="no-print" style="margin-top: 20px;">
                        <button onclick="window.print()" style="padding: 10px 20px; cursor: pointer;">Print Invoice</button>
                    </p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Generate packing slip HTML
     */
    public static function generatePackingSlip(int $orderId): ?string
    {
        $data = self::getOrderData($orderId);
        if (!$data) {
            return null;
        }

        $company = self::getCompanyInfo();
        $order = $data['order'];
        $items = $data['items'];
        $shipping = $data['shipping'];

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>Packing Slip - Order #<?php echo esc_html($order->order_number); ?></title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; font-size: 14px; line-height: 1.5; color: #333; padding: 40px; }
                .slip { max-width: 800px; margin: 0 auto; }
                .header { display: flex; justify-content: space-between; margin-bottom: 30px; padding-bottom: 20px; border-bottom: 2px solid #333; }
                .logo img { max-height: 50px; }
                .title { font-size: 24px; font-weight: bold; }
                .addresses { display: flex; gap: 40px; margin-bottom: 30px; }
                .address-block { flex: 1; }
                .address-block h3 { font-size: 12px; text-transform: uppercase; color: #666; margin-bottom: 10px; border-bottom: 1px solid #ddd; padding-bottom: 5px; }
                .order-info { background: #f5f5f7; padding: 15px; margin-bottom: 30px; display: flex; gap: 40px; }
                .order-info div { }
                .order-info label { font-size: 11px; text-transform: uppercase; color: #666; }
                .order-info strong { display: block; font-size: 14px; }
                table { width: 100%; border-collapse: collapse; }
                th { text-align: left; padding: 12px; background: #f5f5f7; border: 1px solid #ddd; font-size: 12px; text-transform: uppercase; }
                td { padding: 12px; border: 1px solid #ddd; }
                .text-center { text-align: center; }
                .checkbox { width: 20px; height: 20px; border: 2px solid #333; display: inline-block; }
                .notes { margin-top: 30px; padding: 15px; background: #fffbe6; border: 1px solid #ffe58f; }
                .footer { margin-top: 40px; text-align: center; font-size: 12px; color: #666; }
                @media print { body { padding: 20px; } .no-print { display: none; } }
            </style>
        </head>
        <body>
            <div class="slip">
                <div class="header">
                    <div class="logo">
                        <?php if ($company['logo']): ?>
                            <img src="<?php echo esc_url($company['logo']); ?>" alt="">
                        <?php else: ?>
                            <strong><?php echo esc_html($company['name']); ?></strong>
                        <?php endif; ?>
                    </div>
                    <div class="title">PACKING SLIP</div>
                </div>

                <div class="order-info">
                    <div>
                        <label>Order Number</label>
                        <strong><?php echo esc_html($order->order_number); ?></strong>
                    </div>
                    <div>
                        <label>Order Date</label>
                        <strong><?php echo date('M j, Y', strtotime($order->created_at)); ?></strong>
                    </div>
                    <div>
                        <label>Shipping Method</label>
                        <strong><?php echo esc_html($order->shipping_method ?: 'Standard'); ?></strong>
                    </div>
                </div>

                <div class="addresses">
                    <div class="address-block">
                        <h3>Ship From</h3>
                        <p><?php echo esc_html($company['name']); ?></p>
                        <?php if ($company['address']): ?><p><?php echo esc_html($company['address']); ?></p><?php endif; ?>
                        <p><?php echo esc_html($company['city'] . ', ' . $company['state'] . ' ' . $company['zip']); ?></p>
                    </div>
                    <div class="address-block">
                        <h3>Ship To</h3>
                        <p><strong><?php echo esc_html(($shipping['first_name'] ?? '') . ' ' . ($shipping['last_name'] ?? '')); ?></strong></p>
                        <?php if (!empty($shipping['company'])): ?><p><?php echo esc_html($shipping['company']); ?></p><?php endif; ?>
                        <p><?php echo esc_html($shipping['address_1'] ?? ''); ?></p>
                        <?php if (!empty($shipping['address_2'])): ?><p><?php echo esc_html($shipping['address_2']); ?></p><?php endif; ?>
                        <p><?php echo esc_html(($shipping['city'] ?? '') . ', ' . ($shipping['state'] ?? '') . ' ' . ($shipping['postcode'] ?? '')); ?></p>
                        <p><?php echo esc_html($shipping['country'] ?? ''); ?></p>
                        <?php if (!empty($shipping['phone'])): ?><p>Phone: <?php echo esc_html($shipping['phone']); ?></p><?php endif; ?>
                    </div>
                </div>

                <table>
                    <thead>
                        <tr>
                            <th style="width: 40px;">Packed</th>
                            <th>Item</th>
                            <th>SKU</th>
                            <th class="text-center" style="width: 80px;">Qty</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($items as $item): ?>
                        <tr>
                            <td class="text-center"><span class="checkbox"></span></td>
                            <td><?php echo esc_html($item->name); ?></td>
                            <td><?php echo esc_html($item->sku ?: '-'); ?></td>
                            <td class="text-center"><strong><?php echo esc_html($item->quantity); ?></strong></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <?php if ($order->customer_note): ?>
                <div class="notes">
                    <strong>Customer Notes:</strong><br>
                    <?php echo esc_html($order->customer_note); ?>
                </div>
                <?php endif; ?>

                <div class="footer">
                    <p>Packed by: _________________ Date: _________________</p>
                    <p class="no-print" style="margin-top: 20px;">
                        <button onclick="window.print()" style="padding: 10px 20px; cursor: pointer;">Print Packing Slip</button>
                    </p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Generate bulk invoices
     */
    public static function generateBulkInvoices(array $orderIds): string
    {
        $html = '<!DOCTYPE html><html><head><style>
            .page-break { page-break-after: always; }
        </style></head><body>';

        foreach ($orderIds as $i => $orderId) {
            $invoice = self::generateInvoice($orderId);
            if ($invoice) {
                // Extract just the invoice div content
                preg_match('/<div class="invoice">.*?<\/div>\s*<\/body>/s', $invoice, $matches);
                if ($matches) {
                    $html .= '<div class="invoice">' . $matches[0];
                    if ($i < count($orderIds) - 1) {
                        $html .= '<div class="page-break"></div>';
                    }
                }
            }
        }

        $html .= '</body></html>';
        return $html;
    }
}
