<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Inventory Service
 * Manages stock levels, notifications, and backorders
 */
class InventoryService
{
    /**
     * Get inventory settings
     */
    public static function getSettings(): array
    {
        return [
            'manage_stock' => get_option('cartt_manage_stock', true),
            'low_stock_threshold' => intval(get_option('cartt_low_stock_threshold', 5)),
            'out_of_stock_visibility' => get_option('cartt_out_of_stock_visibility', 'show'),
            'backorders_allowed' => get_option('cartt_backorders_allowed', false),
            'notify_low_stock' => get_option('cartt_notify_low_stock', true),
            'notify_out_of_stock' => get_option('cartt_notify_out_of_stock', true),
            'stock_display' => get_option('cartt_stock_display', 'always'), // always, low, never
        ];
    }

    /**
     * Check stock for product
     */
    public static function checkStock(int $productId, int $quantity = 1, ?int $variationId = null): array
    {
        global $wpdb;

        if ($variationId) {
            $item = $wpdb->get_row($wpdb->prepare(
                "SELECT stock_quantity, stock_status FROM {$wpdb->prefix}cartt_product_variations WHERE id = %d",
                $variationId
            ));
        } else {
            $item = $wpdb->get_row($wpdb->prepare(
                "SELECT stock_quantity, stock_status, manage_stock FROM {$wpdb->prefix}cartt_products WHERE id = %d",
                $productId
            ));
        }

        if (!$item) {
            return ['available' => false, 'message' => 'Product not found'];
        }

        // If not managing stock, just check status
        if (!$variationId && empty($item->manage_stock)) {
            return [
                'available' => $item->stock_status === 'instock',
                'message' => $item->stock_status === 'instock' ? 'In stock' : 'Out of stock',
            ];
        }

        $stockQty = intval($item->stock_quantity);
        $settings = self::getSettings();

        if ($stockQty <= 0) {
            if ($settings['backorders_allowed']) {
                return [
                    'available' => true,
                    'backorder' => true,
                    'quantity' => 0,
                    'message' => 'Available on backorder',
                ];
            }
            return ['available' => false, 'quantity' => 0, 'message' => 'Out of stock'];
        }

        if ($stockQty < $quantity) {
            return [
                'available' => false,
                'quantity' => $stockQty,
                'message' => "Only {$stockQty} available",
            ];
        }

        $lowStock = $stockQty <= $settings['low_stock_threshold'];

        return [
            'available' => true,
            'quantity' => $stockQty,
            'low_stock' => $lowStock,
            'message' => $lowStock ? "Only {$stockQty} left!" : 'In stock',
        ];
    }

    /**
     * Reduce stock for order
     */
    public static function reduceStock(int $orderId): void
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_order_items';

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT product_id, variation_id, quantity FROM $items_table WHERE order_id = %d",
            $orderId
        ));

        foreach ($items as $item) {
            self::adjustStock($item->product_id, -$item->quantity, $item->variation_id);
        }
    }

    /**
     * Restore stock for cancelled/refunded order
     */
    public static function restoreStock(int $orderId): void
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_order_items';

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT product_id, variation_id, quantity FROM $items_table WHERE order_id = %d",
            $orderId
        ));

        foreach ($items as $item) {
            self::adjustStock($item->product_id, $item->quantity, $item->variation_id);
        }
    }

    /**
     * Adjust stock level
     */
    public static function adjustStock(int $productId, int $quantity, ?int $variationId = null): bool
    {
        global $wpdb;

        if ($variationId) {
            $table = $wpdb->prefix . 'cartt_product_variations';
            $result = $wpdb->query($wpdb->prepare(
                "UPDATE $table SET stock_quantity = stock_quantity + %d WHERE id = %d",
                $quantity, $variationId
            ));

            // Update stock status
            $newQty = $wpdb->get_var($wpdb->prepare("SELECT stock_quantity FROM $table WHERE id = %d", $variationId));
            $wpdb->update($table, [
                'stock_status' => $newQty > 0 ? 'instock' : 'outofstock'
            ], ['id' => $variationId]);
        } else {
            $table = $wpdb->prefix . 'cartt_products';
            $result = $wpdb->query($wpdb->prepare(
                "UPDATE $table SET stock_quantity = stock_quantity + %d WHERE id = %d AND manage_stock = 1",
                $quantity, $productId
            ));

            // Update stock status
            $product = $wpdb->get_row($wpdb->prepare("SELECT stock_quantity, manage_stock FROM $table WHERE id = %d", $productId));
            if ($product && $product->manage_stock) {
                $wpdb->update($table, [
                    'stock_status' => $product->stock_quantity > 0 ? 'instock' : 'outofstock'
                ], ['id' => $productId]);
            }
        }

        // Check for notifications
        self::checkStockNotifications($productId, $variationId);

        return $result !== false;
    }

    /**
     * Set stock level
     */
    public static function setStock(int $productId, int $quantity, ?int $variationId = null): bool
    {
        global $wpdb;

        if ($variationId) {
            $table = $wpdb->prefix . 'cartt_product_variations';
            $result = $wpdb->update($table, [
                'stock_quantity' => $quantity,
                'stock_status' => $quantity > 0 ? 'instock' : 'outofstock',
            ], ['id' => $variationId]);
        } else {
            $table = $wpdb->prefix . 'cartt_products';
            $result = $wpdb->update($table, [
                'stock_quantity' => $quantity,
                'stock_status' => $quantity > 0 ? 'instock' : 'outofstock',
            ], ['id' => $productId]);
        }

        self::checkStockNotifications($productId, $variationId);

        return $result !== false;
    }

    /**
     * Check and send stock notifications
     */
    private static function checkStockNotifications(int $productId, ?int $variationId = null): void
    {
        global $wpdb;
        $settings = self::getSettings();

        // Get current stock
        if ($variationId) {
            $stock = $wpdb->get_var($wpdb->prepare(
                "SELECT stock_quantity FROM {$wpdb->prefix}cartt_product_variations WHERE id = %d",
                $variationId
            ));
        } else {
            $stock = $wpdb->get_var($wpdb->prepare(
                "SELECT stock_quantity FROM {$wpdb->prefix}cartt_products WHERE id = %d AND manage_stock = 1",
                $productId
            ));
        }

        if ($stock === null) return;

        $stock = intval($stock);

        // Low stock notification
        if ($settings['notify_low_stock'] && $stock > 0 && $stock <= $settings['low_stock_threshold']) {
            self::sendLowStockNotification($productId, $variationId, $stock);
        }

        // Out of stock notification
        if ($settings['notify_out_of_stock'] && $stock <= 0) {
            self::sendOutOfStockNotification($productId, $variationId);
        }

        // Notify back-in-stock subscribers
        if ($stock > 0) {
            self::notifyBackInStock($productId, $variationId);
        }
    }

    /**
     * Send low stock email to admin
     */
    private static function sendLowStockNotification(int $productId, ?int $variationId, int $stock): void
    {
        global $wpdb;
        
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT name, sku FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $productId
        ));

        if (!$product) return;

        $subject = 'Low Stock Alert: ' . $product->name;
        $message = sprintf(
            "The following product is running low on stock:\n\nProduct: %s\nSKU: %s\nCurrent Stock: %d\n\nView in admin: %s",
            $product->name,
            $product->sku ?: 'N/A',
            $stock,
            admin_url('admin.php?page=cartt-products&action=edit&id=' . $productId)
        );

        wp_mail(get_option('admin_email'), $subject, $message);
    }

    /**
     * Send out of stock email to admin
     */
    private static function sendOutOfStockNotification(int $productId, ?int $variationId): void
    {
        global $wpdb;
        
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT name, sku FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $productId
        ));

        if (!$product) return;

        $subject = 'Out of Stock: ' . $product->name;
        $message = sprintf(
            "The following product is now out of stock:\n\nProduct: %s\nSKU: %s\n\nView in admin: %s",
            $product->name,
            $product->sku ?: 'N/A',
            admin_url('admin.php?page=cartt-products&action=edit&id=' . $productId)
        );

        wp_mail(get_option('admin_email'), $subject, $message);
    }

    /**
     * Register for back in stock notification
     */
    public static function registerStockNotification(int $productId, string $email, ?int $variationId = null, ?int $userId = null): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_stock_notifications';

        // Check if already registered
        $exists = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE product_id = %d AND email = %s AND (variation_id = %d OR (variation_id IS NULL AND %d IS NULL)) AND notified = 0",
            $productId, $email, $variationId, $variationId
        ));

        if ($exists) {
            return true; // Already registered
        }

        return $wpdb->insert($table, [
            'product_id' => $productId,
            'variation_id' => $variationId,
            'email' => sanitize_email($email),
            'user_id' => $userId,
        ]) !== false;
    }

    /**
     * Notify back in stock subscribers
     */
    private static function notifyBackInStock(int $productId, ?int $variationId = null): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_stock_notifications';

        $sql = "SELECT * FROM $table WHERE product_id = %d AND notified = 0";
        $params = [$productId];

        if ($variationId) {
            $sql .= " AND variation_id = %d";
            $params[] = $variationId;
        } else {
            $sql .= " AND variation_id IS NULL";
        }

        $subscribers = $wpdb->get_results($wpdb->prepare($sql, ...$params));

        if (empty($subscribers)) return;

        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT name, slug, price, sale_price, featured_image FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $productId
        ));

        if (!$product) return;

        $productUrl = home_url('/product/' . $product->slug);
        $productPrice = CurrencyService::formatPrice($product->sale_price ?: $product->price);

        foreach ($subscribers as $sub) {
            EmailService::send('back_in_stock', $sub->email, [
                'product_name' => $product->name,
                'product_url' => $productUrl,
                'product_image' => $product->featured_image,
                'product_price' => $productPrice,
            ]);
            
            $wpdb->update($table, [
                'notified' => 1,
                'notified_at' => current_time('mysql'),
            ], ['id' => $sub->id]);
        }
    }

    /**
     * Get low stock products
     */
    public static function getLowStockProducts(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $threshold = intval(get_option('cartt_low_stock_threshold', 5));

        return $wpdb->get_results($wpdb->prepare(
            "SELECT id, name, sku, stock_quantity, stock_status
             FROM $table
             WHERE manage_stock = 1 AND stock_quantity > 0 AND stock_quantity <= %d
             ORDER BY stock_quantity ASC",
            $threshold
        ));
    }

    /**
     * Get out of stock products
     */
    public static function getOutOfStockProducts(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        return $wpdb->get_results(
            "SELECT id, name, sku, stock_quantity, stock_status
             FROM $table
             WHERE (manage_stock = 1 AND stock_quantity <= 0) OR stock_status = 'outofstock'
             ORDER BY name ASC"
        );
    }

    /**
     * Get stock report
     */
    public static function getStockReport(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $settings = self::getSettings();

        return [
            'total_products' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status = 'publish'"),
            'in_stock' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status = 'publish' AND stock_status = 'instock'"),
            'low_stock' => (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE status = 'publish' AND manage_stock = 1 AND stock_quantity > 0 AND stock_quantity <= %d",
                $settings['low_stock_threshold']
            )),
            'out_of_stock' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status = 'publish' AND (stock_status = 'outofstock' OR (manage_stock = 1 AND stock_quantity <= 0))"),
            'backorder' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE status = 'publish' AND stock_status = 'onbackorder'"),
        ];
    }

    /**
     * Get stock display text for frontend
     */
    public static function getStockDisplayText(int $productId, ?int $variationId = null): string
    {
        $stock = self::checkStock($productId, 1, $variationId);
        $settings = self::getSettings();

        if ($settings['stock_display'] === 'never') {
            return $stock['available'] ? 'In stock' : 'Out of stock';
        }

        if (!$stock['available']) {
            return $stock['message'];
        }

        if ($settings['stock_display'] === 'low' && empty($stock['low_stock'])) {
            return 'In stock';
        }

        return $stock['message'];
    }
}
