<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Gift Card Service
 * Manages digital gift cards with balance tracking
 */
class GiftCardService
{
    private static array $designs = [
        'default' => 'Classic',
        'birthday' => 'Birthday',
        'holiday' => 'Holiday',
        'thank_you' => 'Thank You',
        'congrats' => 'Congratulations',
        'love' => 'With Love',
    ];

    /**
     * Generate unique gift card code
     */
    public static function generateCode(): string
    {
        $prefix = 'GC';
        $code = $prefix . strtoupper(wp_generate_password(12, false));
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';
        
        // Ensure uniqueness
        while ($wpdb->get_var($wpdb->prepare("SELECT id FROM $table WHERE code = %s", $code))) {
            $code = $prefix . strtoupper(wp_generate_password(12, false));
        }
        
        return $code;
    }

    /**
     * Create a new gift card
     */
    public static function create(array $data): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $code = $data['code'] ?? self::generateCode();
        $amount = floatval($data['amount'] ?? 0);

        if ($amount <= 0) {
            return null;
        }

        $expiresAt = null;
        if (!empty($data['expires_days'])) {
            $expiresAt = date('Y-m-d H:i:s', strtotime('+' . intval($data['expires_days']) . ' days'));
        } elseif (!empty($data['expires_at'])) {
            $expiresAt = $data['expires_at'];
        }

        $deliverAt = null;
        if (!empty($data['deliver_at'])) {
            $deliverAt = $data['deliver_at'];
        }

        $wpdb->insert($table, [
            'code' => $code,
            'initial_balance' => $amount,
            'current_balance' => $amount,
            'currency' => sanitize_text_field($data['currency'] ?? 'USD'),
            'sender_name' => sanitize_text_field($data['sender_name'] ?? ''),
            'sender_email' => sanitize_email($data['sender_email'] ?? ''),
            'recipient_name' => sanitize_text_field($data['recipient_name'] ?? ''),
            'recipient_email' => sanitize_email($data['recipient_email'] ?? ''),
            'message' => sanitize_textarea_field($data['message'] ?? ''),
            'design' => sanitize_text_field($data['design'] ?? 'default'),
            'order_id' => !empty($data['order_id']) ? intval($data['order_id']) : null,
            'deliver_at' => $deliverAt,
            'expires_at' => $expiresAt,
            'is_delivered' => empty($data['recipient_email']) ? 1 : 0,
        ]);

        $id = $wpdb->insert_id;
        
        // Log initial credit
        self::logTransaction($id, $amount, 'credit', null, 'Gift card created');

        return self::getById($id);
    }

    /**
     * Get gift card by ID
     */
    public static function getById(int $id): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));
    }

    /**
     * Get gift card by code
     */
    public static function getByCode(string $code): ?object
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE code = %s", strtoupper($code)));
    }

    /**
     * Check if gift card is valid for use
     */
    public static function isValid(string $code): array
    {
        $card = self::getByCode($code);

        if (!$card) {
            return ['valid' => false, 'error' => 'Gift card not found'];
        }

        if ($card->status !== 'active') {
            return ['valid' => false, 'error' => 'Gift card is ' . $card->status];
        }

        if ($card->current_balance <= 0) {
            return ['valid' => false, 'error' => 'Gift card has no remaining balance'];
        }

        if ($card->expires_at && strtotime($card->expires_at) < time()) {
            self::updateStatus($card->id, 'expired');
            return ['valid' => false, 'error' => 'Gift card has expired'];
        }

        return ['valid' => true, 'card' => $card];
    }

    /**
     * Apply gift card to order
     */
    public static function apply(string $code, float $orderTotal, int $orderId): array
    {
        $validation = self::isValid($code);
        
        if (!$validation['valid']) {
            return $validation;
        }

        $card = $validation['card'];
        $amountToUse = min($card->current_balance, $orderTotal);
        $newBalance = $card->current_balance - $amountToUse;

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $wpdb->update($table, [
            'current_balance' => $newBalance,
            'status' => $newBalance <= 0 ? 'used' : 'active',
        ], ['id' => $card->id]);

        self::logTransaction($card->id, -$amountToUse, 'debit', $orderId, 'Applied to order #' . $orderId);

        return [
            'valid' => true,
            'amount_applied' => $amountToUse,
            'remaining_balance' => $newBalance,
            'card' => self::getById($card->id),
        ];
    }

    /**
     * Refund amount back to gift card
     */
    public static function refund(int $cardId, float $amount, int $orderId): bool
    {
        $card = self::getById($cardId);
        
        if (!$card) {
            return false;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $newBalance = $card->current_balance + $amount;

        $wpdb->update($table, [
            'current_balance' => $newBalance,
            'status' => 'active',
        ], ['id' => $cardId]);

        self::logTransaction($cardId, $amount, 'refund', $orderId, 'Refund from order #' . $orderId);

        return true;
    }

    /**
     * Use gift card balance (wrapper for apply)
     */
    public static function useBalance(int $cardId, float $amount, int $orderId): bool
    {
        $card = self::getById($cardId);
        
        if (!$card || $card->current_balance <= 0) {
            return false;
        }

        $amountToUse = min($card->current_balance, $amount);
        $newBalance = $card->current_balance - $amountToUse;

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $wpdb->update($table, [
            'current_balance' => $newBalance,
            'status' => $newBalance <= 0 ? 'used' : 'active',
        ], ['id' => $cardId]);

        self::logTransaction($cardId, -$amountToUse, 'debit', $orderId, 'Applied to order');

        return true;
    }

    /**
     * Check balance
     */
    public static function checkBalance(string $code): array
    {
        $card = self::getByCode($code);

        if (!$card) {
            return ['found' => false];
        }

        return [
            'found' => true,
            'code' => $card->code,
            'balance' => $card->current_balance,
            'currency' => $card->currency,
            'status' => $card->status,
            'expires_at' => $card->expires_at,
        ];
    }

    /**
     * Log transaction
     */
    private static function logTransaction(int $cardId, float $amount, string $type, ?int $orderId = null, ?string $note = null): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_card_transactions';

        $card = self::getById($cardId);

        $wpdb->insert($table, [
            'gift_card_id' => $cardId,
            'order_id' => $orderId,
            'amount' => $amount,
            'balance_after' => $card->current_balance,
            'type' => $type,
            'note' => $note,
        ]);
    }

    /**
     * Get transaction history
     */
    public static function getTransactions(int $cardId): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_card_transactions';

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE gift_card_id = %d ORDER BY created_at DESC",
            $cardId
        ));
    }

    /**
     * Update gift card status
     */
    public static function updateStatus(int $id, string $status): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        return $wpdb->update($table, ['status' => $status], ['id' => $id]) !== false;
    }

    /**
     * Get all gift cards
     */
    public static function getAll(array $args = []): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $where = '1=1';
        $params = [];

        if (!empty($args['status'])) {
            $where .= ' AND status = %s';
            $params[] = $args['status'];
        }

        if (!empty($args['search'])) {
            $where .= ' AND (code LIKE %s OR recipient_email LIKE %s)';
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $params[] = $search;
            $params[] = $search;
        }

        $limit = intval($args['limit'] ?? 50);
        $offset = intval($args['offset'] ?? 0);

        $sql = "SELECT * FROM $table WHERE $where ORDER BY created_at DESC LIMIT %d OFFSET %d";
        $params[] = $limit;
        $params[] = $offset;

        return $wpdb->get_results($wpdb->prepare($sql, ...$params));
    }

    /**
     * Get cards pending delivery
     */
    public static function getPendingDelivery(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        return $wpdb->get_results(
            "SELECT * FROM $table 
             WHERE is_delivered = 0 
             AND recipient_email != '' 
             AND (deliver_at IS NULL OR deliver_at <= NOW())
             AND status = 'active'"
        );
    }

    /**
     * Mark as delivered
     */
    public static function markDelivered(int $id): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        return $wpdb->update($table, ['is_delivered' => 1], ['id' => $id]) !== false;
    }

    /**
     * Send gift card email
     */
    public static function sendEmail(int $cardId): bool
    {
        $card = self::getById($cardId);
        
        if (!$card || !$card->recipient_email) {
            return false;
        }

        $subject = sprintf('You received a %s gift card!', CurrencyService::formatPrice($card->initial_balance));
        
        $message = self::getEmailTemplate($card);
        
        $sent = wp_mail($card->recipient_email, $subject, $message, ['Content-Type: text/html; charset=UTF-8']);

        if ($sent) {
            self::markDelivered($cardId);
        }

        return $sent;
    }

    /**
     * Get email template
     */
    private static function getEmailTemplate(object $card): string
    {
        $siteName = get_bloginfo('name');
        $siteUrl = home_url();
        
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; margin: 0; padding: 20px; background: #f5f5f7; }
                .card { max-width: 500px; margin: 0 auto; background: #fff; border-radius: 12px; overflow: hidden; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
                .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #fff; padding: 40px; text-align: center; }
                .amount { font-size: 48px; font-weight: bold; margin: 20px 0; }
                .body { padding: 30px; }
                .code { background: #f5f5f7; padding: 15px; border-radius: 8px; text-align: center; font-size: 24px; font-family: monospace; letter-spacing: 2px; margin: 20px 0; }
                .message { background: #fafafa; padding: 20px; border-radius: 8px; margin: 20px 0; font-style: italic; }
                .footer { padding: 20px 30px; background: #fafafa; text-align: center; font-size: 14px; color: #666; }
            </style>
        </head>
        <body>
            <div class="card">
                <div class="header">
                    <h1>🎁 You've Received a Gift Card!</h1>
                    <div class="amount"><?php echo esc_html(CurrencyService::formatPrice($card->initial_balance)); ?></div>
                </div>
                <div class="body">
                    <?php if ($card->sender_name): ?>
                    <p><strong><?php echo esc_html($card->sender_name); ?></strong> sent you a gift card<?php echo $card->recipient_name ? ' for ' . esc_html($card->recipient_name) : ''; ?>.</p>
                    <?php endif; ?>
                    
                    <?php if ($card->message): ?>
                    <div class="message"><?php echo esc_html($card->message); ?></div>
                    <?php endif; ?>
                    
                    <p>Your gift card code:</p>
                    <div class="code"><?php echo esc_html($card->code); ?></div>
                    
                    <p>Use this code at checkout to redeem your gift card.</p>
                    
                    <?php if ($card->expires_at): ?>
                    <p><small>Expires: <?php echo date('F j, Y', strtotime($card->expires_at)); ?></small></p>
                    <?php endif; ?>
                </div>
                <div class="footer">
                    <p>From <?php echo esc_html($siteName); ?></p>
                    <p><a href="<?php echo esc_url($siteUrl); ?>"><?php echo esc_html($siteUrl); ?></a></p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Get available designs
     */
    public static function getDesigns(): array
    {
        return self::$designs;
    }

    /**
     * Process scheduled deliveries (cron job)
     */
    public static function processScheduledDeliveries(): int
    {
        $pending = self::getPendingDelivery();
        $sent = 0;

        foreach ($pending as $card) {
            if (self::sendEmail($card->id)) {
                $sent++;
            }
        }

        return $sent;
    }
}
