<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Email Service
 * Manages email templates and sending
 */
class EmailService
{
    private const OPTION_KEY = 'cartt_email_templates';

    public static function getTemplates(): array
    {
        $saved = get_option(self::OPTION_KEY, []);
        return array_merge(self::getDefaultTemplates(), $saved);
    }

    public static function getTemplate(string $type): array
    {
        $templates = self::getTemplates();
        return $templates[$type] ?? [];
    }

    public static function saveTemplate(string $type, array $data): bool
    {
        $templates = get_option(self::OPTION_KEY, []);
        $templates[$type] = [
            'subject' => sanitize_text_field($data['subject'] ?? ''),
            'heading' => sanitize_text_field($data['heading'] ?? ''),
            'body' => wp_kses_post($data['body'] ?? ''),
            'enabled' => !empty($data['enabled']),
        ];
        return update_option(self::OPTION_KEY, $templates);
    }

    public static function getDefaultTemplates(): array
    {
        $storeName = get_bloginfo('name');
        
        return [
            'order_confirmation' => [
                'subject' => __('Your order #{order_number} has been received', 'cartt'),
                'heading' => __('Thank you for your order!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nWe've received your order and it's being processed.\n\n{order_details}\n\nThank you for shopping with us!", 'cartt'),
                'enabled' => true,
            ],
            'order_processing' => [
                'subject' => __('Your order #{order_number} is being processed', 'cartt'),
                'heading' => __('Order Processing', 'cartt'),
                'body' => __("Hi {customer_name},\n\nGood news! Your order is now being prepared.\n\n{order_details}\n\nWe'll notify you when it ships.", 'cartt'),
                'enabled' => true,
            ],
            'order_shipped' => [
                'subject' => __('Your order #{order_number} has shipped!', 'cartt'),
                'heading' => __('Your order is on its way!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nGreat news! Your order has been shipped.\n\n{tracking_info}\n\n{order_details}\n\nThank you for your purchase!", 'cartt'),
                'enabled' => true,
            ],
            'order_completed' => [
                'subject' => __('Your order #{order_number} is complete', 'cartt'),
                'heading' => __('Order Complete', 'cartt'),
                'body' => __("Hi {customer_name},\n\nYour order has been completed.\n\n{order_details}\n\nWe hope you enjoy your purchase! If you have any questions, please don't hesitate to contact us.", 'cartt'),
                'enabled' => true,
            ],
            'order_cancelled' => [
                'subject' => __('Your order #{order_number} has been cancelled', 'cartt'),
                'heading' => __('Order Cancelled', 'cartt'),
                'body' => __("Hi {customer_name},\n\nYour order has been cancelled.\n\n{order_details}\n\nIf you have any questions about this cancellation, please contact us.", 'cartt'),
                'enabled' => true,
            ],
            'order_refunded' => [
                'subject' => __('Refund for order #{order_number}', 'cartt'),
                'heading' => __('Refund Processed', 'cartt'),
                'body' => __("Hi {customer_name},\n\nA refund has been processed for your order.\n\nRefund amount: {refund_amount}\n\n{order_details}\n\nThe refund should appear in your account within 5-10 business days.", 'cartt'),
                'enabled' => true,
            ],
            'customer_welcome' => [
                'subject' => __('Welcome to {store_name}!', 'cartt'),
                'heading' => __('Welcome!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nThank you for creating an account with us.\n\nYou can now:\n- Track your orders\n- Save your addresses\n- Checkout faster\n\nHappy shopping!", 'cartt'),
                'enabled' => true,
            ],
            'customer_reset_password' => [
                'subject' => __('Password reset for {store_name}', 'cartt'),
                'heading' => __('Reset Your Password', 'cartt'),
                'body' => __("Hi {customer_name},\n\nA password reset was requested for your account.\n\nClick here to reset your password:\n{reset_link}\n\nIf you didn't request this, you can safely ignore this email.", 'cartt'),
                'enabled' => true,
            ],
            'low_stock' => [
                'subject' => __('Low stock alert: {product_name}', 'cartt'),
                'heading' => __('Low Stock Alert', 'cartt'),
                'body' => __("{product_name} is running low on stock.\n\nCurrent stock: {stock_quantity}\n\nPlease restock soon.", 'cartt'),
                'enabled' => true,
            ],
            'new_order_admin' => [
                'subject' => __('New order #{order_number}', 'cartt'),
                'heading' => __('New Order Received', 'cartt'),
                'body' => __("You have received a new order!\n\n{order_details}\n\nCustomer: {customer_name}\nEmail: {customer_email}", 'cartt'),
                'enabled' => true,
            ],
            // v1.3.0 Templates
            'gift_card_received' => [
                'subject' => __('You received a gift card from {sender_name}!', 'cartt'),
                'heading' => __('You\'ve Got a Gift!', 'cartt'),
                'body' => __("Hi {recipient_name},\n\n{sender_name} sent you a {amount} gift card!\n\n{personal_message}\n\nYour gift card code: {gift_card_code}\n\nRedeem it at: {store_url}\n\nThis gift card expires: {expires_at}", 'cartt'),
                'enabled' => true,
            ],
            'loyalty_welcome' => [
                'subject' => __('Welcome to our rewards program!', 'cartt'),
                'heading' => __('You\'re In!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nWelcome to our loyalty rewards program!\n\nYou've earned {points} points just for signing up.\n\nEarn more points:\n- 1 point per dollar spent\n- 50 points for reviews\n- 100 points for referrals\n\nStart shopping and earning!", 'cartt'),
                'enabled' => true,
            ],
            'loyalty_reminder' => [
                'subject' => __('You have {points} points waiting!', 'cartt'),
                'heading' => __('Don\'t Forget Your Points!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nYou have {points} loyalty points worth {points_value}!\n\nYour tier: {tier}\n\nUse them on your next order before they expire.\n\nShop now: {store_url}", 'cartt'),
                'enabled' => true,
            ],
            'loyalty_expired' => [
                'subject' => __('Your loyalty points have expired', 'cartt'),
                'heading' => __('Points Expired', 'cartt'),
                'body' => __("Hi {customer_name},\n\nUnfortunately, {points_expired} loyalty points have expired due to inactivity.\n\nKeep your points active by making a purchase!\n\nShop now: {store_url}", 'cartt'),
                'enabled' => true,
            ],
            'loyalty_tier_upgrade' => [
                'subject' => __('Congratulations! You\'ve reached {new_tier} status!', 'cartt'),
                'heading' => __('You Leveled Up!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nAmazing news! You've been upgraded to {new_tier} tier!\n\nYour new benefits:\n{tier_benefits}\n\nThank you for being a loyal customer!", 'cartt'),
                'enabled' => true,
            ],
            'rma_approved' => [
                'subject' => __('Your return request #{rma_number} has been approved', 'cartt'),
                'heading' => __('Return Approved', 'cartt'),
                'body' => __("Hi {customer_name},\n\nYour return request has been approved.\n\nRMA Number: {rma_number}\nOrder: #{order_number}\n\nPlease ship items to:\n{return_address}\n\nInclude your RMA number on the package.\n\nOnce we receive your items, we'll process your {refund_type}.", 'cartt'),
                'enabled' => true,
            ],
            'rma_received' => [
                'subject' => __('We received your return #{rma_number}', 'cartt'),
                'heading' => __('Return Received', 'cartt'),
                'body' => __("Hi {customer_name},\n\nWe've received your returned items for RMA #{rma_number}.\n\nWe're inspecting your items and will process your {refund_type} within 3-5 business days.\n\nThank you for your patience!", 'cartt'),
                'enabled' => true,
            ],
            'rma_completed' => [
                'subject' => __('Your return #{rma_number} is complete', 'cartt'),
                'heading' => __('Return Complete', 'cartt'),
                'body' => __("Hi {customer_name},\n\nYour return has been completed!\n\nRMA Number: {rma_number}\nRefund Amount: {refund_amount}\n\nThe refund has been processed to your original payment method.", 'cartt'),
                'enabled' => true,
            ],
            'question_answered' => [
                'subject' => __('Your question about {product_name} was answered', 'cartt'),
                'heading' => __('Your Question Was Answered!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nSomeone answered your question about {product_name}.\n\nYour question:\n\"{question}\"\n\nAnswer:\n\"{answer}\"\n\nView the full Q&A: {product_url}", 'cartt'),
                'enabled' => true,
            ],
            'order_delivered' => [
                'subject' => __('Your order #{order_number} has been delivered!', 'cartt'),
                'heading' => __('Package Delivered!', 'cartt'),
                'body' => __("Hi {customer_name},\n\nGreat news! Your order has been delivered.\n\nOrder: #{order_number}\nCarrier: {carrier}\nTracking: {tracking_number}\n\nWe hope you love your purchase! If you have any issues, please let us know.", 'cartt'),
                'enabled' => true,
            ],
            'back_in_stock' => [
                'subject' => __('{product_name} is back in stock!', 'cartt'),
                'heading' => __('It\'s Back!', 'cartt'),
                'body' => __("Hi there,\n\nGood news! {product_name} is back in stock.\n\nDon't miss out - grab it before it sells out again!\n\nPrice: {product_price}\n\nShop now: {product_url}", 'cartt'),
                'enabled' => true,
            ],
            'wishlist_sale' => [
                'subject' => __('An item on your wishlist is on sale!', 'cartt'),
                'heading' => __('Wishlist Alert!', 'cartt'),
                'body' => __("Hi {customer_name},\n\n{product_name} from your wishlist is now on sale!\n\nWas: {original_price}\nNow: {sale_price}\n\nShop now: {product_url}", 'cartt'),
                'enabled' => true,
            ],
        ];
    }

    public static function getPlaceholders(): array
    {
        return [
            '{order_number}' => __('Order number', 'cartt'),
            '{order_date}' => __('Order date', 'cartt'),
            '{order_total}' => __('Order total', 'cartt'),
            '{order_details}' => __('Order items table', 'cartt'),
            '{customer_name}' => __('Customer full name', 'cartt'),
            '{customer_first_name}' => __('Customer first name', 'cartt'),
            '{customer_email}' => __('Customer email', 'cartt'),
            '{billing_address}' => __('Billing address', 'cartt'),
            '{shipping_address}' => __('Shipping address', 'cartt'),
            '{tracking_number}' => __('Tracking number', 'cartt'),
            '{tracking_url}' => __('Tracking URL', 'cartt'),
            '{tracking_info}' => __('Full tracking info block', 'cartt'),
            '{store_name}' => __('Store name', 'cartt'),
            '{store_url}' => __('Store URL', 'cartt'),
            '{refund_amount}' => __('Refund amount', 'cartt'),
            '{product_name}' => __('Product name', 'cartt'),
            '{stock_quantity}' => __('Stock quantity', 'cartt'),
            '{reset_link}' => __('Password reset link', 'cartt'),
            // v1.3.0 placeholders
            '{gift_card_code}' => __('Gift card code', 'cartt'),
            '{sender_name}' => __('Gift card sender name', 'cartt'),
            '{recipient_name}' => __('Gift card recipient name', 'cartt'),
            '{amount}' => __('Gift card amount', 'cartt'),
            '{personal_message}' => __('Gift card personal message', 'cartt'),
            '{expires_at}' => __('Expiration date', 'cartt'),
            '{points}' => __('Loyalty points', 'cartt'),
            '{points_value}' => __('Loyalty points value', 'cartt'),
            '{points_expired}' => __('Expired points count', 'cartt'),
            '{tier}' => __('Loyalty tier', 'cartt'),
            '{new_tier}' => __('New loyalty tier', 'cartt'),
            '{tier_benefits}' => __('Tier benefits list', 'cartt'),
            '{rma_number}' => __('RMA number', 'cartt'),
            '{refund_type}' => __('Refund type', 'cartt'),
            '{return_address}' => __('Return shipping address', 'cartt'),
            '{question}' => __('Customer question', 'cartt'),
            '{answer}' => __('Q&A answer', 'cartt'),
            '{product_url}' => __('Product URL', 'cartt'),
            '{product_price}' => __('Product price', 'cartt'),
            '{original_price}' => __('Original price', 'cartt'),
            '{sale_price}' => __('Sale price', 'cartt'),
            '{carrier}' => __('Shipping carrier', 'cartt'),
        ];
    }

    public static function send(string $type, string $to, array $data = []): bool
    {
        $template = self::getTemplate($type);
        
        if (empty($template) || !$template['enabled']) {
            return false;
        }

        $subject = self::replacePlaceholders($template['subject'], $data);
        $heading = self::replacePlaceholders($template['heading'], $data);
        $body = self::replacePlaceholders($template['body'], $data);

        $html = self::buildEmailHtml($heading, $body, $data);

        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>',
        ];

        return wp_mail($to, $subject, $html, $headers);
    }

    public static function sendOrderEmail(string $type, int $orderId): bool
    {
        global $wpdb;
        
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));

        if (!$order) {
            return false;
        }

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_order_items WHERE order_id = %d",
            $orderId
        ));

        $billing = json_decode($order->billing_address, true) ?: [];
        $shipping = json_decode($order->shipping_address, true) ?: [];

        $data = [
            'order_number' => $order->order_number,
            'order_date' => date_i18n(get_option('date_format'), strtotime($order->created_at)),
            'order_total' => '$' . number_format($order->total, 2),
            'order_details' => self::buildOrderDetailsTable($items),
            'customer_name' => trim(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? '')),
            'customer_first_name' => $billing['first_name'] ?? '',
            'customer_email' => $order->customer_email,
            'billing_address' => self::formatAddress($billing),
            'shipping_address' => self::formatAddress($shipping),
            'store_name' => get_bloginfo('name'),
            'store_url' => home_url(),
            'tracking_number' => $order->tracking_number ?? '',
            'tracking_url' => $order->tracking_url ?? '',
            'tracking_info' => self::buildTrackingInfo($order),
        ];

        return self::send($type, $order->customer_email, $data);
    }

    private static function replacePlaceholders(string $text, array $data): string
    {
        foreach ($data as $key => $value) {
            $placeholder = '{' . $key . '}';
            $text = str_replace($placeholder, $value, $text);
        }
        return $text;
    }

    private static function buildEmailHtml(string $heading, string $body, array $data): string
    {
        $storeName = get_bloginfo('name');
        $storeUrl = home_url();
        $primaryColor = '#1d1d1f';
        
        $bodyHtml = nl2br(esc_html($body));
        // Allow certain HTML in body
        $bodyHtml = str_replace(
            ['&lt;strong&gt;', '&lt;/strong&gt;', '&lt;br&gt;', '&lt;br/&gt;'],
            ['<strong>', '</strong>', '<br>', '<br>'],
            $bodyHtml
        );
        // Restore order details table
        if (isset($data['order_details'])) {
            $bodyHtml = str_replace(esc_html($data['order_details']), $data['order_details'], $bodyHtml);
        }
        if (isset($data['tracking_info'])) {
            $bodyHtml = str_replace(esc_html($data['tracking_info']), $data['tracking_info'], $bodyHtml);
        }

        return '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
</head>
<body style="margin:0; padding:0; background:#f5f5f7; font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,sans-serif;">
    <table width="100%" cellpadding="0" cellspacing="0" style="background:#f5f5f7; padding:40px 20px;">
        <tr>
            <td align="center">
                <table width="600" cellpadding="0" cellspacing="0" style="background:#fff; border-radius:8px; overflow:hidden;">
                    <tr>
                        <td style="padding:32px 40px; background:' . esc_attr($primaryColor) . '; color:#fff;">
                            <h1 style="margin:0; font-size:24px; font-weight:600;">' . esc_html($heading) . '</h1>
                        </td>
                    </tr>
                    <tr>
                        <td style="padding:32px 40px; font-size:15px; line-height:1.6; color:#1d1d1f;">
                            ' . $bodyHtml . '
                        </td>
                    </tr>
                    <tr>
                        <td style="padding:24px 40px; background:#f5f5f7; text-align:center; font-size:13px; color:#86868b;">
                            <a href="' . esc_url($storeUrl) . '" style="color:#1d1d1f; text-decoration:none; font-weight:500;">' . esc_html($storeName) . '</a>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';
    }

    private static function buildOrderDetailsTable(array $items): string
    {
        $html = '<table width="100%" cellpadding="8" cellspacing="0" style="border:1px solid #e8e8ed; border-radius:6px; margin:16px 0;">';
        $html .= '<tr style="background:#f5f5f7;"><th align="left" style="border-bottom:1px solid #e8e8ed;">Item</th><th align="right" style="border-bottom:1px solid #e8e8ed;">Qty</th><th align="right" style="border-bottom:1px solid #e8e8ed;">Price</th></tr>';
        
        foreach ($items as $item) {
            $html .= '<tr>';
            $html .= '<td style="border-bottom:1px solid #e8e8ed;">' . esc_html($item->product_name) . '</td>';
            $html .= '<td align="right" style="border-bottom:1px solid #e8e8ed;">' . intval($item->quantity) . '</td>';
            $html .= '<td align="right" style="border-bottom:1px solid #e8e8ed;">$' . number_format($item->total, 2) . '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</table>';
        return $html;
    }

    private static function buildTrackingInfo(object $order): string
    {
        if (empty($order->tracking_number)) {
            return '';
        }

        $html = '<div style="background:#f5f5f7; padding:16px; border-radius:6px; margin:16px 0;">';
        $html .= '<strong>Tracking Number:</strong> ' . esc_html($order->tracking_number);
        
        if (!empty($order->tracking_url)) {
            $html .= '<br><a href="' . esc_url($order->tracking_url) . '" style="color:#0066cc;">Track your package</a>';
        }
        
        $html .= '</div>';
        return $html;
    }

    private static function formatAddress(array $address): string
    {
        $parts = array_filter([
            trim(($address['first_name'] ?? '') . ' ' . ($address['last_name'] ?? '')),
            $address['address_1'] ?? '',
            $address['address_2'] ?? '',
            trim(($address['city'] ?? '') . ', ' . ($address['state'] ?? '') . ' ' . ($address['postcode'] ?? '')),
            $address['country'] ?? '',
        ]);
        return implode('<br>', $parts);
    }
}
