<?php
/**
 * Dropshipping Service
 * 
 * Features:
 * - Supplier management
 * - AliExpress product import
 * - Printful/Printify integration
 * - Auto-order forwarding to suppliers
 * - Inventory sync
 * - Shipping tracking sync
 * - Profit margin calculator
 * - Supplier price monitoring
 * 
 * @package Cartt
 * @since 1.4.0
 */

declare(strict_types=1);

namespace Cartt\Services;

class DropshippingService
{
    private static ?self $instance = null;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        // Hooks
        add_action('cartt_order_completed', [$this, 'process_dropship_order'], 20, 1);
        add_action('cartt_cron_hourly', [$this, 'sync_inventory']);
        add_action('cartt_cron_hourly', [$this, 'check_tracking_updates']);

        // AJAX
        add_action('wp_ajax_cartt_import_aliexpress', [$this, 'ajax_import_aliexpress']);
        add_action('wp_ajax_cartt_sync_printful', [$this, 'ajax_sync_printful']);
        add_action('wp_ajax_cartt_add_supplier', [$this, 'ajax_add_supplier']);
        add_action('wp_ajax_cartt_sync_supplier_inventory', [$this, 'ajax_sync_inventory']);
    }

    /**
     * Add a supplier
     */
    public function add_supplier(array $data): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $required = ['name', 'type'];
        foreach ($required as $field) {
            if (empty($data[$field])) {
                return ['success' => false, 'error' => "Field '{$field}' is required"];
            }
        }

        $valid_types = ['aliexpress', 'printful', 'printify', 'cjdropshipping', 'custom'];
        if (!in_array($data['type'], $valid_types)) {
            return ['success' => false, 'error' => 'Invalid supplier type'];
        }

        $result = $wpdb->insert($table, [
            'name' => sanitize_text_field($data['name']),
            'type' => $data['type'],
            'api_key' => sanitize_text_field($data['api_key'] ?? ''),
            'api_secret' => sanitize_text_field($data['api_secret'] ?? ''),
            'store_id' => sanitize_text_field($data['store_id'] ?? ''),
            'webhook_url' => esc_url_raw($data['webhook_url'] ?? ''),
            'settings' => json_encode($data['settings'] ?? []),
            'auto_order' => !empty($data['auto_order']) ? 1 : 0,
            'auto_sync' => !empty($data['auto_sync']) ? 1 : 0,
            'status' => 'active',
            'created_at' => current_time('mysql')
        ]);

        if (!$result) {
            return ['success' => false, 'error' => 'Failed to add supplier'];
        }

        return [
            'success' => true,
            'supplier_id' => $wpdb->insert_id
        ];
    }

    /**
     * Get supplier by ID
     */
    public function get_supplier(int $supplier_id): ?array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $supplier = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table} WHERE id = %d",
            $supplier_id
        ), ARRAY_A);

        if ($supplier) {
            $supplier['settings'] = json_decode($supplier['settings'], true) ?: [];
        }

        return $supplier;
    }

    /**
     * Update supplier
     */
    public function update_supplier(int $supplier_id, array $data): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $allowed = ['name', 'api_key', 'api_secret', 'store_id', 'webhook_url', 'auto_order', 'auto_sync', 'status'];
        $update_data = [];

        foreach ($allowed as $field) {
            if (isset($data[$field])) {
                $update_data[$field] = $data[$field];
            }
        }

        if (isset($data['settings'])) {
            $update_data['settings'] = json_encode($data['settings']);
        }

        if (empty($update_data)) return false;

        $update_data['updated_at'] = current_time('mysql');

        return $wpdb->update($table, $update_data, ['id' => $supplier_id]) !== false;
    }

    /**
     * List all suppliers
     */
    public function list_suppliers(string $status = null): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $where = $status ? $wpdb->prepare("WHERE status = %s", $status) : "";

        $suppliers = $wpdb->get_results("SELECT * FROM {$table} {$where} ORDER BY name ASC", ARRAY_A);

        foreach ($suppliers as &$supplier) {
            $supplier['settings'] = json_decode($supplier['settings'], true) ?: [];
            $supplier['product_count'] = $this->get_supplier_product_count($supplier['id']);
        }

        return $suppliers;
    }

    /**
     * Get supplier product count
     */
    private function get_supplier_product_count(int $supplier_id): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_products';
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE supplier_id = %d",
            $supplier_id
        ));
    }

    /**
     * Import product from AliExpress URL
     */
    public function import_aliexpress(string $url, array $options = []): array
    {
        // Extract product ID from URL
        preg_match('/item\/(\d+)\.html/', $url, $matches);
        if (empty($matches[1])) {
            preg_match('/(\d{10,})/', $url, $matches);
        }

        if (empty($matches[1])) {
            return ['success' => false, 'error' => 'Could not extract product ID from URL'];
        }

        $product_id = $matches[1];

        // Note: AliExpress API requires affiliate/partner access
        // This is a placeholder for actual API integration
        $api_key = get_option('cartt_aliexpress_api_key');

        if (empty($api_key)) {
            return ['success' => false, 'error' => 'AliExpress API key not configured'];
        }

        // For demo purposes, return structure of what would be imported
        // In production, this would call the AliExpress API
        return [
            'success' => true,
            'message' => 'Product import queued',
            'product_id' => $product_id,
            'requires_api' => true,
            'instructions' => 'Configure AliExpress API credentials in Settings > Dropshipping'
        ];
    }

    /**
     * Link existing product to supplier
     */
    public function link_product(int $product_id, int $supplier_id, array $data): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_products';

        // Check if already linked
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$table} WHERE product_id = %d",
            $product_id
        ));

        if ($existing) {
            return ['success' => false, 'error' => 'Product already linked to a supplier'];
        }

        $result = $wpdb->insert($table, [
            'product_id' => $product_id,
            'supplier_id' => $supplier_id,
            'supplier_product_id' => sanitize_text_field($data['supplier_product_id'] ?? ''),
            'supplier_sku' => sanitize_text_field($data['supplier_sku'] ?? ''),
            'supplier_url' => esc_url_raw($data['supplier_url'] ?? ''),
            'cost_price' => (float) ($data['cost_price'] ?? 0),
            'shipping_cost' => (float) ($data['shipping_cost'] ?? 0),
            'processing_days' => (int) ($data['processing_days'] ?? 3),
            'variant_mapping' => json_encode($data['variant_mapping'] ?? []),
            'auto_order' => !empty($data['auto_order']) ? 1 : 0,
            'last_synced' => null,
            'created_at' => current_time('mysql')
        ]);

        if (!$result) {
            return ['success' => false, 'error' => 'Failed to link product'];
        }

        // Update product meta
        $this->update_product_profit_margin($product_id);

        return [
            'success' => true,
            'link_id' => $wpdb->insert_id
        ];
    }

    /**
     * Get dropship info for a product
     */
    public function get_product_dropship_info(int $product_id): ?array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_dropship_products';
        $suppliers_table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $info = $wpdb->get_row($wpdb->prepare("
            SELECT 
                dp.*,
                s.name as supplier_name,
                s.type as supplier_type,
                s.auto_order as supplier_auto_order
            FROM {$products_table} dp
            JOIN {$suppliers_table} s ON s.id = dp.supplier_id
            WHERE dp.product_id = %d
        ", $product_id), ARRAY_A);

        if ($info) {
            $info['variant_mapping'] = json_decode($info['variant_mapping'], true) ?: [];
            $info['profit_margin'] = $this->calculate_profit_margin($product_id);
        }

        return $info;
    }

    /**
     * Calculate profit margin for dropship product
     */
    public function calculate_profit_margin(int $product_id): array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        $dropship_table = $wpdb->prefix . 'cartt_dropship_products';

        $data = $wpdb->get_row($wpdb->prepare("
            SELECT 
                p.price as sell_price,
                dp.cost_price,
                dp.shipping_cost
            FROM {$products_table} p
            JOIN {$dropship_table} dp ON dp.product_id = p.id
            WHERE p.id = %d
        ", $product_id), ARRAY_A);

        if (!$data) {
            return ['profit' => 0, 'margin_percent' => 0];
        }

        $total_cost = (float) $data['cost_price'] + (float) $data['shipping_cost'];
        $sell_price = (float) $data['sell_price'];
        $profit = $sell_price - $total_cost;
        $margin_percent = $sell_price > 0 ? ($profit / $sell_price) * 100 : 0;

        return [
            'sell_price' => $sell_price,
            'cost_price' => (float) $data['cost_price'],
            'shipping_cost' => (float) $data['shipping_cost'],
            'total_cost' => $total_cost,
            'profit' => round($profit, 2),
            'margin_percent' => round($margin_percent, 1)
        ];
    }

    /**
     * Update product profit margin meta
     */
    private function update_product_profit_margin(int $product_id): void
    {
        $margin = $this->calculate_profit_margin($product_id);

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $wpdb->update($table, [
            'profit_margin' => $margin['margin_percent']
        ], ['id' => $product_id]);
    }

    /**
     * Process dropship order (auto-forward to supplier)
     */
    public function process_dropship_order(int $order_id): void
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $dropship_table = $wpdb->prefix . 'cartt_dropship_products';
        $suppliers_table = $wpdb->prefix . 'cartt_dropship_suppliers';

        // Get order details
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$orders_table} WHERE id = %d",
            $order_id
        ));

        if (!$order) return;

        // Get dropship items
        $dropship_items = $wpdb->get_results($wpdb->prepare("
            SELECT 
                oi.*,
                dp.supplier_id,
                dp.supplier_product_id,
                dp.supplier_sku,
                dp.variant_mapping,
                dp.auto_order as product_auto_order,
                s.name as supplier_name,
                s.type as supplier_type,
                s.api_key,
                s.api_secret,
                s.auto_order as supplier_auto_order
            FROM {$items_table} oi
            JOIN {$dropship_table} dp ON dp.product_id = oi.product_id
            JOIN {$suppliers_table} s ON s.id = dp.supplier_id
            WHERE oi.order_id = %d
            AND s.status = 'active'
        ", $order_id), ARRAY_A);

        if (empty($dropship_items)) return;

        // Group by supplier
        $by_supplier = [];
        foreach ($dropship_items as $item) {
            $by_supplier[$item['supplier_id']][] = $item;
        }

        // Process each supplier order
        foreach ($by_supplier as $supplier_id => $items) {
            $supplier = $items[0]; // Get supplier info from first item

            // Check if auto-order is enabled
            if (!$supplier['supplier_auto_order']) {
                // Just create a pending fulfillment record
                $this->create_fulfillment_record($order_id, $supplier_id, $items, 'pending');
                continue;
            }

            // Process based on supplier type
            switch ($supplier['supplier_type']) {
                case 'printful':
                    $this->submit_printful_order($order, $items);
                    break;
                case 'printify':
                    $this->submit_printify_order($order, $items);
                    break;
                case 'aliexpress':
                case 'cjdropshipping':
                    // These typically require manual ordering or API partner access
                    $this->create_fulfillment_record($order_id, $supplier_id, $items, 'requires_manual');
                    $this->notify_dropship_order($order_id, $supplier_id, $items);
                    break;
                default:
                    $this->create_fulfillment_record($order_id, $supplier_id, $items, 'pending');
                    $this->notify_dropship_order($order_id, $supplier_id, $items);
            }
        }
    }

    /**
     * Create fulfillment record
     */
    private function create_fulfillment_record(int $order_id, int $supplier_id, array $items, string $status): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_fulfillments';

        $wpdb->insert($table, [
            'order_id' => $order_id,
            'supplier_id' => $supplier_id,
            'items' => json_encode(array_map(fn($i) => [
                'order_item_id' => $i['id'],
                'product_id' => $i['product_id'],
                'supplier_product_id' => $i['supplier_product_id'],
                'quantity' => $i['quantity']
            ], $items)),
            'status' => $status,
            'created_at' => current_time('mysql')
        ]);

        return $wpdb->insert_id;
    }

    /**
     * Submit order to Printful
     */
    private function submit_printful_order(object $order, array $items): void
    {
        $api_key = $items[0]['api_key'];
        if (empty($api_key)) {
            $this->create_fulfillment_record($order->id, $items[0]['supplier_id'], $items, 'api_error');
            return;
        }

        $printful_items = [];
        foreach ($items as $item) {
            $printful_items[] = [
                'sync_variant_id' => $item['supplier_product_id'],
                'quantity' => $item['quantity'],
                'retail_price' => $item['subtotal'] / $item['quantity']
            ];
        }

        $order_data = [
            'recipient' => [
                'name' => $order->billing_first_name . ' ' . $order->billing_last_name,
                'address1' => $order->shipping_address_1 ?: $order->billing_address_1,
                'city' => $order->shipping_city ?: $order->billing_city,
                'state_code' => $order->shipping_state ?: $order->billing_state,
                'country_code' => $order->shipping_country ?: $order->billing_country,
                'zip' => $order->shipping_postcode ?: $order->billing_postcode,
                'email' => $order->billing_email,
                'phone' => $order->billing_phone
            ],
            'items' => $printful_items,
            'external_id' => (string) $order->id
        ];

        $response = wp_remote_post('https://api.printful.com/orders', [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($order_data),
            'timeout' => 30
        ]);

        $fulfillment_id = $this->create_fulfillment_record(
            $order->id,
            $items[0]['supplier_id'],
            $items,
            is_wp_error($response) ? 'api_error' : 'submitted'
        );

        if (!is_wp_error($response)) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($body['result']['id'])) {
                global $wpdb;
                $wpdb->update(
                    $wpdb->prefix . 'cartt_dropship_fulfillments',
                    [
                        'supplier_order_id' => $body['result']['id'],
                        'status' => 'submitted'
                    ],
                    ['id' => $fulfillment_id]
                );
            }
        }
    }

    /**
     * Submit order to Printify
     */
    private function submit_printify_order(object $order, array $items): void
    {
        $api_key = $items[0]['api_key'];
        $shop_id = $items[0]['store_id'] ?? '';

        if (empty($api_key) || empty($shop_id)) {
            $this->create_fulfillment_record($order->id, $items[0]['supplier_id'], $items, 'api_error');
            return;
        }

        $line_items = [];
        foreach ($items as $item) {
            $line_items[] = [
                'product_id' => $item['supplier_product_id'],
                'variant_id' => $item['variant_mapping']['printify_variant_id'] ?? 0,
                'quantity' => $item['quantity']
            ];
        }

        $order_data = [
            'external_id' => (string) $order->id,
            'line_items' => $line_items,
            'shipping_method' => 1,
            'address_to' => [
                'first_name' => $order->billing_first_name,
                'last_name' => $order->billing_last_name,
                'email' => $order->billing_email,
                'phone' => $order->billing_phone,
                'address1' => $order->shipping_address_1 ?: $order->billing_address_1,
                'city' => $order->shipping_city ?: $order->billing_city,
                'region' => $order->shipping_state ?: $order->billing_state,
                'country' => $order->shipping_country ?: $order->billing_country,
                'zip' => $order->shipping_postcode ?: $order->billing_postcode
            ]
        ];

        $response = wp_remote_post("https://api.printify.com/v1/shops/{$shop_id}/orders.json", [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($order_data),
            'timeout' => 30
        ]);

        $fulfillment_id = $this->create_fulfillment_record(
            $order->id,
            $items[0]['supplier_id'],
            $items,
            is_wp_error($response) ? 'api_error' : 'submitted'
        );

        if (!is_wp_error($response)) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            if (!empty($body['id'])) {
                global $wpdb;
                $wpdb->update(
                    $wpdb->prefix . 'cartt_dropship_fulfillments',
                    [
                        'supplier_order_id' => $body['id'],
                        'status' => 'submitted'
                    ],
                    ['id' => $fulfillment_id]
                );
            }
        }
    }

    /**
     * Sync inventory from suppliers
     */
    public function sync_inventory(): void
    {
        $suppliers = $this->list_suppliers('active');

        foreach ($suppliers as $supplier) {
            if (!$supplier['auto_sync']) continue;

            switch ($supplier['type']) {
                case 'printful':
                    $this->sync_printful_inventory($supplier);
                    break;
                case 'printify':
                    $this->sync_printify_inventory($supplier);
                    break;
            }
        }
    }

    /**
     * Sync Printful inventory
     */
    private function sync_printful_inventory(array $supplier): void
    {
        if (empty($supplier['api_key'])) return;

        $response = wp_remote_get('https://api.printful.com/sync/products', [
            'headers' => ['Authorization' => 'Bearer ' . $supplier['api_key']],
            'timeout' => 30
        ]);

        if (is_wp_error($response)) return;

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($body['result'])) return;

        global $wpdb;
        $dropship_table = $wpdb->prefix . 'cartt_dropship_products';
        $products_table = $wpdb->prefix . 'cartt_products';

        foreach ($body['result'] as $product) {
            // Find linked product
            $link = $wpdb->get_row($wpdb->prepare("
                SELECT * FROM {$dropship_table}
                WHERE supplier_id = %d AND supplier_product_id = %s
            ", $supplier['id'], $product['id']), ARRAY_A);

            if (!$link) continue;

            // Update stock if applicable
            if (isset($product['synced'])) {
                $wpdb->update(
                    $dropship_table,
                    ['last_synced' => current_time('mysql')],
                    ['id' => $link['id']]
                );
            }
        }
    }

    /**
     * Sync Printify inventory
     */
    private function sync_printify_inventory(array $supplier): void
    {
        if (empty($supplier['api_key']) || empty($supplier['store_id'])) return;

        $shop_id = $supplier['store_id'];

        $response = wp_remote_get("https://api.printify.com/v1/shops/{$shop_id}/products.json", [
            'headers' => ['Authorization' => 'Bearer ' . $supplier['api_key']],
            'timeout' => 30
        ]);

        if (is_wp_error($response)) return;

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($body['data'])) return;

        global $wpdb;
        $dropship_table = $wpdb->prefix . 'cartt_dropship_products';

        foreach ($body['data'] as $product) {
            $link = $wpdb->get_row($wpdb->prepare("
                SELECT * FROM {$dropship_table}
                WHERE supplier_id = %d AND supplier_product_id = %s
            ", $supplier['id'], $product['id']), ARRAY_A);

            if (!$link) continue;

            $wpdb->update(
                $dropship_table,
                ['last_synced' => current_time('mysql')],
                ['id' => $link['id']]
            );
        }
    }

    /**
     * Check for tracking updates
     */
    public function check_tracking_updates(): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_fulfillments';
        $suppliers_table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $pending = $wpdb->get_results("
            SELECT f.*, s.type, s.api_key, s.store_id
            FROM {$table} f
            JOIN {$suppliers_table} s ON s.id = f.supplier_id
            WHERE f.status IN ('submitted', 'in_production')
            AND f.tracking_number IS NULL
        ", ARRAY_A);

        foreach ($pending as $fulfillment) {
            switch ($fulfillment['type']) {
                case 'printful':
                    $this->check_printful_tracking($fulfillment);
                    break;
                case 'printify':
                    $this->check_printify_tracking($fulfillment);
                    break;
            }
        }
    }

    /**
     * Check Printful tracking
     */
    private function check_printful_tracking(array $fulfillment): void
    {
        if (empty($fulfillment['api_key']) || empty($fulfillment['supplier_order_id'])) return;

        $response = wp_remote_get(
            "https://api.printful.com/orders/{$fulfillment['supplier_order_id']}",
            ['headers' => ['Authorization' => 'Bearer ' . $fulfillment['api_key']]]
        );

        if (is_wp_error($response)) return;

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($body['result'])) return;

        $order = $body['result'];

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_fulfillments';

        $update = ['status' => $order['status']];

        if (!empty($order['shipments'])) {
            $shipment = $order['shipments'][0];
            $update['tracking_number'] = $shipment['tracking_number'] ?? null;
            $update['tracking_url'] = $shipment['tracking_url'] ?? null;
            $update['carrier'] = $shipment['carrier'] ?? null;

            // Update main order tracking
            if (!empty($shipment['tracking_number'])) {
                OrderTrackingService::instance()->add_tracking(
                    $fulfillment['order_id'],
                    $shipment['carrier'] ?? 'Printful',
                    $shipment['tracking_number'],
                    $shipment['tracking_url'] ?? null
                );
            }
        }

        $wpdb->update($table, $update, ['id' => $fulfillment['id']]);
    }

    /**
     * Check Printify tracking
     */
    private function check_printify_tracking(array $fulfillment): void
    {
        if (empty($fulfillment['api_key']) || empty($fulfillment['store_id']) || empty($fulfillment['supplier_order_id'])) {
            return;
        }

        $response = wp_remote_get(
            "https://api.printify.com/v1/shops/{$fulfillment['store_id']}/orders/{$fulfillment['supplier_order_id']}.json",
            ['headers' => ['Authorization' => 'Bearer ' . $fulfillment['api_key']]]
        );

        if (is_wp_error($response)) return;

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($body)) return;

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_fulfillments';

        $update = ['status' => $body['status'] ?? 'unknown'];

        if (!empty($body['shipments'])) {
            $shipment = $body['shipments'][0];
            $update['tracking_number'] = $shipment['tracking_number'] ?? null;
            $update['tracking_url'] = $shipment['url'] ?? null;
            $update['carrier'] = $shipment['carrier'] ?? null;

            if (!empty($shipment['tracking_number'])) {
                OrderTrackingService::instance()->add_tracking(
                    $fulfillment['order_id'],
                    $shipment['carrier'] ?? 'Printify',
                    $shipment['tracking_number'],
                    $shipment['url'] ?? null
                );
            }
        }

        $wpdb->update($table, $update, ['id' => $fulfillment['id']]);
    }

    /**
     * Notify admin of dropship order requiring manual action
     */
    private function notify_dropship_order(int $order_id, int $supplier_id, array $items): void
    {
        $supplier = $this->get_supplier($supplier_id);
        $admin_email = get_option('admin_email');

        $product_list = array_map(fn($i) => "- {$i['name']} x{$i['quantity']} (SKU: {$i['supplier_sku']})", $items);

        $subject = "Dropship Order #{$order_id} - Manual Fulfillment Required";
        $message = sprintf(
            "Order #%d requires manual fulfillment from supplier: %s\n\n" .
            "Products:\n%s\n\n" .
            "Please log in to your %s account to process this order.",
            $order_id,
            $supplier['name'],
            implode("\n", $product_list),
            ucfirst($supplier['type'])
        );

        wp_mail($admin_email, $subject, $message);
    }

    /**
     * Get fulfillments for an order
     */
    public function get_order_fulfillments(int $order_id): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_dropship_fulfillments';
        $suppliers_table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $fulfillments = $wpdb->get_results($wpdb->prepare("
            SELECT f.*, s.name as supplier_name, s.type as supplier_type
            FROM {$table} f
            JOIN {$suppliers_table} s ON s.id = f.supplier_id
            WHERE f.order_id = %d
            ORDER BY f.created_at DESC
        ", $order_id), ARRAY_A);

        foreach ($fulfillments as &$f) {
            $f['items'] = json_decode($f['items'], true) ?: [];
        }

        return $fulfillments;
    }

    /**
     * Get all dropship products with margins
     */
    public function get_dropship_products(array $args = []): array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        $dropship_table = $wpdb->prefix . 'cartt_dropship_products';
        $suppliers_table = $wpdb->prefix . 'cartt_dropship_suppliers';

        $defaults = ['limit' => 50, 'offset' => 0, 'supplier_id' => null];
        $args = wp_parse_args($args, $defaults);

        $where = "";
        if ($args['supplier_id']) {
            $where = $wpdb->prepare("AND dp.supplier_id = %d", $args['supplier_id']);
        }

        $products = $wpdb->get_results("
            SELECT 
                p.id, p.name, p.sku, p.price, p.stock_quantity, p.status,
                dp.cost_price, dp.shipping_cost, dp.supplier_product_id, dp.last_synced,
                s.name as supplier_name, s.type as supplier_type,
                (p.price - dp.cost_price - dp.shipping_cost) as profit,
                CASE WHEN p.price > 0 
                    THEN ((p.price - dp.cost_price - dp.shipping_cost) / p.price) * 100 
                    ELSE 0 
                END as margin_percent
            FROM {$dropship_table} dp
            JOIN {$products_table} p ON p.id = dp.product_id
            JOIN {$suppliers_table} s ON s.id = dp.supplier_id
            {$where}
            ORDER BY margin_percent DESC
            LIMIT {$args['limit']} OFFSET {$args['offset']}
        ", ARRAY_A);

        return $products;
    }

    /**
     * AJAX: Import from AliExpress
     */
    public function ajax_import_aliexpress(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $url = esc_url_raw($_POST['url'] ?? '');
        $result = $this->import_aliexpress($url);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * AJAX: Sync Printful products
     */
    public function ajax_sync_printful(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $supplier_id = intval($_POST['supplier_id'] ?? 0);
        $supplier = $this->get_supplier($supplier_id);

        if (!$supplier || $supplier['type'] !== 'printful') {
            wp_send_json_error('Invalid Printful supplier');
        }

        $this->sync_printful_inventory($supplier);
        wp_send_json_success(['message' => 'Sync completed']);
    }

    /**
     * AJAX: Add supplier
     */
    public function ajax_add_supplier(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $result = $this->add_supplier($_POST);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['error']);
        }
    }

    /**
     * AJAX: Sync inventory
     */
    public function ajax_sync_inventory(): void
    {
        check_ajax_referer('cartt_admin', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        $this->sync_inventory();
        wp_send_json_success(['message' => 'Inventory sync completed']);
    }
}
