<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Cron Service
 * Handles all scheduled tasks
 */
class CronService
{
    /**
     * Register cron hooks
     */
    public static function init(): void
    {
        // Schedule events on plugin activation
        add_action('cartt_hourly_cron', [self::class, 'runHourlyTasks']);
        add_action('cartt_daily_cron', [self::class, 'runDailyTasks']);
        add_action('cartt_weekly_cron', [self::class, 'runWeeklyTasks']);
    }

    /**
     * Schedule cron events (call on activation)
     */
    public static function schedule(): void
    {
        if (!wp_next_scheduled('cartt_hourly_cron')) {
            wp_schedule_event(time(), 'hourly', 'cartt_hourly_cron');
        }

        if (!wp_next_scheduled('cartt_daily_cron')) {
            wp_schedule_event(time(), 'daily', 'cartt_daily_cron');
        }

        if (!wp_next_scheduled('cartt_weekly_cron')) {
            wp_schedule_event(time(), 'weekly', 'cartt_weekly_cron');
        }
    }

    /**
     * Unschedule cron events (call on deactivation)
     */
    public static function unschedule(): void
    {
        wp_clear_scheduled_hook('cartt_hourly_cron');
        wp_clear_scheduled_hook('cartt_daily_cron');
        wp_clear_scheduled_hook('cartt_weekly_cron');
    }

    /**
     * Hourly tasks
     */
    public static function runHourlyTasks(): void
    {
        self::processScheduledGiftCards();
        self::updateOrderTracking();
        self::processAbandonedCartEmails();
    }

    /**
     * Daily tasks
     */
    public static function runDailyTasks(): void
    {
        self::expireLoyaltyPoints();
        self::expireGiftCards();
        self::sendLowStockAlerts();
        self::cleanupExpiredSessions();
        self::updateCurrencyRates();
    }

    /**
     * Weekly tasks
     */
    public static function runWeeklyTasks(): void
    {
        self::sendLoyaltyReminders();
        self::cleanupOldData();
    }

    /**
     * Send scheduled gift cards
     */
    private static function processScheduledGiftCards(): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $cards = $wpdb->get_results(
            "SELECT * FROM $table 
             WHERE status = 'pending' 
             AND deliver_at IS NOT NULL 
             AND deliver_at <= NOW() 
             AND recipient_email != ''"
        );

        foreach ($cards as $card) {
            if (GiftCardService::sendEmail($card->id)) {
                $wpdb->update($table, ['status' => 'active'], ['id' => $card->id]);
            }
        }
    }

    /**
     * Update order tracking from carriers
     */
    private static function updateOrderTracking(): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_order_tracking';

        // Get tracking that needs updates (not delivered, updated more than 2 hours ago)
        $tracking = $wpdb->get_results(
            "SELECT * FROM $table 
             WHERE status NOT IN ('delivered', 'failed') 
             AND updated_at < DATE_SUB(NOW(), INTERVAL 2 HOUR)
             LIMIT 50"
        );

        foreach ($tracking as $t) {
            $status = self::fetchTrackingStatus($t->carrier, $t->tracking_number);
            
            if ($status && $status !== $t->status) {
                OrderTrackingService::updateStatus($t->id, $status['status'], $status['detail'] ?? null);
                
                // Send notification if delivered
                if ($status['status'] === 'delivered') {
                    self::sendDeliveryNotification($t->order_id, $t);
                }
            }
        }
    }

    /**
     * Fetch tracking status from carrier API
     */
    private static function fetchTrackingStatus(string $carrier, string $trackingNumber): ?array
    {
        // This would integrate with actual carrier APIs
        // For now, return null (no update)
        // In production, you'd call USPS, UPS, FedEx APIs here
        
        switch ($carrier) {
            case 'usps':
                return self::fetchUSPSStatus($trackingNumber);
            case 'ups':
                return self::fetchUPSStatus($trackingNumber);
            case 'fedex':
                return self::fetchFedExStatus($trackingNumber);
            default:
                return null;
        }
    }

    /**
     * Fetch USPS tracking status
     */
    private static function fetchUSPSStatus(string $trackingNumber): ?array
    {
        $userId = get_option('cartt_usps_user_id', '');
        if (!$userId) {
            return null;
        }

        $xml = '<?xml version="1.0"?>
            <TrackRequest USERID="' . esc_attr($userId) . '">
                <TrackID ID="' . esc_attr($trackingNumber) . '"></TrackID>
            </TrackRequest>';

        $response = wp_remote_get(
            'https://secure.shippingapis.com/ShippingAPI.dll?API=TrackV2&XML=' . urlencode($xml),
            ['timeout' => 15]
        );

        if (is_wp_error($response)) {
            return null;
        }

        $body = wp_remote_retrieve_body($response);
        $xml = @simplexml_load_string($body);

        if (!$xml || !isset($xml->TrackInfo->TrackSummary)) {
            return null;
        }

        $summary = (string) $xml->TrackInfo->TrackSummary;
        
        if (stripos($summary, 'delivered') !== false) {
            return ['status' => 'delivered', 'detail' => $summary];
        } elseif (stripos($summary, 'out for delivery') !== false) {
            return ['status' => 'out_for_delivery', 'detail' => $summary];
        } elseif (stripos($summary, 'in transit') !== false || stripos($summary, 'processed') !== false) {
            return ['status' => 'in_transit', 'detail' => $summary];
        }

        return null;
    }

    /**
     * Fetch UPS tracking status (placeholder)
     */
    private static function fetchUPSStatus(string $trackingNumber): ?array
    {
        // Would implement UPS API call here
        return null;
    }

    /**
     * Fetch FedEx tracking status (placeholder)
     */
    private static function fetchFedExStatus(string $trackingNumber): ?array
    {
        // Would implement FedEx API call here
        return null;
    }

    /**
     * Send delivery notification
     */
    private static function sendDeliveryNotification(int $orderId, object $tracking): void
    {
        global $wpdb;
        
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT o.*, c.email, c.first_name 
             FROM {$wpdb->prefix}cartt_orders o
             JOIN {$wpdb->prefix}cartt_customers c ON o.customer_id = c.id
             WHERE o.id = %d",
            $orderId
        ));

        if (!$order || !$order->email) {
            return;
        }

        EmailService::send('order_delivered', $order->email, [
            'customer_name' => $order->first_name,
            'order_number' => $order->order_number,
            'carrier' => $tracking->carrier_name ?? $tracking->carrier,
            'tracking_number' => $tracking->tracking_number,
        ]);
    }

    /**
     * Process abandoned cart emails
     */
    private static function processAbandonedCartEmails(): void
    {
        $abandonedService = new AbandonedCartService();
        $abandonedService->processScheduledEmails();
    }

    /**
     * Expire loyalty points
     */
    private static function expireLoyaltyPoints(): void
    {
        $settings = LoyaltyService::getSettings();
        
        if (!$settings['enabled'] || empty($settings['points_expire_days'])) {
            return;
        }

        global $wpdb;
        $expireDays = intval($settings['points_expire_days']);
        
        // Find points that should expire
        $expiringPoints = $wpdb->get_results($wpdb->prepare(
            "SELECT lp.*, c.email, c.first_name 
             FROM {$wpdb->prefix}cartt_loyalty_points lp
             JOIN {$wpdb->prefix}cartt_customers c ON lp.customer_id = c.id
             WHERE lp.points > 0 
             AND lp.updated_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $expireDays
        ));

        foreach ($expiringPoints as $record) {
            $pointsToExpire = $record->points;
            
            // Zero out points
            $wpdb->update(
                $wpdb->prefix . 'cartt_loyalty_points',
                ['points' => 0],
                ['id' => $record->id]
            );

            // Log expiration
            $wpdb->insert($wpdb->prefix . 'cartt_loyalty_transactions', [
                'customer_id' => $record->customer_id,
                'points' => -$pointsToExpire,
                'type' => 'expire',
                'description' => 'Points expired after ' . $expireDays . ' days of inactivity',
            ]);

            // Notify customer
            if ($record->email) {
                EmailService::send('loyalty_expired', $record->email, [
                    'customer_name' => $record->first_name,
                    'points_expired' => $pointsToExpire,
                ]);
            }
        }
    }

    /**
     * Expire gift cards
     */
    private static function expireGiftCards(): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_gift_cards';

        $wpdb->query(
            "UPDATE $table 
             SET status = 'expired' 
             WHERE status = 'active' 
             AND expires_at IS NOT NULL 
             AND expires_at < NOW()"
        );
    }

    /**
     * Send low stock alerts
     */
    private static function sendLowStockAlerts(): void
    {
        $settings = InventoryService::getSettings();
        
        if (!$settings['notify_low_stock']) {
            return;
        }

        $lowStock = InventoryService::getLowStockProducts();
        
        if (empty($lowStock)) {
            return;
        }

        // Check if we already sent alert today
        $lastAlert = get_option('cartt_last_low_stock_alert', '');
        $today = date('Y-m-d');
        
        if ($lastAlert === $today) {
            return;
        }

        $adminEmail = get_option('admin_email');
        
        $productList = '';
        foreach ($lowStock as $product) {
            $productList .= "- {$product->name} (SKU: {$product->sku}) - {$product->stock_quantity} left\n";
        }

        wp_mail(
            $adminEmail,
            '[' . get_bloginfo('name') . '] Low Stock Alert',
            "The following products are running low on stock:\n\n" . $productList . "\n\nManage inventory: " . admin_url('admin.php?page=cartt-inventory'),
            ['Content-Type: text/plain; charset=UTF-8']
        );

        update_option('cartt_last_low_stock_alert', $today);
    }

    /**
     * Cleanup expired sessions
     */
    private static function cleanupExpiredSessions(): void
    {
        global $wpdb;

        // Clean old abandoned carts (30+ days)
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}cartt_abandoned_carts 
             WHERE status = 'abandoned' 
             AND created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)"
        );

        // Clean old wishlists without items (60+ days)
        $wpdb->query(
            "DELETE w FROM {$wpdb->prefix}cartt_wishlists w
             LEFT JOIN {$wpdb->prefix}cartt_wishlist_items i ON w.id = i.wishlist_id
             WHERE i.id IS NULL 
             AND w.updated_at < DATE_SUB(NOW(), INTERVAL 60 DAY)"
        );

        // Clean old stock notifications that were sent (90+ days)
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}cartt_stock_notifications 
             WHERE notified_at IS NOT NULL 
             AND notified_at < DATE_SUB(NOW(), INTERVAL 90 DAY)"
        );
    }

    /**
     * Update currency exchange rates
     */
    private static function updateCurrencyRates(): void
    {
        CurrencyService::updateExchangeRates();
    }

    /**
     * Send loyalty reminders for inactive customers
     */
    private static function sendLoyaltyReminders(): void
    {
        $settings = LoyaltyService::getSettings();
        
        if (!$settings['enabled']) {
            return;
        }

        global $wpdb;

        // Find customers with points who haven't ordered in 30+ days
        $customers = $wpdb->get_results(
            "SELECT lp.*, c.email, c.first_name, c.last_name,
                    (SELECT MAX(created_at) FROM {$wpdb->prefix}cartt_orders WHERE customer_id = c.id) as last_order
             FROM {$wpdb->prefix}cartt_loyalty_points lp
             JOIN {$wpdb->prefix}cartt_customers c ON lp.customer_id = c.id
             WHERE lp.points >= 100
             HAVING last_order < DATE_SUB(NOW(), INTERVAL 30 DAY)
             LIMIT 50"
        );

        foreach ($customers as $customer) {
            // Check if we've already sent reminder this month
            $sentKey = 'cartt_loyalty_reminder_' . $customer->customer_id . '_' . date('Y-m');
            if (get_transient($sentKey)) {
                continue;
            }

            $pointsValue = $customer->points * $settings['points_value'];

            EmailService::send('loyalty_reminder', $customer->email, [
                'customer_name' => $customer->first_name,
                'points' => $customer->points,
                'points_value' => CurrencyService::formatPrice($pointsValue),
                'tier' => ucfirst($customer->tier),
            ]);

            set_transient($sentKey, true, 30 * DAY_IN_SECONDS);
        }
    }

    /**
     * Cleanup old data
     */
    private static function cleanupOldData(): void
    {
        global $wpdb;

        // Clean old search logs (90+ days)
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}cartt_search_logs 
             WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)"
        );

        // Clean old Q&A that was rejected (30+ days)
        $wpdb->query(
            "DELETE FROM {$wpdb->prefix}cartt_product_qa 
             WHERE status = 'rejected' 
             AND created_at < DATE_SUB(NOW(), INTERVAL 30 DAY)"
        );
    }

    /**
     * Send back-in-stock notifications
     */
    public static function sendBackInStockNotifications(int $productId): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_stock_notifications';

        $subscribers = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE product_id = %d AND notified_at IS NULL",
            $productId
        ));

        if (empty($subscribers)) {
            return;
        }

        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $productId
        ));

        if (!$product) {
            return;
        }

        foreach ($subscribers as $sub) {
            EmailService::send('back_in_stock', $sub->email, [
                'product_name' => $product->name,
                'product_url' => home_url('/product/' . $product->slug),
                'product_image' => $product->featured_image,
                'product_price' => CurrencyService::formatPrice($product->sale_price ?: $product->price),
            ]);

            $wpdb->update($table, ['notified_at' => current_time('mysql')], ['id' => $sub->id]);
        }
    }
}
