<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Checkout Service - handles order creation and payment
 */
class CheckoutService
{
    private CartService $cart;

    public function __construct(CartService $cart)
    {
        $this->cart = $cart;
    }

    public function processCheckout(array $data): array
    {
        $cart = $this->cart->getCart();

        if (empty($cart['items'])) {
            return ['success' => false, 'error' => 'Cart is empty'];
        }

        // Validate billing data
        $required = ['email', 'first_name', 'last_name'];
        foreach ($required as $field) {
            if (empty($data['billing'][$field])) {
                return ['success' => false, 'error' => "Missing required field: $field"];
            }
        }

        // Create or get customer
        $customerId = $this->getOrCreateCustomer($data['billing']);

        // Create order
        $order = $this->createOrder($cart, $customerId, $data);

        if (!$order['success']) {
            return $order;
        }

        // Process payment
        $paymentMethod = $data['payment_method'] ?? 'stripe';
        $skipPayment = $data['skip_payment'] ?? false;
        
        // Use final_total from order which includes gift card and loyalty adjustments
        $paymentTotal = $order['final_total'] ?? $cart['totals']['total'];
        
        // If total is zero (fully covered by gift card/loyalty), skip payment
        if ($paymentTotal <= 0) {
            $payment = [
                'success' => true,
                'transaction_id' => 'gc_loyalty_paid',
                'payment_status' => 'paid',
            ];
        } elseif ($skipPayment) {
            // Payment already processed (e.g., PayPal)
            $payment = [
                'success' => true,
                'transaction_id' => $data['paypal_order_id'] ?? null,
                'payment_status' => 'paid',
            ];
        } elseif ($paymentMethod === 'stripe') {
            $payment = $this->processStripePayment($order['order_id'], $paymentTotal, $data);
        } else {
            // Demo mode or other - mark as pending
            $payment = ['success' => true, 'requires_action' => false];
        }

        if (!$payment['success']) {
            // Update order to failed
            $this->updateOrderStatus($order['order_id'], 'failed');
            return $payment;
        }

        // If payment needs confirmation (3D Secure, etc)
        if (!empty($payment['requires_action'])) {
            return [
                'success' => true,
                'requires_action' => true,
                'client_secret' => $payment['client_secret'] ?? null,
                'order_id' => $order['order_id'],
            ];
        }

        // Payment successful
        $this->updateOrderStatus($order['order_id'], 'processing');
        $this->updateOrderPayment($order['order_id'], $payment);
        
        // Process gift card deduction
        if (!empty($order['gift_card'])) {
            GiftCardService::useBalance(
                $order['gift_card']['id'],
                min($order['gift_card']['amount_to_use'], $order['final_total'] + ($order['gift_card']['amount_to_use'] ?? 0)),
                $order['order_id']
            );
            // Clear from session
            if (isset($_SESSION['cartt_gift_card'])) {
                unset($_SESSION['cartt_gift_card']);
            }
        }

        // Process loyalty points deduction and earn new points
        if (!empty($order['loyalty'])) {
            LoyaltyService::redeemPoints(
                $order['loyalty']['customer_id'],
                $order['loyalty']['points'],
                $order['order_id']
            );
            // Clear from session
            if (isset($_SESSION['cartt_loyalty'])) {
                unset($_SESSION['cartt_loyalty']);
            }
        }

        // Award loyalty points for purchase
        if (get_option('cartt_loyalty_enabled', false)) {
            $customerId = $this->getCustomerIdFromOrder($order['order_id']);
            if ($customerId) {
                LoyaltyService::addPoints(
                    $customerId,
                    'purchase',
                    $order['order_id'],
                    $order['final_total']
                );
            }
        }
        
        // Reduce stock
        $this->reduceStock($cart['items']);
        
        // Clear cart
        $this->cart->clear();

        // Send confirmation email
        $this->sendOrderConfirmation($order['order_id']);

        // Mark any abandoned carts as converted
        $abandonedService = new AbandonedCartService();
        $abandonedService->markConverted($data['billing']['email'] ?? '', $order['order_id']);

        // Get order received page URL
        $orderReceivedPageId = get_option('cartt_order_received_page_id');
        $redirectUrl = $orderReceivedPageId 
            ? add_query_arg('order', $order['order_id'], get_permalink($orderReceivedPageId))
            : home_url('/?order=' . $order['order_id']);

        return [
            'success' => true,
            'order_id' => $order['order_id'],
            'order_number' => $order['order_number'],
            'redirect' => $redirectUrl,
        ];
    }

    public function createPaymentIntent(float $amount, string $currency = 'usd'): array
    {
        $stripeSecret = get_option('cartt_stripe_secret_key');
        
        if (!$stripeSecret) {
            return ['success' => false, 'error' => 'Stripe not configured'];
        }

        $response = wp_remote_post('https://api.stripe.com/v1/payment_intents', [
            'headers' => [
                'Authorization' => 'Bearer ' . $stripeSecret,
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => [
                'amount' => (int) ($amount * 100), // Convert to cents
                'currency' => strtolower($currency),
                'automatic_payment_methods' => ['enabled' => 'true'],
            ],
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['error'])) {
            return ['success' => false, 'error' => $body['error']['message']];
        }

        return [
            'success' => true,
            'client_secret' => $body['client_secret'],
            'payment_intent_id' => $body['id'],
        ];
    }

    public function confirmPayment(int $orderId, string $paymentIntentId): array
    {
        $stripeSecret = get_option('cartt_stripe_secret_key');
        
        $response = wp_remote_get("https://api.stripe.com/v1/payment_intents/$paymentIntentId", [
            'headers' => [
                'Authorization' => 'Bearer ' . $stripeSecret,
            ],
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($body['status'] === 'succeeded') {
            $this->updateOrderStatus($orderId, 'processing');
            $this->updateOrderPayment($orderId, [
                'transaction_id' => $paymentIntentId,
                'payment_status' => 'paid',
            ]);
            
            // Clear cart and send email
            $this->cart->clear();
            $this->sendOrderConfirmation($orderId);

            return ['success' => true, 'order_id' => $orderId];
        }

        return ['success' => false, 'error' => 'Payment not completed'];
    }

    private function processStripePayment(int $orderId, float $amount, array $data): array
    {
        $stripeSecret = get_option('cartt_stripe_secret_key');
        
        if (!$stripeSecret) {
            // Demo mode - simulate successful payment
            return [
                'success' => true,
                'transaction_id' => 'demo_' . wp_generate_password(16, false),
                'payment_status' => 'paid',
            ];
        }

        // Create payment intent for client-side confirmation
        $currency = get_option('cartt_currency', 'USD');
        $intent = $this->createPaymentIntent($amount, $currency);

        if (!$intent['success']) {
            return $intent;
        }

        // Store intent ID on order
        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'cartt_orders',
            ['transaction_id' => $intent['payment_intent_id']],
            ['id' => $orderId]
        );

        return [
            'success' => true,
            'requires_action' => true,
            'client_secret' => $intent['client_secret'],
        ];
    }

    private function getOrCreateCustomer(array $billing): int
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customers';

        $email = sanitize_email($billing['email']);
        
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM $table WHERE email = %s",
            $email
        ));

        if ($existing) {
            // Update customer info
            $wpdb->update($table, [
                'first_name' => sanitize_text_field($billing['first_name']),
                'last_name' => sanitize_text_field($billing['last_name']),
                'phone' => sanitize_text_field($billing['phone'] ?? ''),
                'billing_address' => json_encode($billing),
            ], ['id' => $existing->id]);

            return (int) $existing->id;
        }

        // Create new customer
        $wpdb->insert($table, [
            'user_id' => get_current_user_id() ?: null,
            'email' => $email,
            'first_name' => sanitize_text_field($billing['first_name']),
            'last_name' => sanitize_text_field($billing['last_name']),
            'phone' => sanitize_text_field($billing['phone'] ?? ''),
            'billing_address' => json_encode($billing),
        ]);

        return (int) $wpdb->insert_id;
    }

    private function createOrder(array $cart, int $customerId, array $data): array
    {
        global $wpdb;
        $ordersTable = $wpdb->prefix . 'cartt_orders';
        $itemsTable = $wpdb->prefix . 'cartt_order_items';

        // Generate order number
        $orderNumber = $this->generateOrderNumber();

        // Get gift card and loyalty from session
        if (!isset($_SESSION)) {
            session_start();
        }
        $giftCard = $_SESSION['cartt_gift_card'] ?? null;
        $loyalty = $_SESSION['cartt_loyalty'] ?? null;

        // Calculate final totals with gift card and loyalty
        $giftCardAmount = $giftCard ? min($giftCard['amount_to_use'], $cart['totals']['total']) : 0;
        $loyaltyDiscount = $loyalty ? min($loyalty['discount'], $cart['totals']['total'] - $giftCardAmount) : 0;
        $finalTotal = max(0, $cart['totals']['total'] - $giftCardAmount - $loyaltyDiscount);

        // Insert order
        $wpdb->insert($ordersTable, [
            'order_number' => $orderNumber,
            'customer_id' => $customerId,
            'status' => 'pending',
            'currency' => get_option('cartt_currency', 'USD'),
            'subtotal' => $cart['totals']['subtotal'],
            'tax_total' => $cart['totals']['tax'],
            'shipping_total' => $cart['totals']['shipping'],
            'discount_total' => ($cart['totals']['discount'] ?? 0) + $giftCardAmount + $loyaltyDiscount,
            'total' => $finalTotal,
            'payment_method' => sanitize_text_field($data['payment_method'] ?? 'stripe'),
            'payment_status' => 'pending',
            'billing_address' => json_encode($data['billing']),
            'shipping_address' => json_encode($data['shipping'] ?? $data['billing']),
            'customer_note' => sanitize_textarea_field($data['notes'] ?? ''),
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'meta' => json_encode([
                'gift_card_id' => $giftCard['id'] ?? null,
                'gift_card_code' => $giftCard['code'] ?? null,
                'gift_card_amount' => $giftCardAmount,
                'loyalty_points_used' => $loyalty['points'] ?? 0,
                'loyalty_discount' => $loyaltyDiscount,
                'coupon_code' => $cart['coupon']['code'] ?? null,
                'coupon_discount' => $cart['totals']['discount'] ?? 0,
            ]),
        ]);

        $orderId = $wpdb->insert_id;

        if (!$orderId) {
            return ['success' => false, 'error' => 'Failed to create order'];
        }

        // Insert order items
        foreach ($cart['items'] as $item) {
            $wpdb->insert($itemsTable, [
                'order_id' => $orderId,
                'product_id' => $item['product_id'],
                'name' => $item['name'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'subtotal' => $item['price'] * $item['quantity'],
                'total' => $item['price'] * $item['quantity'],
                'meta' => json_encode($item['meta'] ?? []),
            ]);
        }

        // Store gift card and loyalty info for post-payment processing
        return [
            'success' => true,
            'order_id' => $orderId,
            'order_number' => $orderNumber,
            'gift_card' => $giftCard,
            'loyalty' => $loyalty,
            'final_total' => $finalTotal,
        ];
    }

    private function generateOrderNumber(): string
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_orders';
        
        $lastOrder = $wpdb->get_var("SELECT MAX(id) FROM $table");
        $nextId = ($lastOrder ?: 0) + 1;
        
        return 'CT-' . str_pad((string) $nextId, 5, '0', STR_PAD_LEFT);
    }

    private function getCustomerIdFromOrder(int $orderId): ?int
    {
        global $wpdb;
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT customer_id FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        )) ?: null;
    }

    private function updateOrderStatus(int $orderId, string $status): void
    {
        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'cartt_orders',
            [
                'status' => $status,
                'completed_at' => $status === 'completed' ? current_time('mysql') : null,
            ],
            ['id' => $orderId]
        );
    }

    private function updateOrderPayment(int $orderId, array $payment): void
    {
        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'cartt_orders',
            [
                'transaction_id' => $payment['transaction_id'] ?? null,
                'payment_status' => $payment['payment_status'] ?? 'paid',
            ],
            ['id' => $orderId]
        );

        // Update customer totals
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT customer_id, total FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));

        if ($order) {
            $wpdb->query($wpdb->prepare(
                "UPDATE {$wpdb->prefix}cartt_customers 
                 SET total_spent = total_spent + %f, order_count = order_count + 1 
                 WHERE id = %d",
                $order->total,
                $order->customer_id
            ));
        }
    }

    private function reduceStock(array $items): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        foreach ($items as $item) {
            $wpdb->query($wpdb->prepare(
                "UPDATE $table SET stock_quantity = stock_quantity - %d 
                 WHERE id = %d AND manage_stock = 1",
                $item['quantity'],
                $item['product_id']
            ));
        }
    }

    private function sendOrderConfirmation(int $orderId): void
    {
        global $wpdb;
        
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT o.*, c.email, c.first_name 
             FROM {$wpdb->prefix}cartt_orders o
             JOIN {$wpdb->prefix}cartt_customers c ON o.customer_id = c.id
             WHERE o.id = %d",
            $orderId
        ));

        if (!$order || !$order->email) {
            return;
        }

        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_order_items WHERE order_id = %d",
            $orderId
        ));

        $storeName = get_option('cartt_store_name', get_bloginfo('name'));
        $subject = sprintf(__('Order #%s confirmed', 'cartt'), $order->order_number);

        // Build email
        $message = sprintf(__('Hi %s,', 'cartt'), $order->first_name) . "\n\n";
        $message .= sprintf(__('Thank you for your order #%s.', 'cartt'), $order->order_number) . "\n\n";
        $message .= __('Order summary:', 'cartt') . "\n";
        $message .= "---\n";

        foreach ($items as $item) {
            $message .= sprintf("%s x %d - $%.2f\n", $item->name, $item->quantity, $item->total);
        }

        $message .= "---\n";
        $message .= sprintf(__('Total: $%.2f', 'cartt'), $order->total) . "\n\n";
        $message .= sprintf(__('View your order: %s', 'cartt'), home_url('/my-account/')) . "\n\n";
        $message .= $storeName;

        $headers = [
            'Content-Type: text/plain; charset=UTF-8',
            'From: ' . $storeName . ' <' . get_option('admin_email') . '>',
        ];

        wp_mail($order->email, $subject, $message, $headers);

        // Also notify admin
        wp_mail(
            get_option('admin_email'),
            sprintf(__('New order #%s', 'cartt'), $order->order_number),
            sprintf(__('New order received for $%.2f. View in admin.', 'cartt'), $order->total),
            $headers
        );
    }
}
