<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Checkout Configuration Service
 * Manages checkout layout, fields, and settings
 */
class CheckoutConfig
{
    private const OPTION_KEY = 'cartt_checkout_config';

    public static function getDefaults(): array
    {
        return [
            'layout' => 'one-page', // one-page, multi-step, express
            'style' => 'modern', // modern, classic, minimal
            'guest_checkout' => true,
            'create_account_option' => true,
            'email_first' => true, // Capture email first for abandoned cart
            'address_autocomplete' => false,
            'google_places_key' => '',
            'express_payments' => [
                'enabled' => false,
                'apple_pay' => false,
                'google_pay' => false,
                'link' => false, // Stripe Link
            ],
            'order_bumps' => [],
            'fields' => self::getDefaultFields(),
            'field_order' => ['email', 'name', 'address', 'phone', 'notes'],
        ];
    }

    public static function getDefaultFields(): array
    {
        return [
            'email' => [
                'id' => 'email',
                'type' => 'email',
                'label' => 'Email',
                'placeholder' => 'you@example.com',
                'required' => true,
                'enabled' => true,
                'width' => 'full',
                'section' => 'contact',
                'locked' => true, // Cannot be deleted
            ],
            'first_name' => [
                'id' => 'first_name',
                'type' => 'text',
                'label' => 'First name',
                'placeholder' => '',
                'required' => true,
                'enabled' => true,
                'width' => 'half',
                'section' => 'billing',
                'locked' => true,
            ],
            'last_name' => [
                'id' => 'last_name',
                'type' => 'text',
                'label' => 'Last name',
                'placeholder' => '',
                'required' => true,
                'enabled' => true,
                'width' => 'half',
                'section' => 'billing',
                'locked' => true,
            ],
            'company' => [
                'id' => 'company',
                'type' => 'text',
                'label' => 'Company',
                'placeholder' => '',
                'required' => false,
                'enabled' => false,
                'width' => 'full',
                'section' => 'billing',
                'locked' => false,
            ],
            'address_1' => [
                'id' => 'address_1',
                'type' => 'text',
                'label' => 'Address',
                'placeholder' => 'Street address',
                'required' => true,
                'enabled' => true,
                'width' => 'full',
                'section' => 'billing',
                'locked' => true,
                'autocomplete' => 'address', // Trigger for Google Places
            ],
            'address_2' => [
                'id' => 'address_2',
                'type' => 'text',
                'label' => 'Apartment, suite, etc.',
                'placeholder' => '',
                'required' => false,
                'enabled' => true,
                'width' => 'full',
                'section' => 'billing',
                'locked' => false,
            ],
            'city' => [
                'id' => 'city',
                'type' => 'text',
                'label' => 'City',
                'placeholder' => '',
                'required' => true,
                'enabled' => true,
                'width' => 'half',
                'section' => 'billing',
                'locked' => true,
            ],
            'state' => [
                'id' => 'state',
                'type' => 'text',
                'label' => 'State / Province',
                'placeholder' => '',
                'required' => true,
                'enabled' => true,
                'width' => 'quarter',
                'section' => 'billing',
                'locked' => true,
            ],
            'postcode' => [
                'id' => 'postcode',
                'type' => 'text',
                'label' => 'ZIP / Postal code',
                'placeholder' => '',
                'required' => true,
                'enabled' => true,
                'width' => 'quarter',
                'section' => 'billing',
                'locked' => true,
            ],
            'country' => [
                'id' => 'country',
                'type' => 'country',
                'label' => 'Country',
                'placeholder' => '',
                'required' => true,
                'enabled' => true,
                'width' => 'full',
                'section' => 'billing',
                'locked' => true,
            ],
            'phone' => [
                'id' => 'phone',
                'type' => 'tel',
                'label' => 'Phone',
                'placeholder' => '',
                'required' => false,
                'enabled' => true,
                'width' => 'full',
                'section' => 'billing',
                'locked' => false,
            ],
            'notes' => [
                'id' => 'notes',
                'type' => 'textarea',
                'label' => 'Order notes',
                'placeholder' => 'Special instructions for delivery...',
                'required' => false,
                'enabled' => false,
                'width' => 'full',
                'section' => 'additional',
                'locked' => false,
            ],
        ];
    }

    public static function get(): array
    {
        $saved = get_option(self::OPTION_KEY, []);
        return array_merge(self::getDefaults(), $saved);
    }

    public static function save(array $config): bool
    {
        // Validate
        $allowed_layouts = ['one-page', 'multi-step', 'express'];
        if (!in_array($config['layout'] ?? '', $allowed_layouts)) {
            $config['layout'] = 'one-page';
        }

        return update_option(self::OPTION_KEY, $config);
    }

    public static function getFields(): array
    {
        $config = self::get();
        $fields = $config['fields'] ?? self::getDefaultFields();
        
        // Return only enabled fields, sorted by order
        $enabled = array_filter($fields, fn($f) => $f['enabled'] ?? false);
        
        // Sort by section, then by custom order if set
        uasort($enabled, function($a, $b) {
            $sections = ['contact' => 0, 'billing' => 1, 'shipping' => 2, 'additional' => 3];
            $secA = $sections[$a['section'] ?? 'additional'] ?? 99;
            $secB = $sections[$b['section'] ?? 'additional'] ?? 99;
            
            if ($secA !== $secB) {
                return $secA - $secB;
            }
            
            return ($a['order'] ?? 0) - ($b['order'] ?? 0);
        });

        return $enabled;
    }

    public static function addField(array $field): bool
    {
        $config = self::get();
        
        if (empty($field['id'])) {
            $field['id'] = 'custom_' . uniqid();
        }

        $field = array_merge([
            'type' => 'text',
            'label' => 'Custom Field',
            'placeholder' => '',
            'required' => false,
            'enabled' => true,
            'width' => 'full',
            'section' => 'additional',
            'locked' => false,
        ], $field);

        $config['fields'][$field['id']] = $field;

        return self::save($config);
    }

    public static function updateField(string $id, array $updates): bool
    {
        $config = self::get();
        
        if (!isset($config['fields'][$id])) {
            return false;
        }

        // Prevent unlocking locked fields
        if ($config['fields'][$id]['locked'] ?? false) {
            unset($updates['locked']);
            unset($updates['id']); // Can't change ID of locked fields
        }

        $config['fields'][$id] = array_merge($config['fields'][$id], $updates);

        return self::save($config);
    }

    public static function removeField(string $id): bool
    {
        $config = self::get();
        
        if (!isset($config['fields'][$id])) {
            return false;
        }

        // Can't remove locked fields
        if ($config['fields'][$id]['locked'] ?? false) {
            return false;
        }

        unset($config['fields'][$id]);

        return self::save($config);
    }

    public static function reorderFields(array $order): bool
    {
        $config = self::get();
        
        $i = 0;
        foreach ($order as $fieldId) {
            if (isset($config['fields'][$fieldId])) {
                $config['fields'][$fieldId]['order'] = $i++;
            }
        }

        return self::save($config);
    }

    public static function getOrderBumps(): array
    {
        $config = self::get();
        return $config['order_bumps'] ?? [];
    }

    public static function addOrderBump(array $bump): bool
    {
        $config = self::get();
        
        $bump = array_merge([
            'id' => 'bump_' . uniqid(),
            'product_id' => null,
            'title' => '',
            'description' => '',
            'discount_type' => 'percent', // percent, fixed
            'discount_value' => 0,
            'enabled' => true,
        ], $bump);

        $config['order_bumps'][$bump['id']] = $bump;

        return self::save($config);
    }

    public static function removeOrderBump(string $id): bool
    {
        $config = self::get();
        
        if (isset($config['order_bumps'][$id])) {
            unset($config['order_bumps'][$id]);
            return self::save($config);
        }

        return false;
    }
}
