<?php
/**
 * Chat Service
 * 
 * Live chat with SMS/WhatsApp notifications and cart visibility.
 * Requires 'chat' add-on to be active.
 * 
 * @package Cartt
 * @since 1.4.2
 */

declare(strict_types=1);

namespace Cartt\Services;

class ChatService
{
    private static ?self $instance = null;
    
    private string $table_conversations;
    private string $table_messages;
    private string $table_agents;
    private string $relay_url = 'https://cartt.app/wp-content/uploads/cartt-relay/relay.php';

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        global $wpdb;
        $this->table_conversations = $wpdb->prefix . 'cartt_chat_conversations';
        $this->table_messages = $wpdb->prefix . 'cartt_chat_messages';
        $this->table_agents = $wpdb->prefix . 'cartt_chat_agents';
        
        // Basic chat is always enabled (free feature)
        // PRO features (SMS, WhatsApp, AI, screenshots, attachments) require addon
        
        // Admin menu
        add_action('admin_menu', [$this, 'addAdminMenu']);
        add_action('admin_bar_menu', [$this, 'addAdminBarIcon'], 100);
        add_action('admin_footer', [$this, 'addNotificationPolling']);
        
        // Frontend widget - both wp_footer and shortcode for flexibility
        add_action('wp_footer', [$this, 'renderWidget']);
        add_action('wp_enqueue_scripts', [$this, 'enqueueFrontendAssets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueAdminAssets']);
        
        // Shortcode for Elementor/page builder pages where wp_footer doesn't work
        add_shortcode('cartt_chat', [$this, 'renderShortcode']);
        
        // AJAX handlers - Visitor (basic chat - always enabled)
        add_action('wp_ajax_cartt_chat_send', [$this, 'ajaxSendMessage']);
        add_action('wp_ajax_nopriv_cartt_chat_send', [$this, 'ajaxSendMessage']);
        add_action('wp_ajax_cartt_chat_get_messages', [$this, 'ajaxGetMessages']);
        add_action('wp_ajax_nopriv_cartt_chat_get_messages', [$this, 'ajaxGetMessages']);
        add_action('wp_ajax_cartt_chat_poll_messages', [$this, 'ajaxPollMessages']);
        add_action('wp_ajax_nopriv_cartt_chat_poll_messages', [$this, 'ajaxPollMessages']);
        
        // PRO features - screenshots and file uploads (require addon)
        add_action('wp_ajax_cartt_chat_send_screenshot', [$this, 'ajaxSendScreenshot']);
        add_action('wp_ajax_nopriv_cartt_chat_send_screenshot', [$this, 'ajaxSendScreenshot']);
        add_action('wp_ajax_cartt_chat_send_file', [$this, 'ajaxSendFile']);
        add_action('wp_ajax_nopriv_cartt_chat_send_file', [$this, 'ajaxSendFile']);
        
        // AJAX handlers - Admin
        add_action('wp_ajax_cartt_chat_admin_reply', [$this, 'ajaxAdminReply']);
        add_action('wp_ajax_cartt_chat_get_conversation', [$this, 'ajaxGetConversation']);
        add_action('wp_ajax_cartt_chat_get_conversations', [$this, 'ajaxGetConversations']);
        add_action('wp_ajax_cartt_chat_mark_read', [$this, 'ajaxMarkRead']);
        add_action('wp_ajax_cartt_chat_close', [$this, 'ajaxCloseConversation']);
        add_action('wp_ajax_cartt_chat_save_settings', [$this, 'ajaxSaveSettings']);
        add_action('wp_ajax_cartt_chat_save_agent', [$this, 'ajaxSaveAgent']);
        add_action('wp_ajax_cartt_chat_delete_agent', [$this, 'ajaxDeleteAgent']);
        add_action('wp_ajax_cartt_chat_test_notification', [$this, 'ajaxTestNotification']);
        add_action('wp_ajax_cartt_chat_get_usage', [$this, 'ajaxGetUsage']);
        
        // Billing AJAX handlers (PRO)
        add_action('wp_ajax_cartt_chat_get_balance', [$this, 'ajaxGetBalance']);
        add_action('wp_ajax_cartt_chat_get_payment_method', [$this, 'ajaxGetPaymentMethod']);
        add_action('wp_ajax_cartt_chat_get_transactions', [$this, 'ajaxGetTransactions']);
        add_action('wp_ajax_cartt_chat_add_funds', [$this, 'ajaxAddFunds']);
        add_action('wp_ajax_cartt_chat_get_card_update_url', [$this, 'ajaxGetCardUpdateUrl']);
        add_action('wp_ajax_cartt_chat_set_auto_refill', [$this, 'ajaxSetAutoRefill']);
        add_action('wp_ajax_cartt_chat_save_quick_actions', [$this, 'ajaxSaveQuickActions']);
        add_action('wp_ajax_cartt_chat_get_agents', [$this, 'ajaxGetAgents']);
        add_action('wp_ajax_cartt_chat_add_agent', [$this, 'ajaxAddAgent']);
        add_action('wp_ajax_cartt_chat_update_agent', [$this, 'ajaxUpdateAgent']);
        add_action('wp_ajax_cartt_chat_remove_agent', [$this, 'ajaxRemoveAgent']);
        add_action('wp_ajax_cartt_chat_get_quick_actions', [$this, 'ajaxGetQuickActions']);
        add_action('wp_ajax_cartt_chat_get_unread_count', [$this, 'ajaxGetUnreadCount']);
        add_action('wp_ajax_cartt_chat_check_handoff', [$this, 'ajaxCheckHandoff']);
        add_action('wp_ajax_nopriv_cartt_chat_check_handoff', [$this, 'ajaxCheckHandoff']);
        
        // REST API for incoming SMS/WhatsApp (PRO)
        add_action('rest_api_init', [$this, 'registerRestRoutes']);
    }

    /**
     * Check if chat addon is active (for PRO features)
     */
    private function isAddonActive(): bool
    {
        $activeAddons = get_option('cartt_active_addons', []);
        
        // Always active on cartt.app
        if (strpos(home_url(), 'cartt.app') !== false) {
            return true;
        }
        
        return in_array('chat', $activeAddons) || in_array('all_access', $activeAddons);
    }
    
    /**
     * Check if PRO chat features are available (SMS, WhatsApp, screenshots, attachments, AI)
     */
    public function hasProFeatures(): bool
    {
        return $this->isAddonActive();
    }
    
    /**
     * Check if basic chat is enabled (always true - free feature)
     */
    private function isChatEnabled(): bool
    {
        return true; // Basic chat is always available
    }

    /**
     * Get license key for relay
     */
    private function getLicenseKey(): string
    {
        // For cartt.app, use special key
        if (strpos(home_url(), 'cartt.app') !== false) {
            return 'CARTT_APP_INTERNAL';
        }
        
        $licenses = get_option('cartt_addon_licenses', []);
        return $licenses['chat']['key'] ?? $licenses['all_access']['key'] ?? '';
    }

    /**
     * Register REST routes for incoming messages
     */
    public function registerRestRoutes(): void
    {
        // Route for relay to forward SMS/WhatsApp replies
        register_rest_route('cartt/v1', '/relay-reply', [
            'methods' => 'POST',
            'callback' => [$this, 'handleRelayReply'],
            'permission_callback' => '__return_true'
        ]);
        
        // Legacy route (in case relay uses old path)
        register_rest_route('woopro/v1', '/relay-reply', [
            'methods' => 'POST',
            'callback' => [$this, 'handleRelayReply'],
            'permission_callback' => '__return_true'
        ]);
        
        // Route for balance notifications from relay
        register_rest_route('cartt/v1', '/balance-notification', [
            'methods' => 'POST',
            'callback' => [$this, 'handleBalanceNotification'],
            'permission_callback' => '__return_true'
        ]);
    }

    /**
     * Handle balance notification from relay
     * Types: low_balance, refill_success, refill_failed
     */
    public function handleBalanceNotification(\WP_REST_Request $request): \WP_REST_Response
    {
        $params = $request->get_json_params();
        $type = sanitize_text_field($params['type'] ?? '');
        $balance = floatval($params['balance'] ?? 0);
        $amount = floatval($params['amount'] ?? 0);
        $error = sanitize_text_field($params['error'] ?? '');
        
        $adminEmail = get_option('admin_email');
        $siteName = get_bloginfo('name');
        
        switch ($type) {
            case 'low_balance':
                $subject = "[{$siteName}] Chat Balance Low";
                $message = "Your chat balance is \${$balance}.\n\n";
                $message .= "We'll automatically refill \$20 when it reaches \$5 to keep your SMS, WhatsApp, and AI chat features running.\n\n";
                $message .= "Manage your balance: " . admin_url('admin.php?page=cartt-chat-settings');
                break;
                
            case 'refill_success':
                $subject = "[{$siteName}] Chat Balance Refilled";
                $message = "Your chat balance was automatically refilled.\n\n";
                $message .= "Amount added: \${$amount}\n";
                $message .= "New balance: \${$balance}\n\n";
                $message .= "Manage your balance: " . admin_url('admin.php?page=cartt-chat-settings');
                break;
                
            case 'refill_failed':
                $subject = "[{$siteName}] Chat Payment Failed - Action Required";
                $message = "We couldn't process your automatic balance refill.\n\n";
                $message .= "Error: {$error}\n";
                $message .= "Current balance: \${$balance}\n\n";
                $message .= "Please update your payment method to continue using SMS, WhatsApp, and AI chat features:\n";
                $message .= admin_url('admin.php?page=cartt-chat-settings');
                break;
                
            default:
                return new \WP_REST_Response(['success' => false, 'error' => 'Unknown type'], 400);
        }
        
        $this->sendEmailViaRelay($adminEmail, $subject, $message);
        
        return new \WP_REST_Response(['success' => true]);
    }

    /**
     * Add admin menu
     */
    public function addAdminMenu(): void
    {
        $unread = $this->getUnreadCount();
        $badge = $unread > 0 ? ' <span class="awaiting-mod">' . $unread . '</span>' : '';
        
        add_submenu_page(
            'cartt',
            'Chat',
            'Chat' . $badge,
            'manage_options',
            'cartt-chat',
            [$this, 'renderInbox']
        );
        
        // Hidden page (no menu item) - accessed via link from Chat page
        add_submenu_page(
            null,
            'Chat Settings',
            'Chat Settings',
            'manage_options',
            'cartt-chat-settings',
            [$this, 'renderSettings']
        );
        
        // Fix menu URL (WordPress sometimes generates incorrect URLs)
        add_action('admin_menu', [$this, 'fixChatMenuUrl'], 999);
    }
    
    /**
     * Fix chat menu URL
     */
    public function fixChatMenuUrl(): void
    {
        global $submenu;
        if (isset($submenu['cartt'])) {
            foreach ($submenu['cartt'] as $key => $item) {
                if ($item[2] === 'cartt-chat') {
                    $submenu['cartt'][$key][2] = 'admin.php?page=cartt-chat';
                }
            }
        }
    }

    /**
     * Add admin bar icon
     */
    public function addAdminBarIcon($wp_admin_bar): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $unread = $this->getUnreadCount();
        
        $wp_admin_bar->add_node([
            'id' => 'cartt-chat',
            'title' => '<span class="ab-icon dashicons dashicons-format-chat" style="font-family:dashicons;"></span>' . 
                      '<span class="cartt-chat-ab-badge" id="cartt-chat-ab-badge" style="' . ($unread > 0 ? '' : 'display:none;') . 'background:#e74c3c;color:#fff;border-radius:9px;padding:0 5px;font-size:10px;margin-left:4px;">' . $unread . '</span>',
            'href' => admin_url('admin.php?page=cartt-chat'),
            'meta' => ['title' => 'Cartt Chat']
        ]);
    }

    /**
     * Add notification polling script to admin
     */
    public function addNotificationPolling(): void
    {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <script>
        (function() {
            function updateChatBadge() {
                jQuery.post(ajaxurl, {
                    action: 'cartt_chat_get_unread_count',
                    nonce: '<?php echo wp_create_nonce('cartt_chat_admin'); ?>'
                }, function(response) {
                    if (response.success) {
                        var count = response.data.count || 0;
                        var badge = document.getElementById('cartt-chat-ab-badge');
                        if (badge) {
                            badge.textContent = count;
                            badge.style.display = count > 0 ? '' : 'none';
                        }
                        // Also update menu item badge if exists
                        var menuBadge = document.querySelector('#toplevel_page_cartt .cartt-chat-badge');
                        if (menuBadge) {
                            menuBadge.textContent = count;
                            menuBadge.style.display = count > 0 ? '' : 'none';
                        }
                    }
                });
            }
            // Poll every 10 seconds
            setInterval(updateChatBadge, 10000);
            // Initial check after 2 seconds
            setTimeout(updateChatBadge, 2000);
        })();
        </script>
        <?php
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueueFrontendAssets(): void
    {
        if (is_admin()) {
            return;
        }
        
        wp_enqueue_style(
            'cartt-chat-widget',
            CARTT_PLUGIN_URL . 'assets/css/chat-widget.css',
            [],
            CARTT_VERSION
        );
        
        wp_enqueue_script(
            'cartt-chat-widget',
            CARTT_PLUGIN_URL . 'assets/js/chat-widget.js',
            ['jquery'],
            CARTT_VERSION,
            true
        );
        
        wp_localize_script('cartt-chat-widget', 'carttChat', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cartt_chat'),
            'settings' => $this->getWidgetSettings()
        ]);
    }

    /**
     * Enqueue admin assets
     */
    public function enqueueAdminAssets($hook): void
    {
        if (strpos($hook, 'cartt-chat') === false) {
            return;
        }
        
        wp_enqueue_style('dashicons');
        
        wp_enqueue_style(
            'cartt-chat-admin',
            CARTT_PLUGIN_URL . 'assets/css/chat-admin.css',
            ['dashicons'],
            CARTT_VERSION
        );
        
        wp_enqueue_script(
            'cartt-chat-admin',
            CARTT_PLUGIN_URL . 'assets/js/chat-admin.js',
            ['jquery'],
            CARTT_VERSION,
            true
        );
        
        wp_localize_script('cartt-chat-admin', 'carttChatAdmin', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cartt_chat_admin')
        ]);
    }

    /**
     * Get widget settings
     */
    private function getWidgetSettings(): array
    {
        return [
            'enabled' => get_option('cartt_chat_enabled', '1'),
            'title' => get_option('cartt_chat_title', 'Chat with us'),
            'greeting' => get_option('cartt_chat_greeting', 'Hi! How can we help you today?'),
            'position' => get_option('cartt_chat_position', 'right'),
            'avatar' => get_option('cartt_chat_avatar', ''),
            'online_text' => get_option('cartt_chat_online_text', 'Online now'),
            'show_online' => get_option('cartt_chat_show_online', '1'),
            'sound_enabled' => get_option('cartt_chat_sound_enabled', '1'),
            'screenshots' => get_option('cartt_chat_screenshots', '1'),
            'attachments' => get_option('cartt_chat_attachments', '1'),
            // Colors
            'color_header' => get_option('cartt_chat_color_header', '#000000'),
            'color_header_text' => get_option('cartt_chat_color_header_text', 'auto'),
            'color_visitor_bubble' => get_option('cartt_chat_color_visitor_bubble', '#000000'),
            'color_agent_bubble' => get_option('cartt_chat_color_agent_bubble', '#f0f0f0'),
            'color_background' => get_option('cartt_chat_color_background', '#f5f5f5'),
            'color_input_border' => get_option('cartt_chat_color_input_border', '#e0e0e0'),
            'color_input_bg' => get_option('cartt_chat_color_input_bg', '#fafafa'),
            'color_input_field' => get_option('cartt_chat_color_input_field', '#ffffff'),
            'input_border_width' => get_option('cartt_chat_input_border_width', '1'),
            'color_icons' => get_option('cartt_chat_color_icons', '#666666'),
            'color_send_icon' => get_option('cartt_chat_color_send_icon', '#000000'),
            // Pre-chat
            'collect_name' => get_option('cartt_chat_collect_name', '0'),
            'collect_email' => get_option('cartt_chat_collect_email', '0'),
            'collect_phone' => get_option('cartt_chat_collect_phone', '0'),
        ];
    }

    /**
     * Render chat widget on frontend
     */
    public function renderWidget(): void
    {
        static $rendered = false;
        
        if ($rendered || is_admin()) {
            return;
        }
        
        $settings = $this->getWidgetSettings();
        
        if ($settings['enabled'] !== '1') {
            return;
        }
        
        $rendered = true;
        
        // Use the shortcode version which is more feature-complete
        echo $this->renderShortcode();
    }

    /**
     * Get visitor ID (from cookie or generate new)
     */
    private function getVisitorId(): string
    {
        if (isset($_COOKIE['cartt_visitor_id'])) {
            return sanitize_text_field($_COOKIE['cartt_visitor_id']);
        }
        
        $visitorId = 'v_' . bin2hex(random_bytes(16));
        setcookie('cartt_visitor_id', $visitorId, time() + (86400 * 365), '/');
        return $visitorId;
    }

    /**
     * Get current cart data
     */
    private function getCartData(): array
    {
        try {
            if (!class_exists('\\Cartt\\Services\\CartService')) {
                return ['items' => [], 'total' => 0, 'total_formatted' => '$0.00', 'count' => 0];
            }
            
            $cartService = CartService::instance();
            $cart = $cartService->getCart();
            
            $items = [];
            foreach ($cart['items'] ?? [] as $item) {
                $items[] = [
                    'id' => $item['product_id'] ?? 0,
                    'name' => $item['name'] ?? '',
                    'qty' => $item['quantity'] ?? 1,
                    'price' => $item['price'] ?? 0,
                    'total' => $item['subtotal'] ?? 0
                ];
            }
            
            $total = $cart['total'] ?? 0;
            $formatted = class_exists('\\Cartt\\Services\\CurrencyService') 
                ? CurrencyService::formatPrice($total) 
                : '$' . number_format($total, 2);
            
            return [
                'items' => $items,
                'total' => $total,
                'total_formatted' => $formatted,
                'count' => count($items)
            ];
        } catch (\Throwable $e) {
            error_log('Cartt Chat getCartData error: ' . $e->getMessage());
            return ['items' => [], 'total' => 0, 'total_formatted' => '$0.00', 'count' => 0];
        }
    }

    /**
     * AJAX: Send message from visitor
     */
    public function ajaxSendMessage(): void
    {
        try {
            check_ajax_referer('cartt_chat', 'nonce');
            
            $visitorId = isset($_POST['visitor_id']) ? sanitize_text_field($_POST['visitor_id']) : '';
            $message = isset($_POST['message']) ? sanitize_textarea_field($_POST['message']) : '';
            $name = isset($_POST['name']) ? sanitize_text_field($_POST['name']) : '';
            $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
            $phone = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';
            $cart = isset($_POST['cart']) ? $_POST['cart'] : null;
            $pageUrl = isset($_POST['page_url']) ? esc_url_raw($_POST['page_url']) : '';
            
            if (empty($visitorId) || empty($message)) {
                wp_send_json_error('Missing required fields');
                return;
            }
            
            // Get or create conversation
            $conversation = $this->getConversationByVisitor($visitorId);
            
            if (!$conversation) {
                $convId = $this->createConversation([
                    'visitor_id' => $visitorId,
                    'visitor_name' => $name,
                    'visitor_email' => $email,
                    'visitor_phone' => $phone,
                    'cart_data' => $cart ? wp_json_encode($cart) : null,
                    'page_url' => $pageUrl,
                    'ip_address' => $_SERVER['REMOTE_ADDR'] ?? ''
                ]);
                $isNewConversation = true;
            } else {
                $convId = (int) $conversation->id;
                $isNewConversation = false;
                
                // Update visitor info if provided
                $updateData = ['cart_data' => $cart ? wp_json_encode($cart) : $conversation->cart_data];
                if ($name && !$conversation->visitor_name) $updateData['visitor_name'] = $name;
                if ($email && !$conversation->visitor_email) $updateData['visitor_email'] = $email;
                if ($phone && !$conversation->visitor_phone) $updateData['visitor_phone'] = $phone;
                $this->updateConversation($convId, $updateData);
            }
            
            // Add message
            $messageId = $this->addMessage($convId, 'visitor', $message);
            
            // Notify agents (non-blocking, errors caught)
            try {
                $notifyEvery = get_option('cartt_chat_notify_every_message', '0');
                if ($isNewConversation || $notifyEvery === '1') {
                    $this->notifyAgents($convId, $message, $name, $cart);
                }
            } catch (\Throwable $e) {
                error_log('Cartt Chat notify error: ' . $e->getMessage());
            }
            
            // Get AI reply if enabled (non-blocking, errors caught)
            $aiReply = null;
            $aiMessageId = null;
            $needsHandoff = false;
            
            try {
                $aiResult = $this->getAIReply($convId, $message, $name, $cart);
                if ($aiResult) {
                    $aiReply = $aiResult['reply'];
                    $needsHandoff = $aiResult['needs_handoff'] ?? false;
                    
                    // Handle handoff - notify agents immediately and mark conversation
                    if ($needsHandoff) {
                        $this->handleHandoff($convId, $name, $message);
                    }
                    
                    $aiMessageId = $this->addMessage($convId, 'admin', $aiReply);
                }
            } catch (\Throwable $e) {
                error_log('Cartt Chat AI error: ' . $e->getMessage());
            }
            
            wp_send_json_success([
                'conversation_id' => $convId,
                'message_id' => $messageId,
                'ai_reply' => $aiReply,
                'ai_message_id' => $aiMessageId,
                'needs_handoff' => $needsHandoff
            ]);
        } catch (\Throwable $e) {
            error_log('Cartt Chat send error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
            wp_send_json_error('Server error: ' . $e->getMessage());
        }
    }

    /**
     * AJAX: Get messages for visitor
     */
    public function ajaxGetMessages(): void
    {
        try {
            check_ajax_referer('cartt_chat', 'nonce');
            
            $visitorId = isset($_POST['visitor_id']) ? sanitize_text_field($_POST['visitor_id']) : '';
            
            if (empty($visitorId)) {
                wp_send_json_success(['messages' => []]);
                return;
            }
            
            $conversation = $this->getConversationByVisitor($visitorId);
            
            if (!$conversation) {
                wp_send_json_success(['messages' => []]);
                return;
            }
            
            $messages = $this->getMessages($conversation->id);
            
            // Mark admin messages as read
            $this->markMessagesRead($conversation->id, 'admin');
            
            wp_send_json_success(['messages' => $messages]);
        } catch (\Throwable $e) {
            error_log('Cartt Chat getMessages error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
            wp_send_json_success(['messages' => [], 'debug_error' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Admin reply
     */
    public function ajaxAdminReply(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $convId = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        $message = isset($_POST['message']) ? sanitize_textarea_field($_POST['message']) : '';
        $sendSms = isset($_POST['send_sms']) && $_POST['send_sms'] === '1';
        $sendWhatsApp = isset($_POST['send_whatsapp']) && $_POST['send_whatsapp'] === '1';
        $includeCart = isset($_POST['include_cart']) && $_POST['include_cart'] === '1';
        
        if (empty($convId) || empty($message)) {
            wp_send_json_error('Missing required fields');
        }
        
        // Add message to database
        $messageId = $this->addMessage($convId, 'admin', $message);
        
        // Clear handoff status when human responds
        global $wpdb;
        $wpdb->update(
            $this->table_conversations,
            ['handoff_status' => 'resolved'],
            ['id' => $convId]
        );
        
        // Send SMS/WhatsApp if requested
        $conversation = $this->getConversation($convId);
        $smsSent = false;
        $whatsAppSent = false;
        
        if ($conversation && !empty($conversation->visitor_phone)) {
            if ($sendSms) {
                $smsSent = $this->sendReplyToVisitor($conversation, $message, 'sms', $includeCart);
            }
            if ($sendWhatsApp) {
                $whatsAppSent = $this->sendReplyToVisitor($conversation, $message, 'whatsapp', $includeCart);
            }
        }
        
        wp_send_json_success([
            'message_id' => $messageId,
            'sms_sent' => $smsSent,
            'whatsapp_sent' => $whatsAppSent
        ]);
    }

    /**
     * AJAX: Get conversation for admin
     */
    public function ajaxGetConversation(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $convId = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        
        $conversation = $this->getConversation($convId);
        $messages = $this->getMessages($convId);
        
        // Mark visitor messages as read
        $this->markMessagesRead($convId, 'visitor');
        
        wp_send_json_success([
            'conversation' => $conversation,
            'messages' => $messages
        ]);
    }

    /**
     * AJAX: Get all conversations
     */
    public function ajaxGetConversations(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $conversations = $this->getConversations();
        wp_send_json_success(['conversations' => $conversations]);
    }

    /**
     * AJAX: Mark messages as read
     */
    public function ajaxMarkRead(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        $convId = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        $this->markMessagesRead($convId, 'visitor');
        wp_send_json_success();
    }

    /**
     * AJAX: Close conversation
     */
    public function ajaxCloseConversation(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $convId = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        $this->updateConversation($convId, ['status' => 'closed']);
        wp_send_json_success();
    }

    /**
     * AJAX: Save settings
     */
    public function ajaxSaveSettings(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $settings = [
            // Basic
            'cartt_chat_enabled' => isset($_POST['enabled']) && $_POST['enabled'] === '1' ? '1' : '0',
            'cartt_chat_title' => sanitize_text_field($_POST['title'] ?? 'Chat with us'),
            'cartt_chat_greeting' => sanitize_textarea_field($_POST['greeting'] ?? ''),
            'cartt_chat_online_text' => sanitize_text_field($_POST['online_text'] ?? 'Online now'),
            'cartt_chat_position' => in_array($_POST['position'] ?? '', ['left', 'right']) ? $_POST['position'] : 'right',
            'cartt_chat_avatar' => esc_url_raw($_POST['avatar'] ?? ''),
            // Display
            'cartt_chat_show_online' => isset($_POST['show_online']) && $_POST['show_online'] === '1' ? '1' : '0',
            'cartt_chat_sound_enabled' => isset($_POST['sound_enabled']) && $_POST['sound_enabled'] === '1' ? '1' : '0',
            'cartt_chat_screenshots' => isset($_POST['screenshots']) && $_POST['screenshots'] === '1' ? '1' : '0',
            'cartt_chat_attachments' => isset($_POST['attachments']) && $_POST['attachments'] === '1' ? '1' : '0',
            // Colors
            'cartt_chat_color_header' => sanitize_hex_color($_POST['color_header'] ?? '#000000'),
            'cartt_chat_color_header_text' => in_array($_POST['color_header_text'] ?? '', ['auto', '#ffffff', '#000000']) ? $_POST['color_header_text'] : 'auto',
            'cartt_chat_color_visitor_bubble' => sanitize_hex_color($_POST['color_visitor_bubble'] ?? '#000000'),
            'cartt_chat_color_agent_bubble' => sanitize_hex_color($_POST['color_agent_bubble'] ?? '#f0f0f0'),
            'cartt_chat_color_background' => sanitize_hex_color($_POST['color_background'] ?? '#f5f5f5'),
            'cartt_chat_color_input_border' => sanitize_hex_color($_POST['color_input_border'] ?? '#e0e0e0'),
            'cartt_chat_color_input_bg' => sanitize_hex_color($_POST['color_input_bg'] ?? '#fafafa'),
            'cartt_chat_color_input_field' => sanitize_hex_color($_POST['color_input_field'] ?? '#ffffff'),
            'cartt_chat_input_border_width' => max(0, min(5, intval($_POST['input_border_width'] ?? 1))),
            'cartt_chat_color_icons' => sanitize_hex_color($_POST['color_icons'] ?? '#666666'),
            'cartt_chat_color_send_icon' => sanitize_hex_color($_POST['color_send_icon'] ?? '#000000'),
            // Pre-chat
            'cartt_chat_collect_name' => isset($_POST['collect_name']) && $_POST['collect_name'] === '1' ? '1' : '0',
            'cartt_chat_collect_email' => isset($_POST['collect_email']) && $_POST['collect_email'] === '1' ? '1' : '0',
            'cartt_chat_collect_phone' => isset($_POST['collect_phone']) && $_POST['collect_phone'] === '1' ? '1' : '0',
            // Email
            'cartt_chat_notify_email' => isset($_POST['notify_email']) && $_POST['notify_email'] === '1' ? '1' : '0',
            'cartt_chat_email_address' => sanitize_email($_POST['email_address'] ?? ''),
            'cartt_chat_email_include_cart' => isset($_POST['email_include_cart']) && $_POST['email_include_cart'] === '1' ? '1' : '0',
            'cartt_chat_email_frequency' => in_array($_POST['email_frequency'] ?? 'every', ['every', 'first']) ? $_POST['email_frequency'] : 'every',
            // SMS
            'cartt_chat_notify_sms' => isset($_POST['notify_sms']) && $_POST['notify_sms'] === '1' ? '1' : '0',
            'cartt_chat_sms_number' => sanitize_text_field($_POST['sms_number'] ?? ''),
            'cartt_chat_sms_include_cart' => isset($_POST['sms_include_cart']) && $_POST['sms_include_cart'] === '1' ? '1' : '0',
            // WhatsApp
            'cartt_chat_notify_whatsapp' => isset($_POST['notify_whatsapp']) && $_POST['notify_whatsapp'] === '1' ? '1' : '0',
            'cartt_chat_whatsapp_number' => sanitize_text_field($_POST['whatsapp_number'] ?? ''),
            'cartt_chat_whatsapp_include_cart' => isset($_POST['whatsapp_include_cart']) && $_POST['whatsapp_include_cart'] === '1' ? '1' : '0',
            // AI
            'cartt_chat_ai_enabled' => isset($_POST['ai_enabled']) && $_POST['ai_enabled'] === '1' ? '1' : '0',
            'cartt_chat_ai_delay' => max(0, min(300, intval($_POST['ai_delay'] ?? 30))),
            'cartt_chat_ai_store_info' => sanitize_textarea_field($_POST['ai_store_info'] ?? ''),
            // Handoff
            'cartt_chat_handoff_timeout' => max(30, min(300, intval($_POST['handoff_timeout'] ?? 60))),
            'cartt_chat_handoff_message' => sanitize_textarea_field($_POST['handoff_message'] ?? ''),
        ];
        
        foreach ($settings as $key => $value) {
            update_option($key, $value);
        }
        
        wp_send_json_success();
    }

    /**
     * AJAX: Test notification
     */
    public function ajaxTestNotification(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $channel = sanitize_text_field($_POST['channel'] ?? '');
        $to = sanitize_text_field($_POST['to'] ?? '');
        
        if (empty($to)) {
            wp_send_json_error('Please enter a recipient');
        }
        
        $siteName = get_bloginfo('name');
        $testMessage = "This is a test message from {$siteName} Chat.";
        $errorMessage = '';
        
        switch ($channel) {
            case 'email':
                // Check if email is valid
                if (!is_email($to)) {
                    wp_send_json_error('Invalid email address');
                    return;
                }
                $sent = $this->sendEmailViaRelay($to, "[{$siteName}] Test Chat Notification", $testMessage);
                if (!$sent) {
                    $errorMessage = 'Email delivery failed. Please contact support if this persists.';
                }
                break;
                
            case 'sms':
            case 'whatsapp':
                $licenseKey = $this->getLicenseKey();
                if (empty($licenseKey)) {
                    wp_send_json_error('No license key found. Please activate the Chat addon first.');
                    return;
                }
                
                $result = $this->callRelay('send', [
                    'license_key' => $licenseKey,
                    'store_url' => home_url(),
                    'agent_phone' => $to,
                    'visitor_name' => 'Test',
                    'message' => $testMessage,
                    'conversation_id' => 0,
                    'channel' => $channel
                ]);
                
                $sent = !empty($result['success']);
                if (!$sent) {
                    $errorMessage = $result['error'] ?? 'Relay server error. Check your balance and phone number format (+1234567890).';
                }
                break;
                
            default:
                wp_send_json_error('Invalid channel');
                return;
        }
        
        if ($sent) {
            wp_send_json_success(['message' => ucfirst($channel) . ' sent successfully to ' . $to]);
        } else {
            wp_send_json_error($errorMessage ?: 'Failed to send');
        }
    }

    /**
     * AJAX: Get usage stats
     */
    public function ajaxGetUsage(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $result = $this->callRelay('quota', [
            'license_key' => $this->getLicenseKey()
        ]);
        
        if (!empty($result['success'])) {
            wp_send_json_success([
                'plan' => $result['plan'] ?? 'Free',
                'usage' => $result['usage'] ?? 0,
                'quota' => $result['quota'] ?? 0,
                'remaining' => $result['remaining'] ?? 0,
                'overage' => $result['overage'] ?? 0
            ]);
        } else {
            wp_send_json_error('Could not fetch usage');
        }
    }

    /**
     * AJAX: Get balance
     */
    public function ajaxGetBalance(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $result = $this->callRelay('get_balance', [
            'license_key' => $this->getLicenseKey()
        ]);
        
        if (!empty($result['success'])) {
            wp_send_json_success([
                'balance' => $result['balance'] ?? 0
            ]);
        } else {
            wp_send_json_error('Could not fetch balance');
        }
    }

    /**
     * AJAX: Get payment method
     */
    public function ajaxGetPaymentMethod(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $result = $this->callRelay('get_payment_method', [
            'license_key' => $this->getLicenseKey()
        ]);
        
        if (!empty($result['success']) && !empty($result['card'])) {
            wp_send_json_success([
                'card' => $result['card']
            ]);
        } else {
            wp_send_json_success(['card' => null]);
        }
    }

    /**
     * AJAX: Get transactions
     */
    public function ajaxGetTransactions(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $result = $this->callRelay('get_transactions', [
            'license_key' => $this->getLicenseKey(),
            'limit' => 20
        ]);
        
        if (!empty($result['success'])) {
            wp_send_json_success([
                'transactions' => $result['transactions'] ?? []
            ]);
        } else {
            wp_send_json_success(['transactions' => []]);
        }
    }

    /**
     * AJAX: Add funds (creates Stripe checkout session)
     */
    public function ajaxAddFunds(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $amount = floatval($_POST['amount'] ?? 20);
        if ($amount < 5) {
            $amount = 20;
        }
        
        $result = $this->callRelay('create_checkout', [
            'license_key' => $this->getLicenseKey(),
            'amount' => $amount,
            'return_url' => admin_url('admin.php?page=cartt-chat-settings&payment=success'),
            'cancel_url' => admin_url('admin.php?page=cartt-chat-settings&payment=cancelled'),
            'store_name' => get_bloginfo('name'),
            'store_email' => get_option('admin_email')
        ]);
        
        if (!empty($result['success']) && !empty($result['checkout_url'])) {
            wp_send_json_success([
                'checkout_url' => $result['checkout_url']
            ]);
        } else {
            wp_send_json_error($result['error'] ?? 'Could not create checkout session');
        }
    }

    /**
     * AJAX: Get card update URL
     */
    public function ajaxGetCardUpdateUrl(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $result = $this->callRelay('get_card_update_url', [
            'license_key' => $this->getLicenseKey(),
            'return_url' => admin_url('admin.php?page=cartt-chat-settings')
        ]);
        
        if (!empty($result['success']) && !empty($result['url'])) {
            wp_send_json_success([
                'url' => $result['url']
            ]);
        } else {
            wp_send_json_error($result['error'] ?? 'Could not get update link');
        }
    }

    /**
     * AJAX: Set auto-refill
     */
    public function ajaxSetAutoRefill(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $enabled = isset($_POST['enabled']) && $_POST['enabled'] === '1' ? '1' : '0';
        
        // Save locally
        update_option('cartt_chat_auto_refill', $enabled);
        
        // Notify relay
        $this->callRelay('set_auto_refill', [
            'license_key' => $this->getLicenseKey(),
            'enabled' => $enabled
        ]);
        
        wp_send_json_success();
    }

    /**
     * AJAX: Save quick actions
     */
    public function ajaxSaveQuickActions(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $quickActions = isset($_POST['quick_actions']) ? array_map('sanitize_text_field', $_POST['quick_actions']) : [];
        
        // Limit to 20 quick actions
        $quickActions = array_slice($quickActions, 0, 20);
        
        update_option('cartt_chat_quick_actions', $quickActions);
        
        wp_send_json_success(['quick_actions' => $quickActions]);
    }

    /**
     * AJAX: Get quick actions
     */
    public function ajaxGetQuickActions(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $quickActions = get_option('cartt_chat_quick_actions', [
            'Thanks for reaching out!',
            'Let me check on that for you.',
            'Is there anything else I can help with?'
        ]);
        
        wp_send_json_success(['quick_actions' => $quickActions]);
    }

    /**
     * AJAX: Get agents
     */
    public function ajaxGetAgents(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        global $wpdb;
        $agents = $wpdb->get_results("SELECT * FROM {$this->table_agents} ORDER BY created_at DESC");
        
        wp_send_json_success(['agents' => $agents ?: []]);
    }
    
    /**
     * AJAX: Add agent
     */
    public function ajaxAddAgent(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $displayName = sanitize_text_field($_POST['display_name'] ?? '');
        $email = sanitize_email($_POST['email'] ?? '');
        $smsNumber = sanitize_text_field($_POST['sms_number'] ?? '');
        $whatsappNumber = sanitize_text_field($_POST['whatsapp_number'] ?? '');
        $avatar = esc_url_raw($_POST['avatar'] ?? '');
        
        if (empty($displayName)) {
            wp_send_json_error('Name is required');
            return;
        }
        
        global $wpdb;
        
        $result = $wpdb->insert(
            $this->table_agents,
            [
                'user_id' => 0, // No WP user linked
                'display_name' => $displayName,
                'email' => $email,
                'sms_number' => $smsNumber,
                'whatsapp_number' => $whatsappNumber,
                'avatar' => $avatar,
                'is_active' => 1
            ]
        );
        
        if ($result) {
            wp_send_json_success(['id' => $wpdb->insert_id]);
        } else {
            wp_send_json_error('Failed to add agent');
        }
    }
    
    /**
     * AJAX: Update agent
     */
    public function ajaxUpdateAgent(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $id = intval($_POST['id'] ?? 0);
        $isActive = intval($_POST['is_active'] ?? 1);
        
        if (!$id) {
            wp_send_json_error('Invalid agent');
            return;
        }
        
        global $wpdb;
        $wpdb->update(
            $this->table_agents,
            ['is_active' => $isActive],
            ['id' => $id]
        );
        
        wp_send_json_success();
    }
    
    /**
     * AJAX: Remove agent
     */
    public function ajaxRemoveAgent(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $id = intval($_POST['id'] ?? 0);
        
        if (!$id) {
            wp_send_json_error('Invalid agent');
            return;
        }
        
        global $wpdb;
        $wpdb->delete($this->table_agents, ['id' => $id]);
        
        wp_send_json_success();
    }

    /**
     * AJAX: Get unread message count
     */
    public function ajaxGetUnreadCount(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $count = $this->getUnreadCount();
        wp_send_json_success(['count' => $count]);
    }

    /**
     * AJAX: Save agent
     */
    public function ajaxSaveAgent(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $userId = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $smsNumber = isset($_POST['sms_number']) ? sanitize_text_field($_POST['sms_number']) : '';
        $whatsappNumber = isset($_POST['whatsapp_number']) ? sanitize_text_field($_POST['whatsapp_number']) : '';
        
        if (empty($userId)) {
            wp_send_json_error('User ID required');
        }
        
        $this->saveAgent($userId, [
            'sms_number' => $smsNumber,
            'whatsapp_number' => $whatsappNumber,
            'is_active' => 1
        ]);
        
        wp_send_json_success();
    }

    /**
     * AJAX: Delete agent
     */
    public function ajaxDeleteAgent(): void
    {
        check_ajax_referer('cartt_chat_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $userId = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $this->deleteAgent($userId);
        wp_send_json_success();
    }

    /**
     * Handle incoming SMS/WhatsApp reply from relay
     * Agent replies via SMS/WhatsApp get forwarded here
     */
    public function handleRelayReply(\WP_REST_Request $request): \WP_REST_Response
    {
        $params = $request->get_json_params();
        $convId = intval($params['conversation_id'] ?? 0);
        $message = sanitize_textarea_field($params['message'] ?? '');
        $agentPhone = sanitize_text_field($params['agent_phone'] ?? '');
        
        if (empty($convId) || empty($message)) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Missing params'], 400);
        }
        
        // Verify conversation exists
        $conversation = $this->getConversation($convId);
        if (!$conversation) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Not found'], 404);
        }
        
        // Determine sender and channel
        $senderType = 'admin';
        $channel = 'sms';
        
        // Try to find agent by phone
        if (!empty($agentPhone)) {
            $agent = $this->getAgentByPhone($agentPhone);
            if ($agent) {
                $channel = !empty($agent->whatsapp_number) && $agent->whatsapp_number === $agentPhone ? 'whatsapp' : 'sms';
            }
        }
        
        // Add message with channel indicator
        $fullMessage = "[Via {$channel}] {$message}";
        $messageId = $this->addMessage($convId, $senderType, $fullMessage);
        
        return new \WP_REST_Response([
            'success' => true,
            'message_id' => $messageId
        ]);
    }

    /**
     * Get agent by phone number
     */
    private function getAgentByPhone(string $phone): ?object
    {
        global $wpdb;
        
        // Normalize phone (remove non-digits except +)
        $normalized = preg_replace('/[^0-9+]/', '', $phone);
        
        // Try SMS number first
        $agent = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_agents} WHERE REPLACE(REPLACE(sms_number, '-', ''), ' ', '') LIKE %s",
            '%' . $normalized . '%'
        ));
        
        if ($agent) {
            return $agent;
        }
        
        // Try WhatsApp number
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_agents} WHERE REPLACE(REPLACE(whatsapp_number, '-', ''), ' ', '') LIKE %s",
            '%' . $normalized . '%'
        ));
    }

    /**
     * Notify agents of new message
     */
    private function notifyAgents(int $convId, string $message, string $visitorName, $cartData): void
    {
        $licenseKey = $this->getLicenseKey();
        $siteUrl = home_url();
        $siteName = get_bloginfo('name');
        
        // Get conversation to retrieve visitor email
        $conversation = $this->getConversation($convId);
        $visitorEmail = $conversation->visitor_email ?? '';
        
        // Check email frequency setting
        $emailFrequency = get_option('cartt_chat_email_frequency', 'every');
        $shouldSendEmail = true;
        
        if ($emailFrequency === 'first') {
            // Only send email on first visitor message in conversation
            global $wpdb;
            $visitorMessageCount = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_messages} WHERE conversation_id = %d AND sender = 'visitor'",
                $convId
            ));
            // If more than 1 visitor message (the current one already saved), skip email
            $shouldSendEmail = ($visitorMessageCount <= 1);
        }
        
        // Prepare cart text for SMS/WhatsApp
        $cartText = '';
        if (!empty($cartData)) {
            $cart = is_string($cartData) ? json_decode($cartData, true) : $cartData;
            if (!empty($cart['items']) && is_array($cart['items'])) {
                $cartText = " | Cart: ";
                $items = [];
                foreach ($cart['items'] as $item) {
                    $qty = $item['quantity'] ?? $item['qty'] ?? 1;
                    $name = $item['name'] ?? 'Item';
                    $items[] = "{$name} x{$qty}";
                }
                $cartText .= implode(', ', array_slice($items, 0, 3));
                if (count($items) > 3) $cartText .= '...';
            }
        }
        
        // Get all active agents from database
        global $wpdb;
        $agents = $wpdb->get_results("SELECT * FROM {$this->table_agents} WHERE is_active = 1");
        
        $emailsSent = [];
        $smsSent = [];
        $whatsappSent = [];
        
        // Notify each agent
        foreach ($agents as $agent) {
            // Email (respect frequency setting)
            if ($shouldSendEmail && !empty($agent->email) && !in_array($agent->email, $emailsSent)) {
                $subject = "[{$siteName}] New chat from " . ($visitorName ?: 'Visitor');
                
                $this->sendEmailViaRelay($agent->email, $subject, $message, [
                    'visitor_name' => $visitorName ?: 'Visitor',
                    'visitor_email' => $visitorEmail,
                    'conversation_url' => admin_url('admin.php?page=cartt-chat&conv=' . $convId)
                ]);
                $emailsSent[] = $agent->email;
            }
            
            // SMS
            if (!empty($agent->sms_number) && !in_array($agent->sms_number, $smsSent)) {
                $smsBody = ($visitorName ?: 'Visitor') . ": " . substr($message, 0, 100) . $cartText;
                $this->callRelay('send', [
                    'license_key' => $licenseKey,
                    'store_url' => $siteUrl,
                    'agent_phone' => $agent->sms_number,
                    'visitor_name' => $visitorName ?: 'Visitor',
                    'message' => $smsBody,
                    'conversation_id' => $convId,
                    'channel' => 'sms'
                ]);
                $smsSent[] = $agent->sms_number;
            }
            
            // WhatsApp
            if (!empty($agent->whatsapp_number) && !in_array($agent->whatsapp_number, $whatsappSent)) {
                $waBody = "*{$siteName}*\nNew chat from " . ($visitorName ?: 'Visitor') . "\n\n" . substr($message, 0, 200);
                $this->callRelay('send', [
                    'license_key' => $licenseKey,
                    'store_url' => $siteUrl,
                    'agent_phone' => $agent->whatsapp_number,
                    'visitor_name' => $visitorName ?: 'Visitor',
                    'message' => $waBody,
                    'conversation_id' => $convId,
                    'channel' => 'whatsapp'
                ]);
                $whatsappSent[] = $agent->whatsapp_number;
            }
        }
        
        // Fallback: Use settings-based notifications if no agents configured
        if (empty($agents)) {
            // Email notification (respect frequency setting)
            $notifyEmail = get_option('cartt_chat_notify_email', '1') === '1';
            if ($notifyEmail && $shouldSendEmail) {
                $emailAddress = get_option('cartt_chat_email_address', get_option('admin_email'));
                $includeCart = get_option('cartt_chat_email_include_cart', '1') === '1';
                
                $subject = "[{$siteName}] New chat from " . ($visitorName ?: 'Visitor');
                
                $this->sendEmailViaRelay($emailAddress, $subject, $message, [
                    'visitor_name' => $visitorName ?: 'Visitor',
                    'visitor_email' => $visitorEmail,
                    'cart_data' => $includeCart && $cartData ? wp_json_encode($cartData) : '',
                    'conversation_url' => admin_url('admin.php?page=cartt-chat&conv=' . $convId)
                ]);
            }
            
            // SMS notification (from settings) - PRO feature
            $notifySms = get_option('cartt_chat_notify_sms', '0') === '1';
            $smsNumber = get_option('cartt_chat_sms_number', '');
            if ($notifySms && !empty($smsNumber) && $this->hasProFeatures()) {
                $includeCart = get_option('cartt_chat_sms_include_cart', '0') === '1';
                $this->callRelay('send', [
                    'license_key' => $licenseKey,
                    'store_url' => $siteUrl,
                    'agent_phone' => $smsNumber,
                    'visitor_name' => $visitorName ?: 'Visitor',
                    'message' => $message,
                    'cart_data' => $includeCart && $cartData ? wp_json_encode($cartData) : '',
                    'conversation_id' => $convId,
                    'channel' => 'sms'
                ]);
            }
            
            // WhatsApp notification (from settings) - PRO feature
            $notifyWhatsApp = get_option('cartt_chat_notify_whatsapp', '0') === '1';
            $whatsappNumber = get_option('cartt_chat_whatsapp_number', '');
            if ($notifyWhatsApp && !empty($whatsappNumber) && $this->hasProFeatures()) {
                $includeCart = get_option('cartt_chat_whatsapp_include_cart', '0') === '1';
                $this->callRelay('send', [
                    'license_key' => $licenseKey,
                    'store_url' => $siteUrl,
                    'agent_phone' => $whatsappNumber,
                    'visitor_name' => $visitorName ?: 'Visitor',
                    'message' => $message,
                    'cart_data' => $includeCart && $cartData ? wp_json_encode($cartData) : '',
                    'conversation_id' => $convId,
                    'channel' => 'whatsapp'
                ]);
            }
        }
    }

    /**
     * Send reply to visitor via SMS/WhatsApp
     */
    private function sendReplyToVisitor(object $conversation, string $message, string $channel, bool $includeCart): bool
    {
        $licenseKey = $this->getLicenseKey();
        $siteUrl = home_url();
        
        $data = [
            'license_key' => $licenseKey,
            'site_url' => $siteUrl,
            'to_phone' => $conversation->visitor_phone,
            'message' => $message,
            'channel' => $channel
        ];
        
        // If including cart, append it to message
        if ($includeCart && !empty($conversation->cart_data)) {
            $cart = json_decode($conversation->cart_data, true);
            if (!empty($cart['items'])) {
                $cartLines = ["\n--- Their Cart ---"];
                foreach ($cart['items'] as $item) {
                    $cartLines[] = "{$item['qty']}x {$item['name']}";
                }
                $cartLines[] = "Total: " . ($cart['total_formatted'] ?? '');
                $data['message'] .= implode("\n", $cartLines);
            }
        }
        
        $result = $this->callRelay('send_reply', $data);
        return $result['success'] ?? false;
    }

    /**
     * Call relay service
     */
    private function callRelay(string $action, array $data): array
    {
        $response = wp_remote_post($this->relay_url . '?action=' . $action, [
            'body' => $data,
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }
        
        return json_decode(wp_remote_retrieve_body($response), true) ?: ['success' => false];
    }

    /**
     * Send email via relay server (uses SendGrid for reliable delivery)
     */
    private function sendEmailViaRelay(string $toEmail, string $subject, string $message, array $options = []): bool
    {
        $licenseKey = $this->getLicenseKey();
        $storeName = get_bloginfo('name');
        $storeUrl = home_url();
        
        $result = $this->callRelay('send_email', [
            'license_key' => $licenseKey,
            'store_url' => $storeUrl,
            'store_name' => $storeName,
            'to_email' => $toEmail,
            'subject' => $subject,
            'visitor_name' => $options['visitor_name'] ?? '',
            'visitor_email' => $options['visitor_email'] ?? '',
            'message' => $message,
            'cart_data' => $options['cart_data'] ?? '',
            'conversation_url' => $options['conversation_url'] ?? ''
        ]);
        
        return !empty($result['success']);
    }

    // ==================== DATABASE METHODS ====================

    public function getConversations(string $status = 'all', int $limit = 50): array
    {
        global $wpdb;
        
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_conversations}'");
        if (!$tableExists) {
            return [];
        }
        
        $where = $status !== 'all' ? $wpdb->prepare("WHERE c.status = %s", $status) : "";
        
        $sql = "SELECT c.*, 
                (SELECT COUNT(*) FROM {$this->table_messages} m WHERE m.conversation_id = c.id AND m.is_read = 0 AND m.sender = 'visitor') as unread_count,
                (SELECT message FROM {$this->table_messages} m2 WHERE m2.conversation_id = c.id ORDER BY m2.created_at DESC LIMIT 1) as last_message
                FROM {$this->table_conversations} c 
                $where 
                ORDER BY c.updated_at DESC 
                LIMIT $limit";
        
        return $wpdb->get_results($sql) ?: [];
    }

    public function getConversation(int $id): ?object
    {
        global $wpdb;
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_conversations} WHERE id = %d",
            $id
        ));
    }

    public function getConversationByVisitor(string $visitorId): ?object
    {
        global $wpdb;
        
        // Check if table exists
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_conversations}'");
        if (!$tableExists) {
            return null;
        }
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_conversations} WHERE visitor_id = %s AND status = 'active' ORDER BY created_at DESC LIMIT 1",
            $visitorId
        ));
    }

    public function createConversation(array $data): int
    {
        global $wpdb;
        
        // Ensure table exists
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_conversations}'");
        if (!$tableExists) {
            $this->createTables();
        }
        
        $wpdb->insert($this->table_conversations, [
            'visitor_id' => $data['visitor_id'],
            'visitor_name' => $data['visitor_name'] ?? '',
            'visitor_email' => $data['visitor_email'] ?? '',
            'visitor_phone' => $data['visitor_phone'] ?? '',
            'cart_data' => $data['cart_data'] ?? null,
            'page_url' => $data['page_url'] ?? null,
            'ip_address' => $data['ip_address'] ?? null,
            'status' => 'active'
        ]);
        
        return $wpdb->insert_id;
    }
    
    /**
     * Create chat tables if they don't exist
     */
    private function createTables(): void
    {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        $sql1 = "CREATE TABLE IF NOT EXISTS {$this->table_conversations} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            visitor_id VARCHAR(100) NOT NULL,
            visitor_name VARCHAR(255) DEFAULT '',
            visitor_email VARCHAR(255) DEFAULT '',
            visitor_phone VARCHAR(50) DEFAULT '',
            cart_data TEXT,
            page_url TEXT,
            ip_address VARCHAR(45),
            status ENUM('active','closed') DEFAULT 'active',
            handoff_at DATETIME DEFAULT NULL,
            handoff_status VARCHAR(20) DEFAULT 'none',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_visitor (visitor_id),
            INDEX idx_status (status)
        ) $charset_collate;";
        
        $sql2 = "CREATE TABLE IF NOT EXISTS {$this->table_messages} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            conversation_id BIGINT UNSIGNED NOT NULL,
            sender ENUM('visitor','admin') NOT NULL,
            message TEXT NOT NULL,
            is_read TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_conversation (conversation_id),
            INDEX idx_read (is_read)
        ) $charset_collate;";
        
        dbDelta($sql1);
        dbDelta($sql2);
        
        // Add handoff columns if they don't exist (for upgrades)
        $wpdb->query("ALTER TABLE {$this->table_conversations} ADD COLUMN IF NOT EXISTS handoff_at DATETIME DEFAULT NULL");
        $wpdb->query("ALTER TABLE {$this->table_conversations} ADD COLUMN IF NOT EXISTS handoff_status VARCHAR(20) DEFAULT 'none'");
        
        // Create agents table
        $sql3 = "CREATE TABLE IF NOT EXISTS {$this->table_agents} (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT UNSIGNED DEFAULT 0,
            display_name VARCHAR(100) DEFAULT '',
            avatar VARCHAR(500) DEFAULT '',
            email VARCHAR(200) DEFAULT '',
            sms_number VARCHAR(30) DEFAULT '',
            whatsapp_number VARCHAR(30) DEFAULT '',
            is_active TINYINT(1) DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_active (is_active)
        ) $charset_collate;";
        
        dbDelta($sql3);
        
        // Add new columns to agents table if they don't exist (for upgrades)
        $wpdb->query("ALTER TABLE {$this->table_agents} ADD COLUMN IF NOT EXISTS display_name VARCHAR(100) DEFAULT ''");
        $wpdb->query("ALTER TABLE {$this->table_agents} ADD COLUMN IF NOT EXISTS avatar VARCHAR(500) DEFAULT ''");
        $wpdb->query("ALTER TABLE {$this->table_agents} ADD COLUMN IF NOT EXISTS email VARCHAR(200) DEFAULT ''");
    }

    public function updateConversation(int $id, array $data): bool
    {
        global $wpdb;
        return $wpdb->update($this->table_conversations, $data, ['id' => $id]) !== false;
    }

    public function getMessages(int $conversationId): array
    {
        global $wpdb;
        
        // Check if table exists
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_messages}'");
        if (!$tableExists) {
            return [];
        }
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_messages} WHERE conversation_id = %d ORDER BY created_at ASC",
            $conversationId
        )) ?: [];
    }

    public function addMessage(int $conversationId, string $sender, string $message): int
    {
        global $wpdb;
        
        // Ensure table exists
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_messages}'");
        if (!$tableExists) {
            $this->createTables();
        }
        
        $result = $wpdb->insert($this->table_messages, [
            'conversation_id' => $conversationId,
            'sender' => $sender,
            'message' => $message
        ]);
        
        if ($result === false) {
            error_log('Cartt Chat addMessage failed: ' . $wpdb->last_error);
            return 0;
        }
        
        // Update conversation timestamp
        $wpdb->update(
            $this->table_conversations,
            ['updated_at' => current_time('mysql')],
            ['id' => $conversationId]
        );
        
        return $wpdb->insert_id;
    }

    public function markMessagesRead(int $conversationId, string $sender): void
    {
        global $wpdb;
        $wpdb->update(
            $this->table_messages,
            ['is_read' => 1],
            ['conversation_id' => $conversationId, 'sender' => $sender, 'is_read' => 0]
        );
    }

    public function getUnreadCount(): int
    {
        global $wpdb;
        
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_conversations}'");
        if (!$tableExists) {
            return 0;
        }
        
        return (int) $wpdb->get_var("
            SELECT COUNT(DISTINCT c.id) 
            FROM {$this->table_conversations} c 
            INNER JOIN {$this->table_messages} m ON m.conversation_id = c.id 
            WHERE m.sender = 'visitor' AND m.is_read = 0
        ");
    }

    public function getAgents(): array
    {
        global $wpdb;
        
        $tableExists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_agents}'");
        if (!$tableExists) {
            return [];
        }
        
        return $wpdb->get_results("SELECT * FROM {$this->table_agents} WHERE is_active = 1") ?: [];
    }

    public function saveAgent(int $userId, array $data): bool
    {
        global $wpdb;
        
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$this->table_agents} WHERE user_id = %d",
            $userId
        ));
        
        if ($existing) {
            return $wpdb->update($this->table_agents, $data, ['user_id' => $userId]) !== false;
        } else {
            $data['user_id'] = $userId;
            return $wpdb->insert($this->table_agents, $data) !== false;
        }
    }

    public function deleteAgent(int $userId): bool
    {
        global $wpdb;
        return $wpdb->delete($this->table_agents, ['user_id' => $userId]) !== false;
    }

    /**
     * Render admin inbox
     */
    public function renderInbox(): void
    {
        $conversations = $this->getConversations();
        $agents = $this->getAgents();
        include CARTT_PLUGIN_DIR . 'admin/views/chat.php';
    }

    /**
     * Render settings page
     */
    public function renderSettings(): void
    {
        $agents = $this->getAgents();
        $admins = get_users(['role' => 'administrator']);
        include CARTT_PLUGIN_DIR . 'admin/views/chat-settings.php';
    }

    /**
     * Render chat widget via shortcode
     * Use [cartt_chat] in Elementor or page builders where wp_footer doesn't work
     */
    public function renderShortcode(): string
    {
        static $rendered = false;
        
        if ($rendered || is_admin()) {
            return '';
        }
        
        $settings = $this->getWidgetSettings();
        
        if ($settings['enabled'] !== '1') {
            return '';
        }
        
        $rendered = true;
        
        $visitorId = $this->getVisitorId();
        $show_prechat = ($settings['collect_name'] === '1' || $settings['collect_email'] === '1' || $settings['collect_phone'] === '1');
        $nonce = wp_create_nonce('cartt_chat');
        $ajax_url = admin_url('admin-ajax.php');
        
        ob_start();
        
        // Calculate header text color
        $headerTextColor = $settings['color_header_text'];
        if ($headerTextColor === 'auto') {
            // Simple luminance check
            $hex = ltrim($settings['color_header'], '#');
            $r = hexdec(substr($hex, 0, 2));
            $g = hexdec(substr($hex, 2, 2));
            $b = hexdec(substr($hex, 4, 2));
            $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
            $headerTextColor = $luminance > 0.5 ? '#000000' : '#ffffff';
        }
        
        // Visitor bubble text color (same logic)
        $visitorHex = ltrim($settings['color_visitor_bubble'], '#');
        $vr = hexdec(substr($visitorHex, 0, 2));
        $vg = hexdec(substr($visitorHex, 2, 2));
        $vb = hexdec(substr($visitorHex, 4, 2));
        $visitorTextColor = (0.299 * $vr + 0.587 * $vg + 0.114 * $vb) / 255 > 0.5 ? '#000000' : '#ffffff';
        
        // Agent bubble text color
        $agentHex = ltrim($settings['color_agent_bubble'], '#');
        $ar = hexdec(substr($agentHex, 0, 2));
        $ag = hexdec(substr($agentHex, 2, 2));
        $ab = hexdec(substr($agentHex, 4, 2));
        $agentTextColor = (0.299 * $ar + 0.587 * $ag + 0.114 * $ab) / 255 > 0.5 ? '#333333' : '#ffffff';
        
        $borderWidth = intval($settings['input_border_width']);
        ?>
        <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
        <style>
        #cartt-chat{position:fixed!important;bottom:24px!important;right:24px!important;z-index:999999!important;font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif!important}
        #cartt-chat[data-position="left"]{right:auto!important;left:24px!important}
        #cartt-chat *{box-sizing:border-box!important}
        .cartt-chat-btn{width:60px!important;height:60px!important;border-radius:50%!important;background:<?php echo esc_attr($settings['color_header']); ?>!important;cursor:pointer!important;display:flex!important;align-items:center!important;justify-content:center!important;box-shadow:0 4px 20px rgba(0,0,0,.2)!important;transition:transform .2s,box-shadow .2s!important;border:none!important;position:relative!important}
        .cartt-chat-btn:hover{transform:scale(1.08)!important;box-shadow:0 6px 28px rgba(0,0,0,.25)!important}
        .cartt-chat-btn .material-icons{font-size:28px!important;color:<?php echo esc_attr($headerTextColor); ?>!important}
        .cartt-icon-close{display:none!important}
        #cartt-chat.open .cartt-icon-open{display:none!important}
        #cartt-chat.open .cartt-icon-close{display:block!important}
        .cartt-badge{position:absolute!important;top:-4px!important;right:-4px!important;background:#e53e3e!important;color:#fff!important;font-size:11px!important;font-weight:600!important;min-width:20px!important;height:20px!important;border-radius:10px!important;display:none!important;align-items:center!important;justify-content:center!important;padding:0 6px!important}
        @keyframes cartt-pulse{0%,100%{transform:scale(1)}50%{transform:scale(1.1)}}
        @keyframes cartt-slideIn{from{opacity:0;transform:translateY(10px)}to{opacity:1;transform:translateY(0)}}
        @keyframes cartt-dot{0%,80%,100%{transform:scale(0)}40%{transform:scale(1)}}
        .cartt-chat-box{position:absolute!important;bottom:76px!important;right:0!important;width:380px!important;max-height:calc(100vh - 120px)!important;background:#fff!important;border-radius:8px!important;box-shadow:0 8px 40px rgba(0,0,0,.15)!important;display:none!important;flex-direction:column!important;overflow:hidden!important;animation:cartt-slideIn .25s ease-out!important}
        #cartt-chat[data-position="left"] .cartt-chat-box{right:auto!important;left:0!important}
        #cartt-chat.open .cartt-chat-box{display:flex!important}
        .cartt-chat-header{padding:20px!important;background:<?php echo esc_attr($settings['color_header']); ?>!important;display:flex!important;justify-content:space-between!important;align-items:center!important}
        .cartt-header-info{display:flex!important;align-items:center!important;gap:14px!important}
        .cartt-avatar{width:44px!important;height:44px!important;border-radius:50%!important;background:rgba(255,255,255,.15)!important;display:flex!important;align-items:center!important;justify-content:center!important;flex-shrink:0!important;overflow:hidden!important}
        .cartt-avatar img{width:100%!important;height:100%!important;object-fit:cover!important}
        .cartt-avatar .material-icons{font-size:24px!important;color:<?php echo esc_attr($headerTextColor); ?>!important}
        .cartt-header-text{display:flex!important;flex-direction:column!important;gap:2px!important}
        .cartt-title{font-size:16px!important;font-weight:600!important;color:<?php echo esc_attr($headerTextColor); ?>!important;margin:0!important}
        .cartt-status{font-size:13px!important;color:<?php echo esc_attr($headerTextColor); ?>!important;opacity:0.8!important;display:flex!important;align-items:center!important;gap:6px!important}
        .cartt-online-dot{width:8px!important;height:8px!important;background:#4ade80!important;border-radius:50%!important;animation:cartt-pulse 2s infinite!important}
        .cartt-close{background:rgba(255,255,255,.15)!important;border:none!important;cursor:pointer!important;padding:8px!important;border-radius:50%!important;transition:background .2s!important}
        .cartt-close:hover{background:rgba(255,255,255,.25)!important}
        .cartt-close .material-icons{font-size:18px!important;color:<?php echo esc_attr($headerTextColor); ?>!important}
        .cartt-messages{flex:1!important;min-height:200px!important;max-height:380px!important;overflow-y:auto!important;padding:20px!important;display:flex!important;flex-direction:column!important;gap:12px!important;background:<?php echo esc_attr($settings['color_background']); ?>!important}
        .cartt-greeting{padding:14px 18px!important;background:<?php echo esc_attr($settings['color_agent_bubble']); ?>!important;color:<?php echo esc_attr($agentTextColor); ?>!important;border-radius:16px!important;border-bottom-left-radius:4px!important;font-size:14px!important;line-height:1.5!important;box-shadow:0 2px 8px rgba(0,0,0,.06)!important;align-self:flex-start!important;max-width:85%!important}
        .cartt-msg{padding:12px 16px!important;border-radius:16px!important;font-size:14px!important;line-height:1.5!important;max-width:85%!important;word-wrap:break-word!important;animation:cartt-slideIn .3s ease-out!important}
        .cartt-msg.visitor{align-self:flex-end!important;background:<?php echo esc_attr($settings['color_visitor_bubble']); ?>!important;color:<?php echo esc_attr($visitorTextColor); ?>!important;border-bottom-right-radius:4px!important}
        .cartt-msg.admin{align-self:flex-start!important;background:<?php echo esc_attr($settings['color_agent_bubble']); ?>!important;color:<?php echo esc_attr($agentTextColor); ?>!important;border-bottom-left-radius:4px!important;box-shadow:0 2px 8px rgba(0,0,0,.06)!important}
        .cartt-msg-image{padding:0!important;background:transparent!important;box-shadow:none!important}
        .cartt-msg-image img{max-width:200px!important;border-radius:8px!important;cursor:pointer!important;display:block!important}
        .cartt-typing{display:flex!important;gap:4px!important;padding:12px 16px!important;background:<?php echo esc_attr($settings['color_agent_bubble']); ?>!important;border-radius:16px!important;border-bottom-left-radius:4px!important;align-self:flex-start!important;box-shadow:0 2px 8px rgba(0,0,0,.06)!important}
        .cartt-typing span{width:8px!important;height:8px!important;background:#999!important;border-radius:50%!important;animation:cartt-dot 1.4s infinite ease-in-out!important}
        .cartt-typing span:nth-child(1){animation-delay:-.32s!important}
        .cartt-typing span:nth-child(2){animation-delay:-.16s!important}
        .cartt-prechat-form{padding:20px!important;background:#fff!important;flex-direction:column!important;gap:12px!important;border-top:1px solid <?php echo esc_attr($settings['color_input_border']); ?>!important}
        .cartt-prechat-input{padding:14px 16px!important;border:<?php echo $borderWidth; ?>px solid <?php echo esc_attr($settings['color_input_border']); ?>!important;border-radius:8px!important;font-size:14px!important;outline:none!important;width:100%!important;transition:border-color .2s!important;background:<?php echo esc_attr($settings['color_input_field']); ?>!important}
        .cartt-prechat-input:focus{border-color:<?php echo esc_attr($settings['color_header']); ?>!important}
        .cartt-start-btn{padding:14px 24px!important;border:none!important;border-radius:8px!important;font-size:14px!important;font-weight:600!important;cursor:pointer!important;background:<?php echo esc_attr($settings['color_header']); ?>!important;color:<?php echo esc_attr($headerTextColor); ?>!important;transition:opacity .2s!important}
        .cartt-start-btn:hover{opacity:0.9!important}
        .cartt-input-wrap{padding:12px 16px 16px!important;display:flex!important;gap:12px!important;align-items:center!important;background:<?php echo esc_attr($settings['color_input_bg']); ?>!important;border-top:1px solid <?php echo esc_attr($settings['color_input_border']); ?>!important}
        #cartt-input-container{flex:1!important;position:relative!important;display:flex!important;align-items:center!important}
        #cartt-input{width:100%!important;padding:12px 80px 12px 16px!important;border:<?php echo $borderWidth; ?>px solid <?php echo esc_attr($settings['color_input_border']); ?>!important;border-radius:8px!important;font-size:14px!important;background:<?php echo esc_attr($settings['color_input_field']); ?>!important;height:46px!important;outline:none!important}
        #cartt-input:focus{border-color:<?php echo esc_attr($settings['color_header']); ?>!important}
        #cartt-input::placeholder{color:#999!important}
        .cartt-screenshot,.cartt-attach{position:absolute!important;top:50%!important;transform:translateY(-50%)!important;width:28px!important;height:28px!important;border:none!important;cursor:pointer!important;display:flex!important;align-items:center!important;justify-content:center!important;background:transparent!important;padding:0!important}
        .cartt-screenshot{right:40px!important}
        .cartt-attach{right:8px!important}
        .cartt-screenshot:hover,.cartt-attach:hover{opacity:.7!important}
        .cartt-screenshot .material-icons,.cartt-attach .material-icons{font-size:20px!important;color:<?php echo esc_attr($settings['color_icons']); ?>!important}
        #cartt-attach-input{display:none!important}
        .cartt-send{border:none!important;cursor:pointer!important;display:flex!important;align-items:center!important;justify-content:center!important;background:transparent!important;padding:0!important}
        .cartt-send:hover{opacity:.8!important}
        .cartt-send .material-icons{font-size:28px!important;color:<?php echo esc_attr($settings['color_send_icon']); ?>!important}
        #cartt-screenshot-preview,#cartt-file-preview{padding:12px 16px!important;background:#f0f0f0!important;border-bottom:1px solid <?php echo esc_attr($settings['color_input_border']); ?>!important;display:none}
        .cartt-preview-inner{position:relative!important;display:inline-block!important}
        .cartt-preview-inner img{max-width:120px!important;max-height:80px!important;border-radius:8px!important;display:block!important}
        .cartt-preview-remove,.cartt-file-remove{position:absolute!important;top:4px!important;right:4px!important;width:24px!important;height:24px!important;background:rgba(0,0,0,0.5)!important;color:#fff!important;border:none!important;border-radius:4px!important;cursor:pointer!important;font-size:16px!important;display:flex!important;align-items:center!important;justify-content:center!important;line-height:1!important}
        .cartt-preview-remove:hover,.cartt-file-remove:hover{background:rgba(0,0,0,0.7)!important}
        .cartt-file-inner{display:flex!important;align-items:center!important;gap:10px!important;background:#fff!important;padding:8px 12px!important;border-radius:8px!important;border:1px solid #e0e0e0!important;position:relative!important}
        .cartt-file-inner .material-icons{font-size:20px!important;color:#666!important}
        .cartt-file-name{font-size:13px!important;color:#333!important;flex:1!important;overflow:hidden!important;text-overflow:ellipsis!important;white-space:nowrap!important;max-width:200px!important}
        .cartt-file-inner .cartt-file-remove{position:relative!important;top:0!important;right:0!important}
        .cartt-branding{padding:6px 10px!important;background:#fff!important;text-align:left!important;border-radius:0 6px 0 6px!important}
        .cartt-branding a{color:#999!important;font-size:10px!important;text-decoration:none!important}
        .cartt-branding a:hover{color:#666!important}
        .cartt-branding strong{font-weight:600!important}
        #cartt-screenshot-overlay{position:fixed!important;top:0!important;left:0!important;width:100%!important;height:100%!important;background:rgba(0,0,0,.3)!important;z-index:9999999!important;cursor:crosshair!important}
        .cartt-screenshot-hint{position:absolute!important;top:20px!important;left:50%!important;transform:translateX(-50%)!important;background:#000!important;color:#fff!important;padding:10px 20px!important;border-radius:8px!important;font-size:14px!important}
        .cartt-selection-box{position:absolute!important;border:2px dashed #fff!important;background:rgba(255,255,255,.1)!important;pointer-events:none!important}
        .cartt-toast{position:fixed!important;bottom:100px!important;left:50%!important;transform:translateX(-50%)!important;background:#333!important;color:#fff!important;padding:12px 24px!important;border-radius:8px!important;font-size:14px!important;z-index:99999999!important}
        @media(max-width:420px){
            #cartt-chat{bottom:16px!important;right:16px!important}
            #cartt-chat[data-position="left"]{left:16px!important}
            .cartt-chat-box{width:calc(100vw - 32px)!important;bottom:72px!important;max-height:calc(100vh - 100px)!important}
            .cartt-messages{max-height:none!important;flex:1!important}
        }
        </style>
        
        <div id="cartt-chat" data-position="<?php echo esc_attr($settings['position']); ?>">
            <div class="cartt-chat-btn" onclick="CarttChat.toggle()">
                <span class="material-icons cartt-icon-open">chat</span>
                <span class="material-icons cartt-icon-close">close</span>
                <span class="cartt-badge">0</span>
            </div>
            <div class="cartt-chat-box">
                <div class="cartt-chat-header">
                    <div class="cartt-header-info">
                        <div class="cartt-avatar">
                            <?php if (!empty($settings['avatar'])) : ?>
                            <img src="<?php echo esc_url($settings['avatar']); ?>" alt="">
                            <?php else : ?>
                            <span class="material-icons">support_agent</span>
                            <?php endif; ?>
                        </div>
                        <div class="cartt-header-text">
                            <div class="cartt-title"><?php echo esc_html($settings['title']); ?></div>
                            <?php if ($settings['show_online'] === '1') : ?>
                            <div class="cartt-status"><span class="cartt-online-dot"></span> <?php echo esc_html($settings['online_text']); ?></div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <button class="cartt-close" onclick="CarttChat.toggle()">
                        <span class="material-icons">close</span>
                    </button>
                </div>
                <div class="cartt-messages" id="cartt-messages">
                    <div class="cartt-greeting"><?php echo esc_html($settings['greeting']); ?></div>
                </div>
                <div id="cartt-screenshot-preview"></div>
                <div id="cartt-file-preview"></div>
                <div class="cartt-prechat-form" id="cartt-prechat" style="<?php echo $show_prechat ? 'display:flex' : 'display:none'; ?>">
                    <?php if ($settings['collect_name'] === '1') : ?>
                    <input type="text" id="cartt-name" placeholder="Your name" class="cartt-prechat-input">
                    <?php endif; ?>
                    <?php if ($settings['collect_email'] === '1') : ?>
                    <input type="email" id="cartt-email" placeholder="Your email" class="cartt-prechat-input">
                    <?php endif; ?>
                    <?php if ($settings['collect_phone'] === '1') : ?>
                    <input type="tel" id="cartt-phone" placeholder="Your phone" class="cartt-prechat-input">
                    <?php endif; ?>
                    <button class="cartt-start-btn" onclick="CarttChat.startChat()">Start Chat</button>
                </div>
                <div class="cartt-input-wrap" id="cartt-input-wrap" style="<?php echo $show_prechat ? 'display:none' : 'display:flex'; ?>">
                    <div id="cartt-input-container">
                        <input type="text" id="cartt-input" placeholder="Type your message..." onkeypress="if(event.key==='Enter')CarttChat.send()">
                        <?php if ($settings['screenshots'] === '1' && $this->hasProFeatures()) : ?>
                        <button class="cartt-screenshot" onclick="CarttChat.screenshot()" title="Capture screen area">
                            <span class="material-icons">screenshot_monitor</span>
                        </button>
                        <?php endif; ?>
                        <?php if ($settings['attachments'] === '1' && $this->hasProFeatures()) : ?>
                        <button class="cartt-attach" onclick="document.getElementById('cartt-attach-input').click()" title="Attach file">
                            <span class="material-icons">attach_file</span>
                        </button>
                        <input type="file" id="cartt-attach-input" accept="image/*,.pdf,.doc,.docx,.txt" onchange="CarttChat.handleFile(this)">
                        <?php endif; ?>
                    </div>
                    <button class="cartt-send" onclick="CarttChat.send()">
                        <span class="material-icons">send</span>
                    </button>
                </div>
                <?php if (strpos(home_url(), 'cartt.app') === false): ?>
                <div class="cartt-branding">
                    <a href="https://cartt.app" target="_blank">Powered by <strong>Cartt</strong></a>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <script>
        var CarttChat = {
            isOpen: false,
            visitorId: '<?php echo esc_js($visitorId); ?>',
            lastMessageId: 0,
            pollInterval: null,
            visitorName: '',
            visitorEmail: '',
            visitorPhone: '',
            chatStarted: <?php echo $show_prechat ? 'false' : 'true'; ?>,
            pendingScreenshot: null,
            pendingFile: null,
            pendingFileName: null,
            isSending: false,
            ajaxurl: '<?php echo esc_js($ajax_url); ?>',
            nonce: '<?php echo esc_js($nonce); ?>',
            
            init: function() {
                this.loadVisitorInfo();
                this.loadMessages();
                this.startPolling();
            },
            
            loadVisitorInfo: function() {
                this.visitorName = localStorage.getItem('cartt_visitor_name') || '';
                this.visitorEmail = localStorage.getItem('cartt_visitor_email') || '';
                this.visitorPhone = localStorage.getItem('cartt_visitor_phone') || '';
                if (this.visitorName || this.visitorEmail || this.visitorPhone) {
                    this.chatStarted = true;
                    var prechat = document.getElementById('cartt-prechat');
                    var inputWrap = document.getElementById('cartt-input-wrap');
                    if (prechat) prechat.style.display = 'none';
                    if (inputWrap) inputWrap.style.display = 'flex';
                }
            },
            
            startChat: function() {
                var nameInput = document.getElementById('cartt-name');
                var emailInput = document.getElementById('cartt-email');
                var phoneInput = document.getElementById('cartt-phone');
                if (nameInput) { this.visitorName = nameInput.value.trim(); localStorage.setItem('cartt_visitor_name', this.visitorName); }
                if (emailInput) { this.visitorEmail = emailInput.value.trim(); localStorage.setItem('cartt_visitor_email', this.visitorEmail); }
                if (phoneInput) { this.visitorPhone = phoneInput.value.trim(); localStorage.setItem('cartt_visitor_phone', this.visitorPhone); }
                this.chatStarted = true;
                document.getElementById('cartt-prechat').style.display = 'none';
                document.getElementById('cartt-input-wrap').style.display = 'flex';
                document.getElementById('cartt-input').focus();
            },
            
            toggle: function() {
                var widget = document.getElementById('cartt-chat');
                this.isOpen = !this.isOpen;
                widget.classList.toggle('open', this.isOpen);
                if (this.isOpen) { var input = document.getElementById('cartt-input'); if (input) input.focus(); this.hideBadge(); }
            },
            
            send: function() {
                var input = document.getElementById('cartt-input');
                var message = input.value.trim();
                if (this.pendingFile) { this.sendFile(this.pendingFile, this.pendingFileName); this.removeFile(); if (message) { input.value = ''; this.appendMessage(message, 'visitor'); this.sendTextMessage(message); } return; }
                if (this.pendingScreenshot) { this.sendScreenshot(this.pendingScreenshot); this.removeScreenshot(); if (message) { input.value = ''; this.appendMessage(message, 'visitor'); this.sendTextMessage(message); } return; }
                if (!message) return;
                input.value = '';
                this.appendMessage(message, 'visitor');
                this.showTyping();
                this.sendTextMessage(message);
            },
            
            sendTextMessage: function(message) {
                var self = this;
                this.isSending = true;
                var data = new FormData();
                data.append('action', 'cartt_chat_send');
                data.append('nonce', this.nonce);
                data.append('message', message);
                data.append('visitor_id', this.visitorId);
                data.append('name', this.visitorName);
                data.append('email', this.visitorEmail);
                data.append('phone', this.visitorPhone);
                data.append('page_url', window.location.href);
                fetch(this.ajaxurl, {method: 'POST', body: data}).then(function(r) { return r.json(); }).then(function(response) {
                    self.hideTyping();
                    self.isSending = false;
                    if (response.success && response.data) {
                        self.conversationId = response.data.conversation_id;
                        self.lastMessageId = response.data.message_id || self.lastMessageId;
                        if (response.data.ai_reply) {
                            self.appendMessage(response.data.ai_reply, 'admin');
                            if (response.data.ai_message_id) { self.lastMessageId = response.data.ai_message_id; }
                        }
                        if (response.data.needs_handoff) {
                            self.startHandoffCheck();
                        }
                    }
                }).catch(function() { self.hideTyping(); self.isSending = false; });
            },
            
            handoffCheckInterval: null,
            startHandoffCheck: function() {
                var self = this;
                if (this.handoffCheckInterval) clearInterval(this.handoffCheckInterval);
                var checkCount = 0;
                this.handoffCheckInterval = setInterval(function() {
                    checkCount++;
                    if (checkCount > 20) { clearInterval(self.handoffCheckInterval); return; }
                    var data = new FormData();
                    data.append('action', 'cartt_chat_check_handoff');
                    data.append('nonce', self.nonce);
                    data.append('conversation_id', self.conversationId);
                    fetch(self.ajaxurl, {method: 'POST', body: data}).then(function(r) { return r.json(); }).then(function(response) {
                        if (response.success && response.data) {
                            if (response.data.human_responded || response.data.needs_followup) {
                                clearInterval(self.handoffCheckInterval);
                                if (response.data.needs_followup && response.data.followup_message) {
                                    self.appendMessage(response.data.followup_message, 'admin');
                                }
                            }
                        }
                    }).catch(function() {});
                }, 5000);
            },
            
            screenshot: function() {
                var self = this;
                var widget = document.getElementById('cartt-chat');
                widget.style.display = 'none';
                var overlay = document.createElement('div');
                overlay.id = 'cartt-screenshot-overlay';
                overlay.innerHTML = '<div class="cartt-screenshot-hint">Click and drag to select area. Press ESC to cancel.</div>';
                document.body.appendChild(overlay);
                var startX, startY, selectionBox, isSelecting = false;
                overlay.onmousedown = function(e) { isSelecting = true; startX = e.clientX; startY = e.clientY; selectionBox = document.createElement('div'); selectionBox.className = 'cartt-selection-box'; selectionBox.style.left = startX + 'px'; selectionBox.style.top = startY + 'px'; overlay.appendChild(selectionBox); };
                overlay.onmousemove = function(e) { if (!isSelecting || !selectionBox) return; var width = e.clientX - startX; var height = e.clientY - startY; selectionBox.style.width = Math.abs(width) + 'px'; selectionBox.style.height = Math.abs(height) + 'px'; selectionBox.style.left = (width < 0 ? e.clientX : startX) + 'px'; selectionBox.style.top = (height < 0 ? e.clientY : startY) + 'px'; };
                overlay.onmouseup = function(e) { if (!isSelecting) return; isSelecting = false; var rect = selectionBox.getBoundingClientRect(); overlay.remove(); if (rect.width < 10 || rect.height < 10) { widget.style.display = ''; return; } self.captureArea(rect, function(dataUrl) { widget.style.display = ''; if (dataUrl) self.showScreenshotPreview(dataUrl); }); };
                var escHandler = function(e) { if (e.key === 'Escape') { overlay.remove(); widget.style.display = ''; document.removeEventListener('keydown', escHandler); } };
                document.addEventListener('keydown', escHandler);
            },
            
            captureArea: function(rect, callback) {
                var self = this;
                var doCapture = function() { html2canvas(document.body, { x: rect.left + window.scrollX, y: rect.top + window.scrollY, width: rect.width, height: rect.height, useCORS: true, logging: false, scale: 1 }).then(function(canvas) { callback(canvas.toDataURL('image/png')); }).catch(function() { self.showToast('Screenshot capture failed'); callback(null); }); };
                if (typeof html2canvas !== 'undefined') { doCapture(); } else { var script = document.createElement('script'); script.src = 'https://unpkg.com/html2canvas@1.4.1/dist/html2canvas.min.js'; script.onload = doCapture; script.onerror = function() { self.showToast('Could not load screenshot library'); callback(null); }; document.head.appendChild(script); }
            },
            
            showScreenshotPreview: function(dataUrl) { this.pendingScreenshot = dataUrl; var preview = document.getElementById('cartt-screenshot-preview'); preview.innerHTML = '<div class="cartt-preview-inner"><img src="' + dataUrl + '"><button class="cartt-preview-remove" onclick="CarttChat.removeScreenshot()">x</button></div>'; preview.style.display = 'block'; },
            removeScreenshot: function() { this.pendingScreenshot = null; var preview = document.getElementById('cartt-screenshot-preview'); preview.style.display = 'none'; preview.innerHTML = ''; },
            sendScreenshot: function(dataUrl) { var self = this; this.isSending = true; this.appendImage(dataUrl, 'visitor'); var data = new FormData(); data.append('action', 'cartt_chat_send_screenshot'); data.append('nonce', this.nonce); data.append('visitor_id', this.visitorId); data.append('visitor_name', this.visitorName); data.append('image', dataUrl); fetch(this.ajaxurl, {method: 'POST', body: data}).then(function(r) { return r.json(); }).then(function(response) { self.isSending = false; if (response.success && response.data) { self.lastMessageId = response.data.message_id || self.lastMessageId; } }).catch(function() { self.isSending = false; }); },
            
            handleFile: function(input) { var self = this; var file = input.files[0]; if (!file) return; if (file.size > 5 * 1024 * 1024) { this.showToast('File too large. Maximum size is 5MB.'); input.value = ''; return; } var reader = new FileReader(); reader.onload = function(e) { self.pendingFile = e.target.result; self.pendingFileName = file.name; self.showFilePreview(file.name); }; reader.readAsDataURL(file); input.value = ''; },
            showFilePreview: function(fileName) { var preview = document.getElementById('cartt-file-preview'); preview.innerHTML = '<div class="cartt-file-inner"><span class="material-icons">description</span><span class="cartt-file-name">' + this.escapeHtml(fileName) + '</span><button class="cartt-file-remove" onclick="CarttChat.removeFile()">x</button></div>'; preview.style.display = 'block'; },
            removeFile: function() { this.pendingFile = null; this.pendingFileName = null; var preview = document.getElementById('cartt-file-preview'); preview.style.display = 'none'; preview.innerHTML = ''; },
            sendFile: function(dataUrl, fileName) { var self = this; this.isSending = true; var shortName = fileName.length > 30 ? fileName.substring(0, 27) + '...' : fileName; this.appendMessage('File sent: ' + shortName, 'visitor'); var data = new FormData(); data.append('action', 'cartt_chat_send_file'); data.append('nonce', this.nonce); data.append('visitor_id', this.visitorId); data.append('visitor_name', this.visitorName); data.append('file', dataUrl); data.append('filename', fileName); fetch(this.ajaxurl, {method: 'POST', body: data}).then(function(r) { return r.json(); }).then(function(response) { self.isSending = false; if (response.success && response.data) { self.lastMessageId = response.data.message_id || self.lastMessageId; } }).catch(function() { self.isSending = false; }); },
            
            loadMessages: function() { var self = this; var data = new FormData(); data.append('action', 'cartt_chat_get_messages'); data.append('nonce', this.nonce); data.append('visitor_id', this.visitorId); fetch(this.ajaxurl, {method: 'POST', body: data}).then(function(r) { return r.json(); }).then(function(response) { if (response.success && response.data && response.data.messages && response.data.messages.length > 0) { var container = document.getElementById('cartt-messages'); var greeting = container.querySelector('.cartt-greeting'); container.innerHTML = ''; if (greeting) container.appendChild(greeting); response.data.messages.forEach(function(msg) { self.appendMessage(msg.message, msg.sender_type, false); self.lastMessageId = Math.max(self.lastMessageId, parseInt(msg.id)); }); } }); },
            startPolling: function() { var self = this; this.pollInterval = setInterval(function() { self.pollMessages(); }, 3000); },
            pollMessages: function() { if (this.isSending) return; var self = this; var data = new FormData(); data.append('action', 'cartt_chat_poll_messages'); data.append('nonce', this.nonce); data.append('visitor_id', this.visitorId); data.append('last_id', this.lastMessageId); fetch(this.ajaxurl, {method: 'POST', body: data}).then(function(r) { return r.json(); }).then(function(response) { if (response.success && response.data && response.data.messages) { var hasNewAdmin = false; response.data.messages.forEach(function(msg) { if (parseInt(msg.id) > self.lastMessageId) { self.appendMessage(msg.message, msg.sender_type, true); self.lastMessageId = parseInt(msg.id); if (msg.sender_type === 'admin') hasNewAdmin = true; } }); if (hasNewAdmin && !self.isOpen) { self.showBadge(); self.playSound(); } } }); },
            
            appendMessage: function(text, sender, scroll) { var container = document.getElementById('cartt-messages'); var div = document.createElement('div'); div.className = 'cartt-msg ' + sender; var imageMatch = text.match(/\[image\](.*?)\[\/image\]/); if (imageMatch) { div.className += ' cartt-msg-image'; div.innerHTML = '<img src="' + imageMatch[1] + '" onclick="window.open(this.src)">'; } else { var fileMatch = text.match(/\[file\](.*?)\|(.*?)\[\/file\]/); if (fileMatch) { div.innerHTML = '<a href="' + fileMatch[1] + '" target="_blank" style="display:flex;align-items:center;gap:8px;color:inherit;text-decoration:none;"><span class="material-icons" style="font-size:20px;">description</span>' + this.escapeHtml(fileMatch[2]) + '</a>'; } else { div.textContent = text; } } container.appendChild(div); if (scroll !== false) container.scrollTop = container.scrollHeight; },
            appendImage: function(src, sender) { var container = document.getElementById('cartt-messages'); var div = document.createElement('div'); div.className = 'cartt-msg cartt-msg-image ' + sender; div.innerHTML = '<img src="' + src + '" onclick="window.open(this.src)">'; container.appendChild(div); container.scrollTop = container.scrollHeight; },
            showTyping: function() { var container = document.getElementById('cartt-messages'); if (!document.getElementById('cartt-typing')) { var typing = document.createElement('div'); typing.id = 'cartt-typing'; typing.className = 'cartt-typing'; typing.innerHTML = '<span></span><span></span><span></span>'; container.appendChild(typing); container.scrollTop = container.scrollHeight; } },
            hideTyping: function() { var typing = document.getElementById('cartt-typing'); if (typing) typing.remove(); },
            showBadge: function() { var badge = document.querySelector('.cartt-badge'); var count = parseInt(badge.textContent) || 0; badge.textContent = count + 1; badge.style.display = 'flex'; },
            hideBadge: function() { var badge = document.querySelector('.cartt-badge'); badge.textContent = '0'; badge.style.display = 'none'; },
            playSound: function() { try { var audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdH2Onp+dmZaUmKGrraqloJyboqy4vby4sKylqLO+xsfCurKrqay3xMzNx7+1rKqttcLN0czDuK+rrbXDz9XQx7yyrK62xdHW0Mq+s62vusjU2dHMwLWurrjK19zTzcG2r7C6zNne1c/Ct7GyvM7a4NbQwrexs77Q3OHX0cO4srS/0t3i2NLEubO1wdTf49rTxbq0t8PW4OTb1cbAtrfE2OHl3NfIwre5xtrj5t3ZycO5u8jc5efi28rFur3K3+fo49zMyLy+zOHo6uTdz8rAv87k6+vl39HOw8LR5+3s5uDT0cbE1Ont7efh1dPJx9fr7+3o49fVzMvZ7/Hu6eTY1s7O3fHy7+rl2tfQ0OHz8/Dq5tvZ0dPl9fXx7Ofc29TW6fb28u3o3t3X2ez39/Pu6d/f2dzv+Pjz7+rh4Nve8fn58/Hq4+Hd4fP6+vTy7OXj3+T1+/v19O3n5eDo9/z89vXu6OfC'); audio.volume = 0.3; audio.play(); } catch (e) {} },
            showToast: function(message) { var existing = document.querySelector('.cartt-toast'); if (existing) existing.remove(); var toast = document.createElement('div'); toast.className = 'cartt-toast'; toast.textContent = message; document.body.appendChild(toast); setTimeout(function() { toast.remove(); }, 3000); },
            escapeHtml: function(text) { var div = document.createElement('div'); div.textContent = text; return div.innerHTML; }
        };
        document.addEventListener('DOMContentLoaded', function() { CarttChat.init(); });
        </script>
        <?php
        return ob_get_clean();
    }

    /**
     * AJAX: Poll for new messages
     */
    public function ajaxPollMessages(): void
    {
        check_ajax_referer('cartt_chat', 'nonce');
        
        $visitorId = sanitize_text_field($_POST['visitor_id'] ?? '');
        $lastId = intval($_POST['last_id'] ?? 0);
        
        global $wpdb;
        
        $conversation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_conversations} WHERE visitor_id = %s ORDER BY id DESC LIMIT 1",
            $visitorId
        ));
        
        if (!$conversation) {
            wp_send_json_success(['messages' => []]);
            return;
        }
        
        $messages = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->table_messages} WHERE conversation_id = %d AND id > %d ORDER BY created_at ASC",
            $conversation->id,
            $lastId
        ));
        
        wp_send_json_success(['messages' => $messages]);
    }

    /**
     * AJAX: Send screenshot
     */
    public function ajaxSendScreenshot(): void
    {
        check_ajax_referer('cartt_chat', 'nonce');
        
        $imageData = $_POST['image'] ?? '';
        $visitorId = sanitize_text_field($_POST['visitor_id'] ?? '');
        $visitorName = sanitize_text_field($_POST['visitor_name'] ?? '');
        
        if (empty($imageData) || empty($visitorId)) {
            wp_send_json_error('Invalid data');
        }
        
        if (strpos($imageData, 'data:image/') !== 0) {
            wp_send_json_error('Invalid image format');
        }
        
        $parts = explode(',', $imageData);
        if (count($parts) !== 2) {
            wp_send_json_error('Invalid image data');
        }
        
        $decoded = base64_decode($parts[1]);
        if ($decoded === false || strlen($decoded) > 5 * 1024 * 1024) {
            wp_send_json_error('Image too large or invalid');
        }
        
        $filename = 'screenshot-' . time() . '-' . wp_generate_password(6, false) . '.png';
        $upload_dir = wp_upload_dir();
        $chat_dir = $upload_dir['basedir'] . '/cartt-chat';
        
        if (!file_exists($chat_dir)) {
            wp_mkdir_p($chat_dir);
        }
        
        $filepath = $chat_dir . '/' . $filename;
        file_put_contents($filepath, $decoded);
        
        $imageUrl = $upload_dir['baseurl'] . '/cartt-chat/' . $filename;
        $message = '[image]' . $imageUrl . '[/image]';
        
        global $wpdb;
        
        $conversation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_conversations} WHERE visitor_id = %s ORDER BY id DESC LIMIT 1",
            $visitorId
        ));
        
        if (!$conversation) {
            $wpdb->insert($this->table_conversations, [
                'visitor_id' => $visitorId,
                'visitor_name' => $visitorName,
                'status' => 'active',
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql')
            ]);
            $convId = $wpdb->insert_id;
        } else {
            $convId = $conversation->id;
            $wpdb->update($this->table_conversations, ['updated_at' => current_time('mysql')], ['id' => $convId]);
        }
        
        $wpdb->insert($this->table_messages, [
            'conversation_id' => $convId,
            'sender' => 'visitor',
            'message' => $message,
            'created_at' => current_time('mysql')
        ]);
        
        $messageId = $wpdb->insert_id;
        
        // Notify agents about screenshot
        $isNewConversation = !$conversation;
        $notifyEvery = get_option('cartt_chat_notify_every_message', '0');
        if ($isNewConversation || $notifyEvery === '1') {
            $this->notifyAgents($convId, 'Sent a screenshot: ' . $imageUrl, $visitorName, null);
        }
        
        wp_send_json_success([
            'message_id' => $messageId,
            'conversation_id' => $convId,
            'image_url' => $imageUrl
        ]);
    }

    /**
     * AJAX: Send file
     */
    public function ajaxSendFile(): void
    {
        check_ajax_referer('cartt_chat', 'nonce');
        
        $fileData = $_POST['file'] ?? '';
        $filename = sanitize_file_name($_POST['filename'] ?? 'file');
        $visitorId = sanitize_text_field($_POST['visitor_id'] ?? '');
        $visitorName = sanitize_text_field($_POST['visitor_name'] ?? '');
        
        if (empty($fileData) || empty($visitorId)) {
            wp_send_json_error('Invalid data');
        }
        
        $parts = explode(',', $fileData);
        if (count($parts) !== 2) {
            wp_send_json_error('Invalid file data');
        }
        
        $decoded = base64_decode($parts[1]);
        if ($decoded === false || strlen($decoded) > 5 * 1024 * 1024) {
            wp_send_json_error('File too large or invalid');
        }
        
        $ext = pathinfo($filename, PATHINFO_EXTENSION);
        $safeFilename = time() . '-' . wp_generate_password(6, false) . '.' . $ext;
        
        $upload_dir = wp_upload_dir();
        $chat_dir = $upload_dir['basedir'] . '/cartt-chat';
        
        if (!file_exists($chat_dir)) {
            wp_mkdir_p($chat_dir);
        }
        
        $filepath = $chat_dir . '/' . $safeFilename;
        file_put_contents($filepath, $decoded);
        
        $fileUrl = $upload_dir['baseurl'] . '/cartt-chat/' . $safeFilename;
        $message = '[file]' . $fileUrl . '|' . $filename . '[/file]';
        
        global $wpdb;
        
        $conversation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_conversations} WHERE visitor_id = %s ORDER BY id DESC LIMIT 1",
            $visitorId
        ));
        
        if (!$conversation) {
            $wpdb->insert($this->table_conversations, [
                'visitor_id' => $visitorId,
                'visitor_name' => $visitorName,
                'status' => 'active',
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql')
            ]);
            $convId = $wpdb->insert_id;
        } else {
            $convId = $conversation->id;
            $wpdb->update($this->table_conversations, ['updated_at' => current_time('mysql')], ['id' => $convId]);
        }
        
        $wpdb->insert($this->table_messages, [
            'conversation_id' => $convId,
            'sender' => 'visitor',
            'message' => $message,
            'created_at' => current_time('mysql')
        ]);
        
        $messageId = $wpdb->insert_id;
        
        // Notify agents about file
        $isNewConversation = !$conversation;
        $notifyEvery = get_option('cartt_chat_notify_every_message', '0');
        if ($isNewConversation || $notifyEvery === '1') {
            $this->notifyAgents($convId, 'Sent a file: ' . $filename, $visitorName, null);
        }
        
        wp_send_json_success([
            'message_id' => $messageId,
            'conversation_id' => $convId,
            'file_url' => $fileUrl
        ]);
    }

    /**
     * Get AI reply from relay
     */
    /**
     * Handle AI handoff - notify agents and mark conversation
     */
    private function handleHandoff(int $convId, string $visitorName, string $lastMessage): void
    {
        global $wpdb;
        
        // Mark conversation as needing handoff
        $wpdb->update(
            $this->table_conversations,
            [
                'handoff_at' => current_time('mysql'),
                'handoff_status' => 'pending'
            ],
            ['id' => $convId]
        );
        
        // Send urgent notification to agents (always, regardless of notify settings)
        $this->notifyAgentsUrgent($convId, $visitorName, $lastMessage);
    }
    
    /**
     * Send urgent notification to agents for handoff
     */
    private function notifyAgentsUrgent(int $convId, string $visitorName, string $lastMessage): void
    {
        $storeName = get_bloginfo('name');
        $chatUrl = admin_url('admin.php?page=cartt-chat&conv=' . $convId);
        
        // Get visitor email from conversation
        $conversation = $this->getConversation($convId);
        $visitorEmail = $conversation->visitor_email ?? '';
        
        // Email notification
        $adminEmail = get_option('cartt_chat_email_address', get_option('admin_email'));
        $subject = '[URGENT] ' . $storeName . ' - Customer needs human assistance';
        $urgentMessage = "A customer needs immediate assistance!\n\n" . substr($lastMessage, 0, 200);
        
        $this->sendEmailViaRelay($adminEmail, $subject, $urgentMessage, [
            'visitor_name' => $visitorName ?: 'Visitor',
            'visitor_email' => $visitorEmail,
            'conversation_url' => $chatUrl
        ]);
        
        // SMS notification (if enabled) - PRO feature
        if (get_option('cartt_chat_notify_sms', '0') === '1' && $this->hasProFeatures()) {
            $smsNumber = get_option('cartt_chat_sms_number', '');
            if ($smsNumber) {
                $smsBody = "URGENT: Customer needs help. " . ($visitorName ? $visitorName . ": " : "") . substr($lastMessage, 0, 100);
                $this->callRelay('send', [
                    'license_key' => $this->getLicenseKey(),
                    'channel' => 'sms',
                    'to' => $smsNumber,
                    'message' => $smsBody,
                    'conversation_id' => $convId,
                    'site_url' => home_url()
                ]);
            }
        }
        
        // WhatsApp notification (if enabled) - PRO feature
        if (get_option('cartt_chat_notify_whatsapp', '0') === '1' && $this->hasProFeatures()) {
            $waNumber = get_option('cartt_chat_whatsapp_number', '');
            if ($waNumber) {
                $waBody = "URGENT: Customer needs help\n" . ($visitorName ? "Customer: " . $visitorName . "\n" : "") . "Message: " . substr($lastMessage, 0, 100);
                $this->callRelay('send', [
                    'license_key' => $this->getLicenseKey(),
                    'channel' => 'whatsapp',
                    'to' => $waNumber,
                    'message' => $waBody,
                    'conversation_id' => $convId,
                    'site_url' => home_url()
                ]);
            }
        }
    }
    
    /**
     * AJAX: Check handoff status and send follow-up if needed
     */
    public function ajaxCheckHandoff(): void
    {
        check_ajax_referer('cartt_chat', 'nonce');
        
        $convId = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        
        if (!$convId) {
            wp_send_json_error('Invalid conversation');
            return;
        }
        
        global $wpdb;
        
        $conversation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_conversations} WHERE id = %d",
            $convId
        ));
        
        if (!$conversation || empty($conversation->handoff_at) || $conversation->handoff_status !== 'pending') {
            wp_send_json_success(['needs_followup' => false]);
            return;
        }
        
        // Check if human has responded since handoff
        $humanResponse = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_messages} 
             WHERE conversation_id = %d 
             AND sender = 'admin' 
             AND created_at > %s
             AND message NOT LIKE '%%[AI]%%'",
            $convId,
            $conversation->handoff_at
        ));
        
        if ($humanResponse > 0) {
            // Human responded, clear handoff
            $wpdb->update(
                $this->table_conversations,
                ['handoff_status' => 'resolved'],
                ['id' => $convId]
            );
            wp_send_json_success(['needs_followup' => false, 'human_responded' => true]);
            return;
        }
        
        // Check timeout
        $timeout = intval(get_option('cartt_chat_handoff_timeout', '60'));
        $handoffTime = strtotime($conversation->handoff_at);
        $elapsed = time() - $handoffTime;
        
        if ($elapsed >= $timeout && $conversation->handoff_status === 'pending') {
            // Mark as followup sent
            $wpdb->update(
                $this->table_conversations,
                ['handoff_status' => 'followup_sent'],
                ['id' => $convId]
            );
            
            // Send follow-up message asking for contact info
            $followupMsg = get_option('cartt_chat_handoff_message', 
                "Our team members are currently assisting other customers. Please leave your email or phone number and we'll get back to you as soon as possible!");
            
            $this->addMessage($convId, 'admin', $followupMsg);
            
            wp_send_json_success([
                'needs_followup' => true,
                'followup_message' => $followupMsg
            ]);
            return;
        }
        
        // Still waiting
        wp_send_json_success([
            'needs_followup' => false,
            'seconds_remaining' => max(0, $timeout - $elapsed)
        ]);
    }

    private function getAIReply(int $convId, string $message, string $visitorName, $cartData): ?array
    {
        // AI is a PRO feature
        if (!$this->hasProFeatures()) {
            return null;
        }
        
        if (get_option('cartt_chat_ai_enabled', '0') !== '1') {
            return null;
        }
        
        $licenseKey = $this->getLicenseKey();
        if (empty($licenseKey)) {
            error_log('Cartt Chat AI: No license key');
            return null;
        }
        
        $storeName = get_bloginfo('name');
        $storeInfo = get_option('cartt_chat_ai_store_info', '');
        
        // Get conversation history (last 10 messages)
        $messages = $this->getMessages($convId);
        $history = array_slice($messages, -10);
        $historyJson = wp_json_encode(array_map(function($msg) {
            return [
                'sender' => $msg->sender ?? 'visitor',
                'message' => $msg->message ?? ''
            ];
        }, $history));
        
        // Format cart info
        $cartInfo = '';
        if (!empty($cartData)) {
            $cart = is_string($cartData) ? json_decode($cartData, true) : $cartData;
            if (is_array($cart) && !empty($cart['items']) && is_array($cart['items'])) {
                $items = [];
                foreach ($cart['items'] as $item) {
                    $name = $item['name'] ?? 'Item';
                    $qty = $item['quantity'] ?? $item['qty'] ?? 1;
                    $price = $item['line_total'] ?? $item['price'] ?? '';
                    $items[] = $name . ' (qty: ' . $qty . ($price ? ', $' . $price : '') . ')';
                }
                $cartInfo = 'Items: ' . implode(', ', $items) . '. Total: $' . ($cart['total'] ?? '');
            }
        }
        
        error_log('Cartt Chat AI: Calling relay for store ' . $storeName);
        
        $result = $this->callRelay('ai_reply', [
            'license_key' => $licenseKey,
            'store_url' => home_url(),
            'store_name' => $storeName,
            'store_info' => $storeInfo,
            'message' => $message,
            'history' => $historyJson,
            'cart' => $cartInfo,
            'visitor_name' => $visitorName
        ]);
        
        if (!empty($result['success']) && !empty($result['reply'])) {
            $reply = $result['reply'];
            $needsHandoff = false;
            
            // Check for handoff request
            if (strpos($reply, '[HANDOFF]') !== false) {
                $needsHandoff = true;
                $reply = trim(str_replace('[HANDOFF]', '', $reply));
            }
            
            error_log('Cartt Chat AI: Got reply: ' . substr($reply, 0, 50) . '...');
            
            return [
                'reply' => $reply,
                'needs_handoff' => $needsHandoff
            ];
        }
        
        error_log('Cartt Chat AI: Relay failed - ' . wp_json_encode($result));
        
        return null;
    }
}
