<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Category Service
 * Manages product categories
 */
class CategoryService
{
    public static function getCategories(?int $parentId = null): array
    {
        global $wpdb;
        
        $where = $parentId === null ? "parent_id IS NULL" : $wpdb->prepare("parent_id = %d", $parentId);
        
        return $wpdb->get_results(
            "SELECT * FROM {$wpdb->prefix}cartt_categories WHERE $where ORDER BY sort_order ASC, name ASC"
        );
    }

    public static function getAllCategories(): array
    {
        global $wpdb;
        return $wpdb->get_results(
            "SELECT * FROM {$wpdb->prefix}cartt_categories ORDER BY parent_id ASC, sort_order ASC, name ASC"
        );
    }

    public static function getCategory(int $id): ?object
    {
        global $wpdb;
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_categories WHERE id = %d",
            $id
        ));
    }

    public static function getCategoryBySlug(string $slug): ?object
    {
        global $wpdb;
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_categories WHERE slug = %s",
            $slug
        ));
    }

    public static function createCategory(array $data): int
    {
        global $wpdb;
        
        $slug = sanitize_title($data['name']);
        $baseSlug = $slug;
        $counter = 1;
        
        while (self::getCategoryBySlug($slug)) {
            $slug = $baseSlug . '-' . $counter++;
        }
        
        $wpdb->insert(
            $wpdb->prefix . 'cartt_categories',
            [
                'name' => sanitize_text_field($data['name']),
                'slug' => $slug,
                'description' => sanitize_textarea_field($data['description'] ?? ''),
                'parent_id' => !empty($data['parent_id']) ? (int) $data['parent_id'] : null,
                'image' => !empty($data['image']) ? (int) $data['image'] : null,
                'sort_order' => (int) ($data['sort_order'] ?? 0),
            ],
            ['%s', '%s', '%s', '%d', '%d', '%d']
        );
        
        return (int) $wpdb->insert_id;
    }

    public static function updateCategory(int $id, array $data): bool
    {
        global $wpdb;
        
        $updateData = [];
        $formats = [];
        
        if (isset($data['name'])) {
            $updateData['name'] = sanitize_text_field($data['name']);
            $formats[] = '%s';
            
            // Update slug if name changed
            $slug = sanitize_title($data['name']);
            $existing = self::getCategoryBySlug($slug);
            if (!$existing || $existing->id === $id) {
                $updateData['slug'] = $slug;
                $formats[] = '%s';
            }
        }
        
        if (isset($data['description'])) {
            $updateData['description'] = sanitize_textarea_field($data['description']);
            $formats[] = '%s';
        }
        
        if (array_key_exists('parent_id', $data)) {
            $updateData['parent_id'] = !empty($data['parent_id']) ? (int) $data['parent_id'] : null;
            $formats[] = '%d';
        }
        
        if (array_key_exists('image', $data)) {
            $updateData['image'] = !empty($data['image']) ? (int) $data['image'] : null;
            $formats[] = '%d';
        }
        
        if (isset($data['sort_order'])) {
            $updateData['sort_order'] = (int) $data['sort_order'];
            $formats[] = '%d';
        }
        
        if (empty($updateData)) {
            return false;
        }
        
        return $wpdb->update(
            $wpdb->prefix . 'cartt_categories',
            $updateData,
            ['id' => $id],
            $formats,
            ['%d']
        ) !== false;
    }

    public static function deleteCategory(int $id): bool
    {
        global $wpdb;
        
        // Remove product associations
        $wpdb->delete(
            $wpdb->prefix . 'cartt_product_categories',
            ['category_id' => $id],
            ['%d']
        );
        
        // Update children to have no parent
        $wpdb->update(
            $wpdb->prefix . 'cartt_categories',
            ['parent_id' => null],
            ['parent_id' => $id],
            ['%d'],
            ['%d']
        );
        
        return $wpdb->delete(
            $wpdb->prefix . 'cartt_categories',
            ['id' => $id],
            ['%d']
        ) !== false;
    }

    public static function getProductCategories(int $productId): array
    {
        global $wpdb;
        return $wpdb->get_results($wpdb->prepare(
            "SELECT c.* FROM {$wpdb->prefix}cartt_categories c
             JOIN {$wpdb->prefix}cartt_product_categories pc ON c.id = pc.category_id
             WHERE pc.product_id = %d
             ORDER BY c.name ASC",
            $productId
        ));
    }

    public static function setProductCategories(int $productId, array $categoryIds): void
    {
        global $wpdb;
        
        // Remove existing
        $wpdb->delete(
            $wpdb->prefix . 'cartt_product_categories',
            ['product_id' => $productId],
            ['%d']
        );
        
        // Add new
        foreach ($categoryIds as $catId) {
            $wpdb->insert(
                $wpdb->prefix . 'cartt_product_categories',
                [
                    'product_id' => $productId,
                    'category_id' => (int) $catId,
                ],
                ['%d', '%d']
            );
        }
    }

    public static function getCategoryProductCount(int $categoryId): int
    {
        global $wpdb;
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_product_categories WHERE category_id = %d",
            $categoryId
        ));
    }

    public static function getCategoriesHierarchy(): array
    {
        $categories = self::getAllCategories();
        return self::buildTree($categories);
    }

    private static function buildTree(array $categories, ?int $parentId = null): array
    {
        $tree = [];
        foreach ($categories as $cat) {
            if ($cat->parent_id == $parentId) {
                $cat->children = self::buildTree($categories, (int) $cat->id);
                $tree[] = $cat;
            }
        }
        return $tree;
    }
}
