<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Cart Service - handles server-side cart session
 */
class CartService
{
    private const SESSION_KEY = 'cartt_cart';

    public function __construct()
    {
        add_action('init', [$this, 'initSession'], 1);
    }

    public function initSession(): void
    {
        if (!session_id() && !headers_sent()) {
            session_start();
        }
    }

    public function getCart(): array
    {
        return $_SESSION[self::SESSION_KEY] ?? [
            'items' => [],
            'bumps' => [],
            'coupon' => null,
            'shipping_address' => [],
            'shipping_method' => null,
            'totals' => [
                'subtotal' => 0,
                'bumps' => 0,
                'discount' => 0,
                'shipping' => 0,
                'tax' => 0,
                'total' => 0,
            ],
        ];
    }

    public function applyCoupon(array $coupon, float $discount): void
    {
        $cart = $this->getCart();
        $cart['coupon'] = $coupon;
        $cart['totals']['discount'] = $discount;
        $this->calculateTotals($cart);
        $this->saveCart($cart);
    }

    public function removeCoupon(): void
    {
        $cart = $this->getCart();
        $cart['coupon'] = null;
        $cart['totals']['discount'] = 0;
        $this->calculateTotals($cart);
        $this->saveCart($cart);
    }

    public function setShippingAddress(array $address): array
    {
        $cart = $this->getCart();
        $cart['shipping_address'] = $address;
        
        // Calculate available shipping methods
        $shippingService = new ShippingService();
        $shipping = $shippingService->calculateShipping($cart, $address);
        
        // Auto-select first method
        if (!empty($shipping['selected'])) {
            $cart['shipping_method'] = $shipping['selected'];
            $cart['totals']['shipping'] = $shipping['selected']['cost'];
        }
        
        $this->calculateTotals($cart);
        $this->saveCart($cart);
        
        return [
            'success' => true,
            'shipping' => $shipping,
            'cart' => $cart,
        ];
    }

    public function setShippingMethod(int $methodId): array
    {
        $cart = $this->getCart();
        
        $shippingService = new ShippingService();
        $method = $shippingService->getMethod($methodId);
        
        if (!$method) {
            return ['success' => false, 'error' => 'Invalid shipping method'];
        }
        
        $cart['shipping_method'] = [
            'id' => $method->id,
            'name' => $method->name,
            'type' => $method->method_type,
            'cost' => (float) $method->cost,
        ];
        
        $this->calculateTotals($cart);
        $this->saveCart($cart);
        
        return ['success' => true, 'cart' => $cart];
    }

    public function addBump(string $bumpId): array
    {
        global $wpdb;
        
        $cart = $this->getCart();
        
        // Check if bump already added
        if (isset($cart['bumps'][$bumpId])) {
            return ['success' => false, 'error' => 'Already added'];
        }
        
        // Get bump config
        $bumps = CheckoutConfig::getOrderBumps();
        if (!isset($bumps[$bumpId]) || !$bumps[$bumpId]['enabled']) {
            return ['success' => false, 'error' => 'Invalid bump'];
        }
        
        $bump = $bumps[$bumpId];
        $productId = (int) $bump['product_id'];
        
        // Get product
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_products WHERE id = %d AND status = 'publish'",
            $productId
        ));
        
        if (!$product) {
            return ['success' => false, 'error' => 'Product not found'];
        }
        
        // Calculate bump price
        $originalPrice = $product->sale_price && $product->sale_price < $product->price 
            ? (float) $product->sale_price 
            : (float) $product->price;
            
        $bumpPrice = $originalPrice;
        if ($bump['discount_type'] === 'percent' && $bump['discount_value'] > 0) {
            $bumpPrice = $originalPrice * (1 - $bump['discount_value'] / 100);
        } elseif ($bump['discount_type'] === 'fixed' && $bump['discount_value'] > 0) {
            $bumpPrice = max(0, $originalPrice - $bump['discount_value']);
        }
        
        $cart['bumps'][$bumpId] = [
            'bump_id' => $bumpId,
            'product_id' => $productId,
            'name' => $product->name,
            'original_price' => $originalPrice,
            'price' => round($bumpPrice, 2),
            'image' => $product->featured_image ? wp_get_attachment_image_url($product->featured_image, 'thumbnail') : '',
        ];
        
        $this->calculateTotals($cart);
        $this->saveCart($cart);
        
        return ['success' => true, 'cart' => $cart];
    }

    public function removeBump(string $bumpId): array
    {
        $cart = $this->getCart();
        
        if (isset($cart['bumps'][$bumpId])) {
            unset($cart['bumps'][$bumpId]);
            $this->calculateTotals($cart);
            $this->saveCart($cart);
        }
        
        return ['success' => true, 'cart' => $cart];
    }

    public function addItem(int $productId, int $quantity = 1, array $meta = []): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d AND status = 'publish'",
            $productId
        ));

        if (!$product) {
            return ['success' => false, 'error' => 'Product not found'];
        }

        // Check stock
        if ($product->manage_stock && $product->stock_quantity < $quantity) {
            return ['success' => false, 'error' => 'Insufficient stock'];
        }

        $cart = $this->getCart();
        $itemKey = $this->generateItemKey($productId, $meta);
        
        $price = $product->sale_price && $product->sale_price < $product->price 
            ? (float) $product->sale_price 
            : (float) $product->price;

        if (isset($cart['items'][$itemKey])) {
            $cart['items'][$itemKey]['quantity'] += $quantity;
        } else {
            $cart['items'][$itemKey] = [
                'product_id' => $productId,
                'name' => $product->name,
                'price' => $price,
                'quantity' => $quantity,
                'image' => $product->featured_image ? wp_get_attachment_image_url($product->featured_image, 'thumbnail') : '',
                'meta' => $meta,
            ];
        }

        $this->calculateTotals($cart);
        $this->saveCart($cart);

        return ['success' => true, 'cart' => $cart];
    }

    public function updateItem(string $itemKey, int $quantity): array
    {
        $cart = $this->getCart();

        if (!isset($cart['items'][$itemKey])) {
            return ['success' => false, 'error' => 'Item not found'];
        }

        if ($quantity <= 0) {
            return $this->removeItem($itemKey);
        }

        // Check stock
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $productId = $cart['items'][$itemKey]['product_id'];
        
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT manage_stock, stock_quantity FROM $table WHERE id = %d",
            $productId
        ));

        if ($product && $product->manage_stock && $product->stock_quantity < $quantity) {
            return ['success' => false, 'error' => 'Insufficient stock'];
        }

        $cart['items'][$itemKey]['quantity'] = $quantity;
        $this->calculateTotals($cart);
        $this->saveCart($cart);

        return ['success' => true, 'cart' => $cart];
    }

    public function removeItem(string $itemKey): array
    {
        $cart = $this->getCart();

        if (isset($cart['items'][$itemKey])) {
            unset($cart['items'][$itemKey]);
            $this->calculateTotals($cart);
            $this->saveCart($cart);
        }

        return ['success' => true, 'cart' => $cart];
    }

    public function clear(): array
    {
        $cart = [
            'items' => [],
            'totals' => [
                'subtotal' => 0,
                'tax' => 0,
                'shipping' => 0,
                'total' => 0,
            ],
        ];
        
        $this->saveCart($cart);
        return ['success' => true, 'cart' => $cart];
    }

    public function getItemCount(): int
    {
        $cart = $this->getCart();
        $count = 0;
        
        foreach ($cart['items'] as $item) {
            $count += $item['quantity'];
        }
        
        return $count;
    }

    private function calculateTotals(array &$cart): void
    {
        $subtotal = 0;

        foreach ($cart['items'] as $item) {
            $subtotal += $item['price'] * $item['quantity'];
        }

        $cart['totals']['subtotal'] = round($subtotal, 2);
        
        // Calculate bumps total
        $bumpsTotal = 0;
        foreach ($cart['bumps'] ?? [] as $bump) {
            $bumpsTotal += $bump['price'];
        }
        $cart['totals']['bumps'] = round($bumpsTotal, 2);
        
        // Recalculate coupon discount if coupon exists
        if (!empty($cart['coupon'])) {
            $couponService = new CouponService();
            $coupon = $couponService->getCouponById($cart['coupon']['id']);
            if ($coupon) {
                $cart['totals']['discount'] = $couponService->calculateDiscount($coupon, $cart);
                
                // Check if coupon grants free shipping
                if ($coupon->free_shipping) {
                    $cart['coupon']['free_shipping'] = true;
                }
            }
        }
        
        $discount = $cart['totals']['discount'] ?? 0;
        
        // Shipping - use selected method or 0
        $shipping = 0;
        if (!empty($cart['shipping_method']) && empty($cart['coupon']['free_shipping'])) {
            $shipping = $cart['shipping_method']['cost'] ?? 0;
        }
        $cart['totals']['shipping'] = round($shipping, 2);
        
        // Tax calculation using TaxService
        $taxEnabled = get_option('cartt_tax_enabled', false);
        if ($taxEnabled && !empty($cart['shipping_address'])) {
            $taxService = new TaxService();
            $taxableAmount = max(0, $subtotal + $bumpsTotal - $discount);
            $taxResult = $taxService->calculateTax($taxableAmount, $cart['shipping_address'], true, $shipping);
            $cart['totals']['tax'] = $taxResult['total'];
            $cart['totals']['tax_rates'] = $taxResult['rates'];
        } else {
            $cart['totals']['tax'] = 0;
            $cart['totals']['tax_rates'] = [];
        }
        
        $cart['totals']['total'] = round(
            $cart['totals']['subtotal'] + $cart['totals']['bumps'] - $discount + $cart['totals']['shipping'] + $cart['totals']['tax'],
            2
        );
    }

    private function saveCart(array $cart): void
    {
        $_SESSION[self::SESSION_KEY] = $cart;
    }

    private function generateItemKey(int $productId, array $meta = []): string
    {
        return md5($productId . serialize($meta));
    }
}
