<?php

declare(strict_types=1);

namespace Cartt\Services;

/**
 * Abandoned Cart Service
 * Tracks abandoned carts and sends recovery emails
 */
class AbandonedCartService
{
    private const TABLE_SUFFIX = 'cartt_abandoned_carts';

    public static function createTable(): void
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;
        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            session_id varchar(100) NOT NULL,
            email varchar(255) DEFAULT NULL,
            cart_data longtext NOT NULL,
            cart_total decimal(10,2) NOT NULL DEFAULT 0,
            billing_data longtext DEFAULT NULL,
            status enum('active','abandoned','recovered','converted') NOT NULL DEFAULT 'active',
            emails_sent tinyint(1) NOT NULL DEFAULT 0,
            last_email_at datetime DEFAULT NULL,
            recovered_order_id bigint(20) unsigned DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            user_agent text DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            abandoned_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY session_id (session_id),
            KEY email (email),
            KEY status (status),
            KEY abandoned_at (abandoned_at)
        ) $charset;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    public function captureEmail(string $email, array $cart, array $billing = []): int
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        $sessionId = $this->getSessionId();
        $email = sanitize_email($email);

        if (!is_email($email)) {
            return 0;
        }

        // Check if session exists
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id, status FROM $table WHERE session_id = %s ORDER BY id DESC LIMIT 1",
            $sessionId
        ));

        $data = [
            'email' => $email,
            'cart_data' => json_encode($cart),
            'cart_total' => $cart['totals']['total'] ?? 0,
            'billing_data' => json_encode($billing),
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? '',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'updated_at' => current_time('mysql'),
        ];

        if ($existing && $existing->status === 'active') {
            $wpdb->update($table, $data, ['id' => $existing->id]);
            return (int) $existing->id;
        }

        $data['session_id'] = $sessionId;
        $data['status'] = 'active';
        $data['created_at'] = current_time('mysql');

        $wpdb->insert($table, $data);
        return (int) $wpdb->insert_id;
    }

    public function markConverted(string $email, int $orderId): void
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        $wpdb->update($table, [
            'status' => 'converted',
            'recovered_order_id' => $orderId,
        ], [
            'email' => $email,
            'status' => 'active',
        ]);

        // Also mark abandoned ones as recovered if same email
        $wpdb->update($table, [
            'status' => 'recovered',
            'recovered_order_id' => $orderId,
        ], [
            'email' => $email,
            'status' => 'abandoned',
        ]);
    }

    public function markAbandoned(): int
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        // Mark carts as abandoned if no activity for 1 hour and has email
        $cutoff = date('Y-m-d H:i:s', strtotime('-1 hour'));

        $updated = $wpdb->query($wpdb->prepare(
            "UPDATE $table SET status = 'abandoned', abandoned_at = NOW() 
             WHERE status = 'active' 
             AND email IS NOT NULL 
             AND email != ''
             AND updated_at < %s",
            $cutoff
        ));

        return (int) $updated;
    }

    public function getAbandonedCarts(int $limit = 50): array
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE status = 'abandoned' ORDER BY abandoned_at DESC LIMIT %d",
            $limit
        ));
    }

    public function getCartsToEmail(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        // Get abandoned carts that haven't received email in last 24 hours
        // and have been abandoned for at least 1 hour but less than 7 days
        $minTime = date('Y-m-d H:i:s', strtotime('-7 days'));
        $maxTime = date('Y-m-d H:i:s', strtotime('-1 hour'));
        $emailCooldown = date('Y-m-d H:i:s', strtotime('-24 hours'));

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table 
             WHERE status = 'abandoned' 
             AND emails_sent < 3
             AND (last_email_at IS NULL OR last_email_at < %s)
             AND abandoned_at BETWEEN %s AND %s
             ORDER BY abandoned_at ASC
             LIMIT 50",
            $emailCooldown,
            $minTime,
            $maxTime
        ));
    }

    public function sendRecoveryEmail(int $cartId): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        $cart = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $cartId
        ));

        if (!$cart || !$cart->email) {
            return false;
        }

        $cartData = json_decode($cart->cart_data, true);
        $items = $cartData['items'] ?? [];
        $total = $cart->cart_total;

        if (empty($items)) {
            return false;
        }

        // Build email
        $storeName = get_option('cartt_store_name', get_bloginfo('name'));
        $cartUrl = $this->getRecoveryUrl($cartId);

        $subject = sprintf(__('You left something behind at %s', 'cartt'), $storeName);

        // Email sequence varies by send count
        $emailNum = $cart->emails_sent + 1;
        
        if ($emailNum === 1) {
            $subject = sprintf(__('You left something behind at %s', 'cartt'), $storeName);
            $intro = __('Looks like you didn\'t finish checking out. Your cart is still waiting for you.', 'cartt');
        } elseif ($emailNum === 2) {
            $subject = sprintf(__('Still thinking it over? Your cart at %s', 'cartt'), $storeName);
            $intro = __('We noticed you haven\'t completed your purchase. Need help with anything?', 'cartt');
        } else {
            $subject = sprintf(__('Last chance: Your cart at %s', 'cartt'), $storeName);
            $intro = __('Your cart will expire soon. Complete your order before it\'s too late.', 'cartt');
        }

        $message = $intro . "\n\n";
        $message .= __('Your cart:', 'cartt') . "\n";
        $message .= "---\n";

        foreach ($items as $item) {
            $message .= sprintf("%s x %d - $%.2f\n", $item['name'], $item['quantity'], $item['price'] * $item['quantity']);
        }

        $message .= "---\n";
        $message .= sprintf(__('Total: $%.2f', 'cartt'), $total) . "\n\n";
        $message .= sprintf(__('Complete your order: %s', 'cartt'), $cartUrl) . "\n\n";
        $message .= $storeName;

        $headers = [
            'Content-Type: text/plain; charset=UTF-8',
            'From: ' . $storeName . ' <' . get_option('admin_email') . '>',
        ];

        $sent = wp_mail($cart->email, $subject, $message, $headers);

        if ($sent) {
            $wpdb->update($table, [
                'emails_sent' => $emailNum,
                'last_email_at' => current_time('mysql'),
            ], ['id' => $cartId]);
        }

        return $sent;
    }

    public function getRecoveryUrl(int $cartId): string
    {
        $checkoutPageId = get_option('cartt_checkout_page_id');
        $baseUrl = $checkoutPageId ? get_permalink($checkoutPageId) : home_url('/checkout/');
        
        $token = $this->generateRecoveryToken($cartId);
        
        return add_query_arg('recover', $token, $baseUrl);
    }

    public function recoverCart(string $token): bool
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        // Decode token
        $decoded = $this->decodeRecoveryToken($token);
        if (!$decoded) {
            return false;
        }

        $cart = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $decoded['cart_id']
        ));

        if (!$cart) {
            return false;
        }

        // Restore cart to session
        $cartData = json_decode($cart->cart_data, true);
        $_SESSION['cartt_cart'] = $cartData;

        // Update status
        $wpdb->update($table, [
            'status' => 'active',
            'updated_at' => current_time('mysql'),
        ], ['id' => $cart->id]);

        return true;
    }

    private function getSessionId(): string
    {
        if (!session_id()) {
            session_start();
        }
        return session_id();
    }

    private function generateRecoveryToken(int $cartId): string
    {
        $data = [
            'cart_id' => $cartId,
            'exp' => time() + (7 * 24 * 60 * 60), // 7 days
        ];
        
        $payload = base64_encode(json_encode($data));
        $signature = hash_hmac('sha256', $payload, wp_salt('auth'));
        
        return $payload . '.' . substr($signature, 0, 16);
    }

    private function decodeRecoveryToken(string $token): ?array
    {
        $parts = explode('.', $token);
        if (count($parts) !== 2) {
            return null;
        }

        [$payload, $sig] = $parts;
        
        // Verify signature
        $expectedSig = substr(hash_hmac('sha256', $payload, wp_salt('auth')), 0, 16);
        if (!hash_equals($expectedSig, $sig)) {
            return null;
        }

        $data = json_decode(base64_decode($payload), true);
        
        // Check expiry
        if (($data['exp'] ?? 0) < time()) {
            return null;
        }

        return $data;
    }

    public function getStats(): array
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE_SUFFIX;

        $stats = [];

        $stats['total_abandoned'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM $table WHERE status = 'abandoned'"
        );

        $stats['total_recovered'] = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM $table WHERE status = 'recovered'"
        );

        $stats['total_value_abandoned'] = (float) $wpdb->get_var(
            "SELECT SUM(cart_total) FROM $table WHERE status = 'abandoned'"
        ) ?: 0;

        $stats['total_value_recovered'] = (float) $wpdb->get_var(
            "SELECT SUM(cart_total) FROM $table WHERE status = 'recovered'"
        ) ?: 0;

        $stats['recovery_rate'] = $stats['total_abandoned'] > 0 
            ? round(($stats['total_recovered'] / ($stats['total_abandoned'] + $stats['total_recovered'])) * 100, 1)
            : 0;

        return $stats;
    }
}
