<?php
/**
 * AI Recommendation Service
 * 
 * Provides intelligent product recommendations using multiple algorithms:
 * - Collaborative filtering (customers who bought X also bought Y)
 * - Content-based filtering (similar products by attributes)
 * - Popularity-based (trending, bestsellers)
 * - AI-powered (OpenAI/Claude integration for personalized suggestions)
 * 
 * @package Cartt
 * @since 1.4.0
 */

declare(strict_types=1);

namespace Cartt\Services;

class AIRecommendationService
{
    private static ?self $instance = null;

    public static function instance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('wp_ajax_cartt_get_recommendations', [$this, 'ajax_get_recommendations']);
        add_action('wp_ajax_nopriv_cartt_get_recommendations', [$this, 'ajax_get_recommendations']);
        add_action('cartt_order_completed', [$this, 'record_purchase_pattern'], 10, 1);
        add_action('cartt_product_viewed', [$this, 'record_product_view'], 10, 2);
    }

    /**
     * Get frequently bought together products
     */
    public function get_frequently_bought_together(int $product_id, int $limit = 4): array
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_purchase_patterns';
        
        $results = $wpdb->get_results($wpdb->prepare("
            SELECT related_product_id, SUM(count) as frequency
            FROM {$table}
            WHERE product_id = %d
            GROUP BY related_product_id
            ORDER BY frequency DESC
            LIMIT %d
        ", $product_id, $limit), ARRAY_A);

        if (empty($results)) {
            // Fallback to same-category products
            return $this->get_similar_products($product_id, $limit);
        }

        return array_column($results, 'related_product_id');
    }

    /**
     * Get similar products by category and attributes
     */
    public function get_similar_products(int $product_id, int $limit = 4): array
    {
        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';
        
        // Get source product
        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT category_id, price FROM {$products_table} WHERE id = %d",
            $product_id
        ));

        if (!$product) {
            return [];
        }

        // Find similar by category and price range (±30%)
        $price_min = $product->price * 0.7;
        $price_max = $product->price * 1.3;

        $results = $wpdb->get_col($wpdb->prepare("
            SELECT id FROM {$products_table}
            WHERE id != %d
            AND category_id = %d
            AND price BETWEEN %f AND %f
            AND status = 'published'
            ORDER BY RAND()
            LIMIT %d
        ", $product_id, $product->category_id, $price_min, $price_max, $limit));

        return $results;
    }

    /**
     * Get personalized recommendations for a customer
     */
    public function get_personalized(int $customer_id, int $limit = 8): array
    {
        global $wpdb;
        
        // Get customer's purchase history
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $products_table = $wpdb->prefix . 'cartt_products';
        
        // Get categories customer has purchased from
        $purchased_categories = $wpdb->get_col($wpdb->prepare("
            SELECT DISTINCT p.category_id
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE o.customer_id = %d
            AND o.status IN ('completed', 'processing')
        ", $customer_id));

        if (empty($purchased_categories)) {
            return $this->get_trending($limit);
        }

        // Get products from same categories not yet purchased
        $purchased_products = $wpdb->get_col($wpdb->prepare("
            SELECT DISTINCT oi.product_id
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            WHERE o.customer_id = %d
        ", $customer_id));

        $exclude = implode(',', array_map('intval', $purchased_products ?: [0]));
        $categories = implode(',', array_map('intval', $purchased_categories));

        $results = $wpdb->get_col("
            SELECT id FROM {$products_table}
            WHERE id NOT IN ({$exclude})
            AND category_id IN ({$categories})
            AND status = 'published'
            ORDER BY RAND()
            LIMIT {$limit}
        ");

        return $results;
    }

    /**
     * Get trending products (most viewed/purchased recently)
     */
    public function get_trending(int $limit = 8, int $days = 7): array
    {
        global $wpdb;
        $views_table = $wpdb->prefix . 'cartt_product_views';
        $products_table = $wpdb->prefix . 'cartt_products';
        
        $results = $wpdb->get_col($wpdb->prepare("
            SELECT pv.product_id, COUNT(*) as views
            FROM {$views_table} pv
            JOIN {$products_table} p ON p.id = pv.product_id
            WHERE pv.viewed_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
            AND p.status = 'published'
            GROUP BY pv.product_id
            ORDER BY views DESC
            LIMIT %d
        ", $days, $limit));

        return $results;
    }

    /**
     * Get bestsellers
     */
    public function get_bestsellers(int $limit = 8, int $days = 30): array
    {
        global $wpdb;
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $products_table = $wpdb->prefix . 'cartt_products';

        $results = $wpdb->get_col($wpdb->prepare("
            SELECT oi.product_id, SUM(oi.quantity) as sold
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE o.created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
            AND o.status IN ('completed', 'processing')
            AND p.status = 'published'
            GROUP BY oi.product_id
            ORDER BY sold DESC
            LIMIT %d
        ", $days, $limit));

        return $results;
    }

    /**
     * Get recently viewed products for current session/customer
     */
    public function get_recently_viewed(int $limit = 8): array
    {
        $customer_id = get_current_user_id();
        $session_id = $this->get_session_id();

        global $wpdb;
        $views_table = $wpdb->prefix . 'cartt_product_views';

        if ($customer_id) {
            $results = $wpdb->get_col($wpdb->prepare("
                SELECT DISTINCT product_id FROM {$views_table}
                WHERE customer_id = %d
                ORDER BY viewed_at DESC
                LIMIT %d
            ", $customer_id, $limit));
        } else {
            $results = $wpdb->get_col($wpdb->prepare("
                SELECT DISTINCT product_id FROM {$views_table}
                WHERE session_id = %s
                ORDER BY viewed_at DESC
                LIMIT %d
            ", $session_id, $limit));
        }

        return $results;
    }

    /**
     * AI-powered recommendations using OpenAI
     */
    public function get_ai_recommendations(int $customer_id, array $context = []): array
    {
        $api_key = get_option('cartt_openai_api_key');
        if (empty($api_key)) {
            return $this->get_personalized($customer_id);
        }

        global $wpdb;
        $products_table = $wpdb->prefix . 'cartt_products';

        // Get customer profile
        $purchase_history = $this->get_customer_purchase_history($customer_id);
        $browsing_history = $this->get_recently_viewed();

        // Get available products
        $products = $wpdb->get_results("
            SELECT id, name, short_description, price, category_id
            FROM {$products_table}
            WHERE status = 'published'
            LIMIT 50
        ", ARRAY_A);

        if (empty($products)) {
            return [];
        }

        // Build prompt
        $prompt = $this->build_recommendation_prompt($purchase_history, $browsing_history, $products, $context);

        try {
            $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json',
                ],
                'body' => json_encode([
                    'model' => 'gpt-4o-mini',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a product recommendation engine. Return only a JSON array of product IDs.'],
                        ['role' => 'user', 'content' => $prompt]
                    ],
                    'temperature' => 0.7,
                    'max_tokens' => 100
                ]),
                'timeout' => 10
            ]);

            if (is_wp_error($response)) {
                return $this->get_personalized($customer_id);
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);
            $content = $body['choices'][0]['message']['content'] ?? '';
            
            // Parse JSON array from response
            preg_match('/\[[\d,\s]+\]/', $content, $matches);
            if (!empty($matches[0])) {
                return json_decode($matches[0], true);
            }
        } catch (\Exception $e) {
            error_log('Cartt AI Recommendation Error: ' . $e->getMessage());
        }

        return $this->get_personalized($customer_id);
    }

    /**
     * Build AI recommendation prompt
     */
    private function build_recommendation_prompt(array $purchases, array $viewed, array $products, array $context): string
    {
        $purchase_names = array_column($purchases, 'name');
        $product_list = array_map(fn($p) => "ID:{$p['id']} - {$p['name']} (\${$p['price']})", $products);

        $prompt = "Customer purchase history: " . implode(', ', $purchase_names ?: ['None']) . "\n";
        $prompt .= "Recently viewed: " . implode(', ', array_slice($viewed, 0, 5) ?: ['None']) . "\n";
        
        if (!empty($context['cart_items'])) {
            $prompt .= "Current cart: " . implode(', ', $context['cart_items']) . "\n";
        }
        
        $prompt .= "\nAvailable products:\n" . implode("\n", $product_list) . "\n";
        $prompt .= "\nRecommend 4-6 product IDs that would interest this customer. Return only a JSON array of IDs.";

        return $prompt;
    }

    /**
     * Get customer purchase history
     */
    private function get_customer_purchase_history(int $customer_id): array
    {
        global $wpdb;
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $products_table = $wpdb->prefix . 'cartt_products';

        return $wpdb->get_results($wpdb->prepare("
            SELECT DISTINCT p.id, p.name, p.category_id
            FROM {$items_table} oi
            JOIN {$orders_table} o ON o.id = oi.order_id
            JOIN {$products_table} p ON p.id = oi.product_id
            WHERE o.customer_id = %d
            ORDER BY o.created_at DESC
            LIMIT 20
        ", $customer_id), ARRAY_A);
    }

    /**
     * Record purchase patterns for collaborative filtering
     */
    public function record_purchase_pattern(int $order_id): void
    {
        global $wpdb;
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $patterns_table = $wpdb->prefix . 'cartt_purchase_patterns';

        // Get all products in this order
        $products = $wpdb->get_col($wpdb->prepare(
            "SELECT product_id FROM {$items_table} WHERE order_id = %d",
            $order_id
        ));

        if (count($products) < 2) {
            return;
        }

        // Create pairs of products bought together
        foreach ($products as $product_id) {
            foreach ($products as $related_id) {
                if ($product_id === $related_id) continue;

                $wpdb->query($wpdb->prepare("
                    INSERT INTO {$patterns_table} (product_id, related_product_id, count)
                    VALUES (%d, %d, 1)
                    ON DUPLICATE KEY UPDATE count = count + 1
                ", $product_id, $related_id));
            }
        }
    }

    /**
     * Record product view
     */
    public function record_product_view(int $product_id, ?int $customer_id = null): void
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_product_views';

        $wpdb->insert($table, [
            'product_id' => $product_id,
            'customer_id' => $customer_id ?: null,
            'session_id' => $this->get_session_id(),
            'viewed_at' => current_time('mysql')
        ]);
    }

    /**
     * AJAX handler for recommendations
     */
    public function ajax_get_recommendations(): void
    {
        $type = sanitize_text_field($_POST['type'] ?? 'similar');
        $product_id = intval($_POST['product_id'] ?? 0);
        $customer_id = get_current_user_id();

        switch ($type) {
            case 'frequently_bought':
                $ids = $this->get_frequently_bought_together($product_id);
                break;
            case 'similar':
                $ids = $this->get_similar_products($product_id);
                break;
            case 'personalized':
                $ids = $this->get_personalized($customer_id);
                break;
            case 'trending':
                $ids = $this->get_trending();
                break;
            case 'bestsellers':
                $ids = $this->get_bestsellers();
                break;
            case 'recently_viewed':
                $ids = $this->get_recently_viewed();
                break;
            case 'ai':
                $ids = $this->get_ai_recommendations($customer_id);
                break;
            default:
                $ids = [];
        }

        // Get full product data
        $products = $this->get_products_by_ids($ids);

        wp_send_json_success(['products' => $products]);
    }

    /**
     * Get products by IDs
     */
    private function get_products_by_ids(array $ids): array
    {
        if (empty($ids)) return [];

        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        return $wpdb->get_results($wpdb->prepare(
            "SELECT id, name, slug, price, sale_price, image_url, short_description
             FROM {$table} WHERE id IN ({$placeholders}) AND status = 'published'",
            ...$ids
        ), ARRAY_A);
    }

    /**
     * Get or create session ID
     */
    private function get_session_id(): string
    {
        if (!isset($_COOKIE['cartt_session'])) {
            $session_id = wp_generate_uuid4();
            setcookie('cartt_session', $session_id, time() + DAY_IN_SECONDS * 30, '/');
            return $session_id;
        }
        return sanitize_text_field($_COOKIE['cartt_session']);
    }

    /**
     * Get recommendation widget HTML
     */
    public function render_widget(string $type, int $product_id = 0, array $args = []): string
    {
        $defaults = [
            'title' => $this->get_widget_title($type),
            'limit' => 4,
            'columns' => 4
        ];
        $args = wp_parse_args($args, $defaults);

        switch ($type) {
            case 'frequently_bought':
                $ids = $this->get_frequently_bought_together($product_id, $args['limit']);
                break;
            case 'similar':
                $ids = $this->get_similar_products($product_id, $args['limit']);
                break;
            case 'trending':
                $ids = $this->get_trending($args['limit']);
                break;
            case 'bestsellers':
                $ids = $this->get_bestsellers($args['limit']);
                break;
            case 'recently_viewed':
                $ids = $this->get_recently_viewed($args['limit']);
                break;
            default:
                $ids = [];
        }

        if (empty($ids)) return '';

        $products = $this->get_products_by_ids($ids);
        if (empty($products)) return '';

        ob_start();
        ?>
        <div class="cartt-recommendations cartt-recommendations-<?php echo esc_attr($type); ?>">
            <h3 class="cartt-recommendations-title"><?php echo esc_html($args['title']); ?></h3>
            <div class="cartt-recommendations-grid columns-<?php echo intval($args['columns']); ?>">
                <?php foreach ($products as $product): ?>
                <div class="cartt-recommendation-item">
                    <a href="<?php echo esc_url(home_url('/product/' . $product['slug'])); ?>">
                        <?php if ($product['image_url']): ?>
                        <img src="<?php echo esc_url($product['image_url']); ?>" alt="<?php echo esc_attr($product['name']); ?>">
                        <?php endif; ?>
                        <h4><?php echo esc_html($product['name']); ?></h4>
                        <span class="price">
                            <?php if ($product['sale_price']): ?>
                                <del><?php echo cartt_format_price($product['price']); ?></del>
                                <?php echo cartt_format_price($product['sale_price']); ?>
                            <?php else: ?>
                                <?php echo cartt_format_price($product['price']); ?>
                            <?php endif; ?>
                        </span>
                    </a>
                    <button class="cartt-add-to-cart" data-product-id="<?php echo intval($product['id']); ?>">
                        Add to Cart
                    </button>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Get default widget title
     */
    private function get_widget_title(string $type): string
    {
        $titles = [
            'frequently_bought' => 'Frequently Bought Together',
            'similar' => 'Similar Products',
            'trending' => 'Trending Now',
            'bestsellers' => 'Bestsellers',
            'recently_viewed' => 'Recently Viewed',
            'personalized' => 'Recommended for You'
        ];
        return $titles[$type] ?? 'You May Also Like';
    }
}
