<?php
/**
 * AI Content Generator Service
 * 
 * Generates product titles, descriptions, SEO meta using AI.
 * Requires 'ai_content' addon to be active.
 * 
 * @since 1.5.5
 */

namespace Cartt\Services;

class AIContentService
{
    private static ?self $instance = null;
    
    public static function getInstance(): self
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function __construct()
    {
        // Only load if ai_content addon is active
        if (!$this->isAddonActive()) {
            return;
        }
        
        // Admin hooks
        add_action('wp_ajax_cartt_ai_generate_title', [$this, 'ajaxGenerateTitle']);
        add_action('wp_ajax_cartt_ai_generate_description', [$this, 'ajaxGenerateDescription']);
        add_action('wp_ajax_cartt_ai_generate_seo', [$this, 'ajaxGenerateSEO']);
        add_action('wp_ajax_cartt_ai_generate_bulk', [$this, 'ajaxGenerateBulk']);
        
        // Add metabox to product edit screen
        add_action('add_meta_boxes', [$this, 'addMetaBox']);
        
        // Add admin menu
        add_action('admin_menu', [$this, 'addAdminMenu'], 99);
    }
    
    /**
     * Check if addon is active
     */
    private function isAddonActive(): bool
    {
        $activeAddons = get_option('cartt_active_addons', []);
        return in_array('ai_content', $activeAddons) || in_array('all_access', $activeAddons);
    }
    
    /**
     * Get license key
     */
    private function getLicenseKey(): string
    {
        $licenses = get_option('cartt_addon_licenses', []);
        return $licenses['ai_content']['key'] ?? $licenses['all_access']['key'] ?? '';
    }
    
    /**
     * Add admin menu
     */
    public function addAdminMenu(): void
    {
        add_submenu_page(
            'cartt',
            'AI Content',
            'AI Content',
            'manage_options',
            'cartt-ai-content',
            [$this, 'renderAdminPage']
        );
    }
    
    /**
     * Add metabox to product edit
     */
    public function addMetaBox(): void
    {
        add_meta_box(
            'cartt-ai-content',
            'AI Content Generator',
            [$this, 'renderMetaBox'],
            'product',
            'side',
            'high'
        );
    }
    
    /**
     * Render metabox
     */
    public function renderMetaBox($post): void
    {
        $nonce = wp_create_nonce('cartt_ai_content');
        ?>
        <div class="cartt-ai-metabox">
            <p style="margin-bottom:12px;color:#666;font-size:12px;">Generate content using AI based on product details.</p>
            
            <div style="display:flex;flex-direction:column;gap:8px;">
                <button type="button" class="button cartt-ai-btn" data-action="title" data-post="<?php echo $post->ID; ?>" data-nonce="<?php echo $nonce; ?>">
                    Generate Title
                </button>
                <button type="button" class="button cartt-ai-btn" data-action="description" data-post="<?php echo $post->ID; ?>" data-nonce="<?php echo $nonce; ?>">
                    Generate Description
                </button>
                <button type="button" class="button cartt-ai-btn" data-action="seo" data-post="<?php echo $post->ID; ?>" data-nonce="<?php echo $nonce; ?>">
                    Generate SEO Meta
                </button>
            </div>
            
            <div id="cartt-ai-result" style="margin-top:12px;display:none;">
                <textarea id="cartt-ai-output" rows="4" style="width:100%;"></textarea>
                <div style="margin-top:8px;display:flex;gap:8px;">
                    <button type="button" class="button button-primary" id="cartt-ai-apply">Apply</button>
                    <button type="button" class="button" id="cartt-ai-regenerate">Regenerate</button>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(function($) {
            var currentAction = '';
            var currentPost = 0;
            var currentNonce = '';
            
            $('.cartt-ai-btn').on('click', function() {
                var $btn = $(this);
                currentAction = $btn.data('action');
                currentPost = $btn.data('post');
                currentNonce = $btn.data('nonce');
                
                $btn.prop('disabled', true).text('Generating...');
                
                $.post(ajaxurl, {
                    action: 'cartt_ai_generate_' + currentAction,
                    post_id: currentPost,
                    nonce: currentNonce
                }, function(response) {
                    $btn.prop('disabled', false).text($btn.text().replace('Generating...', 'Generate ' + currentAction.charAt(0).toUpperCase() + currentAction.slice(1)));
                    
                    if (response.success) {
                        $('#cartt-ai-output').val(response.data.content);
                        $('#cartt-ai-result').show();
                    } else {
                        alert(response.data || 'Generation failed');
                    }
                });
            });
            
            $('#cartt-ai-apply').on('click', function() {
                var content = $('#cartt-ai-output').val();
                
                if (currentAction === 'title') {
                    $('#title').val(content);
                } else if (currentAction === 'description') {
                    if (typeof tinymce !== 'undefined' && tinymce.get('content')) {
                        tinymce.get('content').setContent(content);
                    } else {
                        $('#content').val(content);
                    }
                } else if (currentAction === 'seo') {
                    // Try Yoast or RankMath
                    if ($('#yoast_wpseo_metadesc').length) {
                        $('#yoast_wpseo_metadesc').val(content);
                    } else if ($('[name="rank_math_description"]').length) {
                        $('[name="rank_math_description"]').val(content);
                    }
                }
                
                $('#cartt-ai-result').hide();
            });
            
            $('#cartt-ai-regenerate').on('click', function() {
                $('.cartt-ai-btn[data-action="' + currentAction + '"]').click();
            });
        });
        </script>
        <?php
    }
    
    /**
     * Call AI API
     */
    private function callAI(string $prompt, string $context = ''): ?string
    {
        $licenseKey = $this->getLicenseKey();
        
        $response = wp_remote_post('https://cartt.app/wp-content/uploads/cartt-relay/relay.php', [
            'body' => [
                'action' => 'ai_content_generate',
                'license_key' => $licenseKey,
                'site_url' => home_url(),
                'prompt' => $prompt,
                'context' => $context
            ],
            'timeout' => 60
        ]);
        
        if (is_wp_error($response)) {
            return null;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        return $body['content'] ?? null;
    }
    
    /**
     * Get product context for AI
     */
    private function getProductContext(int $postId): array
    {
        $post = get_post($postId);
        if (!$post) {
            return [];
        }
        
        $product = function_exists('wc_get_product') ? wc_get_product($postId) : null;
        
        $context = [
            'title' => $post->post_title,
            'content' => wp_strip_all_tags($post->post_content),
            'excerpt' => $post->post_excerpt,
        ];
        
        if ($product) {
            $context['price'] = $product->get_price();
            $context['sku'] = $product->get_sku();
            $context['categories'] = wp_get_post_terms($postId, 'product_cat', ['fields' => 'names']);
            $context['tags'] = wp_get_post_terms($postId, 'product_tag', ['fields' => 'names']);
            $context['attributes'] = [];
            
            foreach ($product->get_attributes() as $attr) {
                if (is_object($attr)) {
                    $context['attributes'][$attr->get_name()] = $attr->get_options();
                }
            }
        }
        
        return $context;
    }
    
    /**
     * AJAX: Generate title
     */
    public function ajaxGenerateTitle(): void
    {
        check_ajax_referer('cartt_ai_content', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }
        
        $postId = intval($_POST['post_id'] ?? 0);
        $context = $this->getProductContext($postId);
        
        $prompt = "Generate a compelling, SEO-friendly product title for an e-commerce product. ";
        $prompt .= "Current title: " . ($context['title'] ?? 'Untitled') . ". ";
        $prompt .= "Description: " . substr($context['content'] ?? '', 0, 500) . ". ";
        $prompt .= "Categories: " . implode(', ', $context['categories'] ?? []) . ". ";
        $prompt .= "Return ONLY the title, no quotes or explanation. Keep it under 60 characters.";
        
        $result = $this->callAI($prompt, json_encode($context));
        
        if ($result) {
            wp_send_json_success(['content' => trim($result, '"\'')]);
        } else {
            wp_send_json_error('Failed to generate title');
        }
    }
    
    /**
     * AJAX: Generate description
     */
    public function ajaxGenerateDescription(): void
    {
        check_ajax_referer('cartt_ai_content', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }
        
        $postId = intval($_POST['post_id'] ?? 0);
        $context = $this->getProductContext($postId);
        
        $prompt = "Write a compelling product description for e-commerce. ";
        $prompt .= "Product: " . ($context['title'] ?? 'Product') . ". ";
        $prompt .= "Current description: " . substr($context['content'] ?? '', 0, 300) . ". ";
        $prompt .= "Categories: " . implode(', ', $context['categories'] ?? []) . ". ";
        $prompt .= "Price: $" . ($context['price'] ?? '0') . ". ";
        $prompt .= "Write 2-3 paragraphs that highlight benefits, features, and create desire. Use HTML formatting (p, ul, li, strong). ";
        $prompt .= "Be persuasive but honest. Do not include the price in the description.";
        
        $result = $this->callAI($prompt, json_encode($context));
        
        if ($result) {
            wp_send_json_success(['content' => $result]);
        } else {
            wp_send_json_error('Failed to generate description');
        }
    }
    
    /**
     * AJAX: Generate SEO meta
     */
    public function ajaxGenerateSEO(): void
    {
        check_ajax_referer('cartt_ai_content', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }
        
        $postId = intval($_POST['post_id'] ?? 0);
        $context = $this->getProductContext($postId);
        
        $prompt = "Write an SEO meta description for a product page. ";
        $prompt .= "Product: " . ($context['title'] ?? 'Product') . ". ";
        $prompt .= "Description: " . substr($context['content'] ?? '', 0, 300) . ". ";
        $prompt .= "Categories: " . implode(', ', $context['categories'] ?? []) . ". ";
        $prompt .= "Return ONLY the meta description. Must be 150-160 characters. ";
        $prompt .= "Include a call to action. Be compelling and include relevant keywords.";
        
        $result = $this->callAI($prompt, json_encode($context));
        
        if ($result) {
            wp_send_json_success(['content' => trim($result, '"\'')]);
        } else {
            wp_send_json_error('Failed to generate SEO meta');
        }
    }
    
    /**
     * AJAX: Bulk generate
     */
    public function ajaxGenerateBulk(): void
    {
        check_ajax_referer('cartt_ai_content', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Permission denied');
        }
        
        $postIds = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : [];
        $type = sanitize_text_field($_POST['type'] ?? 'description');
        
        if (empty($postIds)) {
            wp_send_json_error('No products selected');
        }
        
        $results = [];
        foreach ($postIds as $postId) {
            $_POST['post_id'] = $postId;
            
            ob_start();
            if ($type === 'title') {
                $this->ajaxGenerateTitle();
            } elseif ($type === 'description') {
                $this->ajaxGenerateDescription();
            } elseif ($type === 'seo') {
                $this->ajaxGenerateSEO();
            }
            $output = ob_get_clean();
            
            $response = json_decode($output, true);
            $results[$postId] = $response;
        }
        
        wp_send_json_success(['results' => $results]);
    }
    
    /**
     * Render admin page
     */
    public function renderAdminPage(): void
    {
        $nonce = wp_create_nonce('cartt_ai_content');
        ?>
        <div class="wrap cartt-admin">
            <h1>AI Content Generator</h1>
            <p>Generate product titles, descriptions, and SEO meta using AI.</p>
            
            <div class="cartt-ai-dashboard" style="display:grid;grid-template-columns:1fr 1fr;gap:24px;margin-top:24px;">
                <!-- Single Product -->
                <div class="cartt-card" style="background:#fff;border:1px solid #e0e0e0;border-radius:8px;padding:24px;">
                    <h2 style="margin-top:0;">Single Product</h2>
                    <p>Generate content for one product at a time from the product edit screen.</p>
                    <p>Look for the "AI Content Generator" box in the sidebar when editing a product.</p>
                    <a href="<?php echo admin_url('edit.php?post_type=product'); ?>" class="button button-primary">Go to Products</a>
                </div>
                
                <!-- Bulk Generation -->
                <div class="cartt-card" style="background:#fff;border:1px solid #e0e0e0;border-radius:8px;padding:24px;">
                    <h2 style="margin-top:0;">Bulk Generation</h2>
                    <p>Generate content for multiple products at once.</p>
                    
                    <div style="margin:16px 0;">
                        <label style="display:block;margin-bottom:8px;font-weight:600;">Content Type</label>
                        <select id="cartt-bulk-type" style="width:100%;">
                            <option value="title">Product Titles</option>
                            <option value="description">Product Descriptions</option>
                            <option value="seo">SEO Meta Descriptions</option>
                        </select>
                    </div>
                    
                    <div style="margin:16px 0;">
                        <label style="display:block;margin-bottom:8px;font-weight:600;">Products</label>
                        <select id="cartt-bulk-products" multiple style="width:100%;min-height:150px;">
                            <?php
                            $products = get_posts([
                                'post_type' => 'product',
                                'posts_per_page' => 100,
                                'orderby' => 'title',
                                'order' => 'ASC'
                            ]);
                            foreach ($products as $product) {
                                echo '<option value="' . $product->ID . '">' . esc_html($product->post_title) . '</option>';
                            }
                            ?>
                        </select>
                        <p class="description">Hold Ctrl/Cmd to select multiple products.</p>
                    </div>
                    
                    <button type="button" class="button button-primary" id="cartt-bulk-generate">Generate Content</button>
                    <div id="cartt-bulk-progress" style="margin-top:12px;display:none;">
                        <div style="background:#e0e0e0;border-radius:4px;height:8px;overflow:hidden;">
                            <div id="cartt-bulk-bar" style="background:#2271b1;height:100%;width:0%;transition:width 0.3s;"></div>
                        </div>
                        <p id="cartt-bulk-status" style="margin:8px 0;font-size:12px;color:#666;">Processing...</p>
                    </div>
                </div>
            </div>
            
            <!-- Usage Stats -->
            <div class="cartt-card" style="background:#fff;border:1px solid #e0e0e0;border-radius:8px;padding:24px;margin-top:24px;">
                <h2 style="margin-top:0;">Tips for Best Results</h2>
                <ul style="margin:0;padding-left:20px;">
                    <li>Add basic product information (title, category, attributes) before generating descriptions</li>
                    <li>Review and edit AI-generated content before publishing</li>
                    <li>Use specific categories and tags to help AI understand your product</li>
                    <li>For bulk generation, start with a small batch to review quality</li>
                </ul>
            </div>
        </div>
        
        <script>
        jQuery(function($) {
            $('#cartt-bulk-generate').on('click', function() {
                var type = $('#cartt-bulk-type').val();
                var products = $('#cartt-bulk-products').val();
                
                if (!products || products.length === 0) {
                    alert('Please select at least one product');
                    return;
                }
                
                if (products.length > 10) {
                    if (!confirm('You selected ' + products.length + ' products. This may take a while. Continue?')) {
                        return;
                    }
                }
                
                var $btn = $(this);
                $btn.prop('disabled', true);
                $('#cartt-bulk-progress').show();
                
                var total = products.length;
                var completed = 0;
                var results = [];
                
                function processNext() {
                    if (completed >= total) {
                        $btn.prop('disabled', false);
                        $('#cartt-bulk-status').text('Completed! ' + total + ' products processed.');
                        return;
                    }
                    
                    var productId = products[completed];
                    var productName = $('#cartt-bulk-products option[value="' + productId + '"]').text();
                    
                    $('#cartt-bulk-status').text('Processing: ' + productName + ' (' + (completed + 1) + '/' + total + ')');
                    $('#cartt-bulk-bar').css('width', ((completed / total) * 100) + '%');
                    
                    $.post(ajaxurl, {
                        action: 'cartt_ai_generate_' + type,
                        post_id: productId,
                        nonce: '<?php echo $nonce; ?>'
                    }, function(response) {
                        if (response.success) {
                            // Auto-apply to product
                            $.post(ajaxurl, {
                                action: 'cartt_ai_apply_content',
                                post_id: productId,
                                type: type,
                                content: response.data.content,
                                nonce: '<?php echo $nonce; ?>'
                            });
                        }
                        completed++;
                        $('#cartt-bulk-bar').css('width', ((completed / total) * 100) + '%');
                        setTimeout(processNext, 500); // Rate limiting
                    }).fail(function() {
                        completed++;
                        setTimeout(processNext, 500);
                    });
                }
                
                processNext();
            });
        });
        </script>
        <?php
    }
}
