<?php

declare(strict_types=1);

namespace Cartt\Frontend;

/**
 * Frontend Controller - handles the storefront
 */
class FrontendController
{
    public function __construct()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueueAssets']);
        add_action('init', [$this, 'registerShortcodes']);
        add_action('init', [$this, 'registerEndpoints']);
        add_filter('template_include', [$this, 'loadTemplates']);
    }

    public function enqueueAssets(): void
    {
        if (!$this->isCarttPage()) {
            return;
        }

        wp_enqueue_style(
            'cartt-frontend',
            CARTT_PLUGIN_URL . 'assets/css/frontend.css',
            [],
            CARTT_VERSION
        );

        wp_enqueue_script(
            'cartt-frontend',
            CARTT_PLUGIN_URL . 'assets/js/frontend.js',
            ['jquery'],
            CARTT_VERSION,
            true
        );

        // Get current currency info
        $currentCurrency = \Cartt\Services\CurrencyService::getCurrentCurrency();
        
        wp_localize_script('cartt-frontend', 'carttFrontend', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'apiUrl' => rest_url('cartt/v1/'),
            'nonce' => wp_create_nonce('cartt_frontend'),
            'currency' => $currentCurrency->code ?? get_option('cartt_currency', 'USD'),
            'currencySymbol' => $currentCurrency->symbol ?? '$',
            'currencyPosition' => $currentCurrency->symbol_position ?? get_option('cartt_currency_position', 'left'),
            'exchangeRate' => floatval($currentCurrency->exchange_rate ?? 1),
            'currencyDecimals' => intval($currentCurrency->decimals ?? 2),
            'thousandSeparator' => $currentCurrency->thousand_separator ?? ',',
            'decimalSeparator' => $currentCurrency->decimal_separator ?? '.',
            'cartUrl' => $this->getPageUrl('cart'),
            'checkoutUrl' => $this->getPageUrl('checkout'),
            'accountUrl' => $this->getPageUrl('account'),
        ]);
        
        // Social Proof settings
        $socialProofSettings = get_option('cartt_social_proof_settings', []);
        if (!empty($socialProofSettings['enabled'])) {
            $productId = 0;
            if (is_singular('cartt_product')) {
                $productId = get_the_ID();
            }
            
            wp_localize_script('cartt-frontend', 'carttSocialProof', [
                'enabled' => true,
                'productId' => $productId,
                'purchaseNotifications' => !empty($socialProofSettings['purchase_notifications']['enabled']),
                'notificationPosition' => $socialProofSettings['purchase_notifications']['position'] ?? 'bottom-left',
                'notificationDuration' => intval($socialProofSettings['purchase_notifications']['duration'] ?? 5),
                'notificationDelay' => intval($socialProofSettings['purchase_notifications']['delay'] ?? 10),
                'notificationMessage' => $socialProofSettings['purchase_notifications']['message_template'] ?? '{buyer} from {location} purchased',
                'anonymize' => !empty($socialProofSettings['purchase_notifications']['anonymize']),
                'visitorCount' => !empty($socialProofSettings['visitor_count']['enabled']),
                'visitorMinThreshold' => intval($socialProofSettings['visitor_count']['min_threshold'] ?? 1),
                'stockUrgency' => !empty($socialProofSettings['stock_urgency']['enabled']),
                'stockLowThreshold' => intval($socialProofSettings['stock_urgency']['low_stock_threshold'] ?? 10),
                'stockCriticalThreshold' => intval($socialProofSettings['stock_urgency']['critical_stock_threshold'] ?? 3),
                'showExactCount' => !empty($socialProofSettings['stock_urgency']['show_exact_count']),
                'cartActivity' => !empty($socialProofSettings['cart_activity']['enabled']),
                'cartActivityMin' => intval($socialProofSettings['cart_activity']['min_threshold'] ?? 1),
                'recentSales' => !empty($socialProofSettings['recent_sales']['enabled']),
                'fomoCountdown' => !empty($socialProofSettings['fomo_countdown']['enabled']),
            ]);
        }
    }

    public function registerShortcodes(): void
    {
        add_shortcode('cartt_products', [$this, 'shortcodeProducts']);
        add_shortcode('cartt_product', [$this, 'shortcodeProduct']);
        add_shortcode('cartt_cart', [$this, 'shortcodeCart']);
        add_shortcode('cartt_checkout', [$this, 'shortcodeCheckout']);
        add_shortcode('cartt_account', [$this, 'shortcodeAccount']);
        add_shortcode('cartt_orders', [$this, 'shortcodeOrders']);
        add_shortcode('cartt_order_received', [$this, 'shortcodeOrderReceived']);
        add_shortcode('cartt_currency_switcher', [$this, 'shortcodeCurrencySwitcher']);
        
        // v1.3.0 shortcodes
        add_shortcode('cartt_wishlist', [$this, 'shortcodeWishlist']);
        add_shortcode('cartt_search', [$this, 'shortcodeSearch']);
        add_shortcode('cartt_gift_card_balance', [$this, 'shortcodeGiftCardBalance']);
        add_shortcode('cartt_loyalty_points', [$this, 'shortcodeLoyaltyPoints']);
        add_shortcode('cartt_order_tracking', [$this, 'shortcodeOrderTracking']);
        
        // v1.4.0 shortcodes - Multi-vendor & Wholesale
        add_shortcode('cartt_vendor_store', [$this, 'shortcodeVendorStore']);
        add_shortcode('cartt_vendor_dashboard', [$this, 'shortcodeVendorDashboard']);
        add_shortcode('cartt_vendor_registration', [$this, 'shortcodeVendorRegistration']);
        add_shortcode('cartt_wholesale_application', [$this, 'shortcodeWholesaleApplication']);
        add_shortcode('cartt_bulk_order_form', [$this, 'shortcodeBulkOrderForm']);
        add_shortcode('cartt_quote_request', [$this, 'shortcodeQuoteRequest']);
        
        // v1.4.0 shortcodes - AI & Social Proof
        add_shortcode('cartt_recommendations', [$this, 'shortcodeRecommendations']);
        add_shortcode('cartt_recently_viewed', [$this, 'shortcodeRecentlyViewed']);
        add_shortcode('cartt_trending_products', [$this, 'shortcodeTrendingProducts']);
        add_shortcode('cartt_social_proof', [$this, 'shortcodeSocialProof']);
        
        // AJAX handler for currency switching
        add_action('wp_ajax_cartt_switch_currency', [$this, 'ajaxSwitchCurrency']);
        add_action('wp_ajax_nopriv_cartt_switch_currency', [$this, 'ajaxSwitchCurrency']);
        
        // v1.3.0 AJAX handlers
        add_action('wp_ajax_cartt_add_to_wishlist', [$this, 'ajaxAddToWishlist']);
        add_action('wp_ajax_nopriv_cartt_add_to_wishlist', [$this, 'ajaxAddToWishlist']);
        add_action('wp_ajax_cartt_remove_from_wishlist', [$this, 'ajaxRemoveFromWishlist']);
        add_action('wp_ajax_nopriv_cartt_remove_from_wishlist', [$this, 'ajaxRemoveFromWishlist']);
        add_action('wp_ajax_cartt_toggle_wishlist_public', [$this, 'ajaxToggleWishlistPublic']);
        add_action('wp_ajax_nopriv_cartt_toggle_wishlist_public', [$this, 'ajaxToggleWishlistPublic']);
        add_action('wp_ajax_cartt_wishlist_add_all_to_cart', [$this, 'ajaxWishlistAddAllToCart']);
        add_action('wp_ajax_nopriv_cartt_wishlist_add_all_to_cart', [$this, 'ajaxWishlistAddAllToCart']);
        
        add_action('wp_ajax_cartt_autocomplete', [$this, 'ajaxAutocomplete']);
        add_action('wp_ajax_nopriv_cartt_autocomplete', [$this, 'ajaxAutocomplete']);
        add_action('wp_ajax_cartt_search_products', [$this, 'ajaxSearchProducts']);
        add_action('wp_ajax_nopriv_cartt_search_products', [$this, 'ajaxSearchProducts']);
        
        add_action('wp_ajax_cartt_submit_question', [$this, 'ajaxSubmitQuestion']);
        add_action('wp_ajax_nopriv_cartt_submit_question', [$this, 'ajaxSubmitQuestion']);
        add_action('wp_ajax_cartt_submit_answer', [$this, 'ajaxSubmitAnswer']);
        add_action('wp_ajax_nopriv_cartt_submit_answer', [$this, 'ajaxSubmitAnswer']);
        add_action('wp_ajax_cartt_qa_upvote', [$this, 'ajaxQAUpvote']);
        add_action('wp_ajax_nopriv_cartt_qa_upvote', [$this, 'ajaxQAUpvote']);
        
        add_action('wp_ajax_cartt_check_gift_card', [$this, 'ajaxCheckGiftCard']);
        add_action('wp_ajax_nopriv_cartt_check_gift_card', [$this, 'ajaxCheckGiftCard']);
        
        add_action('wp_ajax_cartt_notify_stock', [$this, 'ajaxNotifyStock']);
        add_action('wp_ajax_nopriv_cartt_notify_stock', [$this, 'ajaxNotifyStock']);
        
        // v1.4.0 Social Proof AJAX handlers
        add_action('wp_ajax_cartt_get_recent_purchases', [$this, 'ajaxGetRecentPurchases']);
        add_action('wp_ajax_nopriv_cartt_get_recent_purchases', [$this, 'ajaxGetRecentPurchases']);
        add_action('wp_ajax_cartt_register_visitor', [$this, 'ajaxRegisterVisitor']);
        add_action('wp_ajax_nopriv_cartt_register_visitor', [$this, 'ajaxRegisterVisitor']);
        add_action('wp_ajax_cartt_get_visitor_count', [$this, 'ajaxGetVisitorCount']);
        add_action('wp_ajax_nopriv_cartt_get_visitor_count', [$this, 'ajaxGetVisitorCount']);
        add_action('wp_ajax_cartt_get_stock_status', [$this, 'ajaxGetStockStatus']);
        add_action('wp_ajax_nopriv_cartt_get_stock_status', [$this, 'ajaxGetStockStatus']);
        add_action('wp_ajax_cartt_get_cart_activity', [$this, 'ajaxGetCartActivity']);
        add_action('wp_ajax_nopriv_cartt_get_cart_activity', [$this, 'ajaxGetCartActivity']);
        add_action('wp_ajax_cartt_get_recent_sales', [$this, 'ajaxGetRecentSales']);
        add_action('wp_ajax_nopriv_cartt_get_recent_sales', [$this, 'ajaxGetRecentSales']);
    }

    public function registerEndpoints(): void
    {
        add_rewrite_endpoint('view-order', EP_PAGES);
        add_rewrite_endpoint('downloads', EP_PAGES);
        add_rewrite_endpoint('subscriptions', EP_PAGES);
        add_rewrite_endpoint('bookings', EP_PAGES);
    }

    public function loadTemplates(string $template): string
    {
        if (is_singular('cartt_product')) {
            $custom_template = $this->locateTemplate('single-product.php');
            if ($custom_template) {
                return $custom_template;
            }
        }

        if (is_post_type_archive('cartt_product')) {
            $custom_template = $this->locateTemplate('archive-product.php');
            if ($custom_template) {
                return $custom_template;
            }
        }

        return $template;
    }

    public function shortcodeProducts(array $atts = []): string
    {
        $atts = shortcode_atts([
            'columns' => 4,
            'limit' => 12,
            'category' => '',
            'orderby' => 'date',
            'order' => 'DESC',
        ], $atts);

        ob_start();
        include $this->locateTemplate('shortcodes/products.php');
        return ob_get_clean();
    }

    public function shortcodeProduct(array $atts = []): string
    {
        $atts = shortcode_atts([
            'id' => 0,
            'sku' => '',
        ], $atts);

        ob_start();
        include $this->locateTemplate('shortcodes/product.php');
        return ob_get_clean();
    }

    public function shortcodeCart(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/cart.php');
        return ob_get_clean();
    }

    public function shortcodeCheckout(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/checkout.php');
        return ob_get_clean();
    }

    public function shortcodeAccount(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/account.php');
        return ob_get_clean();
    }

    public function shortcodeOrders(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/orders.php');
        return ob_get_clean();
    }

    public function shortcodeOrderReceived(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/order-received.php');
        return ob_get_clean();
    }

    private function locateTemplate(string $template_name): ?string
    {
        // Check theme first
        $theme_template = locate_template(['cartt/' . $template_name]);
        if ($theme_template) {
            return $theme_template;
        }

        // Fall back to plugin
        $plugin_template = CARTT_PLUGIN_DIR . 'templates/' . $template_name;
        if (file_exists($plugin_template)) {
            return $plugin_template;
        }

        return null;
    }

    private function isCarttPage(): bool
    {
        global $post;

        if (!$post) {
            return false;
        }

        $cartt_shortcodes = [
            'cartt_products',
            'cartt_product',
            'cartt_cart',
            'cartt_checkout',
            'cartt_account',
            'cartt_orders',
        ];

        foreach ($cartt_shortcodes as $shortcode) {
            if (has_shortcode($post->post_content, $shortcode)) {
                return true;
            }
        }

        return false;
    }

    private function getPageUrl(string $page): string
    {
        $page_id = get_option("cartt_{$page}_page_id");
        return $page_id ? get_permalink($page_id) : home_url("/{$page}/");
    }

    public function shortcodeCurrencySwitcher(array $atts = []): string
    {
        return \Cartt\Services\CurrencyService::getCurrencySelectorHtml();
    }

    public function ajaxSwitchCurrency(): void
    {
        $code = sanitize_text_field($_POST['currency'] ?? '');
        
        if (empty($code)) {
            wp_send_json_error(['message' => 'No currency specified']);
        }

        $success = \Cartt\Services\CurrencyService::setCurrentCurrency($code);
        
        if ($success) {
            wp_send_json_success(['currency' => $code]);
        } else {
            wp_send_json_error(['message' => 'Invalid currency']);
        }
    }

    // v1.3.0 Shortcodes

    public function shortcodeWishlist(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/wishlist.php');
        return ob_get_clean();
    }

    public function shortcodeSearch(array $atts = []): string
    {
        ob_start();
        include $this->locateTemplate('shortcodes/search.php');
        return ob_get_clean();
    }

    public function shortcodeGiftCardBalance(array $atts = []): string
    {
        ob_start();
        ?>
        <div class="cartt-gift-card-balance">
            <h3>Check Gift Card Balance</h3>
            <form id="cartt-gc-balance-form">
                <input type="text" id="cartt-gc-code" placeholder="Enter gift card code" required>
                <button type="submit" class="cartt-btn cartt-btn-primary">Check Balance</button>
            </form>
            <div id="cartt-gc-result" style="display: none; margin-top: 15px; padding: 15px; background: #f9f9f9; border-radius: 8px;"></div>
        </div>
        <script>
        document.getElementById('cartt-gc-balance-form').addEventListener('submit', function(e) {
            e.preventDefault();
            const code = document.getElementById('cartt-gc-code').value;
            const result = document.getElementById('cartt-gc-result');
            
            fetch(carttFrontend.ajaxUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: 'action=cartt_check_gift_card&code=' + encodeURIComponent(code)
            })
            .then(r => r.json())
            .then(data => {
                result.style.display = 'block';
                if (data.success && data.data.found) {
                    result.innerHTML = '<strong>Balance:</strong> ' + data.data.balance_formatted + 
                        (data.data.expires_at ? '<br><small>Expires: ' + data.data.expires_at + '</small>' : '');
                } else {
                    result.innerHTML = '<span style="color: #dc3545;">Gift card not found</span>';
                }
            });
        });
        </script>
        <?php
        return ob_get_clean();
    }

    public function shortcodeLoyaltyPoints(array $atts = []): string
    {
        if (!is_user_logged_in()) {
            return '<p>Please <a href="' . wp_login_url(get_permalink()) . '">log in</a> to view your loyalty points.</p>';
        }

        $customerId = $this->getCustomerId();
        if (!$customerId) {
            return '';
        }

        $points = \Cartt\Services\LoyaltyService::getCustomerPoints($customerId);
        $settings = \Cartt\Services\LoyaltyService::getSettings();
        $transactions = \Cartt\Services\LoyaltyService::getTransactions($customerId, 10);

        ob_start();
        ?>
        <div class="cartt-loyalty-widget">
            <div class="cartt-loyalty-header">
                <div class="cartt-loyalty-points">
                    <span class="points-value"><?php echo number_format($points->points); ?></span>
                    <span class="points-label">Points</span>
                </div>
                <div class="cartt-loyalty-tier">
                    <span class="tier-badge tier-<?php echo esc_attr($points->tier); ?>"><?php echo esc_html(ucfirst($points->tier)); ?></span>
                    <?php if ($points->next_tier): ?>
                    <small><?php echo number_format($points->next_tier['points_needed']); ?> points to <?php echo esc_html($points->next_tier['tier']['name']); ?></small>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="cartt-loyalty-value">
                Worth: <strong><?php echo \Cartt\Services\CurrencyService::formatPrice(\Cartt\Services\LoyaltyService::getPointsValue($points->points)); ?></strong>
                <small>(<?php echo number_format($settings['points_value'], 3); ?>/point)</small>
            </div>

            <?php if (!empty($transactions)): ?>
            <div class="cartt-loyalty-history">
                <h4>Recent Activity</h4>
                <ul>
                    <?php foreach ($transactions as $t): ?>
                    <li>
                        <span class="<?php echo $t->points > 0 ? 'earned' : 'redeemed'; ?>">
                            <?php echo $t->points > 0 ? '+' : ''; ?><?php echo number_format($t->points); ?>
                        </span>
                        <span class="desc"><?php echo esc_html($t->description ?: ucfirst($t->type)); ?></span>
                        <span class="date"><?php echo date('M j', strtotime($t->created_at)); ?></span>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
        </div>
        <style>
        .cartt-loyalty-widget { background: #fff; border: 1px solid #eee; border-radius: 12px; padding: 20px; max-width: 400px; }
        .cartt-loyalty-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; }
        .cartt-loyalty-points { text-align: center; }
        .cartt-loyalty-points .points-value { display: block; font-size: 36px; font-weight: bold; }
        .cartt-loyalty-points .points-label { font-size: 12px; color: #666; text-transform: uppercase; }
        .tier-badge { display: inline-block; padding: 4px 12px; border-radius: 20px; font-size: 12px; font-weight: 500; }
        .tier-bronze { background: #cd7f32; color: #fff; }
        .tier-silver { background: #c0c0c0; color: #333; }
        .tier-gold { background: #ffd700; color: #333; }
        .tier-platinum { background: #e5e4e2; color: #333; }
        .cartt-loyalty-value { text-align: center; padding: 10px; background: #f9f9f9; border-radius: 8px; margin-bottom: 15px; }
        .cartt-loyalty-history h4 { margin: 0 0 10px; font-size: 14px; }
        .cartt-loyalty-history ul { list-style: none; margin: 0; padding: 0; }
        .cartt-loyalty-history li { display: flex; gap: 10px; padding: 8px 0; border-bottom: 1px solid #f0f0f0; font-size: 13px; }
        .cartt-loyalty-history .earned { color: #28a745; font-weight: 500; }
        .cartt-loyalty-history .redeemed { color: #dc3545; }
        .cartt-loyalty-history .desc { flex: 1; }
        .cartt-loyalty-history .date { color: #999; }
        </style>
        <?php
        return ob_get_clean();
    }

    public function shortcodeOrderTracking(array $atts = []): string
    {
        $orderId = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
        
        if (!$orderId) {
            ob_start();
            ?>
            <div class="cartt-order-tracking-form">
                <h3>Track Your Order</h3>
                <form method="get">
                    <input type="number" name="order_id" placeholder="Order Number" required>
                    <button type="submit" class="cartt-btn cartt-btn-primary">Track</button>
                </form>
            </div>
            <?php
            return ob_get_clean();
        }

        return \Cartt\Services\OrderTrackingService::getTrackingPageHtml($orderId);
    }

    // v1.3.0 AJAX Handlers

    public function ajaxAddToWishlist(): void
    {
        $productId = intval($_POST['product_id'] ?? 0);
        $variationId = !empty($_POST['variation_id']) ? intval($_POST['variation_id']) : null;

        $userId = get_current_user_id();
        $sessionId = !$userId ? (isset($_COOKIE['cartt_session']) ? sanitize_text_field($_COOKIE['cartt_session']) : null) : null;

        $wishlist = \Cartt\Services\WishlistService::getWishlist($userId, $sessionId);
        
        if (\Cartt\Services\WishlistService::addItem($wishlist->id, $productId, $variationId)) {
            wp_send_json_success(['count' => \Cartt\Services\WishlistService::getItemCount($wishlist->id)]);
        } else {
            wp_send_json_error(['message' => 'Failed to add to wishlist']);
        }
    }

    public function ajaxRemoveFromWishlist(): void
    {
        check_ajax_referer('cartt_wishlist', 'nonce');

        $productId = intval($_POST['product_id'] ?? 0);
        $variationId = !empty($_POST['variation_id']) ? intval($_POST['variation_id']) : null;

        $userId = get_current_user_id();
        $sessionId = !$userId ? (isset($_COOKIE['cartt_session']) ? sanitize_text_field($_COOKIE['cartt_session']) : null) : null;

        $wishlist = \Cartt\Services\WishlistService::getWishlist($userId, $sessionId);
        
        if (\Cartt\Services\WishlistService::removeItem($wishlist->id, $productId, $variationId)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to remove from wishlist']);
        }
    }

    public function ajaxToggleWishlistPublic(): void
    {
        check_ajax_referer('cartt_wishlist', 'nonce');

        $isPublic = !empty($_POST['is_public']);

        $userId = get_current_user_id();
        $sessionId = !$userId ? (isset($_COOKIE['cartt_session']) ? sanitize_text_field($_COOKIE['cartt_session']) : null) : null;

        $wishlist = \Cartt\Services\WishlistService::getWishlist($userId, $sessionId);
        
        if (\Cartt\Services\WishlistService::setPublic($wishlist->id, $isPublic)) {
            wp_send_json_success(['share_key' => $wishlist->share_key]);
        } else {
            wp_send_json_error(['message' => 'Failed to update wishlist']);
        }
    }

    public function ajaxWishlistAddAllToCart(): void
    {
        check_ajax_referer('cartt_wishlist', 'nonce');

        $userId = get_current_user_id();
        $sessionId = !$userId ? (isset($_COOKIE['cartt_session']) ? sanitize_text_field($_COOKIE['cartt_session']) : null) : null;

        $wishlist = \Cartt\Services\WishlistService::getWishlist($userId, $sessionId);
        $added = \Cartt\Services\WishlistService::moveAllToCart($wishlist->id);

        wp_send_json_success(['added' => $added]);
    }

    public function ajaxAutocomplete(): void
    {
        $query = sanitize_text_field($_POST['query'] ?? '');
        
        if (strlen($query) < 2) {
            wp_send_json_success([]);
            return;
        }

        $results = \Cartt\Services\ProductSearchService::autocomplete($query);
        wp_send_json_success($results);
    }

    public function ajaxSearchProducts(): void
    {
        $query = sanitize_text_field($_POST['query'] ?? '');
        $page = intval($_POST['page'] ?? 1);
        $limit = 12;
        $offset = ($page - 1) * $limit;

        $filters = [
            'category' => sanitize_text_field($_POST['category'] ?? ''),
            'min_price' => floatval($_POST['min_price'] ?? 0),
            'max_price' => floatval($_POST['max_price'] ?? 0),
            'min_rating' => floatval($_POST['min_rating'] ?? 0),
            'in_stock' => !empty($_POST['in_stock']),
            'on_sale' => !empty($_POST['on_sale']),
            'sort' => sanitize_text_field($_POST['sort'] ?? 'newest'),
        ];

        $results = \Cartt\Services\ProductSearchService::search($query, $filters, $limit, $offset);
        
        // Log search
        if ($query) {
            \Cartt\Services\ProductSearchService::logSearch($query, $results['total']);
        }

        wp_send_json_success($results);
    }

    public function ajaxSubmitQuestion(): void
    {
        check_ajax_referer('cartt_qa', 'nonce');

        $id = \Cartt\Services\ProductQAService::submitQuestion([
            'product_id' => intval($_POST['product_id'] ?? 0),
            'user_id' => get_current_user_id() ?: null,
            'author_name' => sanitize_text_field($_POST['author_name'] ?? ''),
            'author_email' => sanitize_email($_POST['author_email'] ?? ''),
            'content' => sanitize_textarea_field($_POST['content'] ?? ''),
        ]);

        if ($id) {
            wp_send_json_success(['id' => $id]);
        } else {
            wp_send_json_error(['message' => 'Failed to submit question']);
        }
    }

    public function ajaxSubmitAnswer(): void
    {
        check_ajax_referer('cartt_qa', 'nonce');

        $id = \Cartt\Services\ProductQAService::submitAnswer([
            'question_id' => intval($_POST['question_id'] ?? 0),
            'user_id' => get_current_user_id() ?: null,
            'author_name' => sanitize_text_field($_POST['author_name'] ?? ''),
            'author_email' => sanitize_email($_POST['author_email'] ?? ''),
            'content' => sanitize_textarea_field($_POST['content'] ?? ''),
        ]);

        if ($id) {
            wp_send_json_success(['id' => $id]);
        } else {
            wp_send_json_error(['message' => 'Failed to submit answer']);
        }
    }

    public function ajaxQAUpvote(): void
    {
        $id = intval($_POST['id'] ?? 0);
        
        if (\Cartt\Services\ProductQAService::upvote($id)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to upvote']);
        }
    }

    public function ajaxCheckGiftCard(): void
    {
        $code = sanitize_text_field($_POST['code'] ?? '');
        $result = \Cartt\Services\GiftCardService::checkBalance($code);
        
        if ($result['found']) {
            $result['balance_formatted'] = \Cartt\Services\CurrencyService::formatPrice($result['balance']);
        }

        wp_send_json_success($result);
    }

    public function ajaxNotifyStock(): void
    {
        $productId = intval($_POST['product_id'] ?? 0);
        $email = sanitize_email($_POST['email'] ?? '');
        $variationId = !empty($_POST['variation_id']) ? intval($_POST['variation_id']) : null;

        if (!$productId || !$email) {
            wp_send_json_error(['message' => 'Invalid data']);
        }

        if (\Cartt\Services\InventoryService::registerStockNotification($productId, $email, $variationId, get_current_user_id() ?: null)) {
            wp_send_json_success(['message' => 'You will be notified when this item is back in stock']);
        } else {
            wp_send_json_error(['message' => 'Failed to register notification']);
        }
    }

    private function getCustomerId(): ?int
    {
        if (!is_user_logged_in()) {
            return null;
        }

        global $wpdb;
        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}cartt_customers WHERE user_id = %d",
            get_current_user_id()
        )) ?: null;
    }

    // =========================================================================
    // v1.4.0 Shortcodes - Multi-vendor Marketplace
    // =========================================================================

    /**
     * Vendor store front shortcode
     * [cartt_vendor_store vendor="slug-or-id"]
     */
    public function shortcodeVendorStore(array $atts = []): string
    {
        $atts = shortcode_atts([
            'vendor' => get_query_var('cartt_vendor', ''),
            'products_per_page' => 12,
            'show_header' => 'yes',
        ], $atts);

        $service = \Cartt\Services\VendorService::instance();
        
        // Get vendor by slug or ID
        $vendor = is_numeric($atts['vendor']) 
            ? $service->getVendor((int)$atts['vendor'])
            : $service->getVendorBySlug(sanitize_title($atts['vendor']));

        if (!$vendor) {
            return '<div class="cartt-notice cartt-notice-error">Vendor not found.</div>';
        }

        $products = $service->getVendorProducts($vendor['id'], [
            'limit' => (int)$atts['products_per_page'],
            'status' => 'active'
        ]);

        ob_start();
        include $this->locateTemplate('shortcodes/vendor-store.php');
        return ob_get_clean();
    }

    /**
     * Vendor dashboard shortcode (requires logged-in vendor)
     * [cartt_vendor_dashboard]
     */
    public function shortcodeVendorDashboard(array $atts = []): string
    {
        if (!is_user_logged_in()) {
            return '<div class="cartt-notice cartt-notice-warning">Please <a href="' . wp_login_url(get_permalink()) . '">log in</a> to access your vendor dashboard.</div>';
        }

        $service = \Cartt\Services\VendorService::instance();
        $vendor = $service->getVendorByUserId(get_current_user_id());

        if (!$vendor) {
            return '<div class="cartt-notice cartt-notice-info">You are not registered as a vendor. <a href="' . esc_url(get_permalink(get_option('cartt_vendor_registration_page'))) . '">Apply to become a vendor</a>.</div>';
        }

        $tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'overview';
        $commissionService = \Cartt\Services\VendorCommissionService::instance();
        
        $data = [
            'vendor' => $vendor,
            'products' => $service->getVendorProducts($vendor['id']),
            'orders' => $service->getVendorOrders($vendor['id'], ['limit' => 10]),
            'earnings' => $commissionService->getVendorBalance($vendor['id']),
            'tab' => $tab
        ];

        ob_start();
        include $this->locateTemplate('shortcodes/vendor-dashboard.php');
        return ob_get_clean();
    }

    /**
     * Vendor registration form shortcode
     * [cartt_vendor_registration]
     */
    public function shortcodeVendorRegistration(array $atts = []): string
    {
        if (is_user_logged_in()) {
            $service = \Cartt\Services\VendorService::instance();
            $existing = $service->getVendorByUserId(get_current_user_id());
            
            if ($existing) {
                if ($existing['status'] === 'active') {
                    return '<div class="cartt-notice cartt-notice-success">You are already registered as a vendor. <a href="' . esc_url(get_permalink(get_option('cartt_vendor_dashboard_page'))) . '">Go to your dashboard</a>.</div>';
                } elseif ($existing['status'] === 'pending') {
                    return '<div class="cartt-notice cartt-notice-info">Your vendor application is pending approval. We will notify you once it has been reviewed.</div>';
                }
            }
        }

        $submitted = false;
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_vendor_register'])) {
            if (!wp_verify_nonce($_POST['_wpnonce'], 'cartt_vendor_registration')) {
                $error = 'Security check failed. Please try again.';
            } else {
                $service = \Cartt\Services\VendorService::instance();
                $result = $service->registerVendor([
                    'store_name' => sanitize_text_field($_POST['store_name'] ?? ''),
                    'store_description' => wp_kses_post($_POST['store_description'] ?? ''),
                    'email' => sanitize_email($_POST['email'] ?? ''),
                    'phone' => sanitize_text_field($_POST['phone'] ?? ''),
                    'address' => sanitize_textarea_field($_POST['address'] ?? ''),
                    'user_id' => get_current_user_id()
                ]);
                
                if ($result['success']) {
                    $submitted = true;
                } else {
                    $error = $result['message'];
                }
            }
        }

        ob_start();
        include $this->locateTemplate('shortcodes/vendor-registration.php');
        return ob_get_clean();
    }

    // =========================================================================
    // v1.4.0 Shortcodes - Wholesale & B2B
    // =========================================================================

    /**
     * Wholesale application form shortcode
     * [cartt_wholesale_application]
     */
    public function shortcodeWholesaleApplication(array $atts = []): string
    {
        $service = \Cartt\Services\WholesaleService::instance();
        
        if (is_user_logged_in()) {
            $customer = $service->getWholesaleCustomer(get_current_user_id());
            if ($customer) {
                if ($customer['status'] === 'active') {
                    return '<div class="cartt-notice cartt-notice-success">You are already approved as a wholesale customer. Wholesale pricing is automatically applied to your orders.</div>';
                } elseif ($customer['status'] === 'pending') {
                    return '<div class="cartt-notice cartt-notice-info">Your wholesale application is under review. We will notify you once a decision has been made.</div>';
                }
            }
        }

        $submitted = false;
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_wholesale_apply'])) {
            if (!wp_verify_nonce($_POST['_wpnonce'], 'cartt_wholesale_application')) {
                $error = 'Security check failed. Please try again.';
            } else {
                $result = $service->submitApplication([
                    'company_name' => sanitize_text_field($_POST['company_name'] ?? ''),
                    'contact_name' => sanitize_text_field($_POST['contact_name'] ?? ''),
                    'email' => sanitize_email($_POST['email'] ?? ''),
                    'phone' => sanitize_text_field($_POST['phone'] ?? ''),
                    'tax_id' => sanitize_text_field($_POST['tax_id'] ?? ''),
                    'business_type' => sanitize_text_field($_POST['business_type'] ?? ''),
                    'estimated_monthly_volume' => sanitize_text_field($_POST['estimated_volume'] ?? ''),
                    'message' => sanitize_textarea_field($_POST['message'] ?? ''),
                    'user_id' => get_current_user_id()
                ]);
                
                if ($result['success']) {
                    $submitted = true;
                } else {
                    $error = $result['message'];
                }
            }
        }

        ob_start();
        include $this->locateTemplate('shortcodes/wholesale-application.php');
        return ob_get_clean();
    }

    /**
     * Bulk order form shortcode for wholesale customers
     * [cartt_bulk_order_form category="optional-category-slug"]
     */
    public function shortcodeBulkOrderForm(array $atts = []): string
    {
        $atts = shortcode_atts([
            'category' => '',
            'show_quantity_breaks' => 'yes',
        ], $atts);

        $service = \Cartt\Services\WholesaleService::instance();
        $isWholesale = false;
        $tier = null;
        
        if (is_user_logged_in()) {
            $customer = $service->getWholesaleCustomer(get_current_user_id());
            if ($customer && $customer['status'] === 'active') {
                $isWholesale = true;
                $tier = $customer['tier'];
            }
        }

        // Get products with wholesale pricing
        global $wpdb;
        $where = "p.status = 'active'";
        if ($atts['category']) {
            $where .= $wpdb->prepare(" AND p.id IN (SELECT product_id FROM {$wpdb->prefix}cartt_product_categories pc JOIN {$wpdb->prefix}cartt_categories c ON c.id = pc.category_id WHERE c.slug = %s)", $atts['category']);
        }
        
        $products = $wpdb->get_results("SELECT p.* FROM {$wpdb->prefix}cartt_products p WHERE {$where} ORDER BY p.name ASC", ARRAY_A);
        
        // Add quantity pricing for each product
        foreach ($products as &$product) {
            $product['quantity_pricing'] = $service->getQuantityPricing($product['id']);
        }

        ob_start();
        include $this->locateTemplate('shortcodes/bulk-order-form.php');
        return ob_get_clean();
    }

    /**
     * Quote request form shortcode
     * [cartt_quote_request]
     */
    public function shortcodeQuoteRequest(array $atts = []): string
    {
        $service = \Cartt\Services\WholesaleService::instance();
        $submitted = false;
        $error = '';
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cartt_request_quote'])) {
            if (!wp_verify_nonce($_POST['_wpnonce'], 'cartt_quote_request')) {
                $error = 'Security check failed. Please try again.';
            } else {
                $items = [];
                if (!empty($_POST['items']) && is_array($_POST['items'])) {
                    foreach ($_POST['items'] as $item) {
                        if (!empty($item['product_id']) && !empty($item['quantity'])) {
                            $items[] = [
                                'product_id' => intval($item['product_id']),
                                'quantity' => intval($item['quantity']),
                                'notes' => sanitize_text_field($item['notes'] ?? '')
                            ];
                        }
                    }
                }
                
                if (empty($items)) {
                    $error = 'Please add at least one product to your quote request.';
                } else {
                    $result = $service->createQuote([
                        'customer_id' => $this->getCustomerId(),
                        'contact_name' => sanitize_text_field($_POST['contact_name'] ?? ''),
                        'email' => sanitize_email($_POST['email'] ?? ''),
                        'phone' => sanitize_text_field($_POST['phone'] ?? ''),
                        'company' => sanitize_text_field($_POST['company'] ?? ''),
                        'notes' => sanitize_textarea_field($_POST['notes'] ?? ''),
                        'items' => $items
                    ]);
                    
                    if ($result['success']) {
                        $submitted = true;
                    } else {
                        $error = $result['message'];
                    }
                }
            }
        }

        // Get products for selection
        global $wpdb;
        $products = $wpdb->get_results("SELECT id, name, sku, price FROM {$wpdb->prefix}cartt_products WHERE status = 'active' ORDER BY name ASC", ARRAY_A);

        ob_start();
        include $this->locateTemplate('shortcodes/quote-request.php');
        return ob_get_clean();
    }

    // =========================================================================
    // v1.4.0 Shortcodes - AI Recommendations
    // =========================================================================

    /**
     * AI product recommendations shortcode
     * [cartt_recommendations type="personalized" limit="4" product_id="optional"]
     */
    public function shortcodeRecommendations(array $atts = []): string
    {
        $atts = shortcode_atts([
            'type' => 'personalized',
            'limit' => 4,
            'product_id' => 0,
            'title' => '',
            'columns' => 4
        ], $atts);

        $service = \Cartt\Services\AIRecommendationService::instance();
        $products = [];
        $title = $atts['title'];

        switch ($atts['type']) {
            case 'frequently_bought':
                if ($atts['product_id']) {
                    $products = $service->getFrequentlyBoughtTogether($atts['product_id'], $atts['limit']);
                    $title = $title ?: 'Frequently Bought Together';
                }
                break;
            case 'similar':
                if ($atts['product_id']) {
                    $products = $service->getSimilarProducts($atts['product_id'], $atts['limit']);
                    $title = $title ?: 'Similar Products';
                }
                break;
            case 'trending':
                $products = $service->getTrendingProducts($atts['limit']);
                $title = $title ?: 'Trending Now';
                break;
            case 'bestsellers':
                $products = $service->getBestsellers($atts['limit']);
                $title = $title ?: 'Best Sellers';
                break;
            case 'personalized':
            default:
                $products = $service->getPersonalizedRecommendations(get_current_user_id() ?: null, $atts['limit']);
                $title = $title ?: 'Recommended For You';
                break;
        }

        if (empty($products)) {
            return '';
        }

        ob_start();
        include $this->locateTemplate('shortcodes/recommendations.php');
        return ob_get_clean();
    }

    /**
     * Recently viewed products shortcode
     * [cartt_recently_viewed limit="6"]
     */
    public function shortcodeRecentlyViewed(array $atts = []): string
    {
        $atts = shortcode_atts([
            'limit' => 6,
            'title' => 'Recently Viewed',
            'columns' => 4
        ], $atts);

        $service = \Cartt\Services\AIRecommendationService::instance();
        $products = $service->getRecentlyViewed(get_current_user_id() ?: null, $atts['limit']);

        if (empty($products)) {
            return '';
        }

        $title = $atts['title'];
        ob_start();
        include $this->locateTemplate('shortcodes/recommendations.php');
        return ob_get_clean();
    }

    /**
     * Trending products shortcode
     * [cartt_trending_products limit="8" period="7"]
     */
    public function shortcodeTrendingProducts(array $atts = []): string
    {
        $atts = shortcode_atts([
            'limit' => 8,
            'period' => 7,
            'title' => 'Trending Now',
            'columns' => 4
        ], $atts);

        $service = \Cartt\Services\AIRecommendationService::instance();
        $products = $service->getTrendingProducts($atts['limit'], $atts['period']);

        if (empty($products)) {
            return '';
        }

        $title = $atts['title'];
        ob_start();
        include $this->locateTemplate('shortcodes/recommendations.php');
        return ob_get_clean();
    }

    // =========================================================================
    // v1.4.0 Shortcodes - Social Proof
    // =========================================================================

    /**
     * Social proof display shortcode
     * [cartt_social_proof type="visitor_count" product_id="optional"]
     */
    public function shortcodeSocialProof(array $atts = []): string
    {
        $atts = shortcode_atts([
            'type' => 'visitor_count',
            'product_id' => 0,
        ], $atts);

        $settings = get_option('cartt_social_proof_settings', []);
        if (empty($settings['enabled'])) {
            return '';
        }

        $service = \Cartt\Services\SocialProofService::instance();
        $output = '';

        switch ($atts['type']) {
            case 'visitor_count':
                if (empty($settings['visitor_count']['enabled'])) {
                    return '';
                }
                $count = $service->getActiveViewers($atts['product_id'] ?: get_the_ID());
                if ($count >= ($settings['visitor_count']['minimum_threshold'] ?? 2)) {
                    $message = str_replace('{count}', $count, $settings['visitor_count']['message'] ?? '{count} people are viewing this');
                    $output = '<div class="cartt-social-proof cartt-visitor-count"><span class="dashicons dashicons-visibility"></span> ' . esc_html($message) . '</div>';
                }
                break;

            case 'cart_activity':
                if (empty($settings['cart_activity']['enabled'])) {
                    return '';
                }
                $count = $service->getCartCount($atts['product_id'] ?: get_the_ID());
                if ($count >= ($settings['cart_activity']['minimum_threshold'] ?? 1)) {
                    $message = str_replace('{count}', $count, $settings['cart_activity']['message'] ?? '{count} people have this in cart');
                    $output = '<div class="cartt-social-proof cartt-cart-activity"><span class="dashicons dashicons-cart"></span> ' . esc_html($message) . '</div>';
                }
                break;

            case 'recent_sales':
                if (empty($settings['recent_sales']['enabled'])) {
                    return '';
                }
                $count = $service->getRecentSalesCount($atts['product_id'] ?: get_the_ID());
                if ($count > 0) {
                    $message = str_replace('{count}', $count, $settings['recent_sales']['message'] ?? '{count} sold recently');
                    $output = '<div class="cartt-social-proof cartt-recent-sales"><span class="dashicons dashicons-chart-line"></span> ' . esc_html($message) . '</div>';
                }
                break;

            case 'stock_urgency':
                if (empty($settings['stock_urgency']['enabled'])) {
                    return '';
                }
                $productId = $atts['product_id'] ?: get_the_ID();
                global $wpdb;
                $stock = (int)$wpdb->get_var($wpdb->prepare(
                    "SELECT stock_quantity FROM {$wpdb->prefix}cartt_products WHERE id = %d",
                    $productId
                ));
                
                $critical = $settings['stock_urgency']['critical_stock_threshold'] ?? 3;
                $low = $settings['stock_urgency']['low_stock_threshold'] ?? 10;
                
                if ($stock <= $critical && $stock > 0) {
                    $message = $settings['stock_urgency']['show_exact_count']
                        ? str_replace('{count}', $stock, $settings['stock_urgency']['critical_stock_message'])
                        : $settings['stock_urgency']['critical_stock_message'];
                    $output = '<div class="cartt-social-proof cartt-stock-critical"><span class="dashicons dashicons-warning"></span> ' . esc_html($message) . '</div>';
                } elseif ($stock <= $low && $stock > 0) {
                    $message = $settings['stock_urgency']['low_stock_message'];
                    $output = '<div class="cartt-social-proof cartt-stock-low"><span class="dashicons dashicons-warning"></span> ' . esc_html($message) . '</div>';
                }
                break;
        }

        return $output;
    }

    /**
     * AJAX: Get recent purchases for social proof notifications
     */
    public function ajaxGetRecentPurchases(): void
    {
        global $wpdb;
        
        $settings = get_option('cartt_social_proof_settings', []);
        
        if (empty($settings['enabled']) || empty($settings['purchase_notifications']['enabled'])) {
            wp_send_json_success(['purchases' => []]);
        }
        
        // Get recent orders from last 24 hours
        $orders = $wpdb->get_results(
            "SELECT o.id, o.created_at, o.billing_address,
                    oi.product_id, oi.product_name,
                    p.featured_image
             FROM {$wpdb->prefix}cartt_orders o
             JOIN {$wpdb->prefix}cartt_order_items oi ON o.id = oi.order_id
             LEFT JOIN {$wpdb->prefix}cartt_products p ON oi.product_id = p.id
             WHERE o.status IN ('completed', 'processing')
             AND o.created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
             ORDER BY o.created_at DESC
             LIMIT 20"
        );
        
        $purchases = [];
        foreach ($orders as $order) {
            $billing = json_decode($order->billing_address, true);
            $buyerName = $billing['first_name'] ?? 'Someone';
            $location = $billing['city'] ?? ($billing['state'] ?? 'your area');
            
            // Calculate time ago
            $timestamp = strtotime($order->created_at);
            $timeAgo = $this->getTimeAgo($timestamp);
            
            $purchases[] = [
                'buyer_name' => $buyerName,
                'location' => $location,
                'product_name' => $order->product_name,
                'product_image' => $order->featured_image ? wp_get_attachment_url($order->featured_image) : '',
                'time_ago' => $timeAgo
            ];
        }
        
        wp_send_json_success(['purchases' => $purchases]);
    }

    /**
     * AJAX: Register visitor for live visitor count
     */
    public function ajaxRegisterVisitor(): void
    {
        global $wpdb;
        
        $productId = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $pageUrl = isset($_POST['page_url']) ? esc_url_raw($_POST['page_url']) : '';
        $sessionId = $this->getVisitorSessionId();
        
        // Insert or update visitor record
        $wpdb->replace(
            $wpdb->prefix . 'cartt_visitor_sessions',
            [
                'session_id' => $sessionId,
                'product_id' => $productId,
                'page_url' => $pageUrl,
                'ip_address' => $this->getClientIp(),
                'user_agent' => sanitize_text_field($_SERVER['HTTP_USER_AGENT'] ?? ''),
                'last_activity' => current_time('mysql'),
            ],
            ['%s', '%d', '%s', '%s', '%s', '%s']
        );
        
        wp_send_json_success();
    }

    /**
     * AJAX: Get visitor count
     */
    public function ajaxGetVisitorCount(): void
    {
        global $wpdb;
        
        $productId = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $settings = get_option('cartt_social_proof_settings', []);
        $timeWindow = ($settings['visitor_count']['time_window'] ?? 5) * 60; // Convert minutes to seconds
        
        $whereClause = $productId > 0 
            ? $wpdb->prepare("AND product_id = %d", $productId)
            : "";
        
        $count = (int)$wpdb->get_var(
            "SELECT COUNT(DISTINCT session_id) 
             FROM {$wpdb->prefix}cartt_visitor_sessions 
             WHERE last_activity > DATE_SUB(NOW(), INTERVAL {$timeWindow} SECOND)
             {$whereClause}"
        );
        
        wp_send_json_success(['count' => $count]);
    }

    /**
     * AJAX: Get stock status for a product
     */
    public function ajaxGetStockStatus(): void
    {
        global $wpdb;
        
        $productId = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        
        if (!$productId) {
            wp_send_json_error(['message' => 'Product ID required']);
        }
        
        $stock = (int)$wpdb->get_var($wpdb->prepare(
            "SELECT stock_quantity FROM {$wpdb->prefix}cartt_products WHERE id = %d",
            $productId
        ));
        
        wp_send_json_success(['stock' => $stock]);
    }

    /**
     * AJAX: Get cart activity count
     */
    public function ajaxGetCartActivity(): void
    {
        global $wpdb;
        
        $productId = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $settings = get_option('cartt_social_proof_settings', []);
        $timeWindow = ($settings['cart_activity']['time_window'] ?? 60) * 60;
        
        // Count visitors who have this product in their session/cart
        // For simplicity, we're counting recent visitors to product pages
        $count = (int)$wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT session_id) 
             FROM {$wpdb->prefix}cartt_visitor_sessions 
             WHERE product_id = %d
             AND last_activity > DATE_SUB(NOW(), INTERVAL {$timeWindow} SECOND)",
            $productId
        ));
        
        wp_send_json_success(['count' => $count]);
    }

    /**
     * AJAX: Get recent sales for a product
     */
    public function ajaxGetRecentSales(): void
    {
        global $wpdb;
        
        $productId = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
        $limit = isset($_POST['limit']) ? absint($_POST['limit']) : 5;
        $settings = get_option('cartt_social_proof_settings', []);
        $period = ($settings['recent_sales']['period'] ?? 24) * 3600;
        
        $whereProduct = $productId > 0 
            ? $wpdb->prepare("AND oi.product_id = %d", $productId)
            : "";
        
        $sales = $wpdb->get_results($wpdb->prepare(
            "SELECT oi.product_name, p.featured_image, o.created_at
             FROM {$wpdb->prefix}cartt_orders o
             JOIN {$wpdb->prefix}cartt_order_items oi ON o.id = oi.order_id
             LEFT JOIN {$wpdb->prefix}cartt_products p ON oi.product_id = p.id
             WHERE o.status IN ('completed', 'processing')
             AND o.created_at > DATE_SUB(NOW(), INTERVAL %d SECOND)
             {$whereProduct}
             ORDER BY o.created_at DESC
             LIMIT %d",
            $period,
            $limit
        ));
        
        $result = [];
        foreach ($sales as $sale) {
            $result[] = [
                'product_name' => $sale->product_name,
                'image' => $sale->featured_image ? wp_get_attachment_url($sale->featured_image) : '',
                'time_ago' => $this->getTimeAgo(strtotime($sale->created_at))
            ];
        }
        
        wp_send_json_success(['sales' => $result]);
    }

    /**
     * Get human-readable time ago string
     */
    private function getTimeAgo(int $timestamp): string
    {
        $diff = time() - $timestamp;
        
        if ($diff < 60) {
            return 'Just now';
        } elseif ($diff < 3600) {
            $mins = floor($diff / 60);
            return $mins . ' minute' . ($mins > 1 ? 's' : '') . ' ago';
        } elseif ($diff < 86400) {
            $hours = floor($diff / 3600);
            return $hours . ' hour' . ($hours > 1 ? 's' : '') . ' ago';
        } else {
            $days = floor($diff / 86400);
            return $days . ' day' . ($days > 1 ? 's' : '') . ' ago';
        }
    }

    /**
     * Get or create visitor session ID
     */
    private function getVisitorSessionId(): string
    {
        if (!isset($_COOKIE['cartt_visitor_session'])) {
            $sessionId = wp_generate_uuid4();
            setcookie('cartt_visitor_session', $sessionId, time() + 86400, COOKIEPATH, COOKIE_DOMAIN);
            return $sessionId;
        }
        
        return sanitize_text_field($_COOKIE['cartt_visitor_session']);
    }

    /**
     * Get client IP address
     */
    private function getClientIp(): string
    {
        $ip = '';
        
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
        } elseif (!empty($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }
        
        return sanitize_text_field(trim($ip));
    }
}
