<?php

declare(strict_types=1);

namespace Cartt\Core;

/**
 * Setup Wizard - guides new users through store configuration
 * This is the core differentiator: pick your store type, UI adapts
 */
class SetupWizard
{
    public function __construct()
    {
        add_action('admin_menu', [$this, 'addWizardPage']);
        add_action('admin_init', [$this, 'maybeRedirect']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueAssets']);
        add_action('wp_ajax_cartt_save_wizard', [$this, 'saveWizard']);
    }

    public function addWizardPage(): void
    {
        add_dashboard_page(
            __('Cartt Setup', 'cartt'),
            __('Cartt Setup', 'cartt'),
            'manage_options',
            'cartt-setup',
            [$this, 'renderWizard']
        );
    }

    public function maybeRedirect(): void
    {
        if (!get_transient('cartt_activation_redirect')) {
            return;
        }

        delete_transient('cartt_activation_redirect');

        if (wp_doing_ajax() || is_network_admin()) {
            return;
        }

        wp_safe_redirect(admin_url('index.php?page=cartt-setup'));
        exit;
    }

    public function enqueueAssets(string $hook): void
    {
        if ($hook !== 'dashboard_page_cartt-setup') {
            return;
        }

        wp_enqueue_style(
            'cartt-wizard',
            CARTT_PLUGIN_URL . 'assets/css/wizard.css',
            [],
            CARTT_VERSION
        );

        wp_enqueue_script(
            'cartt-wizard',
            CARTT_PLUGIN_URL . 'assets/js/wizard.js',
            ['jquery'],
            CARTT_VERSION,
            true
        );

        wp_localize_script('cartt-wizard', 'carttWizard', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cartt_wizard_nonce'),
            'dashboardUrl' => admin_url('admin.php?page=cartt'),
        ]);
    }

    public function renderWizard(): void
    {
        ?>
        <div class="cartt-wizard-wrap">
            <div class="cartt-wizard">
                <div class="cartt-wizard-header">
                    <h1><?php esc_html_e('Welcome to Cartt', 'cartt'); ?></h1>
                    <p><?php esc_html_e('Let\'s set up your store in 2 minutes.', 'cartt'); ?></p>
                </div>

                <!-- Step 1: Store Type -->
                <div class="cartt-wizard-step active" data-step="1">
                    <h2><?php esc_html_e('What will you sell?', 'cartt'); ?></h2>
                    <p class="cartt-wizard-subtitle"><?php esc_html_e('This determines which features you\'ll see. You can change this later.', 'cartt'); ?></p>
                    
                    <div class="cartt-store-types">
                        <label class="cartt-store-type">
                            <input type="radio" name="store_type" value="physical_goods">
                            <div class="cartt-store-type-card">
                                <strong><?php esc_html_e('Physical Products', 'cartt'); ?></strong>
                                <span><?php esc_html_e('Items you ship to customers', 'cartt'); ?></span>
                            </div>
                        </label>

                        <label class="cartt-store-type">
                            <input type="radio" name="store_type" value="digital_goods">
                            <div class="cartt-store-type-card">
                                <strong><?php esc_html_e('Digital Downloads', 'cartt'); ?></strong>
                                <span><?php esc_html_e('Files, software, media', 'cartt'); ?></span>
                            </div>
                        </label>

                        <label class="cartt-store-type">
                            <input type="radio" name="store_type" value="services">
                            <div class="cartt-store-type-card">
                                <strong><?php esc_html_e('Services & Bookings', 'cartt'); ?></strong>
                                <span><?php esc_html_e('Appointments, consultations', 'cartt'); ?></span>
                            </div>
                        </label>

                        <label class="cartt-store-type">
                            <input type="radio" name="store_type" value="subscriptions">
                            <div class="cartt-store-type-card">
                                <strong><?php esc_html_e('Subscriptions', 'cartt'); ?></strong>
                                <span><?php esc_html_e('Recurring payments, memberships', 'cartt'); ?></span>
                            </div>
                        </label>

                        <label class="cartt-store-type">
                            <input type="radio" name="store_type" value="mixed">
                            <div class="cartt-store-type-card">
                                <strong><?php esc_html_e('Mixed / Not Sure', 'cartt'); ?></strong>
                                <span><?php esc_html_e('Show me all options', 'cartt'); ?></span>
                            </div>
                        </label>
                    </div>

                    <div class="cartt-wizard-actions">
                        <button type="button" class="button button-primary cartt-wizard-next" disabled>
                            <?php esc_html_e('Continue', 'cartt'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 2: Basic Settings -->
                <div class="cartt-wizard-step" data-step="2">
                    <h2><?php esc_html_e('Basic Settings', 'cartt'); ?></h2>
                    
                    <div class="cartt-wizard-form">
                        <div class="cartt-form-row">
                            <label for="store_name"><?php esc_html_e('Store Name', 'cartt'); ?></label>
                            <input type="text" id="store_name" name="store_name" value="<?php echo esc_attr(get_bloginfo('name')); ?>">
                        </div>

                        <div class="cartt-form-row">
                            <label for="currency"><?php esc_html_e('Currency', 'cartt'); ?></label>
                            <select id="currency" name="currency">
                                <option value="USD">USD ($)</option>
                                <option value="EUR">EUR (€)</option>
                                <option value="GBP">GBP (£)</option>
                                <option value="CAD">CAD ($)</option>
                                <option value="AUD">AUD ($)</option>
                                <option value="JPY">JPY (¥)</option>
                                <option value="CNY">CNY (¥)</option>
                                <option value="INR">INR (₹)</option>
                                <option value="BRL">BRL (R$)</option>
                                <option value="MXN">MXN ($)</option>
                            </select>
                        </div>

                        <div class="cartt-form-row cartt-shipping-row">
                            <label for="country"><?php esc_html_e('Store Location', 'cartt'); ?></label>
                            <select id="country" name="country">
                                <option value="US"><?php esc_html_e('United States', 'cartt'); ?></option>
                                <option value="CA"><?php esc_html_e('Canada', 'cartt'); ?></option>
                                <option value="GB"><?php esc_html_e('United Kingdom', 'cartt'); ?></option>
                                <option value="AU"><?php esc_html_e('Australia', 'cartt'); ?></option>
                                <option value="DE"><?php esc_html_e('Germany', 'cartt'); ?></option>
                                <option value="FR"><?php esc_html_e('France', 'cartt'); ?></option>
                                <option value="other"><?php esc_html_e('Other', 'cartt'); ?></option>
                            </select>
                        </div>
                    </div>

                    <div class="cartt-wizard-actions">
                        <button type="button" class="button cartt-wizard-back">
                            <?php esc_html_e('Back', 'cartt'); ?>
                        </button>
                        <button type="button" class="button button-primary cartt-wizard-next">
                            <?php esc_html_e('Continue', 'cartt'); ?>
                        </button>
                    </div>
                </div>

                <!-- Step 3: Payments -->
                <div class="cartt-wizard-step" data-step="3">
                    <h2><?php esc_html_e('Payment Methods', 'cartt'); ?></h2>
                    <p class="cartt-wizard-subtitle"><?php esc_html_e('You can add more payment methods later.', 'cartt'); ?></p>

                    <div class="cartt-payment-methods">
                        <label class="cartt-payment-method">
                            <input type="checkbox" name="payments[]" value="stripe">
                            <div class="cartt-payment-card">
                                <strong>Stripe</strong>
                                <span><?php esc_html_e('Credit cards, Apple Pay, Google Pay', 'cartt'); ?></span>
                            </div>
                        </label>

                        <label class="cartt-payment-method">
                            <input type="checkbox" name="payments[]" value="paypal">
                            <div class="cartt-payment-card">
                                <strong>PayPal</strong>
                                <span><?php esc_html_e('PayPal balance and cards', 'cartt'); ?></span>
                            </div>
                        </label>
                    </div>

                    <p class="cartt-wizard-note">
                        <?php esc_html_e('You\'ll connect your accounts after setup.', 'cartt'); ?>
                    </p>

                    <div class="cartt-wizard-actions">
                        <button type="button" class="button cartt-wizard-back">
                            <?php esc_html_e('Back', 'cartt'); ?>
                        </button>
                        <button type="button" class="button button-primary cartt-wizard-finish">
                            <?php esc_html_e('Finish Setup', 'cartt'); ?>
                        </button>
                    </div>
                </div>

                <!-- Success -->
                <div class="cartt-wizard-step" data-step="4">
                    <div class="cartt-wizard-success">
                        <div class="cartt-success-icon">
                            <svg width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="20 6 9 17 4 12"></polyline>
                            </svg>
                        </div>
                        <h2><?php esc_html_e('You\'re all set!', 'cartt'); ?></h2>
                        <p><?php esc_html_e('Your store is ready. Add your first product to start selling.', 'cartt'); ?></p>
                        <div class="cartt-wizard-success-actions">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=cartt-products&action=new')); ?>" class="cartt-btn cartt-btn-primary">
                                <?php esc_html_e('Add First Product', 'cartt'); ?>
                            </a>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=cartt')); ?>" class="cartt-btn cartt-btn-secondary">
                                <?php esc_html_e('Go to Dashboard', 'cartt'); ?>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    public function saveWizard(): void
    {
        check_ajax_referer('cartt_wizard_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $store_type = sanitize_text_field($_POST['store_type'] ?? 'mixed');
        $store_name = sanitize_text_field($_POST['store_name'] ?? '');
        $currency = sanitize_text_field($_POST['currency'] ?? 'USD');
        $country = sanitize_text_field($_POST['country'] ?? 'US');
        $payments = array_map('sanitize_text_field', $_POST['payments'] ?? []);

        update_option('cartt_store_type', $store_type);
        update_option('cartt_store_name', $store_name);
        update_option('cartt_currency', $currency);
        update_option('cartt_country', $country);
        update_option('cartt_payment_methods', $payments);
        update_option('cartt_setup_complete', true);

        wp_send_json_success(['redirect' => admin_url('admin.php?page=cartt')]);
    }
}
