<?php

declare(strict_types=1);

namespace Cartt\Core;

use Cartt\Admin\AdminController;
use Cartt\Frontend\FrontendController;
use Cartt\Api\RestController;

/**
 * Main plugin orchestrator
 */
final class Plugin
{
    private static ?Plugin $instance = null;
    private bool $initialized = false;

    /**
     * Check if current site is cartt.app (all addons unlocked)
     */
    public static function isCarttApp(): bool
    {
        $site_url = strtolower(get_site_url());
        $host = parse_url($site_url, PHP_URL_HOST);
        
        // Check various formats
        return (
            $host === 'cartt.app' ||
            $host === 'www.cartt.app' ||
            strpos($host, '.cartt.app') !== false ||
            $host === 'cartt.local' ||
            $host === 'www.cartt.local' ||
            strpos($host, '.cartt.local') !== false
        );
    }

    /**
     * Get active addons (all unlocked for cartt.app)
     */
    public static function getActiveAddons(): array
    {
        if (self::isCarttApp()) {
            return [
                'bookings',
                'subscriptions',
                'marketplace',
                'ai_conversion',
                'live_shipping',
                'gift_cards',
                'loyalty',
                'advanced_analytics',
                'dropshipping',
                'wholesale',
                'chat',
                'ai_content',
                'abandoned_cart',
                'multi_currency'
            ];
        }
        return get_option('cartt_active_addons', []);
    }

    public static function instance(): Plugin
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        $this->init();
    }

    private function init(): void
    {
        if ($this->initialized) {
            return;
        }

        // One-time migrations
        $this->runMigrations();

        // Register custom cron schedules
        add_filter('cron_schedules', [$this, 'addCronSchedules']);
        
        // Register cron hooks (legacy)
        add_action('cartt_send_abandoned_cart_emails', [$this, 'processAbandonedCarts']);
        add_action('cartt_check_low_stock', [$this, 'checkLowStock']);

        // Initialize v1.3.0 cron service
        \Cartt\Services\CronService::init();

        // Initialize cart session
        new \Cartt\Services\CartService();

        // =====================================================================
        // v1.4.0 Services - Enterprise & AI Features
        // =====================================================================
        \Cartt\Services\AIRecommendationService::instance();
        \Cartt\Services\SocialProofService::instance();
        \Cartt\Services\AdvancedAnalyticsService::instance();
        \Cartt\Services\VendorService::instance();
        \Cartt\Services\VendorCommissionService::instance();
        \Cartt\Services\WholesaleService::instance();
        \Cartt\Services\DropshippingService::instance();
        
        // v1.4.1 Software Licensing
        \Cartt\Services\SoftwareLicensingService::instance();
        
        // v1.4.2 Chat Service
        try {
            \Cartt\Services\ChatService::instance();
        } catch (\Throwable $e) {
            error_log('Cartt ChatService init error: ' . $e->getMessage());
        }
        
        // v1.5.5 AI Content Service
        try {
            \Cartt\Services\AIContentService::getInstance();
        } catch (\Throwable $e) {
            error_log('Cartt AIContentService init error: ' . $e->getMessage());
        }

        // Check if setup wizard needs to run
        $store_type = get_option('cartt_store_type');
        
        if (is_admin()) {
            new AdminController();
            
            // Show setup wizard if not configured
            if (!$store_type) {
                new SetupWizard();
            }
        }

        if (!is_admin() || wp_doing_ajax()) {
            new FrontendController();
        }

        // REST API always loads
        new RestController();

        $this->initialized = true;
    }

    public function addCronSchedules(array $schedules): array
    {
        $schedules['fifteen_minutes'] = [
            'interval' => 900,
            'display' => __('Every 15 Minutes', 'cartt'),
        ];
        return $schedules;
    }

    public function processAbandonedCarts(): void
    {
        $service = new \Cartt\Services\AbandonedCartService();
        $service->processAbandonedCarts();
    }

    public function checkLowStock(): void
    {
        global $wpdb;
        
        $threshold = get_option('cartt_low_stock_threshold', 5);
        $adminEmail = get_option('admin_email');
        
        $lowStockProducts = $wpdb->get_results($wpdb->prepare(
            "SELECT id, name, sku, stock_quantity FROM {$wpdb->prefix}cartt_products 
             WHERE manage_stock = 1 AND stock_quantity <= %d AND stock_quantity > 0 AND status = 'publish'",
            $threshold
        ));
        
        if (empty($lowStockProducts)) {
            return;
        }
        
        // Check if we've already notified today
        $lastNotified = get_option('cartt_low_stock_last_notified', '');
        $today = date('Y-m-d');
        
        if ($lastNotified === $today) {
            return;
        }
        
        // Send notification
        $emailService = \Cartt\Services\EmailService::class;
        
        foreach ($lowStockProducts as $product) {
            $emailService::send('low_stock', $adminEmail, [
                'product_name' => $product->name,
                'stock_quantity' => $product->stock_quantity,
                'sku' => $product->sku ?: 'N/A',
                'store_name' => get_bloginfo('name'),
            ]);
        }
        
        update_option('cartt_low_stock_last_notified', $today);
    }

    /**
     * Run one-time migrations
     */
    private function runMigrations(): void
    {
        $migrationsRun = get_option('cartt_migrations_run', []);
        
        // Fix handoff message escaping (v1.5.9)
        if (!in_array('fix_handoff_message_escaping', $migrationsRun)) {
            $handoffMessage = get_option('cartt_chat_handoff_message', '');
            if (!empty($handoffMessage)) {
                // Fix double-escaped quotes and backslashes
                $fixed = stripslashes($handoffMessage);
                $fixed = str_replace("\\'", "'", $fixed);
                $fixed = str_replace('\\"', '"', $fixed);
                if ($fixed !== $handoffMessage) {
                    update_option('cartt_chat_handoff_message', $fixed);
                }
            }
            $migrationsRun[] = 'fix_handoff_message_escaping';
            update_option('cartt_migrations_run', $migrationsRun);
        }
    }

    // Prevent cloning
    private function __clone() {}

    // Prevent unserialization
    public function __wakeup()
    {
        throw new \Exception('Cannot unserialize singleton');
    }
}
