<?php

declare(strict_types=1);

namespace Cartt\Core;

/**
 * Handles plugin activation
 */
class Activator
{
    public static function activate(): void
    {
        self::createTables();
        self::setDefaultOptions();
        self::createPages();
        self::scheduleCronJobs();
        
        // Create abandoned cart table
        \Cartt\Services\AbandonedCartService::createTable();
        
        // Set activation flag for redirect to setup wizard
        set_transient('cartt_activation_redirect', true, 30);
        
        // Clear permalinks
        flush_rewrite_rules();
    }

    private static function scheduleCronJobs(): void
    {
        // Abandoned cart recovery emails - runs every 15 minutes
        if (!wp_next_scheduled('cartt_send_abandoned_cart_emails')) {
            wp_schedule_event(time(), 'fifteen_minutes', 'cartt_send_abandoned_cart_emails');
        }
        
        // Low stock notifications - runs daily
        if (!wp_next_scheduled('cartt_check_low_stock')) {
            wp_schedule_event(time(), 'daily', 'cartt_check_low_stock');
        }

        // v1.3.0 Cron jobs
        \Cartt\Services\CronService::schedule();
    }

    private static function createPages(): void
    {
        $pages = [
            'shop' => [
                'title' => __('Shop', 'cartt'),
                'content' => '[cartt_products]',
                'option' => 'cartt_shop_page_id',
            ],
            'cart' => [
                'title' => __('Cart', 'cartt'),
                'content' => '[cartt_cart]',
                'option' => 'cartt_cart_page_id',
            ],
            'checkout' => [
                'title' => __('Checkout', 'cartt'),
                'content' => '[cartt_checkout]',
                'option' => 'cartt_checkout_page_id',
            ],
            'account' => [
                'title' => __('My Account', 'cartt'),
                'content' => '[cartt_account]',
                'option' => 'cartt_account_page_id',
            ],
            'order-received' => [
                'title' => __('Order Received', 'cartt'),
                'content' => '[cartt_order_received]',
                'option' => 'cartt_order_received_page_id',
            ],
        ];

        foreach ($pages as $slug => $page) {
            // Check if page already exists
            $existing_id = get_option($page['option']);
            if ($existing_id && get_post($existing_id)) {
                continue;
            }

            // Check if page with slug exists
            $existing_page = get_page_by_path($slug);
            if ($existing_page) {
                update_option($page['option'], $existing_page->ID);
                continue;
            }

            // Create new page
            $page_id = wp_insert_post([
                'post_title' => $page['title'],
                'post_name' => $slug,
                'post_content' => $page['content'],
                'post_status' => 'publish',
                'post_type' => 'page',
                'comment_status' => 'closed',
            ]);

            if ($page_id && !is_wp_error($page_id)) {
                update_option($page['option'], $page_id);
            }
        }
    }

    private static function createTables(): void
    {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Products table
        $table_products = $wpdb->prefix . 'cartt_products';
        $sql_products = "CREATE TABLE $table_products (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            slug varchar(255) NOT NULL,
            description longtext,
            short_description text,
            product_type varchar(50) NOT NULL DEFAULT 'physical',
            status varchar(20) NOT NULL DEFAULT 'draft',
            price decimal(19,4) DEFAULT NULL,
            sale_price decimal(19,4) DEFAULT NULL,
            sku varchar(100) DEFAULT NULL,
            stock_quantity int(11) DEFAULT NULL,
            stock_status varchar(20) DEFAULT 'instock',
            manage_stock tinyint(1) DEFAULT 0,
            weight decimal(10,3) DEFAULT NULL,
            length decimal(10,3) DEFAULT NULL,
            width decimal(10,3) DEFAULT NULL,
            height decimal(10,3) DEFAULT NULL,
            downloadable tinyint(1) DEFAULT 0,
            virtual tinyint(1) DEFAULT 0,
            featured_image bigint(20) unsigned DEFAULT NULL,
            gallery_images longtext,
            attributes longtext,
            meta longtext,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY slug (slug),
            KEY status (status),
            KEY product_type (product_type),
            KEY sku (sku)
        ) $charset_collate;";

        // Product categories
        $table_categories = $wpdb->prefix . 'cartt_categories';
        $sql_categories = "CREATE TABLE $table_categories (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            slug varchar(255) NOT NULL,
            description text,
            parent_id bigint(20) unsigned DEFAULT NULL,
            image bigint(20) unsigned DEFAULT NULL,
            sort_order int(11) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY slug (slug),
            KEY parent_id (parent_id)
        ) $charset_collate;";

        // Product to category relationship
        $table_product_categories = $wpdb->prefix . 'cartt_product_categories';
        $sql_product_categories = "CREATE TABLE $table_product_categories (
            product_id bigint(20) unsigned NOT NULL,
            category_id bigint(20) unsigned NOT NULL,
            PRIMARY KEY (product_id, category_id),
            KEY category_id (category_id)
        ) $charset_collate;";

        // Customers table
        $table_customers = $wpdb->prefix . 'cartt_customers';
        $sql_customers = "CREATE TABLE $table_customers (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned DEFAULT NULL,
            email varchar(255) NOT NULL,
            first_name varchar(100) DEFAULT NULL,
            last_name varchar(100) DEFAULT NULL,
            phone varchar(50) DEFAULT NULL,
            billing_address longtext,
            shipping_address longtext,
            meta longtext,
            total_spent decimal(19,4) DEFAULT 0,
            order_count int(11) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY email (email),
            KEY user_id (user_id)
        ) $charset_collate;";

        // Orders table
        $table_orders = $wpdb->prefix . 'cartt_orders';
        $sql_orders = "CREATE TABLE $table_orders (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_number varchar(50) NOT NULL,
            customer_id bigint(20) unsigned DEFAULT NULL,
            status varchar(50) NOT NULL DEFAULT 'pending',
            currency varchar(10) NOT NULL DEFAULT 'USD',
            subtotal decimal(19,4) NOT NULL DEFAULT 0,
            discount_total decimal(19,4) NOT NULL DEFAULT 0,
            shipping_total decimal(19,4) NOT NULL DEFAULT 0,
            tax_total decimal(19,4) NOT NULL DEFAULT 0,
            total decimal(19,4) NOT NULL DEFAULT 0,
            payment_method varchar(100) DEFAULT NULL,
            payment_status varchar(50) DEFAULT 'pending',
            transaction_id varchar(255) DEFAULT NULL,
            billing_address longtext,
            shipping_address longtext,
            customer_note text,
            admin_note text,
            ip_address varchar(45) DEFAULT NULL,
            user_agent text,
            meta longtext,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            completed_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY order_number (order_number),
            KEY customer_id (customer_id),
            KEY status (status),
            KEY created_at (created_at)
        ) $charset_collate;";

        // Order items table
        $table_order_items = $wpdb->prefix . 'cartt_order_items';
        $sql_order_items = "CREATE TABLE $table_order_items (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned DEFAULT NULL,
            name varchar(255) NOT NULL,
            sku varchar(100) DEFAULT NULL,
            quantity int(11) NOT NULL DEFAULT 1,
            price decimal(19,4) NOT NULL,
            subtotal decimal(19,4) NOT NULL,
            tax decimal(19,4) DEFAULT 0,
            total decimal(19,4) NOT NULL,
            meta longtext,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY product_id (product_id)
        ) $charset_collate;";

        // Downloads table (for digital products)
        $table_downloads = $wpdb->prefix . 'cartt_downloads';
        $sql_downloads = "CREATE TABLE $table_downloads (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            name varchar(255) NOT NULL,
            file_url text NOT NULL,
            file_type varchar(100) DEFAULT NULL,
            download_limit int(11) DEFAULT NULL,
            expiry_days int(11) DEFAULT NULL,
            sort_order int(11) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id)
        ) $charset_collate;";

        // Customer download permissions
        $table_download_permissions = $wpdb->prefix . 'cartt_download_permissions';
        $sql_download_permissions = "CREATE TABLE $table_download_permissions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            download_id bigint(20) unsigned NOT NULL,
            order_id bigint(20) unsigned NOT NULL,
            customer_id bigint(20) unsigned NOT NULL,
            download_count int(11) DEFAULT 0,
            access_granted datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            access_expires datetime DEFAULT NULL,
            PRIMARY KEY (id),
            KEY download_id (download_id),
            KEY order_id (order_id),
            KEY customer_id (customer_id)
        ) $charset_collate;";

        // Subscriptions table
        $table_subscriptions = $wpdb->prefix . 'cartt_subscriptions';
        $sql_subscriptions = "CREATE TABLE $table_subscriptions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            status varchar(50) NOT NULL DEFAULT 'active',
            price decimal(19,4) NOT NULL,
            period varchar(20) NOT NULL DEFAULT 'month',
            interval_count int(11) NOT NULL DEFAULT 1,
            trial_end datetime DEFAULT NULL,
            current_period_start datetime NOT NULL,
            current_period_end datetime NOT NULL,
            cancel_at_period_end tinyint(1) DEFAULT 0,
            canceled_at datetime DEFAULT NULL,
            payment_method varchar(100) DEFAULT NULL,
            external_id varchar(255) DEFAULT NULL,
            meta longtext,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY customer_id (customer_id),
            KEY product_id (product_id),
            KEY status (status),
            KEY external_id (external_id)
        ) $charset_collate;";

        // Bookings table (for services)
        $table_bookings = $wpdb->prefix . 'cartt_bookings';
        $sql_bookings = "CREATE TABLE $table_bookings (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            order_id bigint(20) unsigned DEFAULT NULL,
            customer_id bigint(20) unsigned NOT NULL,
            status varchar(50) NOT NULL DEFAULT 'pending',
            start_time datetime NOT NULL,
            end_time datetime NOT NULL,
            timezone varchar(100) DEFAULT 'UTC',
            customer_note text,
            admin_note text,
            meta longtext,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY order_id (order_id),
            KEY customer_id (customer_id),
            KEY status (status),
            KEY start_time (start_time)
        ) $charset_collate;";

        // Coupons table
        $table_coupons = $wpdb->prefix . 'cartt_coupons';
        $sql_coupons = "CREATE TABLE $table_coupons (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            code varchar(50) NOT NULL,
            description text,
            discount_type enum('percent','fixed_cart','fixed_product') NOT NULL DEFAULT 'percent',
            amount decimal(10,2) NOT NULL DEFAULT 0,
            free_shipping tinyint(1) NOT NULL DEFAULT 0,
            minimum_amount decimal(10,2) DEFAULT NULL,
            maximum_amount decimal(10,2) DEFAULT NULL,
            usage_limit int(11) DEFAULT NULL,
            usage_limit_per_user int(11) DEFAULT NULL,
            usage_count int(11) NOT NULL DEFAULT 0,
            product_ids text,
            excluded_product_ids text,
            category_ids text,
            start_date datetime DEFAULT NULL,
            end_date datetime DEFAULT NULL,
            status enum('active','inactive','expired') NOT NULL DEFAULT 'active',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY code (code),
            KEY status (status)
        ) $charset_collate;";

        // Shipping zones table
        $table_shipping_zones = $wpdb->prefix . 'cartt_shipping_zones';
        $sql_shipping_zones = "CREATE TABLE $table_shipping_zones (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            countries text,
            regions text,
            postcodes text,
            sort_order int(11) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        // Shipping methods table
        $table_shipping_methods = $wpdb->prefix . 'cartt_shipping_methods';
        $sql_shipping_methods = "CREATE TABLE $table_shipping_methods (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            zone_id bigint(20) unsigned NOT NULL,
            name varchar(255) NOT NULL,
            method_type enum('flat_rate','free_shipping','local_pickup') NOT NULL DEFAULT 'flat_rate',
            cost decimal(10,2) DEFAULT 0,
            min_amount decimal(10,2) DEFAULT NULL,
            enabled tinyint(1) NOT NULL DEFAULT 1,
            sort_order int(11) DEFAULT 0,
            settings longtext,
            PRIMARY KEY (id),
            KEY zone_id (zone_id)
        ) $charset_collate;";

        // Product variations table
        $table_variations = $wpdb->prefix . 'cartt_product_variations';
        $sql_variations = "CREATE TABLE $table_variations (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            sku varchar(100) DEFAULT NULL,
            price decimal(19,4) DEFAULT NULL,
            sale_price decimal(19,4) DEFAULT NULL,
            stock_quantity int(11) DEFAULT NULL,
            stock_status varchar(20) DEFAULT 'instock',
            attributes longtext NOT NULL,
            image bigint(20) unsigned DEFAULT NULL,
            enabled tinyint(1) NOT NULL DEFAULT 1,
            sort_order int(11) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY sku (sku)
        ) $charset_collate;";

        // Product relationships table (related, cross-sells, upsells)
        $table_relationships = $wpdb->prefix . 'cartt_product_relationships';
        $sql_relationships = "CREATE TABLE $table_relationships (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            related_product_id bigint(20) unsigned NOT NULL,
            relationship_type enum('related','crosssell','upsell') NOT NULL DEFAULT 'related',
            sort_order int(11) DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY product_related_type (product_id, related_product_id, relationship_type),
            KEY product_id (product_id),
            KEY related_product_id (related_product_id)
        ) $charset_collate;";

        // Currencies table for multi-currency
        $table_currencies = $wpdb->prefix . 'cartt_currencies';
        $sql_currencies = "CREATE TABLE $table_currencies (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            code varchar(3) NOT NULL,
            name varchar(100) NOT NULL,
            symbol varchar(10) NOT NULL,
            exchange_rate decimal(15,6) NOT NULL DEFAULT 1.000000,
            decimals tinyint(1) NOT NULL DEFAULT 2,
            symbol_position enum('left','right','left_space','right_space') NOT NULL DEFAULT 'left',
            thousand_separator varchar(5) DEFAULT ',',
            decimal_separator varchar(5) DEFAULT '.',
            enabled tinyint(1) NOT NULL DEFAULT 1,
            is_default tinyint(1) NOT NULL DEFAULT 0,
            auto_update tinyint(1) NOT NULL DEFAULT 0,
            sort_order int(11) DEFAULT 0,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY code (code)
        ) $charset_collate;";

        // Tax rates table
        $table_tax_rates = $wpdb->prefix . 'cartt_tax_rates';
        $sql_tax_rates = "CREATE TABLE $table_tax_rates (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            country varchar(2) NOT NULL,
            state varchar(100) DEFAULT '',
            postcode varchar(100) DEFAULT '',
            city varchar(100) DEFAULT '',
            rate decimal(8,4) NOT NULL,
            name varchar(255) NOT NULL,
            priority int(11) NOT NULL DEFAULT 1,
            compound tinyint(1) NOT NULL DEFAULT 0,
            shipping tinyint(1) NOT NULL DEFAULT 1,
            sort_order int(11) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY country_state (country, state)
        ) $charset_collate;";

        // Product reviews table
        $table_reviews = $wpdb->prefix . 'cartt_product_reviews';
        $sql_reviews = "CREATE TABLE $table_reviews (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            customer_id bigint(20) unsigned DEFAULT NULL,
            customer_name varchar(100) NOT NULL,
            customer_email varchar(255) NOT NULL,
            rating tinyint(1) NOT NULL DEFAULT 5,
            title varchar(255) DEFAULT NULL,
            content text,
            status enum('pending','approved','spam') NOT NULL DEFAULT 'pending',
            verified_purchase tinyint(1) DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY customer_id (customer_id),
            KEY status (status),
            KEY rating (rating)
        ) $charset_collate;";

        // Order notes table
        $table_order_notes = $wpdb->prefix . 'cartt_order_notes';
        $sql_order_notes = "CREATE TABLE $table_order_notes (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_id bigint(20) unsigned NOT NULL,
            note text NOT NULL,
            is_customer_note tinyint(1) NOT NULL DEFAULT 0,
            created_by bigint(20) unsigned DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id)
        ) $charset_collate;";

        // Refunds table
        $table_refunds = $wpdb->prefix . 'cartt_refunds';
        $sql_refunds = "CREATE TABLE $table_refunds (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_id bigint(20) unsigned NOT NULL,
            amount decimal(19,4) NOT NULL,
            reason text,
            refunded_by bigint(20) unsigned DEFAULT NULL,
            payment_refund_id varchar(255) DEFAULT NULL,
            line_items longtext,
            status enum('pending','completed','failed') NOT NULL DEFAULT 'completed',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id)
        ) $charset_collate;";

        // Wishlists table
        $table_wishlists = $wpdb->prefix . 'cartt_wishlists';
        $sql_wishlists = "CREATE TABLE $table_wishlists (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned DEFAULT NULL,
            session_id varchar(100) DEFAULT NULL,
            share_key varchar(32) DEFAULT NULL,
            name varchar(255) DEFAULT 'My Wishlist',
            is_public tinyint(1) NOT NULL DEFAULT 0,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY session_id (session_id),
            UNIQUE KEY share_key (share_key)
        ) $charset_collate;";

        // Wishlist items table
        $table_wishlist_items = $wpdb->prefix . 'cartt_wishlist_items';
        $sql_wishlist_items = "CREATE TABLE $table_wishlist_items (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            wishlist_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            variation_id bigint(20) unsigned DEFAULT NULL,
            added_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            notify_on_sale tinyint(1) NOT NULL DEFAULT 0,
            notify_on_stock tinyint(1) NOT NULL DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY wishlist_product (wishlist_id, product_id, variation_id),
            KEY product_id (product_id)
        ) $charset_collate;";

        // Gift cards table
        $table_gift_cards = $wpdb->prefix . 'cartt_gift_cards';
        $sql_gift_cards = "CREATE TABLE $table_gift_cards (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            code varchar(50) NOT NULL,
            initial_balance decimal(19,4) NOT NULL,
            current_balance decimal(19,4) NOT NULL,
            currency varchar(3) NOT NULL DEFAULT 'USD',
            sender_name varchar(255) DEFAULT NULL,
            sender_email varchar(255) DEFAULT NULL,
            recipient_name varchar(255) DEFAULT NULL,
            recipient_email varchar(255) DEFAULT NULL,
            message text,
            design varchar(50) DEFAULT 'default',
            order_id bigint(20) unsigned DEFAULT NULL,
            is_delivered tinyint(1) NOT NULL DEFAULT 0,
            deliver_at datetime DEFAULT NULL,
            expires_at datetime DEFAULT NULL,
            status enum('active','used','expired','disabled') NOT NULL DEFAULT 'active',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY code (code),
            KEY recipient_email (recipient_email),
            KEY order_id (order_id)
        ) $charset_collate;";

        // Gift card transactions
        $table_gift_card_transactions = $wpdb->prefix . 'cartt_gift_card_transactions';
        $sql_gift_card_transactions = "CREATE TABLE $table_gift_card_transactions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            gift_card_id bigint(20) unsigned NOT NULL,
            order_id bigint(20) unsigned DEFAULT NULL,
            amount decimal(19,4) NOT NULL,
            balance_after decimal(19,4) NOT NULL,
            type enum('credit','debit','refund') NOT NULL,
            note varchar(255) DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY gift_card_id (gift_card_id)
        ) $charset_collate;";

        // Loyalty points table
        $table_loyalty_points = $wpdb->prefix . 'cartt_loyalty_points';
        $sql_loyalty_points = "CREATE TABLE $table_loyalty_points (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            points int(11) NOT NULL DEFAULT 0,
            lifetime_points int(11) NOT NULL DEFAULT 0,
            tier varchar(50) DEFAULT 'bronze',
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY customer_id (customer_id)
        ) $charset_collate;";

        // Loyalty transactions
        $table_loyalty_transactions = $wpdb->prefix . 'cartt_loyalty_transactions';
        $sql_loyalty_transactions = "CREATE TABLE $table_loyalty_transactions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            points int(11) NOT NULL,
            type enum('earn','redeem','expire','adjust','bonus') NOT NULL,
            source varchar(50) DEFAULT NULL,
            reference_id bigint(20) unsigned DEFAULT NULL,
            description varchar(255) DEFAULT NULL,
            expires_at datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY customer_id (customer_id),
            KEY type (type)
        ) $charset_collate;";

        // RMA / Returns table
        $table_rma = $wpdb->prefix . 'cartt_rma';
        $sql_rma = "CREATE TABLE $table_rma (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            rma_number varchar(50) NOT NULL,
            order_id bigint(20) unsigned NOT NULL,
            customer_id bigint(20) unsigned NOT NULL,
            reason enum('defective','wrong_item','not_as_described','changed_mind','other') NOT NULL,
            reason_detail text,
            status enum('pending','approved','rejected','shipped','received','refunded','completed') NOT NULL DEFAULT 'pending',
            refund_type enum('original','store_credit','exchange') DEFAULT 'original',
            refund_amount decimal(19,4) DEFAULT NULL,
            restocking_fee decimal(19,4) DEFAULT 0,
            return_label_url varchar(500) DEFAULT NULL,
            tracking_number varchar(100) DEFAULT NULL,
            admin_notes text,
            processed_by bigint(20) unsigned DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY rma_number (rma_number),
            KEY order_id (order_id),
            KEY customer_id (customer_id),
            KEY status (status)
        ) $charset_collate;";

        // RMA items
        $table_rma_items = $wpdb->prefix . 'cartt_rma_items';
        $sql_rma_items = "CREATE TABLE $table_rma_items (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            rma_id bigint(20) unsigned NOT NULL,
            order_item_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            quantity int(11) NOT NULL,
            condition_received enum('new','opened','damaged','missing') DEFAULT NULL,
            restock tinyint(1) NOT NULL DEFAULT 1,
            PRIMARY KEY (id),
            KEY rma_id (rma_id)
        ) $charset_collate;";

        // Product Q&A
        $table_product_qa = $wpdb->prefix . 'cartt_product_qa';
        $sql_product_qa = "CREATE TABLE $table_product_qa (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            parent_id bigint(20) unsigned DEFAULT NULL,
            user_id bigint(20) unsigned DEFAULT NULL,
            author_name varchar(255) NOT NULL,
            author_email varchar(255) NOT NULL,
            content text NOT NULL,
            is_answer tinyint(1) NOT NULL DEFAULT 0,
            is_vendor_answer tinyint(1) NOT NULL DEFAULT 0,
            upvotes int(11) NOT NULL DEFAULT 0,
            status enum('pending','approved','rejected') NOT NULL DEFAULT 'pending',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY parent_id (parent_id),
            KEY status (status)
        ) $charset_collate;";

        // Order tracking
        $table_order_tracking = $wpdb->prefix . 'cartt_order_tracking';
        $sql_order_tracking = "CREATE TABLE $table_order_tracking (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_id bigint(20) unsigned NOT NULL,
            carrier varchar(50) NOT NULL,
            tracking_number varchar(100) NOT NULL,
            tracking_url varchar(500) DEFAULT NULL,
            status varchar(50) DEFAULT 'in_transit',
            status_detail text,
            estimated_delivery date DEFAULT NULL,
            shipped_at datetime DEFAULT NULL,
            delivered_at datetime DEFAULT NULL,
            last_update datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY tracking_number (tracking_number)
        ) $charset_collate;";

        // Product SEO meta
        $table_product_seo = $wpdb->prefix . 'cartt_product_seo';
        $sql_product_seo = "CREATE TABLE $table_product_seo (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            meta_title varchar(255) DEFAULT NULL,
            meta_description text,
            focus_keyword varchar(255) DEFAULT NULL,
            og_title varchar(255) DEFAULT NULL,
            og_description text,
            og_image bigint(20) unsigned DEFAULT NULL,
            canonical_url varchar(500) DEFAULT NULL,
            noindex tinyint(1) NOT NULL DEFAULT 0,
            schema_data longtext,
            PRIMARY KEY (id),
            UNIQUE KEY product_id (product_id)
        ) $charset_collate;";

        // Stock notifications
        $table_stock_notifications = $wpdb->prefix . 'cartt_stock_notifications';
        $sql_stock_notifications = "CREATE TABLE $table_stock_notifications (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            variation_id bigint(20) unsigned DEFAULT NULL,
            email varchar(255) NOT NULL,
            user_id bigint(20) unsigned DEFAULT NULL,
            notified tinyint(1) NOT NULL DEFAULT 0,
            notified_at datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY email (email)
        ) $charset_collate;";

        // =====================================================================
        // v1.4.0 TABLES - Enterprise & AI Features
        // =====================================================================

        // AI Recommendations - Purchase patterns for "frequently bought together"
        $table_purchase_patterns = $wpdb->prefix . 'cartt_purchase_patterns';
        $sql_purchase_patterns = "CREATE TABLE $table_purchase_patterns (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            related_product_id bigint(20) unsigned NOT NULL,
            purchase_count int(11) NOT NULL DEFAULT 1,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY product_pair (product_id, related_product_id),
            KEY idx_product (product_id),
            KEY idx_count (purchase_count)
        ) $charset_collate;";

        // AI Recommendations - Product view tracking
        $table_product_views = $wpdb->prefix . 'cartt_product_views';
        $sql_product_views = "CREATE TABLE $table_product_views (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            customer_id bigint(20) unsigned DEFAULT NULL,
            session_id varchar(64) NOT NULL,
            viewed_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_product (product_id),
            KEY idx_customer (customer_id),
            KEY idx_session (session_id),
            KEY idx_viewed (viewed_at)
        ) $charset_collate;";

        // Advanced Analytics - Event tracking
        $table_analytics_events = $wpdb->prefix . 'cartt_analytics_events';
        $sql_analytics_events = "CREATE TABLE $table_analytics_events (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            event_type varchar(50) NOT NULL,
            session_id varchar(64) NOT NULL,
            customer_id bigint(20) unsigned DEFAULT NULL,
            product_id bigint(20) unsigned DEFAULT NULL,
            order_id bigint(20) unsigned DEFAULT NULL,
            event_data longtext,
            page_url varchar(500) DEFAULT NULL,
            referrer varchar(500) DEFAULT NULL,
            utm_source varchar(100) DEFAULT NULL,
            utm_medium varchar(100) DEFAULT NULL,
            utm_campaign varchar(100) DEFAULT NULL,
            user_agent varchar(500) DEFAULT NULL,
            ip_address varchar(45) DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_event_type (event_type),
            KEY idx_session (session_id),
            KEY idx_customer (customer_id),
            KEY idx_product (product_id),
            KEY idx_created (created_at)
        ) $charset_collate;";

        // Vendors - Store profiles
        $table_vendors = $wpdb->prefix . 'cartt_vendors';
        $sql_vendors = "CREATE TABLE $table_vendors (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned NOT NULL,
            store_name varchar(200) NOT NULL,
            store_slug varchar(200) NOT NULL,
            email varchar(200) NOT NULL,
            phone varchar(50) DEFAULT NULL,
            description text,
            logo_url varchar(500) DEFAULT NULL,
            banner_url varchar(500) DEFAULT NULL,
            address varchar(500) DEFAULT NULL,
            city varchar(100) DEFAULT NULL,
            state varchar(100) DEFAULT NULL,
            country varchar(2) DEFAULT NULL,
            postal_code varchar(20) DEFAULT NULL,
            commission_rate decimal(5,2) DEFAULT NULL,
            status enum('pending','active','suspended','rejected') NOT NULL DEFAULT 'pending',
            verified tinyint(1) NOT NULL DEFAULT 0,
            verified_at datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY user_id (user_id),
            UNIQUE KEY store_slug (store_slug),
            KEY idx_status (status)
        ) $charset_collate;";

        // Vendor settings (key-value store)
        $table_vendor_settings = $wpdb->prefix . 'cartt_vendor_settings';
        $sql_vendor_settings = "CREATE TABLE $table_vendor_settings (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            vendor_id bigint(20) unsigned NOT NULL,
            setting_key varchar(100) NOT NULL,
            setting_value longtext,
            PRIMARY KEY (id),
            UNIQUE KEY vendor_setting (vendor_id, setting_key)
        ) $charset_collate;";

        // Vendor commissions per order item
        $table_vendor_commissions = $wpdb->prefix . 'cartt_vendor_commissions';
        $sql_vendor_commissions = "CREATE TABLE $table_vendor_commissions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            order_id bigint(20) unsigned NOT NULL,
            order_item_id bigint(20) unsigned NOT NULL,
            vendor_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            order_subtotal decimal(10,2) NOT NULL,
            commission_rate decimal(5,2) NOT NULL,
            platform_fee decimal(10,2) NOT NULL,
            vendor_amount decimal(10,2) NOT NULL,
            status enum('pending','available','paid','refunded') NOT NULL DEFAULT 'pending',
            payout_id bigint(20) unsigned DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_order (order_id),
            KEY idx_vendor (vendor_id),
            KEY idx_status (status),
            KEY idx_payout (payout_id)
        ) $charset_collate;";

        // Vendor payout requests
        $table_vendor_payouts = $wpdb->prefix . 'cartt_vendor_payouts';
        $sql_vendor_payouts = "CREATE TABLE $table_vendor_payouts (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            vendor_id bigint(20) unsigned NOT NULL,
            amount decimal(10,2) NOT NULL,
            payment_method varchar(50) NOT NULL,
            payment_email varchar(200) DEFAULT NULL,
            payment_details text,
            status enum('pending','processing','completed','rejected') NOT NULL DEFAULT 'pending',
            transaction_id varchar(200) DEFAULT NULL,
            requested_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            processed_at datetime DEFAULT NULL,
            processed_by bigint(20) unsigned DEFAULT NULL,
            rejection_reason text,
            PRIMARY KEY (id),
            KEY idx_vendor (vendor_id),
            KEY idx_status (status)
        ) $charset_collate;";

        // Wholesale customers
        $table_wholesale_customers = $wpdb->prefix . 'cartt_wholesale_customers';
        $sql_wholesale_customers = "CREATE TABLE $table_wholesale_customers (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            company_name varchar(200) NOT NULL,
            tax_id varchar(100) DEFAULT NULL,
            tax_exempt tinyint(1) NOT NULL DEFAULT 0,
            discount_percent decimal(5,2) NOT NULL DEFAULT 0,
            min_order_amount decimal(10,2) NOT NULL DEFAULT 0,
            payment_terms varchar(50) NOT NULL DEFAULT 'prepaid',
            credit_limit decimal(10,2) NOT NULL DEFAULT 0,
            tier varchar(20) NOT NULL DEFAULT 'bronze',
            status enum('pending','active','suspended') NOT NULL DEFAULT 'pending',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY customer_id (customer_id),
            KEY idx_status (status),
            KEY idx_tier (tier)
        ) $charset_collate;";

        // Wholesale applications
        $table_wholesale_applications = $wpdb->prefix . 'cartt_wholesale_applications';
        $sql_wholesale_applications = "CREATE TABLE $table_wholesale_applications (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            user_id bigint(20) unsigned DEFAULT NULL,
            company_name varchar(200) NOT NULL,
            contact_name varchar(200) NOT NULL,
            email varchar(200) NOT NULL,
            phone varchar(50) DEFAULT NULL,
            website varchar(500) DEFAULT NULL,
            tax_id varchar(100) DEFAULT NULL,
            business_type varchar(100) DEFAULT NULL,
            years_in_business int(11) DEFAULT NULL,
            estimated_monthly_volume decimal(10,2) DEFAULT NULL,
            message text,
            status enum('pending','approved','rejected') NOT NULL DEFAULT 'pending',
            reviewed_by bigint(20) unsigned DEFAULT NULL,
            reviewed_at datetime DEFAULT NULL,
            review_notes text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_status (status),
            KEY idx_email (email)
        ) $charset_collate;";

        // Quantity-based pricing
        $table_quantity_pricing = $wpdb->prefix . 'cartt_quantity_pricing';
        $sql_quantity_pricing = "CREATE TABLE $table_quantity_pricing (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            min_quantity int(11) NOT NULL,
            max_quantity int(11) DEFAULT NULL,
            price decimal(10,2) NOT NULL,
            discount_percent decimal(5,2) DEFAULT NULL,
            tier varchar(20) DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_product (product_id),
            KEY idx_quantity (min_quantity)
        ) $charset_collate;";

        // Customer-specific pricing
        $table_customer_pricing = $wpdb->prefix . 'cartt_customer_pricing';
        $sql_customer_pricing = "CREATE TABLE $table_customer_pricing (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            customer_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            price decimal(10,2) NOT NULL,
            valid_from datetime DEFAULT NULL,
            valid_to datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY customer_product (customer_id, product_id),
            KEY idx_customer (customer_id),
            KEY idx_product (product_id)
        ) $charset_collate;";

        // Quote requests
        $table_quotes = $wpdb->prefix . 'cartt_quotes';
        $sql_quotes = "CREATE TABLE $table_quotes (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            quote_number varchar(50) NOT NULL,
            customer_id bigint(20) unsigned DEFAULT NULL,
            customer_email varchar(200) NOT NULL,
            customer_name varchar(200) NOT NULL,
            company_name varchar(200) DEFAULT NULL,
            phone varchar(50) DEFAULT NULL,
            notes text,
            subtotal decimal(10,2) NOT NULL,
            discount_amount decimal(10,2) NOT NULL DEFAULT 0,
            total decimal(10,2) NOT NULL,
            status enum('pending','sent','accepted','rejected','expired','converted') NOT NULL DEFAULT 'pending',
            admin_notes text,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            expires_at datetime DEFAULT NULL,
            converted_order_id bigint(20) unsigned DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY quote_number (quote_number),
            KEY idx_status (status),
            KEY idx_customer (customer_id)
        ) $charset_collate;";

        // Quote line items
        $table_quote_items = $wpdb->prefix . 'cartt_quote_items';
        $sql_quote_items = "CREATE TABLE $table_quote_items (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            quote_id bigint(20) unsigned NOT NULL,
            product_id bigint(20) unsigned NOT NULL,
            product_name varchar(255) NOT NULL,
            sku varchar(100) DEFAULT NULL,
            quantity int(11) NOT NULL,
            unit_price decimal(10,2) NOT NULL,
            total decimal(10,2) NOT NULL,
            notes text,
            PRIMARY KEY (id),
            KEY idx_quote (quote_id)
        ) $charset_collate;";

        // Dropshipping suppliers
        $table_suppliers = $wpdb->prefix . 'cartt_suppliers';
        $sql_suppliers = "CREATE TABLE $table_suppliers (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            name varchar(200) NOT NULL,
            type enum('generic','aliexpress','printful','printify') NOT NULL DEFAULT 'generic',
            api_key varchar(500) DEFAULT NULL,
            api_secret varchar(500) DEFAULT NULL,
            store_id varchar(100) DEFAULT NULL,
            settings longtext,
            status enum('active','inactive') NOT NULL DEFAULT 'active',
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_type (type),
            KEY idx_status (status)
        ) $charset_collate;";

        // Product-to-supplier mappings
        $table_supplier_products = $wpdb->prefix . 'cartt_supplier_products';
        $sql_supplier_products = "CREATE TABLE $table_supplier_products (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            supplier_id bigint(20) unsigned NOT NULL,
            supplier_sku varchar(200) NOT NULL,
            supplier_url varchar(1000) DEFAULT NULL,
            supplier_price decimal(10,2) NOT NULL,
            our_price decimal(10,2) NOT NULL,
            auto_fulfill tinyint(1) NOT NULL DEFAULT 1,
            last_synced datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY product_supplier (product_id, supplier_id),
            KEY idx_supplier (supplier_id),
            KEY idx_product (product_id)
        ) $charset_collate;";

        // Social proof - visitor tracking for live counts
        $table_visitor_sessions = $wpdb->prefix . 'cartt_visitor_sessions';
        $sql_visitor_sessions = "CREATE TABLE $table_visitor_sessions (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            session_id varchar(64) NOT NULL,
            product_id bigint(20) unsigned DEFAULT NULL,
            page_type varchar(50) NOT NULL DEFAULT 'other',
            ip_address varchar(45) DEFAULT NULL,
            last_activity datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY session_product (session_id, product_id),
            KEY idx_product (product_id),
            KEY idx_activity (last_activity)
        ) $charset_collate;";

        // Social proof - cart activity tracking
        $table_cart_activity = $wpdb->prefix . 'cartt_cart_activity';
        $sql_cart_activity = "CREATE TABLE $table_cart_activity (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            session_id varchar(64) NOT NULL,
            quantity int(11) NOT NULL DEFAULT 1,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY product_session (product_id, session_id),
            KEY idx_product (product_id),
            KEY idx_updated (updated_at)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        
        dbDelta($sql_products);
        dbDelta($sql_categories);
        dbDelta($sql_product_categories);
        dbDelta($sql_customers);
        dbDelta($sql_orders);
        dbDelta($sql_order_items);
        dbDelta($sql_downloads);
        dbDelta($sql_download_permissions);
        dbDelta($sql_subscriptions);
        dbDelta($sql_bookings);
        dbDelta($sql_coupons);
        dbDelta($sql_shipping_zones);
        dbDelta($sql_shipping_methods);
        dbDelta($sql_variations);
        dbDelta($sql_relationships);
        dbDelta($sql_currencies);
        dbDelta($sql_tax_rates);
        dbDelta($sql_reviews);
        dbDelta($sql_order_notes);
        dbDelta($sql_refunds);
        dbDelta($sql_wishlists);
        dbDelta($sql_wishlist_items);
        dbDelta($sql_gift_cards);
        dbDelta($sql_gift_card_transactions);
        dbDelta($sql_loyalty_points);
        dbDelta($sql_loyalty_transactions);
        dbDelta($sql_rma);
        dbDelta($sql_rma_items);
        dbDelta($sql_product_qa);
        dbDelta($sql_order_tracking);
        dbDelta($sql_product_seo);
        dbDelta($sql_stock_notifications);

        // v1.4.0 Tables
        dbDelta($sql_purchase_patterns);
        dbDelta($sql_product_views);
        dbDelta($sql_analytics_events);
        dbDelta($sql_vendors);
        dbDelta($sql_vendor_settings);
        dbDelta($sql_vendor_commissions);
        dbDelta($sql_vendor_payouts);
        dbDelta($sql_wholesale_customers);
        dbDelta($sql_wholesale_applications);
        dbDelta($sql_quantity_pricing);
        dbDelta($sql_customer_pricing);
        dbDelta($sql_quotes);
        dbDelta($sql_quote_items);
        dbDelta($sql_suppliers);
        dbDelta($sql_supplier_products);
        dbDelta($sql_visitor_sessions);
        dbDelta($sql_cart_activity);

        // Software Licensing Tables (v1.4.1)
        $table_licenses = $wpdb->prefix . 'cartt_licenses';
        $sql_licenses = "CREATE TABLE $table_licenses (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            license_key VARCHAR(50) NOT NULL UNIQUE,
            addon VARCHAR(100) NOT NULL,
            customer_id BIGINT UNSIGNED DEFAULT 0,
            customer_email VARCHAR(200),
            order_id BIGINT UNSIGNED DEFAULT 0,
            activation_limit INT UNSIGNED DEFAULT 1,
            status ENUM('active','expired','suspended','revoked') DEFAULT 'active',
            expires_at DATETIME DEFAULT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_customer (customer_id),
            INDEX idx_order (order_id),
            INDEX idx_addon (addon),
            INDEX idx_status (status)
        ) $charset_collate;";
        
        $table_activations = $wpdb->prefix . 'cartt_license_activations';
        $sql_activations = "CREATE TABLE $table_activations (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            license_id BIGINT UNSIGNED NOT NULL,
            site_url VARCHAR(500) NOT NULL,
            activated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            last_check DATETIME DEFAULT NULL,
            status ENUM('active','deactivated') DEFAULT 'active',
            INDEX idx_license (license_id),
            INDEX idx_site (site_url(191)),
            UNIQUE KEY license_site (license_id, site_url(191))
        ) $charset_collate;";
        
        dbDelta($sql_licenses);
        dbDelta($sql_activations);

        // Chat Tables (v1.4.2)
        $table_chat_conversations = $wpdb->prefix . 'cartt_chat_conversations';
        $sql_chat_conversations = "CREATE TABLE $table_chat_conversations (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            visitor_id VARCHAR(64) NOT NULL,
            visitor_name VARCHAR(100) DEFAULT '',
            visitor_email VARCHAR(200) DEFAULT '',
            visitor_phone VARCHAR(30) DEFAULT '',
            status ENUM('active','closed') DEFAULT 'active',
            assigned_agent_id BIGINT UNSIGNED DEFAULT NULL,
            cart_data LONGTEXT DEFAULT NULL,
            page_url TEXT DEFAULT NULL,
            ip_address VARCHAR(45) DEFAULT NULL,
            handoff_at DATETIME DEFAULT NULL,
            handoff_status ENUM('none','pending','followup_sent','resolved') DEFAULT 'none',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            INDEX idx_visitor (visitor_id),
            INDEX idx_status (status),
            INDEX idx_handoff (handoff_status)
        ) $charset_collate;";
        
        $table_chat_messages = $wpdb->prefix . 'cartt_chat_messages';
        $sql_chat_messages = "CREATE TABLE $table_chat_messages (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            conversation_id BIGINT UNSIGNED NOT NULL,
            sender ENUM('visitor','admin') NOT NULL,
            message TEXT NOT NULL,
            is_read TINYINT(1) DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_conversation (conversation_id),
            INDEX idx_read (is_read)
        ) $charset_collate;";
        
        $table_chat_agents = $wpdb->prefix . 'cartt_chat_agents';
        $sql_chat_agents = "CREATE TABLE $table_chat_agents (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT UNSIGNED NOT NULL UNIQUE,
            display_name VARCHAR(100) DEFAULT '',
            avatar VARCHAR(500) DEFAULT '',
            email VARCHAR(200) DEFAULT '',
            sms_number VARCHAR(30) DEFAULT '',
            whatsapp_number VARCHAR(30) DEFAULT '',
            is_active TINYINT(1) DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_user (user_id)
        ) $charset_collate;";
        
        dbDelta($sql_chat_conversations);
        dbDelta($sql_chat_messages);
        dbDelta($sql_chat_agents);

        // Store database version
        update_option('cartt_db_version', CARTT_VERSION);
    }

    private static function setDefaultOptions(): void
    {
        $defaults = [
            'cartt_currency' => 'USD',
            'cartt_currency_position' => 'left',
            'cartt_thousand_separator' => ',',
            'cartt_decimal_separator' => '.',
            'cartt_decimals' => 2,
            'cartt_tax_enabled' => false,
            'cartt_prices_include_tax' => false,
            'cartt_multi_currency_enabled' => false,
            'cartt_auto_detect_currency' => false,
            'cartt_exchange_rate_api' => '',
        ];

        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
    }
}
