<?php

declare(strict_types=1);

namespace Cartt\Api;

/**
 * REST API Controller - provides headless commerce capability
 */
class RestController
{
    private string $namespace = 'cartt/v1';

    public function __construct()
    {
        add_action('rest_api_init', [$this, 'registerRoutes']);
    }

    public function registerRoutes(): void
    {
        // Products
        register_rest_route($this->namespace, '/products', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getProducts'],
                'permission_callback' => '__return_true',
            ],
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'createProduct'],
                'permission_callback' => [$this, 'canManageProducts'],
            ],
        ]);

        register_rest_route($this->namespace, '/products/(?P<id>\d+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getProduct'],
                'permission_callback' => '__return_true',
            ],
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [$this, 'updateProduct'],
                'permission_callback' => [$this, 'canManageProducts'],
            ],
            [
                'methods' => \WP_REST_Server::DELETABLE,
                'callback' => [$this, 'deleteProduct'],
                'permission_callback' => [$this, 'canManageProducts'],
            ],
        ]);

        // Cart
        register_rest_route($this->namespace, '/cart', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getCart'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/cart/add', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'addToCart'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/cart/update', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'updateCart'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/cart/remove', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'removeFromCart'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/cart/capture-email', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'captureCartEmail'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Coupon endpoints
        register_rest_route($this->namespace, '/coupon/apply', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'applyCoupon'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/coupon/remove', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'removeCoupon'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Gift Card endpoints
        register_rest_route($this->namespace, '/gift-card/apply', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'applyGiftCard'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/gift-card/remove', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'removeGiftCard'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Loyalty Points endpoints
        register_rest_route($this->namespace, '/loyalty/apply', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'applyLoyaltyPoints'],
                'permission_callback' => [$this, 'isLoggedIn'],
            ],
        ]);

        register_rest_route($this->namespace, '/loyalty/remove', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'removeLoyaltyPoints'],
                'permission_callback' => [$this, 'isLoggedIn'],
            ],
        ]);

        // Shipping endpoints
        register_rest_route($this->namespace, '/shipping/calculate', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'calculateShipping'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/shipping/select', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'selectShippingMethod'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Order bump endpoints
        register_rest_route($this->namespace, '/cart/add-bump', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'addOrderBump'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/cart/remove-bump', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'removeOrderBump'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Checkout
        register_rest_route($this->namespace, '/checkout', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'processCheckout'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/checkout/payment-intent', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'createPaymentIntent'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/checkout/confirm', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'confirmPayment'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Orders
        register_rest_route($this->namespace, '/orders', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getOrders'],
                'permission_callback' => [$this, 'canViewOrders'],
            ],
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'createOrder'],
                'permission_callback' => [$this, 'canManageOrders'],
            ],
        ]);

        register_rest_route($this->namespace, '/orders/(?P<id>\d+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getOrder'],
                'permission_callback' => [$this, 'canViewOrder'],
            ],
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [$this, 'updateOrder'],
                'permission_callback' => [$this, 'canManageOrders'],
            ],
        ]);

        // Customers (admin only)
        register_rest_route($this->namespace, '/customers', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getCustomers'],
                'permission_callback' => [$this, 'canManageCustomers'],
            ],
        ]);

        register_rest_route($this->namespace, '/customers/(?P<id>\d+)', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getCustomer'],
                'permission_callback' => [$this, 'canManageCustomers'],
            ],
        ]);

        // Categories
        register_rest_route($this->namespace, '/categories', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getCategories'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // Settings (admin only)
        register_rest_route($this->namespace, '/settings', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getSettings'],
                'permission_callback' => [$this, 'canManageSettings'],
            ],
            [
                'methods' => \WP_REST_Server::EDITABLE,
                'callback' => [$this, 'updateSettings'],
                'permission_callback' => [$this, 'canManageSettings'],
            ],
        ]);

        // Stats (admin only)
        register_rest_route($this->namespace, '/stats', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getStats'],
                'permission_callback' => [$this, 'canViewStats'],
            ],
        ]);

        // WooCommerce Migration
        register_rest_route($this->namespace, '/migration/status', [
            [
                'methods' => \WP_REST_Server::READABLE,
                'callback' => [$this, 'getMigrationStatus'],
                'permission_callback' => [$this, 'canManageSettings'],
            ],
        ]);

        register_rest_route($this->namespace, '/migration/run', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'runMigration'],
                'permission_callback' => [$this, 'canManageSettings'],
            ],
        ]);

        // Demo data
        register_rest_route($this->namespace, '/demo/generate', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'generateDemoData'],
                'permission_callback' => [$this, 'canManageSettings'],
            ],
        ]);

        // PayPal
        register_rest_route($this->namespace, '/paypal/create-order', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'createPayPalOrder'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($this->namespace, '/paypal/capture', [
            [
                'methods' => \WP_REST_Server::CREATABLE,
                'callback' => [$this, 'capturePayPalOrder'],
                'permission_callback' => '__return_true',
            ],
        ]);
    }

    // Permission callbacks
    public function canManageProducts(): bool
    {
        return current_user_can('manage_options');
    }

    public function canManageOrders(): bool
    {
        return current_user_can('manage_options');
    }

    public function canViewOrders(\WP_REST_Request $request): bool
    {
        return is_user_logged_in();
    }

    public function canViewOrder(\WP_REST_Request $request): bool
    {
        if (current_user_can('manage_options')) {
            return true;
        }
        // TODO: Check if order belongs to current user
        return is_user_logged_in();
    }

    public function canManageCustomers(): bool
    {
        return current_user_can('manage_options');
    }

    public function canManageSettings(): bool
    {
        return current_user_can('manage_options');
    }

    public function canViewStats(): bool
    {
        return current_user_can('manage_options');
    }

    // Product endpoints
    public function getProducts(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        $page = (int) $request->get_param('page') ?: 1;
        $per_page = (int) $request->get_param('per_page') ?: 20;
        $offset = ($page - 1) * $per_page;
        $status = $request->get_param('status') ?: 'publish';
        $search = $request->get_param('search');

        $where = "WHERE status = %s";
        $params = [$status];

        if ($search) {
            $where .= " AND (name LIKE %s OR sku LIKE %s)";
            $params[] = '%' . $wpdb->esc_like($search) . '%';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }

        $total = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table $where",
            ...$params
        ));

        $products = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table $where ORDER BY created_at DESC LIMIT %d OFFSET %d",
            ...array_merge($params, [$per_page, $offset])
        ));

        return new \WP_REST_Response([
            'data' => $products,
            'meta' => [
                'total' => (int) $total,
                'page' => $page,
                'per_page' => $per_page,
                'total_pages' => ceil($total / $per_page),
            ],
        ]);
    }

    public function getProduct(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $id = (int) $request->get_param('id');

        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));

        if (!$product) {
            return new \WP_REST_Response(['error' => 'Product not found'], 404);
        }

        return new \WP_REST_Response(['data' => $product]);
    }

    public function createProduct(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        $data = [
            'name' => sanitize_text_field($request->get_param('name')),
            'slug' => sanitize_title($request->get_param('name')),
            'description' => wp_kses_post($request->get_param('description')),
            'short_description' => sanitize_textarea_field($request->get_param('short_description')),
            'product_type' => sanitize_text_field($request->get_param('product_type') ?: 'physical'),
            'status' => sanitize_text_field($request->get_param('status') ?: 'draft'),
            'price' => floatval($request->get_param('price')),
            'sale_price' => $request->get_param('sale_price') ? floatval($request->get_param('sale_price')) : null,
            'sku' => sanitize_text_field($request->get_param('sku')),
            'stock_quantity' => $request->get_param('stock_quantity') ? intval($request->get_param('stock_quantity')) : null,
            'manage_stock' => (bool) $request->get_param('manage_stock'),
            'featured_image' => $request->get_param('featured_image') ? intval($request->get_param('featured_image')) : null,
        ];

        $wpdb->insert($table, $data);
        $id = $wpdb->insert_id;

        if (!$id) {
            return new \WP_REST_Response(['error' => 'Failed to create product'], 500);
        }

        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));

        return new \WP_REST_Response(['data' => $product], 201);
    }

    public function updateProduct(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $id = (int) $request->get_param('id');

        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));

        if (!$existing) {
            return new \WP_REST_Response(['error' => 'Product not found'], 404);
        }

        $data = [];
        $fields = ['name', 'description', 'short_description', 'product_type', 'status', 'price', 'sale_price', 'sku', 'stock_quantity', 'manage_stock', 'featured_image', 'gallery_images'];

        foreach ($fields as $field) {
            if ($request->has_param($field)) {
                $value = $request->get_param($field);
                
                switch ($field) {
                    case 'name':
                        $data['name'] = sanitize_text_field($value);
                        $data['slug'] = sanitize_title($value);
                        break;
                    case 'description':
                        $data['description'] = wp_kses_post($value);
                        break;
                    case 'price':
                    case 'sale_price':
                        $data[$field] = $value !== null ? floatval($value) : null;
                        break;
                    case 'stock_quantity':
                    case 'featured_image':
                        $data[$field] = $value !== null ? intval($value) : null;
                        break;
                    case 'manage_stock':
                        $data[$field] = (bool) $value;
                        break;
                    case 'gallery_images':
                        // Validate JSON array of image IDs
                        $data[$field] = is_string($value) ? $value : json_encode($value);
                        break;
                    default:
                        $data[$field] = sanitize_text_field($value);
                }
            }
        }

        if (!empty($data)) {
            $wpdb->update($table, $data, ['id' => $id]);
        }

        // Handle categories
        if ($request->has_param('categories')) {
            $categories = $request->get_param('categories');
            if (is_array($categories)) {
                \Cartt\Services\CategoryService::setProductCategories($id, $categories);
            }
        }

        $product = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));

        return new \WP_REST_Response(['data' => $product]);
    }

    public function deleteProduct(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $id = (int) $request->get_param('id');

        $deleted = $wpdb->delete($table, ['id' => $id]);

        if (!$deleted) {
            return new \WP_REST_Response(['error' => 'Product not found'], 404);
        }

        return new \WP_REST_Response(['success' => true]);
    }

    // Cart endpoints
    public function getCart(\WP_REST_Request $request): \WP_REST_Response
    {
        $cartService = new \Cartt\Services\CartService();
        return new \WP_REST_Response(['data' => $cartService->getCart()]);
    }

    public function addToCart(\WP_REST_Request $request): \WP_REST_Response
    {
        $productId = (int) $request->get_param('product_id');
        $quantity = (int) ($request->get_param('quantity') ?: 1);
        $meta = $request->get_param('meta') ?: [];

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->addItem($productId, $quantity, $meta);

        if (!$result['success']) {
            return new \WP_REST_Response(['error' => $result['error']], 400);
        }

        return new \WP_REST_Response($result);
    }

    public function updateCart(\WP_REST_Request $request): \WP_REST_Response
    {
        $itemKey = $request->get_param('item_key');
        $quantity = (int) $request->get_param('quantity');

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->updateItem($itemKey, $quantity);

        if (!$result['success']) {
            return new \WP_REST_Response(['error' => $result['error']], 400);
        }

        return new \WP_REST_Response($result);
    }

    public function removeFromCart(\WP_REST_Request $request): \WP_REST_Response
    {
        $itemKey = $request->get_param('item_key');

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->removeItem($itemKey);

        return new \WP_REST_Response($result);
    }

    public function captureCartEmail(\WP_REST_Request $request): \WP_REST_Response
    {
        $email = sanitize_email($request->get_param('email'));
        $billing = $request->get_param('billing') ?: [];

        if (!is_email($email)) {
            return new \WP_REST_Response(['success' => false]);
        }

        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();

        if (empty($cart['items'])) {
            return new \WP_REST_Response(['success' => false]);
        }

        $abandonedService = new \Cartt\Services\AbandonedCartService();
        $id = $abandonedService->captureEmail($email, $cart, $billing);

        return new \WP_REST_Response(['success' => $id > 0, 'id' => $id]);
    }

    public function applyCoupon(\WP_REST_Request $request): \WP_REST_Response
    {
        $code = sanitize_text_field($request->get_param('code'));
        $email = sanitize_email($request->get_param('email'));

        if (empty($code)) {
            return new \WP_REST_Response(['success' => false, 'error' => __('Please enter a coupon code.', 'cartt')], 400);
        }

        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();

        if (empty($cart['items'])) {
            return new \WP_REST_Response(['success' => false, 'error' => __('Your cart is empty.', 'cartt')], 400);
        }

        $couponService = new \Cartt\Services\CouponService();
        $result = $couponService->applyCoupon($code, $cart, $email);

        if (!$result['valid']) {
            return new \WP_REST_Response(['success' => false, 'error' => $result['error']], 400);
        }

        // Store coupon in session
        $cartService->applyCoupon($result['coupon'], $result['discount']);

        return new \WP_REST_Response([
            'success' => true,
            'coupon' => $result['coupon'],
            'discount' => $result['discount'],
            'cart' => $cartService->getCart(),
        ]);
    }

    public function removeCoupon(\WP_REST_Request $request): \WP_REST_Response
    {
        $cartService = new \Cartt\Services\CartService();
        $cartService->removeCoupon();

        return new \WP_REST_Response([
            'success' => true,
            'cart' => $cartService->getCart(),
        ]);
    }

    public function calculateShipping(\WP_REST_Request $request): \WP_REST_Response
    {
        $address = [
            'country' => sanitize_text_field($request->get_param('country') ?? ''),
            'state' => sanitize_text_field($request->get_param('state') ?? ''),
            'postcode' => sanitize_text_field($request->get_param('postcode') ?? ''),
            'city' => sanitize_text_field($request->get_param('city') ?? ''),
        ];

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->setShippingAddress($address);

        return new \WP_REST_Response($result);
    }

    public function selectShippingMethod(\WP_REST_Request $request): \WP_REST_Response
    {
        $methodId = (int) $request->get_param('method_id');

        if (!$methodId) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Method ID required'], 400);
        }

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->setShippingMethod($methodId);

        return new \WP_REST_Response($result);
    }

    public function addOrderBump(\WP_REST_Request $request): \WP_REST_Response
    {
        $bumpId = sanitize_text_field($request->get_param('bump_id'));

        if (empty($bumpId)) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Bump ID required'], 400);
        }

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->addBump($bumpId);

        return new \WP_REST_Response($result);
    }

    public function removeOrderBump(\WP_REST_Request $request): \WP_REST_Response
    {
        $bumpId = sanitize_text_field($request->get_param('bump_id'));

        if (empty($bumpId)) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Bump ID required'], 400);
        }

        $cartService = new \Cartt\Services\CartService();
        $result = $cartService->removeBump($bumpId);

        return new \WP_REST_Response($result);
    }

    // Checkout
    public function processCheckout(\WP_REST_Request $request): \WP_REST_Response
    {
        $cartService = new \Cartt\Services\CartService();
        $checkoutService = new \Cartt\Services\CheckoutService($cartService);

        $data = [
            'billing' => $request->get_param('billing') ?: [],
            'shipping' => $request->get_param('shipping'),
            'payment_method' => $request->get_param('payment_method') ?: 'stripe',
            'notes' => $request->get_param('notes'),
        ];

        $result = $checkoutService->processCheckout($data);

        if (!$result['success']) {
            return new \WP_REST_Response(['error' => $result['error']], 400);
        }

        return new \WP_REST_Response($result);
    }

    public function createPaymentIntent(\WP_REST_Request $request): \WP_REST_Response
    {
        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();

        if (empty($cart['items'])) {
            return new \WP_REST_Response(['error' => 'Cart is empty'], 400);
        }

        $checkoutService = new \Cartt\Services\CheckoutService($cartService);
        $currency = get_option('cartt_currency', 'USD');
        $result = $checkoutService->createPaymentIntent($cart['totals']['total'], $currency);

        if (!$result['success']) {
            return new \WP_REST_Response(['error' => $result['error']], 400);
        }

        return new \WP_REST_Response($result);
    }

    public function confirmPayment(\WP_REST_Request $request): \WP_REST_Response
    {
        $orderId = (int) $request->get_param('order_id');
        $paymentIntentId = $request->get_param('payment_intent_id');

        $cartService = new \Cartt\Services\CartService();
        $checkoutService = new \Cartt\Services\CheckoutService($cartService);
        $result = $checkoutService->confirmPayment($orderId, $paymentIntentId);

        if (!$result['success']) {
            return new \WP_REST_Response(['error' => $result['error']], 400);
        }

        return new \WP_REST_Response($result);
    }

    // Orders (placeholder implementations)
    public function getOrders(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_orders';

        $orders = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC LIMIT 50");

        return new \WP_REST_Response(['data' => $orders]);
    }

    public function getOrder(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_orders';
        $id = (int) $request->get_param('id');

        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));

        if (!$order) {
            return new \WP_REST_Response(['error' => 'Order not found'], 404);
        }

        // Get order items
        $items_table = $wpdb->prefix . 'cartt_order_items';
        $order->items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $items_table WHERE order_id = %d",
            $id
        ));

        return new \WP_REST_Response(['data' => $order]);
    }

    public function createOrder(\WP_REST_Request $request): \WP_REST_Response
    {
        // TODO: Implement
        return new \WP_REST_Response(['success' => true]);
    }

    public function updateOrder(\WP_REST_Request $request): \WP_REST_Response
    {
        // TODO: Implement
        return new \WP_REST_Response(['success' => true]);
    }

    // Customers
    public function getCustomers(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customers';

        $customers = $wpdb->get_results("SELECT * FROM $table ORDER BY created_at DESC LIMIT 50");

        return new \WP_REST_Response(['data' => $customers]);
    }

    public function getCustomer(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_customers';
        $id = (int) $request->get_param('id');

        $customer = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d",
            $id
        ));

        if (!$customer) {
            return new \WP_REST_Response(['error' => 'Customer not found'], 404);
        }

        return new \WP_REST_Response(['data' => $customer]);
    }

    // Categories
    public function getCategories(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_categories';

        $categories = $wpdb->get_results("SELECT * FROM $table ORDER BY sort_order ASC, name ASC");

        return new \WP_REST_Response(['data' => $categories]);
    }

    // Settings
    public function getSettings(\WP_REST_Request $request): \WP_REST_Response
    {
        return new \WP_REST_Response([
            'data' => [
                'store_type' => get_option('cartt_store_type', 'mixed'),
                'store_name' => get_option('cartt_store_name', get_bloginfo('name')),
                'currency' => get_option('cartt_currency', 'USD'),
                'country' => get_option('cartt_country', 'US'),
                'payment_methods' => get_option('cartt_payment_methods', []),
                'tax_enabled' => get_option('cartt_tax_enabled', false),
            ],
        ]);
    }

    public function updateSettings(\WP_REST_Request $request): \WP_REST_Response
    {
        $settings = ['store_type', 'store_name', 'currency', 'country', 'tax_enabled'];

        foreach ($settings as $setting) {
            if ($request->has_param($setting)) {
                update_option('cartt_' . $setting, $request->get_param($setting));
            }
        }

        return $this->getSettings($request);
    }

    // Stats
    public function getStats(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        
        $orders_table = $wpdb->prefix . 'cartt_orders';
        $products_table = $wpdb->prefix . 'cartt_products';
        $customers_table = $wpdb->prefix . 'cartt_customers';

        $total_orders = $wpdb->get_var("SELECT COUNT(*) FROM $orders_table");
        $total_revenue = $wpdb->get_var("SELECT SUM(total) FROM $orders_table WHERE status IN ('completed', 'processing')") ?: 0;
        $total_products = $wpdb->get_var("SELECT COUNT(*) FROM $products_table WHERE status = 'publish'");
        $total_customers = $wpdb->get_var("SELECT COUNT(*) FROM $customers_table");

        // Recent orders for dashboard
        $recent_orders = $wpdb->get_results("SELECT * FROM $orders_table ORDER BY created_at DESC LIMIT 5");

        return new \WP_REST_Response([
            'data' => [
                'total_orders' => (int) $total_orders,
                'total_revenue' => (float) $total_revenue,
                'total_products' => (int) $total_products,
                'total_customers' => (int) $total_customers,
                'recent_orders' => $recent_orders,
            ],
        ]);
    }

    // Migration endpoints
    public function getMigrationStatus(\WP_REST_Request $request): \WP_REST_Response
    {
        $migration = new \Cartt\Services\WooMigrationService();

        return new \WP_REST_Response([
            'data' => [
                'woo_active' => $migration->isWooCommerceActive(),
                'has_data' => $migration->hasWooCommerceData(),
                'stats' => $migration->getWooCommerceStats(),
            ],
        ]);
    }

    public function runMigration(\WP_REST_Request $request): \WP_REST_Response
    {
        $migration = new \Cartt\Services\WooMigrationService();

        if (!$migration->hasWooCommerceData()) {
            return new \WP_REST_Response(['error' => 'No WooCommerce data found'], 400);
        }

        $options = [
            'products' => (bool) ($request->get_param('products') ?? true),
            'categories' => (bool) ($request->get_param('categories') ?? true),
            'customers' => (bool) ($request->get_param('customers') ?? true),
            'orders' => (bool) ($request->get_param('orders') ?? true),
        ];

        $results = $migration->migrate($options);

        return new \WP_REST_Response([
            'success' => true,
            'data' => $results,
        ]);
    }

    public function generateDemoData(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';

        $demoProducts = [
            [
                'name' => 'Classic T-Shirt',
                'slug' => 'classic-t-shirt',
                'description' => 'A comfortable, everyday essential. Made from 100% organic cotton.',
                'short_description' => 'Soft organic cotton tee.',
                'product_type' => 'physical',
                'status' => 'publish',
                'price' => 29.99,
                'sale_price' => null,
                'sku' => 'TSH-001',
                'stock_quantity' => 50,
                'manage_stock' => 1,
            ],
            [
                'name' => 'Premium Hoodie',
                'slug' => 'premium-hoodie',
                'description' => 'Stay warm in style with our premium fleece hoodie. Features a kangaroo pocket and adjustable drawstring.',
                'short_description' => 'Cozy fleece hoodie.',
                'product_type' => 'physical',
                'status' => 'publish',
                'price' => 79.99,
                'sale_price' => 59.99,
                'sku' => 'HOD-001',
                'stock_quantity' => 30,
                'manage_stock' => 1,
            ],
            [
                'name' => 'Wireless Earbuds',
                'slug' => 'wireless-earbuds',
                'description' => 'Crystal clear audio with active noise cancellation. 24-hour battery life with charging case.',
                'short_description' => 'Premium wireless earbuds with ANC.',
                'product_type' => 'physical',
                'status' => 'publish',
                'price' => 149.99,
                'sale_price' => null,
                'sku' => 'EAR-001',
                'stock_quantity' => 100,
                'manage_stock' => 1,
            ],
            [
                'name' => 'Digital Course: Photography Basics',
                'slug' => 'photography-basics-course',
                'description' => '10 video lessons covering camera settings, composition, and lighting fundamentals.',
                'short_description' => 'Learn photography from scratch.',
                'product_type' => 'digital',
                'status' => 'publish',
                'price' => 49.99,
                'sale_price' => null,
                'sku' => 'CRS-001',
                'downloadable' => 1,
                'virtual' => 1,
            ],
            [
                'name' => 'Ceramic Coffee Mug',
                'slug' => 'ceramic-coffee-mug',
                'description' => 'Handcrafted ceramic mug with a comfortable handle. Dishwasher and microwave safe.',
                'short_description' => 'Handcrafted 12oz mug.',
                'product_type' => 'physical',
                'status' => 'publish',
                'price' => 24.99,
                'sale_price' => 19.99,
                'sku' => 'MUG-001',
                'stock_quantity' => 75,
                'manage_stock' => 1,
            ],
            [
                'name' => 'Leather Wallet',
                'slug' => 'leather-wallet',
                'description' => 'Genuine full-grain leather wallet with RFID blocking. Slim design fits comfortably in any pocket.',
                'short_description' => 'Slim RFID-blocking wallet.',
                'product_type' => 'physical',
                'status' => 'publish',
                'price' => 59.99,
                'sale_price' => null,
                'sku' => 'WAL-001',
                'stock_quantity' => 40,
                'manage_stock' => 1,
            ],
        ];

        $count = 0;
        foreach ($demoProducts as $product) {
            // Check if already exists
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table WHERE slug = %s",
                $product['slug']
            ));

            if (!$exists) {
                $wpdb->insert($table, $product);
                $count++;
            }
        }

        return new \WP_REST_Response([
            'success' => true,
            'data' => ['count' => $count],
        ]);
    }

    // PayPal endpoints
    public function createPayPalOrder(\WP_REST_Request $request): \WP_REST_Response
    {
        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();

        if (empty($cart['items'])) {
            return new \WP_REST_Response(['error' => 'Cart is empty'], 400);
        }

        $clientId = get_option('cartt_paypal_client_id');
        $secret = get_option('cartt_paypal_secret');
        $sandbox = get_option('cartt_paypal_sandbox', true);

        if (!$clientId || !$secret) {
            return new \WP_REST_Response(['error' => 'PayPal not configured'], 400);
        }

        $baseUrl = $sandbox 
            ? 'https://api-m.sandbox.paypal.com' 
            : 'https://api-m.paypal.com';

        // Get access token
        $authResponse = wp_remote_post($baseUrl . '/v1/oauth2/token', [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($clientId . ':' . $secret),
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => 'grant_type=client_credentials',
        ]);

        if (is_wp_error($authResponse)) {
            return new \WP_REST_Response(['error' => 'PayPal auth failed'], 500);
        }

        $authBody = json_decode(wp_remote_retrieve_body($authResponse), true);
        $accessToken = $authBody['access_token'] ?? null;

        if (!$accessToken) {
            return new \WP_REST_Response(['error' => 'PayPal auth failed'], 500);
        }

        // Create order
        $currency = strtoupper(get_option('cartt_currency', 'USD'));
        $total = number_format($cart['totals']['total'], 2, '.', '');

        $orderData = [
            'intent' => 'CAPTURE',
            'purchase_units' => [[
                'amount' => [
                    'currency_code' => $currency,
                    'value' => $total,
                ],
            ]],
        ];

        $orderResponse = wp_remote_post($baseUrl . '/v2/checkout/orders', [
            'headers' => [
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ],
            'body' => json_encode($orderData),
        ]);

        if (is_wp_error($orderResponse)) {
            return new \WP_REST_Response(['error' => 'Failed to create PayPal order'], 500);
        }

        $orderBody = json_decode(wp_remote_retrieve_body($orderResponse), true);

        if (empty($orderBody['id'])) {
            return new \WP_REST_Response(['error' => $orderBody['message'] ?? 'PayPal error'], 500);
        }

        return new \WP_REST_Response([
            'success' => true,
            'order_id' => $orderBody['id'],
        ]);
    }

    public function capturePayPalOrder(\WP_REST_Request $request): \WP_REST_Response
    {
        $paypalOrderId = $request->get_param('paypal_order_id');
        $billing = $request->get_param('billing') ?: [];

        if (!$paypalOrderId) {
            return new \WP_REST_Response(['error' => 'Missing PayPal order ID'], 400);
        }

        $clientId = get_option('cartt_paypal_client_id');
        $secret = get_option('cartt_paypal_secret');
        $sandbox = get_option('cartt_paypal_sandbox', true);

        $baseUrl = $sandbox 
            ? 'https://api-m.sandbox.paypal.com' 
            : 'https://api-m.paypal.com';

        // Get access token
        $authResponse = wp_remote_post($baseUrl . '/v1/oauth2/token', [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($clientId . ':' . $secret),
                'Content-Type' => 'application/x-www-form-urlencoded',
            ],
            'body' => 'grant_type=client_credentials',
        ]);

        $authBody = json_decode(wp_remote_retrieve_body($authResponse), true);
        $accessToken = $authBody['access_token'] ?? null;

        if (!$accessToken) {
            return new \WP_REST_Response(['error' => 'PayPal auth failed'], 500);
        }

        // Capture payment
        $captureResponse = wp_remote_post($baseUrl . '/v2/checkout/orders/' . $paypalOrderId . '/capture', [
            'headers' => [
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
            ],
            'body' => '{}',
        ]);

        $captureBody = json_decode(wp_remote_retrieve_body($captureResponse), true);

        if (($captureBody['status'] ?? '') !== 'COMPLETED') {
            return new \WP_REST_Response(['error' => 'Payment not completed'], 400);
        }

        // Create order in our system
        $cartService = new \Cartt\Services\CartService();
        $checkoutService = new \Cartt\Services\CheckoutService($cartService);

        $result = $checkoutService->processCheckout([
            'billing' => $billing,
            'payment_method' => 'paypal',
            'paypal_order_id' => $paypalOrderId,
            'skip_payment' => true, // Payment already captured
        ]);

        if (!$result['success']) {
            return new \WP_REST_Response(['error' => $result['error']], 400);
        }

        // Update order with PayPal transaction ID
        global $wpdb;
        $captureId = $captureBody['purchase_units'][0]['payments']['captures'][0]['id'] ?? $paypalOrderId;
        $wpdb->update(
            $wpdb->prefix . 'cartt_orders',
            [
                'transaction_id' => $captureId,
                'payment_status' => 'paid',
                'status' => 'processing',
            ],
            ['id' => $result['order_id']]
        );

        return new \WP_REST_Response([
            'success' => true,
            'order_id' => $result['order_id'],
            'redirect' => $result['redirect'],
        ]);
    }

    // Permission callback for logged-in users
    public function isLoggedIn(): bool
    {
        return is_user_logged_in();
    }

    // Gift Card methods
    public function applyGiftCard(\WP_REST_Request $request): \WP_REST_Response
    {
        $code = strtoupper(sanitize_text_field($request->get_param('code')));

        if (empty($code)) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Gift card code required'], 400);
        }

        // Validate gift card
        $result = \Cartt\Services\GiftCardService::checkBalance($code);
        
        if (!$result['found']) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Invalid gift card code'], 400);
        }

        if ($result['balance'] <= 0) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Gift card has no remaining balance'], 400);
        }

        if ($result['expired']) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Gift card has expired'], 400);
        }

        // Apply to cart session
        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();
        
        // Calculate how much of the gift card to use (can't exceed cart total)
        $amountToUse = min($result['balance'], $cart['totals']['total']);
        
        // Store gift card info in session
        if (!isset($_SESSION)) {
            session_start();
        }
        $_SESSION['cartt_gift_card'] = [
            'id' => $result['id'],
            'code' => $code,
            'balance' => $result['balance'],
            'amount_to_use' => $amountToUse,
        ];

        // Recalculate cart with gift card
        $cart['gift_card'] = ['code' => $code, 'id' => $result['id']];
        $cart['totals']['gift_card_amount'] = $amountToUse;
        $cart['totals']['total'] = max(0, $cart['totals']['total'] - $amountToUse);

        return new \WP_REST_Response([
            'success' => true,
            'cart' => $cart,
        ]);
    }

    public function removeGiftCard(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!isset($_SESSION)) {
            session_start();
        }
        unset($_SESSION['cartt_gift_card']);

        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();
        
        // Remove gift card from cart
        unset($cart['gift_card']);
        $cart['totals']['gift_card_amount'] = 0;

        return new \WP_REST_Response([
            'success' => true,
            'cart' => $cart,
        ]);
    }

    // Loyalty Points methods
    public function applyLoyaltyPoints(\WP_REST_Request $request): \WP_REST_Response
    {
        $points = intval($request->get_param('points'));

        if ($points < 1) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Invalid points amount'], 400);
        }

        // Get customer ID
        global $wpdb;
        $customerId = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$wpdb->prefix}cartt_customers WHERE user_id = %d",
            get_current_user_id()
        ));

        if (!$customerId) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Customer not found'], 400);
        }

        // Check customer has enough points
        $customerPoints = \Cartt\Services\LoyaltyService::getCustomerPoints($customerId);
        
        if ($customerPoints->points < $points) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Insufficient points'], 400);
        }

        // Check minimum redemption
        $minRedeem = intval(get_option('cartt_loyalty_min_redemption', 100));
        if ($points < $minRedeem) {
            return new \WP_REST_Response(['success' => false, 'error' => 'Minimum ' . $minRedeem . ' points required'], 400);
        }

        // Calculate discount value
        $pointsValue = floatval(get_option('cartt_loyalty_points_value', 0.01));
        $discount = $points * $pointsValue;

        // Get cart and check discount doesn't exceed total
        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();
        
        $maxDiscount = $cart['totals']['total'] - ($cart['totals']['gift_card_amount'] ?? 0);
        $discount = min($discount, $maxDiscount);
        $actualPoints = ceil($discount / $pointsValue);

        // Store in session
        if (!isset($_SESSION)) {
            session_start();
        }
        $_SESSION['cartt_loyalty'] = [
            'customer_id' => $customerId,
            'points' => $actualPoints,
            'discount' => $discount,
        ];

        // Update cart
        $cart['loyalty_points'] = $actualPoints;
        $cart['totals']['loyalty_discount'] = $discount;
        $cart['totals']['total'] = max(0, $cart['totals']['total'] - $discount);

        return new \WP_REST_Response([
            'success' => true,
            'cart' => $cart,
            'remaining_points' => $customerPoints->points - $actualPoints,
        ]);
    }

    public function removeLoyaltyPoints(\WP_REST_Request $request): \WP_REST_Response
    {
        if (!isset($_SESSION)) {
            session_start();
        }
        
        $loyaltyData = $_SESSION['cartt_loyalty'] ?? null;
        unset($_SESSION['cartt_loyalty']);

        $cartService = new \Cartt\Services\CartService();
        $cart = $cartService->getCart();
        
        // Remove loyalty from cart
        unset($cart['loyalty_points']);
        $cart['totals']['loyalty_discount'] = 0;

        // Get remaining points
        $remainingPoints = 0;
        if ($loyaltyData) {
            $customerPoints = \Cartt\Services\LoyaltyService::getCustomerPoints($loyaltyData['customer_id']);
            $remainingPoints = $customerPoints->points;
        }

        return new \WP_REST_Response([
            'success' => true,
            'cart' => $cart,
            'remaining_points' => $remainingPoints,
        ]);
    }
}
