<?php

declare(strict_types=1);

namespace Cartt\Admin;

/**
 * Admin Controller - orchestrates the WordPress admin interface
 */
class AdminController
{
    public function __construct()
    {
        add_action('admin_menu', [$this, 'registerMenus']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueAssets']);
        
        // AJAX handlers for variations
        add_action('wp_ajax_cartt_generate_variations', [$this, 'ajaxGenerateVariations']);
        add_action('wp_ajax_cartt_save_variations', [$this, 'ajaxSaveVariations']);
        add_action('wp_ajax_cartt_delete_variation', [$this, 'ajaxDeleteVariation']);
        
        // AJAX handlers for order management
        add_action('wp_ajax_cartt_add_order_note', [$this, 'ajaxAddOrderNote']);
        add_action('wp_ajax_cartt_delete_order_note', [$this, 'ajaxDeleteOrderNote']);
        add_action('wp_ajax_cartt_save_tracking', [$this, 'ajaxSaveTracking']);
        add_action('wp_ajax_cartt_send_tracking_email', [$this, 'ajaxSendTrackingEmail']);
        add_action('wp_ajax_cartt_process_refund', [$this, 'ajaxProcessRefund']);
        
        // AJAX handlers for currency management
        add_action('wp_ajax_cartt_save_currency', [$this, 'ajaxSaveCurrency']);
        add_action('wp_ajax_cartt_delete_currency', [$this, 'ajaxDeleteCurrency']);
        add_action('wp_ajax_cartt_update_exchange_rates', [$this, 'ajaxUpdateExchangeRates']);
        add_action('wp_ajax_cartt_save_option', [$this, 'ajaxSaveOption']);
        
        // v1.3.0 AJAX handlers
        add_action('wp_ajax_cartt_create_gift_card', [$this, 'ajaxCreateGiftCard']);
        add_action('wp_ajax_cartt_send_gift_card', [$this, 'ajaxSendGiftCard']);
        add_action('wp_ajax_cartt_add_tracking', [$this, 'ajaxAddTracking']);
        add_action('wp_ajax_cartt_update_tracking_status', [$this, 'ajaxUpdateTrackingStatus']);
        add_action('wp_ajax_cartt_delete_tracking', [$this, 'ajaxDeleteTracking']);
        add_action('wp_ajax_cartt_update_stock', [$this, 'ajaxUpdateStock']);
        add_action('wp_ajax_cartt_export_inventory', [$this, 'ajaxExportInventory']);
        add_action('wp_ajax_cartt_update_rma_status', [$this, 'ajaxUpdateRMAStatus']);
        add_action('wp_ajax_cartt_approve_qa', [$this, 'ajaxApproveQA']);
        add_action('wp_ajax_cartt_delete_qa', [$this, 'ajaxDeleteQA']);
        add_action('wp_ajax_cartt_print_invoice', [$this, 'ajaxPrintInvoice']);
        add_action('wp_ajax_cartt_print_packing_slip', [$this, 'ajaxPrintPackingSlip']);

        // v1.4.0 AJAX handlers - Vendors
        add_action('wp_ajax_cartt_approve_vendor', [$this, 'ajaxApproveVendor']);
        add_action('wp_ajax_cartt_reject_vendor', [$this, 'ajaxRejectVendor']);
        add_action('wp_ajax_cartt_suspend_vendor', [$this, 'ajaxSuspendVendor']);
        add_action('wp_ajax_cartt_verify_vendor', [$this, 'ajaxVerifyVendor']);
        
        // v1.4.0 AJAX handlers - Commissions
        add_action('wp_ajax_cartt_process_payout', [$this, 'ajaxProcessPayout']);
        add_action('wp_ajax_cartt_reject_payout', [$this, 'ajaxRejectPayout']);
        
        // v1.4.0 AJAX handlers - Wholesale
        add_action('wp_ajax_cartt_approve_wholesale', [$this, 'ajaxApproveWholesale']);
        add_action('wp_ajax_cartt_reject_wholesale', [$this, 'ajaxRejectWholesale']);
        add_action('wp_ajax_cartt_update_quote_status', [$this, 'ajaxUpdateQuoteStatus']);
        
        // v1.4.0 AJAX handlers - Dropshipping
        add_action('wp_ajax_cartt_save_supplier', [$this, 'ajaxSaveSupplier']);
        add_action('wp_ajax_cartt_delete_supplier', [$this, 'ajaxDeleteSupplier']);
        add_action('wp_ajax_cartt_sync_supplier', [$this, 'ajaxSyncSupplier']);

        // Social Proof AJAX
        add_action('wp_ajax_cartt_save_social_proof_settings', [$this, 'ajaxSaveSocialProofSettings']);
        
        // AI Recommendations AJAX
        add_action('wp_ajax_cartt_save_ai_recommendations', [$this, 'ajaxSaveAIRecommendations']);
        
        // v1.7.0 - UX improvements
        add_action('wp_ajax_cartt_bulk_products', [$this, 'ajaxBulkProducts']);
        add_action('wp_ajax_cartt_inline_edit_product', [$this, 'ajaxInlineEditProduct']);
        add_action('wp_ajax_cartt_update_product_status', [$this, 'ajaxUpdateProductStatus']);
        add_action('wp_ajax_cartt_global_search', [$this, 'ajaxGlobalSearch']);
        add_action('wp_ajax_cartt_bulk_orders', [$this, 'ajaxBulkOrders']);
        add_action('wp_ajax_cartt_reorder_categories', [$this, 'ajaxReorderCategories']);
        add_action('wp_ajax_cartt_inline_update', [$this, 'ajaxInlineUpdate']);
        
        // Card Designer AJAX
        add_action('wp_ajax_cartt_save_card_design', [$this, 'ajaxSaveCardDesign']);
        
        // Add-ons AJAX
        add_action('wp_ajax_cartt_activate_addon', [$this, 'ajaxActivateAddon']);
        add_action('wp_ajax_cartt_deactivate_addon', [$this, 'ajaxDeactivateAddon']);
        add_action('wp_ajax_cartt_create_addon_checkout', [$this, 'ajaxCreateAddonCheckout']);
    }

    public function registerMenus(): void
    {
        $store_type = get_option('cartt_store_type', 'mixed');
        $addons = \Cartt\Core\Plugin::getActiveAddons();

        // Main menu (is the dashboard)
        add_menu_page(
            __('Cartt', 'cartt'),
            __('Cartt', 'cartt'),
            'manage_options',
            'cartt',
            [$this, 'renderDashboard'],
            'dashicons-cart',
            55
        );

        // Rename first submenu to Dashboard
        add_submenu_page(
            'cartt',
            __('Dashboard', 'cartt'),
            __('Dashboard', 'cartt'),
            'manage_options',
            'cartt',
            [$this, 'renderDashboard']
        );

        // Orders
        add_submenu_page(
            'cartt',
            __('Orders', 'cartt'),
            __('Orders', 'cartt'),
            'manage_options',
            'cartt-orders',
            [$this, 'renderOrders']
        );

        // Products (includes Categories, Inventory, Q&A)
        $pendingQA = \Cartt\Services\ProductQAService::getPendingCount();
        $productsLabel = __('Products', 'cartt');
        if ($pendingQA > 0) {
            $productsLabel .= ' <span class="awaiting-mod">' . $pendingQA . '</span>';
        }
        add_submenu_page(
            'cartt',
            __('Products', 'cartt'),
            $productsLabel,
            'manage_options',
            'cartt-products',
            [$this, 'renderProducts']
        );

        // Customers
        add_submenu_page(
            'cartt',
            __('Customers', 'cartt'),
            __('Customers', 'cartt'),
            'manage_options',
            'cartt-customers',
            [$this, 'renderCustomers']
        );

        // Checkout
        add_submenu_page(
            'cartt',
            __('Checkout', 'cartt'),
            __('Checkout', 'cartt'),
            'manage_options',
            'cartt-checkout',
            [$this, 'renderCheckout']
        );

        // Shipping (includes Returns, Live Rates & Tracking)
        $pendingRMA = \Cartt\Services\RMAService::getPendingCount();
        $shippingLabel = __('Shipping', 'cartt');
        if ($pendingRMA > 0) {
            $shippingLabel .= ' <span class="awaiting-mod">' . $pendingRMA . '</span>';
        }
        add_submenu_page(
            'cartt',
            __('Shipping', 'cartt'),
            $shippingLabel,
            'manage_options',
            'cartt-shipping',
            [$this, 'renderShipping']
        );

        // Design (Card Designer)
        add_submenu_page(
            'cartt',
            __('Design', 'cartt'),
            __('Design', 'cartt'),
            'manage_options',
            'cartt-design',
            [$this, 'renderDesign']
        );

        // Marketing (Coupons, Gift Cards, Loyalty, SEO, Emails)
        add_submenu_page(
            'cartt',
            __('Marketing', 'cartt'),
            __('Marketing', 'cartt'),
            'manage_options',
            'cartt-marketing',
            [$this, 'renderMarketing']
        );

        // Finance (Tax, Currency, Reports, Analytics)
        add_submenu_page(
            'cartt',
            __('Finance', 'cartt'),
            __('Finance', 'cartt'),
            'manage_options',
            'cartt-finance',
            [$this, 'renderFinance']
        );

        // Conditional menus based on addons
        if (in_array('bookings', $addons)) {
            add_submenu_page(
                'cartt',
                __('Bookings', 'cartt'),
                __('Bookings', 'cartt'),
                'manage_options',
                'cartt-bookings',
                [$this, 'renderBookings']
            );
        }

        if (in_array('subscriptions', $addons)) {
            add_submenu_page(
                'cartt',
                __('Subscriptions', 'cartt'),
                __('Subscriptions', 'cartt'),
                'manage_options',
                'cartt-subscriptions',
                [$this, 'renderSubscriptions']
            );
        }

        // =====================================================================
        // PRO Features (only show if addon is active)
        // =====================================================================

        // Marketplace (Vendors, Commissions, Wholesale, Dropshipping)
        if (in_array('marketplace', $addons)) {
            $pendingVendors = \Cartt\Services\VendorService::getPendingCount();
            $pendingPayouts = \Cartt\Services\VendorCommissionService::getPendingPayoutsCount();
            $pendingWholesale = \Cartt\Services\WholesaleService::getPendingApplicationsCount();
            $marketplaceBadge = $pendingVendors + $pendingPayouts + $pendingWholesale;
            $marketplaceLabel = __('Marketplace', 'cartt');
            if ($marketplaceBadge > 0) {
                $marketplaceLabel .= ' <span class="awaiting-mod">' . $marketplaceBadge . '</span>';
            }
            add_submenu_page(
                'cartt',
                __('Marketplace', 'cartt'),
                $marketplaceLabel,
                'manage_options',
                'cartt-marketplace',
                [$this, 'renderMarketplace']
            );
        }

        // AI & Conversion (AI Recommendations, Social Proof)
        if (in_array('ai_conversion', $addons)) {
            add_submenu_page(
                'cartt',
                __('AI & Conversion', 'cartt'),
                __('AI & Conversion', 'cartt'),
                'manage_options',
                'cartt-ai-conversion',
                [$this, 'renderAIConversion']
            );
        }

        // Add-ons (always show)
        add_submenu_page(
            'cartt',
            __('Add-ons', 'cartt'),
            __('Add-ons', 'cartt'),
            'manage_options',
            'cartt-addons',
            [$this, 'renderAddons']
        );

        // Settings (always last)
        add_submenu_page(
            'cartt',
            __('Settings', 'cartt'),
            __('Settings', 'cartt'),
            'manage_options',
            'cartt-settings',
            [$this, 'renderSettings']
        );
    }

    public function enqueueAssets(string $hook): void
    {
        // Only load on our pages
        if (strpos($hook, 'cartt') === false) {
            return;
        }

        // Enqueue WordPress media uploader on products page
        if (strpos($hook, 'cartt-products') !== false) {
            wp_enqueue_media();
        }

        // Material Icons
        wp_enqueue_style(
            'material-icons',
            'https://fonts.googleapis.com/icon?family=Material+Icons+Outlined',
            [],
            null
        );

        wp_enqueue_style(
            'cartt-admin',
            CARTT_PLUGIN_URL . 'assets/css/admin.css',
            ['material-icons'],
            CARTT_VERSION
        );

        wp_enqueue_script(
            'cartt-admin',
            CARTT_PLUGIN_URL . 'assets/js/admin.js',
            ['jquery', 'wp-api-fetch', 'jquery-ui-sortable'],
            CARTT_VERSION,
            true
        );

        wp_localize_script('cartt-admin', 'carttAdmin', [
            'apiUrl' => rest_url('cartt/v1/'),
            'adminUrl' => admin_url('admin.php'),
            'nonce' => wp_create_nonce('wp_rest'),
            'adminNonce' => wp_create_nonce('cartt_admin'),
            'currency' => get_option('cartt_currency', 'USD'),
            'storeType' => get_option('cartt_store_type', 'mixed'),
        ]);
    }

    public function renderDashboard(): void
    {
        $this->renderPage('dashboard');
    }

    public function renderOrders(): void
    {
        $this->renderPage('orders');
    }

    public function renderProducts(): void
    {
        $this->renderPage('products');
    }

    public function renderDesign(): void
    {
        $this->renderPage('card-designer');
    }

    public function renderCustomers(): void
    {
        $this->renderPage('customers');
    }

    public function renderCheckout(): void
    {
        $this->renderPage('checkout');
    }

    public function renderCategories(): void
    {
        $this->renderPage('categories');
    }

    public function renderReviews(): void
    {
        $this->renderPage('reviews');
    }

    public function renderCoupons(): void
    {
        $this->renderPage('coupons');
    }

    public function renderShipping(): void
    {
        $this->renderPage('shipping');
    }

    public function renderTax(): void
    {
        $this->renderPage('tax');
    }

    public function renderCurrency(): void
    {
        $this->renderPage('currency');
    }

    public function renderEmails(): void
    {
        $this->renderPage('emails');
    }

    public function renderReports(): void
    {
        $this->renderPage('reports');
    }

    public function renderBookings(): void
    {
        $this->renderPage('bookings');
    }

    public function renderSubscriptions(): void
    {
        $this->renderPage('subscriptions');
    }

    public function renderSettings(): void
    {
        $this->renderPage('settings');
    }

    public function renderGiftCards(): void
    {
        $this->renderPage('gift-cards');
    }

    public function renderLoyalty(): void
    {
        $this->renderPage('loyalty');
    }

    public function renderInventory(): void
    {
        $this->renderPage('inventory');
    }

    public function renderTracking(): void
    {
        $this->renderPage('tracking');
    }

    public function renderRMA(): void
    {
        $this->renderPage('rma');
    }

    public function renderQA(): void
    {
        $this->renderPage('qa');
    }

    public function renderSEO(): void
    {
        $this->renderPage('seo');
    }

    // =====================================================================
    // v1.4.0 Render Methods
    // =====================================================================

    public function renderVendors(): void
    {
        $this->renderPage('vendors');
    }

    public function renderCommissions(): void
    {
        $this->renderPage('commissions');
    }

    public function renderWholesale(): void
    {
        $this->renderPage('wholesale');
    }

    public function renderAnalytics(): void
    {
        $this->renderPage('analytics');
    }

    public function renderDropshipping(): void
    {
        $this->renderPage('dropshipping');
    }

    public function renderAIRecommendations(): void
    {
        $this->renderPage('ai-recommendations');
    }

    public function renderSocialProof(): void
    {
        $this->renderPage('social-proof');
    }

    public function renderMarketing(): void
    {
        $this->renderPage('marketing');
    }

    public function renderFinance(): void
    {
        $this->renderPage('finance');
    }

    public function renderMarketplace(): void
    {
        $this->renderPage('marketplace');
    }

    public function renderAIConversion(): void
    {
        $this->renderPage('ai-conversion');
    }

    public function renderAddons(): void
    {
        $this->renderPage('addons');
    }

    private function renderPage(string $page): void
    {
        $template = CARTT_PLUGIN_DIR . "admin/views/{$page}.php";
        
        if (file_exists($template)) {
            include $template;
        } else {
            // Fallback: React app mount point
            echo '<div class="wrap"><div id="cartt-admin-app" data-page="' . esc_attr($page) . '"></div></div>';
        }
    }

    public function ajaxGenerateVariations(): void
    {
        check_ajax_referer('cartt_variations');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $productId = intval($_POST['product_id'] ?? 0);
        $attributes = json_decode(stripslashes($_POST['attributes'] ?? '{}'), true);
        $basePrice = floatval($_POST['base_price'] ?? 0);

        if (!$productId || empty($attributes)) {
            wp_send_json_error('Invalid data');
        }

        $variationService = new \Cartt\Services\VariationService();
        
        // Delete existing variations first
        $variationService->deleteProductVariations($productId);
        
        // Generate new variations
        $count = $variationService->generateVariations($productId, $attributes, $basePrice);

        wp_send_json_success(['count' => $count]);
    }

    public function ajaxSaveVariations(): void
    {
        check_ajax_referer('cartt_variations');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $variations = json_decode(stripslashes($_POST['variations'] ?? '[]'), true);

        if (empty($variations)) {
            wp_send_json_error('No variations to save');
        }

        $variationService = new \Cartt\Services\VariationService();

        foreach ($variations as $v) {
            $variationService->updateVariation((int) $v['id'], [
                'sku' => $v['sku'] ?? '',
                'price' => !empty($v['price']) ? (float) $v['price'] : null,
                'stock_quantity' => isset($v['stock_quantity']) && $v['stock_quantity'] !== '' ? (int) $v['stock_quantity'] : null,
            ]);
        }

        wp_send_json_success();
    }

    public function ajaxDeleteVariation(): void
    {
        check_ajax_referer('cartt_variations');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $variationId = intval($_POST['variation_id'] ?? 0);

        if (!$variationId) {
            wp_send_json_error('Invalid variation');
        }

        $variationService = new \Cartt\Services\VariationService();
        $variationService->deleteVariation($variationId);

        wp_send_json_success();
    }

    public function ajaxAddOrderNote(): void
    {
        check_ajax_referer('cartt_order_actions');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $orderId = intval($_POST['order_id'] ?? 0);
        $note = sanitize_textarea_field($_POST['note'] ?? '');
        $isCustomerNote = !empty($_POST['is_customer_note']);

        if (!$orderId || !$note) {
            wp_send_json_error('Missing required fields');
        }

        $noteId = \Cartt\Services\OrderNoteService::addNote($orderId, $note, $isCustomerNote, $isCustomerNote);

        wp_send_json_success(['note_id' => $noteId]);
    }

    public function ajaxDeleteOrderNote(): void
    {
        check_ajax_referer('cartt_order_actions');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $noteId = intval($_POST['note_id'] ?? 0);

        if (!$noteId) {
            wp_send_json_error('Invalid note');
        }

        \Cartt\Services\OrderNoteService::deleteNote($noteId);
        wp_send_json_success();
    }

    public function ajaxSaveTracking(): void
    {
        check_ajax_referer('cartt_order_actions');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        global $wpdb;
        
        $orderId = intval($_POST['order_id'] ?? 0);
        $trackingNumber = sanitize_text_field($_POST['tracking_number'] ?? '');
        $trackingUrl = esc_url_raw($_POST['tracking_url'] ?? '');

        if (!$orderId) {
            wp_send_json_error('Invalid order');
        }

        // Get existing meta
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT meta FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));

        $meta = json_decode($order->meta ?? '{}', true) ?: [];
        $meta['tracking_number'] = $trackingNumber;
        $meta['tracking_url'] = $trackingUrl;

        $wpdb->update(
            $wpdb->prefix . 'cartt_orders',
            ['meta' => json_encode($meta)],
            ['id' => $orderId]
        );

        // Add note
        if ($trackingNumber) {
            \Cartt\Services\OrderNoteService::addSystemNote(
                $orderId,
                sprintf(__('Tracking number added: %s', 'cartt'), $trackingNumber)
            );
        }

        wp_send_json_success();
    }

    public function ajaxSendTrackingEmail(): void
    {
        check_ajax_referer('cartt_order_actions');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        global $wpdb;
        
        $orderId = intval($_POST['order_id'] ?? 0);

        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));

        if (!$order) {
            wp_send_json_error('Order not found');
        }

        $meta = json_decode($order->meta ?? '{}', true);
        $trackingNumber = $meta['tracking_number'] ?? '';
        $trackingUrl = $meta['tracking_url'] ?? '';

        if (!$trackingNumber) {
            wp_send_json_error('No tracking number set');
        }

        $billing = json_decode($order->billing_address, true) ?: [];
        $customerName = trim(($billing['first_name'] ?? '') . ' ' . ($billing['last_name'] ?? ''));

        // Send email using EmailService
        $result = \Cartt\Services\EmailService::send('order_shipped', $order->customer_email, [
            'order_number' => $order->order_number,
            'customer_name' => $customerName,
            'customer_first_name' => $billing['first_name'] ?? '',
            'tracking_number' => $trackingNumber,
            'tracking_url' => $trackingUrl,
            'store_name' => get_bloginfo('name'),
            'store_url' => home_url(),
        ]);

        if ($result) {
            \Cartt\Services\OrderNoteService::addSystemNote(
                $orderId,
                __('Shipping notification email sent to customer', 'cartt')
            );
            wp_send_json_success();
        } else {
            wp_send_json_error('Failed to send email');
        }
    }

    public function ajaxProcessRefund(): void
    {
        check_ajax_referer('cartt_order_actions');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $orderId = intval($_POST['order_id'] ?? 0);
        $amount = floatval($_POST['amount'] ?? 0);
        $reason = sanitize_text_field($_POST['reason'] ?? '');
        $restoreStock = !empty($_POST['restore_stock']);

        if (!$orderId || $amount <= 0) {
            wp_send_json_error('Invalid order or amount');
        }

        global $wpdb;
        $order = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_orders WHERE id = %d",
            $orderId
        ));

        if (!$order) {
            wp_send_json_error('Order not found');
        }

        // Check refundable amount
        $totalRefunded = \Cartt\Services\RefundService::getOrderTotalRefunded($orderId);
        $refundable = $order->total - $totalRefunded;

        if ($amount > $refundable) {
            wp_send_json_error(sprintf(__('Amount exceeds refundable amount ($%s)', 'cartt'), number_format($refundable, 2)));
        }

        // Get line items if restoring stock
        $lineItems = [];
        if ($restoreStock) {
            $items = $wpdb->get_results($wpdb->prepare(
                "SELECT product_id, quantity FROM {$wpdb->prefix}cartt_order_items WHERE order_id = %d",
                $orderId
            ));
            foreach ($items as $item) {
                $lineItems[] = [
                    'product_id' => $item->product_id,
                    'quantity' => $item->quantity,
                ];
            }
        }

        // Try Stripe refund if payment was via Stripe
        if ($order->payment_method === 'stripe' && !empty($order->payment_intent_id)) {
            $result = \Cartt\Services\RefundService::processStripeRefund($orderId, $amount, $reason);
            if (!$result['success']) {
                wp_send_json_error($result['error']);
            }
        } else {
            // Manual refund (no payment gateway)
            \Cartt\Services\RefundService::createRefund([
                'order_id' => $orderId,
                'amount' => $amount,
                'reason' => $reason,
                'restore_stock' => $restoreStock,
                'line_items' => $lineItems,
            ]);
        }

        wp_send_json_success();
    }

    public function ajaxSaveCurrency(): void
    {
        check_ajax_referer('cartt_currency', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $data = [
            'code' => sanitize_text_field($_POST['code'] ?? ''),
            'name' => sanitize_text_field($_POST['name'] ?? ''),
            'symbol' => sanitize_text_field($_POST['symbol'] ?? ''),
            'exchange_rate' => floatval($_POST['exchange_rate'] ?? 1),
            'decimals' => intval($_POST['decimals'] ?? 2),
            'symbol_position' => sanitize_text_field($_POST['symbol_position'] ?? 'left'),
            'thousand_separator' => sanitize_text_field($_POST['thousand_separator'] ?? ','),
            'decimal_separator' => sanitize_text_field($_POST['decimal_separator'] ?? '.'),
            'enabled' => !empty($_POST['enabled']),
            'is_default' => !empty($_POST['is_default']),
            'auto_update' => !empty($_POST['auto_update']),
            'sort_order' => intval($_POST['sort_order'] ?? 0),
        ];

        if (empty($data['code']) || strlen($data['code']) !== 3) {
            wp_send_json_error(['message' => 'Invalid currency code']);
        }

        $id = \Cartt\Services\CurrencyService::saveCurrency($data);
        
        wp_send_json_success(['id' => $id]);
    }

    public function ajaxDeleteCurrency(): void
    {
        check_ajax_referer('cartt_currency', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);
        
        if (!$id) {
            wp_send_json_error(['message' => 'Invalid currency ID']);
        }

        $success = \Cartt\Services\CurrencyService::deleteCurrency($id);
        
        if (!$success) {
            wp_send_json_error(['message' => 'Cannot delete the default currency']);
        }

        wp_send_json_success();
    }

    public function ajaxUpdateExchangeRates(): void
    {
        check_ajax_referer('cartt_currency', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $success = \Cartt\Services\CurrencyService::updateExchangeRates();
        
        if (!$success) {
            wp_send_json_error(['message' => 'Failed to update rates. Check your API key.']);
        }

        wp_send_json_success(['message' => 'Exchange rates updated successfully']);
    }

    public function ajaxSaveOption(): void
    {
        check_ajax_referer('cartt_currency', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $option = sanitize_text_field($_POST['option'] ?? '');
        $value = sanitize_text_field($_POST['value'] ?? '');
        
        // Only allow specific options
        $allowedOptions = [
            'cartt_exchange_rate_api',
            'cartt_auto_detect_currency',
        ];
        
        if (!in_array($option, $allowedOptions)) {
            wp_send_json_error(['message' => 'Invalid option']);
        }

        update_option($option, $value);
        
        wp_send_json_success();
    }

    // v1.3.0 AJAX Handlers

    public function ajaxCreateGiftCard(): void
    {
        check_ajax_referer('cartt_gift_cards', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $card = \Cartt\Services\GiftCardService::create([
            'amount' => floatval($_POST['amount'] ?? 0),
            'recipient_email' => sanitize_email($_POST['recipient_email'] ?? ''),
            'recipient_name' => sanitize_text_field($_POST['recipient_name'] ?? ''),
            'sender_name' => sanitize_text_field($_POST['sender_name'] ?? ''),
            'message' => sanitize_textarea_field($_POST['message'] ?? ''),
            'design' => sanitize_text_field($_POST['design'] ?? 'default'),
            'expires_days' => intval($_POST['expires_days'] ?? 0),
        ]);

        if (!$card) {
            wp_send_json_error(['message' => 'Failed to create gift card']);
        }

        // Send email if requested
        if (!empty($_POST['send_now']) && $card->recipient_email) {
            \Cartt\Services\GiftCardService::sendEmail($card->id);
        }

        wp_send_json_success(['id' => $card->id, 'code' => $card->code]);
    }

    public function ajaxSendGiftCard(): void
    {
        check_ajax_referer('cartt_gift_cards', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);
        
        if (\Cartt\Services\GiftCardService::sendEmail($id)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to send email']);
        }
    }

    public function ajaxAddTracking(): void
    {
        check_ajax_referer('cartt_tracking', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = \Cartt\Services\OrderTrackingService::addTracking([
            'order_id' => intval($_POST['order_id'] ?? 0),
            'carrier' => sanitize_text_field($_POST['carrier'] ?? ''),
            'tracking_number' => sanitize_text_field($_POST['tracking_number'] ?? ''),
            'tracking_url' => esc_url_raw($_POST['tracking_url'] ?? ''),
            'estimated_delivery' => sanitize_text_field($_POST['estimated_delivery'] ?? ''),
            'notify_customer' => !empty($_POST['notify_customer']),
        ]);

        if (!$id) {
            wp_send_json_error(['message' => 'Failed to add tracking']);
        }

        wp_send_json_success(['id' => $id]);
    }

    public function ajaxUpdateTrackingStatus(): void
    {
        check_ajax_referer('cartt_tracking', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);
        $status = sanitize_text_field($_POST['status'] ?? '');

        if (\Cartt\Services\OrderTrackingService::updateStatus($id, $status)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to update status']);
        }
    }

    public function ajaxDeleteTracking(): void
    {
        check_ajax_referer('cartt_tracking', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);

        if (\Cartt\Services\OrderTrackingService::delete($id)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to delete tracking']);
        }
    }

    public function ajaxUpdateStock(): void
    {
        check_ajax_referer('cartt_inventory', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $productId = intval($_POST['product_id'] ?? 0);
        $quantity = intval($_POST['quantity'] ?? 0);

        if (\Cartt\Services\InventoryService::setStock($productId, $quantity)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to update stock']);
        }
    }

    public function ajaxExportInventory(): void
    {
        check_ajax_referer('cartt_inventory', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Permission denied');
        }

        global $wpdb;
        $products = $wpdb->get_results(
            "SELECT id, name, sku, price, stock_quantity, stock_status 
             FROM {$wpdb->prefix}cartt_products 
             WHERE status = 'publish' 
             ORDER BY name"
        );

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="inventory-' . date('Y-m-d') . '.csv"');

        $output = fopen('php://output', 'w');
        fputcsv($output, ['ID', 'Name', 'SKU', 'Price', 'Stock Quantity', 'Stock Status']);

        foreach ($products as $p) {
            fputcsv($output, [
                $p->id,
                $p->name,
                $p->sku,
                $p->price,
                $p->stock_quantity,
                $p->stock_status,
            ]);
        }

        fclose($output);
        exit;
    }

    public function ajaxUpdateRMAStatus(): void
    {
        check_ajax_referer('cartt_rma', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);
        $status = sanitize_text_field($_POST['status'] ?? '');

        if (\Cartt\Services\RMAService::updateStatus($id, $status, get_current_user_id())) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to update status']);
        }
    }

    public function ajaxApproveQA(): void
    {
        check_ajax_referer('cartt_qa', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);

        if (\Cartt\Services\ProductQAService::updateStatus($id, 'approved')) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to approve']);
        }
    }

    public function ajaxDeleteQA(): void
    {
        check_ajax_referer('cartt_qa', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $id = intval($_POST['id'] ?? 0);

        if (\Cartt\Services\ProductQAService::delete($id)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to delete']);
        }
    }

    public function ajaxPrintInvoice(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die('Permission denied');
        }

        $orderId = intval($_GET['order_id'] ?? 0);
        
        if (!$orderId) {
            wp_die('Invalid order');
        }

        $html = \Cartt\Services\InvoiceService::generateInvoice($orderId);
        
        if (!$html) {
            wp_die('Order not found');
        }

        echo $html;
        exit;
    }

    public function ajaxPrintPackingSlip(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die('Permission denied');
        }

        $orderId = intval($_GET['order_id'] ?? 0);
        
        if (!$orderId) {
            wp_die('Invalid order');
        }

        $html = \Cartt\Services\InvoiceService::generatePackingSlip($orderId);
        
        if (!$html) {
            wp_die('Order not found');
        }

        echo $html;
        exit;
    }

    // =====================================================================
    // v1.4.0 AJAX Handlers
    // =====================================================================

    public function ajaxApproveVendor(): void
    {
        check_ajax_referer('cartt_vendors', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $vendorId = intval($_POST['vendor_id'] ?? 0);
        $service = new \Cartt\Services\VendorService();
        
        if ($service->approve_vendor($vendorId)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to approve vendor']);
        }
    }

    public function ajaxRejectVendor(): void
    {
        check_ajax_referer('cartt_vendors', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $vendorId = intval($_POST['vendor_id'] ?? 0);
        $reason = sanitize_textarea_field($_POST['reason'] ?? '');
        $service = new \Cartt\Services\VendorService();
        
        if ($service->reject_vendor($vendorId, $reason)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to reject vendor']);
        }
    }

    public function ajaxSuspendVendor(): void
    {
        check_ajax_referer('cartt_vendors', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $vendorId = intval($_POST['vendor_id'] ?? 0);
        $reason = sanitize_textarea_field($_POST['reason'] ?? '');
        $service = new \Cartt\Services\VendorService();
        
        if ($service->suspend_vendor($vendorId, $reason)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to suspend vendor']);
        }
    }

    public function ajaxVerifyVendor(): void
    {
        check_ajax_referer('cartt_vendors', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $vendorId = intval($_POST['vendor_id'] ?? 0);
        $service = new \Cartt\Services\VendorService();
        
        if ($service->verify_vendor($vendorId)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to verify vendor']);
        }
    }

    public function ajaxProcessPayout(): void
    {
        check_ajax_referer('cartt_commissions', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $payoutId = intval($_POST['payout_id'] ?? 0);
        $transactionId = sanitize_text_field($_POST['transaction_id'] ?? '');
        $service = new \Cartt\Services\VendorCommissionService();
        
        if ($service->process_payout($payoutId, $transactionId)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to process payout']);
        }
    }

    public function ajaxRejectPayout(): void
    {
        check_ajax_referer('cartt_commissions', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $payoutId = intval($_POST['payout_id'] ?? 0);
        $reason = sanitize_textarea_field($_POST['reason'] ?? '');
        $service = new \Cartt\Services\VendorCommissionService();
        
        if ($service->reject_payout($payoutId, $reason)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to reject payout']);
        }
    }

    public function ajaxApproveWholesale(): void
    {
        check_ajax_referer('cartt_wholesale', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $applicationId = intval($_POST['application_id'] ?? 0);
        $service = new \Cartt\Services\WholesaleService();
        
        if ($service->approve_application($applicationId)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to approve application']);
        }
    }

    public function ajaxRejectWholesale(): void
    {
        check_ajax_referer('cartt_wholesale', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $applicationId = intval($_POST['application_id'] ?? 0);
        $reason = sanitize_textarea_field($_POST['reason'] ?? '');
        $service = new \Cartt\Services\WholesaleService();
        
        if ($service->reject_application($applicationId, $reason)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to reject application']);
        }
    }

    public function ajaxUpdateQuoteStatus(): void
    {
        check_ajax_referer('cartt_wholesale', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $quoteId = intval($_POST['quote_id'] ?? 0);
        $status = sanitize_text_field($_POST['status'] ?? '');
        $service = new \Cartt\Services\WholesaleService();
        
        if ($service->update_quote_status($quoteId, $status)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to update quote status']);
        }
    }

    public function ajaxSaveSupplier(): void
    {
        check_ajax_referer('cartt_dropshipping', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $data = [
            'name' => sanitize_text_field($_POST['name'] ?? ''),
            'type' => sanitize_text_field($_POST['type'] ?? 'generic'),
            'api_key' => sanitize_text_field($_POST['api_key'] ?? ''),
            'api_secret' => sanitize_text_field($_POST['api_secret'] ?? ''),
            'store_id' => sanitize_text_field($_POST['store_id'] ?? ''),
        ];

        $service = new \Cartt\Services\DropshippingService();
        $supplierId = intval($_POST['supplier_id'] ?? 0);
        
        if ($supplierId) {
            $result = $service->update_supplier($supplierId, $data);
        } else {
            $result = $service->add_supplier($data);
        }
        
        if ($result) {
            wp_send_json_success(['id' => $result]);
        } else {
            wp_send_json_error(['message' => 'Failed to save supplier']);
        }
    }

    public function ajaxDeleteSupplier(): void
    {
        check_ajax_referer('cartt_dropshipping', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $supplierId = intval($_POST['supplier_id'] ?? 0);
        $service = new \Cartt\Services\DropshippingService();
        
        if ($service->delete_supplier($supplierId)) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to delete supplier']);
        }
    }

    public function ajaxSyncSupplier(): void
    {
        check_ajax_referer('cartt_dropshipping', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $supplierId = intval($_POST['supplier_id'] ?? 0);
        $syncType = sanitize_text_field($_POST['sync_type'] ?? 'inventory');
        $service = new \Cartt\Services\DropshippingService();
        
        $result = false;
        if ($syncType === 'inventory') {
            $result = $service->sync_inventory($supplierId);
        } elseif ($syncType === 'prices') {
            $result = $service->sync_prices($supplierId);
        }
        
        if ($result) {
            wp_send_json_success();
        } else {
            wp_send_json_error(['message' => 'Failed to sync supplier']);
        }
    }

    /**
     * Save social proof settings via AJAX
     */
    public function ajaxSaveSocialProofSettings(): void
    {
        check_ajax_referer('cartt_social_proof', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $settings = get_option('cartt_social_proof_settings', []);
        
        // Parse the serialized form data if present
        if (!empty($_POST['settings'])) {
            if (is_string($_POST['settings'])) {
                parse_str($_POST['settings'], $formData);
            } else {
                $formData = $_POST['settings'];
            }
            
            // Handle global enable toggle
            if (isset($formData['enabled'])) {
                $settings['enabled'] = !empty($formData['enabled']);
            }
            
            // Handle purchase notifications settings
            if (isset($formData['notifications'])) {
                $n = $formData['notifications'];
                $settings['purchase_notifications'] = [
                    'enabled' => !empty($n['enabled']),
                    'position' => sanitize_text_field($n['position'] ?? 'bottom-left'),
                    'display_duration' => intval($n['display_duration'] ?? 5000),
                    'delay_between' => intval($n['delay_between'] ?? 8000),
                    'show_product_image' => !empty($n['show_product_image']),
                    'show_location' => !empty($n['show_location']),
                    'anonymize_names' => !empty($n['anonymize_names']),
                    'max_age_hours' => intval($n['max_age_hours'] ?? 24),
                    'message_template' => sanitize_text_field($n['message_template'] ?? '{name} from {location} purchased {product}')
                ];
            }
            
            // Handle visitor count settings
            if (isset($formData['visitors'])) {
                $v = $formData['visitors'];
                $settings['visitor_count'] = [
                    'enabled' => !empty($v['enabled']),
                    'show_on_product' => !empty($v['show_on_product']),
                    'show_on_category' => !empty($v['show_on_category']),
                    'time_window' => intval($v['time_window'] ?? 15),
                    'minimum_threshold' => intval($v['minimum_threshold'] ?? 2),
                    'message' => sanitize_text_field($v['message'] ?? '{count} people are viewing this right now')
                ];
            }
            
            // Handle stock urgency settings
            if (isset($formData['urgency'])) {
                $u = $formData['urgency'];
                $settings['stock_urgency'] = [
                    'enabled' => !empty($u['enabled']),
                    'low_stock_threshold' => intval($u['low_stock_threshold'] ?? 10),
                    'critical_stock_threshold' => intval($u['critical_stock_threshold'] ?? 3),
                    'show_exact_count' => !empty($u['show_exact_count']),
                    'low_stock_message' => sanitize_text_field($u['low_stock_message'] ?? 'Only a few left in stock'),
                    'critical_stock_message' => sanitize_text_field($u['critical_stock_message'] ?? 'Almost gone! Only {count} left')
                ];
            }
            
            // Handle cart activity settings
            if (isset($formData['cart_activity'])) {
                $ca = $formData['cart_activity'];
                $settings['cart_activity'] = [
                    'enabled' => !empty($ca['enabled']),
                    'time_window' => intval($ca['time_window'] ?? 60),
                    'minimum_threshold' => intval($ca['minimum_threshold'] ?? 1),
                    'message' => sanitize_text_field($ca['message'] ?? '{count} people have this in their cart')
                ];
            }
            
            // Handle recent sales settings
            if (isset($formData['recent_sales'])) {
                $rs = $formData['recent_sales'];
                $settings['recent_sales'] = [
                    'enabled' => !empty($rs['enabled']),
                    'time_window' => intval($rs['time_window'] ?? 24),
                    'show_on_product' => !empty($rs['show_on_product']),
                    'message' => sanitize_text_field($rs['message'] ?? '{count} sold in the last 24 hours')
                ];
            }
            
            // Handle FOMO countdown settings
            if (isset($formData['fomo_countdown'])) {
                $fc = $formData['fomo_countdown'];
                $settings['fomo_countdown'] = [
                    'enabled' => !empty($fc['enabled']),
                    'type' => sanitize_text_field($fc['type'] ?? 'sale_end'),
                    'message' => sanitize_text_field($fc['message'] ?? 'Sale ends in {countdown}')
                ];
            }
        }
        
        update_option('cartt_social_proof_settings', $settings);
        wp_send_json_success(['message' => 'Settings saved']);
    }

    /**
     * Save AI recommendations settings via AJAX
     */
    public function ajaxSaveAIRecommendations(): void
    {
        check_ajax_referer('cartt_ai_recommendations', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }

        $settings = get_option('cartt_ai_recommendations', []);
        
        // Parse the serialized form data if present
        if (!empty($_POST['settings'])) {
            if (is_string($_POST['settings'])) {
                parse_str($_POST['settings'], $formData);
            } else {
                $formData = $_POST['settings'];
            }
            
            // General settings
            if (isset($formData['enabled'])) {
                $settings['enabled'] = !empty($formData['enabled']);
            }
            
            if (isset($formData['ai_provider'])) {
                $settings['ai_provider'] = sanitize_text_field($formData['ai_provider']);
            }
            
            if (isset($formData['openai_api_key'])) {
                $settings['openai_api_key'] = sanitize_text_field($formData['openai_api_key']);
            }
            
            if (isset($formData['trending_period'])) {
                $settings['trending_period'] = intval($formData['trending_period']);
            }
            
            if (isset($formData['bestseller_period'])) {
                $settings['bestseller_period'] = intval($formData['bestseller_period']);
            }
            
            // Algorithm settings
            if (isset($formData['algorithms'])) {
                $settings['algorithms'] = array_map(function($v) { return !empty($v); }, $formData['algorithms']);
            }
            
            // Display settings
            if (isset($formData['display'])) {
                $settings['display'] = array_map(function($v) { return !empty($v); }, $formData['display']);
            }
            
            // Limits settings
            if (isset($formData['limits'])) {
                $settings['limits'] = array_map('intval', $formData['limits']);
            }
        }
        
        update_option('cartt_ai_recommendations', $settings);
        wp_send_json_success(['message' => 'Settings saved']);
    }

    /**
     * AJAX: Save card design
     */
    public function ajaxSaveCardDesign(): void
    {
        check_ajax_referer('cartt_card_designer', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $design = isset($_POST['design']) ? json_decode(stripslashes($_POST['design']), true) : null;
        
        if (!$design) {
            wp_send_json_error('Invalid design data');
        }
        
        // Sanitize the design data
        $sanitized = $this->sanitizeCardDesign($design);
        
        update_option('cartt_card_design', $sanitized);
        wp_send_json_success(['message' => 'Design saved']);
    }
    
    /**
     * Sanitize card design data
     */
    private function sanitizeCardDesign(array $design): array
    {
        $sanitized = [];
        
        // Preset
        $sanitized['preset'] = sanitize_text_field($design['preset'] ?? 'modern');
        
        // Elements (array of strings)
        $valid_elements = ['image', 'title', 'price', 'rating', 'add_to_cart', 'quick_view', 'wishlist', 'sale_badge', 'description', 'sku', 'stock'];
        $sanitized['elements'] = array_filter(
            array_map('sanitize_text_field', $design['elements'] ?? []),
            function($el) use ($valid_elements) { return in_array($el, $valid_elements); }
        );
        
        // Card settings
        $sanitized['card'] = [
            'background' => sanitize_hex_color($design['card']['background'] ?? '#ffffff'),
            'border_color' => sanitize_hex_color($design['card']['border_color'] ?? '#e0e0e0'),
            'border_width' => max(0, min(5, intval($design['card']['border_width'] ?? 1))),
            'border_radius' => max(0, min(50, intval($design['card']['border_radius'] ?? 8))),
            'shadow' => in_array($design['card']['shadow'] ?? 'small', ['none', 'small', 'medium', 'large']) ? $design['card']['shadow'] : 'small',
            'hover_effect' => in_array($design['card']['hover_effect'] ?? 'lift', ['none', 'lift', 'shadow', 'border']) ? $design['card']['hover_effect'] : 'lift',
            'hover_shadow_size' => max(5, min(40, intval($design['card']['hover_shadow_size'] ?? 15))),
            'padding' => max(0, min(30, intval($design['card']['padding'] ?? 0)))
        ];
        
        // Image settings
        $sanitized['image'] = [
            'aspect_ratio' => in_array($design['image']['aspect_ratio'] ?? 'square', ['square', 'portrait', 'landscape']) ? $design['image']['aspect_ratio'] : 'square',
            'border_radius' => max(0, min(50, intval($design['image']['border_radius'] ?? 8))),
            'hover_effect' => in_array($design['image']['hover_effect'] ?? 'zoom', ['none', 'zoom', 'fade']) ? $design['image']['hover_effect'] : 'zoom'
        ];
        
        // Title settings
        $sanitized['title'] = [
            'font_size' => max(10, min(32, intval($design['title']['font_size'] ?? 16))),
            'font_weight' => in_array($design['title']['font_weight'] ?? '600', ['300', '400', '500', '600', '700', '800']) ? $design['title']['font_weight'] : '600',
            'color' => sanitize_hex_color($design['title']['color'] ?? '#1d2327'),
            'margin_top' => max(0, min(30, intval($design['title']['margin_top'] ?? 12))),
            'lines' => max(1, min(5, intval($design['title']['lines'] ?? 2)))
        ];
        
        // Price settings
        $sanitized['price'] = [
            'font_size' => max(10, min(32, intval($design['price']['font_size'] ?? 18))),
            'font_weight' => in_array($design['price']['font_weight'] ?? '700', ['400', '500', '600', '700', '800']) ? $design['price']['font_weight'] : '700',
            'color' => sanitize_hex_color($design['price']['color'] ?? '#1d2327'),
            'sale_color' => sanitize_hex_color($design['price']['sale_color'] ?? '#dc2626'),
            'margin_top' => max(0, min(30, intval($design['price']['margin_top'] ?? 8)))
        ];
        
        // Rating settings
        $sanitized['rating'] = [
            'star_color' => sanitize_hex_color($design['rating']['star_color'] ?? '#fbbf24'),
            'show_count' => !empty($design['rating']['show_count']),
            'margin_top' => max(0, min(30, intval($design['rating']['margin_top'] ?? 8)))
        ];
        
        // Add to cart settings
        $sanitized['add_to_cart'] = [
            'style' => in_array($design['add_to_cart']['style'] ?? 'button', ['button', 'link', 'icon']) ? $design['add_to_cart']['style'] : 'button',
            'bg_color' => sanitize_hex_color($design['add_to_cart']['bg_color'] ?? '#1d2327'),
            'text_color' => sanitize_hex_color($design['add_to_cart']['text_color'] ?? '#ffffff'),
            'border_radius' => max(0, min(50, intval($design['add_to_cart']['border_radius'] ?? 6))),
            'full_width' => !empty($design['add_to_cart']['full_width']),
            'margin_top' => max(0, min(30, intval($design['add_to_cart']['margin_top'] ?? 12))),
            'button_text' => sanitize_text_field($design['add_to_cart']['button_text'] ?? 'Add to Cart')
        ];
        
        // Sale badge settings
        $sanitized['sale_badge'] = [
            'enabled' => $design['sale_badge']['enabled'] ?? true,
            'bg_color' => sanitize_hex_color($design['sale_badge']['bg_color'] ?? '#dc2626'),
            'text_color' => sanitize_hex_color($design['sale_badge']['text_color'] ?? '#ffffff'),
            'text' => sanitize_text_field($design['sale_badge']['text'] ?? 'Sale')
        ];
        
        // Wishlist settings
        $sanitized['wishlist'] = [
            'enabled' => !empty($design['wishlist']['enabled']),
            'position' => in_array($design['wishlist']['position'] ?? 'top-right', ['top-left', 'top-right']) ? $design['wishlist']['position'] : 'top-right'
        ];
        
        // Quick view settings
        $sanitized['quick_view'] = [
            'enabled' => !empty($design['quick_view']['enabled']),
            'style' => in_array($design['quick_view']['style'] ?? 'icon', ['icon', 'button']) ? $design['quick_view']['style'] : 'icon'
        ];
        
        // Grid settings
        $sanitized['grid'] = [
            'columns' => max(1, min(6, intval($design['grid']['columns'] ?? 4))),
            'columns_tablet' => max(1, min(4, intval($design['grid']['columns_tablet'] ?? 3))),
            'columns_mobile' => max(1, min(3, intval($design['grid']['columns_mobile'] ?? 2))),
            'gap' => max(0, min(60, intval($design['grid']['gap'] ?? 24)))
        ];
        
        return $sanitized;
    }

    /**
     * AJAX: Activate an addon
     */
    public function ajaxActivateAddon(): void
    {
        check_ajax_referer('cartt_addons', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $addon = isset($_POST['addon']) ? sanitize_text_field($_POST['addon']) : '';
        $license = isset($_POST['license']) ? sanitize_text_field($_POST['license']) : '';
        
        if (empty($addon) || empty($license)) {
            wp_send_json_error('Addon and license key required');
        }
        
        // Valid addon keys
        $validAddons = [
            'marketplace', 'wholesale', 'dropshipping', 'ai_conversion',
            'abandoned_cart', 'chat', 'ai_content', 'gift_cards', 'loyalty',
            'advanced_analytics', 'live_shipping', 'multi_currency',
            'bookings', 'subscriptions', 'all_access'
        ];
        
        if (!in_array($addon, $validAddons)) {
            wp_send_json_error('Invalid addon');
        }
        
        // Use licensing service to verify and activate
        $result = \Cartt\Services\SoftwareLicensingService::clientVerifyAndActivate($license, $addon);
        
        if ($result['success'] ?? false) {
            wp_send_json_success(['message' => 'Addon activated']);
        } else {
            wp_send_json_error($result['message'] ?? 'License verification failed');
        }
    }

    /**
     * AJAX: Deactivate an addon
     */
    public function ajaxDeactivateAddon(): void
    {
        check_ajax_referer('cartt_addons', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $addon = isset($_POST['addon']) ? sanitize_text_field($_POST['addon']) : '';
        
        if (empty($addon)) {
            wp_send_json_error('Addon required');
        }
        
        // Deactivate addon locally
        $activeAddons = get_option('cartt_active_addons', []);
        $activeAddons = array_diff($activeAddons, [$addon]);
        update_option('cartt_active_addons', array_values($activeAddons));
        
        // Remove stored license
        $licenses = get_option('cartt_addon_licenses', []);
        if (isset($licenses[$addon])) {
            // Optionally deactivate on remote server
            // SoftwareLicensingService would handle this
            unset($licenses[$addon]);
            update_option('cartt_addon_licenses', $licenses);
        }
        
        wp_send_json_success(['message' => 'Addon deactivated']);
    }

    /**
     * AJAX: Create addon checkout session
     */
    public function ajaxCreateAddonCheckout(): void
    {
        check_ajax_referer('cartt_addons', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $addon = isset($_POST['addon']) ? sanitize_text_field($_POST['addon']) : '';
        $plan = isset($_POST['plan']) ? sanitize_text_field($_POST['plan']) : 'yearly';
        
        if (empty($addon)) {
            wp_send_json_error('Addon required');
        }
        
        // Addon pricing (yearly/monthly)
        $pricing = [
            'chat' => ['yearly' => 12900, 'monthly' => 2900],
            'ai_content' => ['yearly' => 9900, 'monthly' => 1900],
            'marketplace' => ['yearly' => 14900, 'monthly' => 0],
            'wholesale' => ['yearly' => 9900, 'monthly' => 0],
            'dropshipping' => ['yearly' => 7900, 'monthly' => 0],
            'ai_conversion' => ['yearly' => 9900, 'monthly' => 0],
            'abandoned_cart' => ['yearly' => 4900, 'monthly' => 0],
            'gift_cards' => ['yearly' => 4900, 'monthly' => 0],
            'loyalty' => ['yearly' => 5900, 'monthly' => 0],
            'advanced_analytics' => ['yearly' => 7900, 'monthly' => 0],
            'live_shipping' => ['yearly' => 5900, 'monthly' => 0],
            'multi_currency' => ['yearly' => 4900, 'monthly' => 0],
            'bookings' => ['yearly' => 9900, 'monthly' => 0],
            'subscriptions' => ['yearly' => 9900, 'monthly' => 0],
            'all_access' => ['yearly' => 29900, 'monthly' => 4900],
        ];
        
        if (!isset($pricing[$addon])) {
            wp_send_json_error('Invalid addon');
        }
        
        $price = $plan === 'monthly' && $pricing[$addon]['monthly'] > 0 
            ? $pricing[$addon]['monthly'] 
            : $pricing[$addon]['yearly'];
        
        $isSubscription = $plan === 'monthly' && $pricing[$addon]['monthly'] > 0;
        
        // Call relay to create Stripe checkout
        $response = wp_remote_post('https://cartt.app/wp-content/uploads/cartt-relay/relay.php', [
            'body' => [
                'action' => 'create_addon_checkout',
                'addon' => $addon,
                'plan' => $plan,
                'price' => $price,
                'is_subscription' => $isSubscription ? '1' : '0',
                'site_url' => home_url(),
                'admin_email' => get_option('admin_email'),
                'return_url' => admin_url('admin.php?page=cartt-addons&purchase=success&addon=' . $addon),
                'cancel_url' => admin_url('admin.php?page=cartt-addons&purchase=cancelled'),
            ],
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error('Failed to connect to payment server');
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!empty($body['checkout_url'])) {
            wp_send_json_success(['checkout_url' => $body['checkout_url']]);
        } else {
            wp_send_json_error($body['error'] ?? 'Failed to create checkout session');
        }
    }

    /**
     * v1.7.0 - Bulk product actions
     */
    public function ajaxBulkProducts(): void
    {
        check_ajax_referer('cartt_bulk', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $action = sanitize_text_field($_POST['bulk_action'] ?? '');
        $ids = array_map('intval', $_POST['ids'] ?? []);
        
        if (empty($ids)) {
            wp_send_json_error(['message' => 'No products selected']);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $count = count($ids);
        
        switch ($action) {
            case 'publish':
                $wpdb->query($wpdb->prepare(
                    "UPDATE $table SET status = 'published' WHERE id IN (" . implode(',', $ids) . ")"
                ));
                wp_send_json_success(['message' => "$count products published"]);
                break;
                
            case 'draft':
                $wpdb->query($wpdb->prepare(
                    "UPDATE $table SET status = 'draft' WHERE id IN (" . implode(',', $ids) . ")"
                ));
                wp_send_json_success(['message' => "$count products set to draft"]);
                break;
                
            case 'delete':
                $wpdb->query("DELETE FROM $table WHERE id IN (" . implode(',', $ids) . ")");
                wp_send_json_success(['message' => "$count products deleted"]);
                break;
                
            default:
                wp_send_json_error(['message' => 'Invalid action']);
        }
    }

    /**
     * v1.7.0 - Inline edit product field
     */
    public function ajaxInlineEditProduct(): void
    {
        check_ajax_referer('cartt_inline', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $id = intval($_POST['id'] ?? 0);
        $field = sanitize_text_field($_POST['field'] ?? '');
        $value = sanitize_text_field($_POST['value'] ?? '');
        
        if (!$id || !$field) {
            wp_send_json_error(['message' => 'Invalid request']);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        
        $allowed_fields = ['price', 'stock_quantity', 'sale_price', 'sku'];
        if (!in_array($field, $allowed_fields)) {
            wp_send_json_error(['message' => 'Invalid field']);
        }
        
        // Map field names
        $db_field = $field === 'stock' ? 'stock_quantity' : $field;
        
        $wpdb->update($table, [$db_field => $value], ['id' => $id]);
        
        wp_send_json_success(['message' => 'Updated']);
    }

    /**
     * v1.7.0 - Update product status inline
     */
    public function ajaxUpdateProductStatus(): void
    {
        check_ajax_referer('cartt_inline', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $id = intval($_POST['id'] ?? 0);
        $status = sanitize_text_field($_POST['status'] ?? '');
        
        if (!$id || !in_array($status, ['published', 'draft'])) {
            wp_send_json_error(['message' => 'Invalid request']);
        }
        
        global $wpdb;
        $wpdb->update($wpdb->prefix . 'cartt_products', ['status' => $status], ['id' => $id]);
        
        wp_send_json_success(['message' => 'Status updated']);
    }

    /**
     * v1.7.0 - Global search
     */
    public function ajaxGlobalSearch(): void
    {
        $query = sanitize_text_field($_POST['query'] ?? '');
        
        if (strlen($query) < 2) {
            wp_send_json_success(['html' => '']);
        }
        
        global $wpdb;
        $like = '%' . $wpdb->esc_like($query) . '%';
        $html = '';
        
        // Search products
        $products = $wpdb->get_results($wpdb->prepare(
            "SELECT id, name FROM {$wpdb->prefix}cartt_products WHERE name LIKE %s OR sku LIKE %s LIMIT 5",
            $like, $like
        ));
        
        foreach ($products as $p) {
            $url = admin_url('admin.php?page=cartt-products&action=edit&id=' . $p->id);
            $html .= '<a href="' . esc_url($url) . '" class="search-result-item"><span class="search-result-type">Product</span>' . esc_html($p->name) . '</a>';
        }
        
        // Search orders
        $orders = $wpdb->get_results($wpdb->prepare(
            "SELECT o.id, o.order_number, c.first_name, c.last_name 
             FROM {$wpdb->prefix}cartt_orders o
             LEFT JOIN {$wpdb->prefix}cartt_customers c ON o.customer_id = c.id
             WHERE o.order_number LIKE %s OR c.first_name LIKE %s OR c.last_name LIKE %s OR c.email LIKE %s
             LIMIT 5",
            $like, $like, $like, $like
        ));
        
        foreach ($orders as $o) {
            $url = admin_url('admin.php?page=cartt-orders&action=view&id=' . $o->id);
            $name = trim($o->first_name . ' ' . $o->last_name) ?: 'Guest';
            $html .= '<a href="' . esc_url($url) . '" class="search-result-item"><span class="search-result-type">Order</span>#' . esc_html($o->order_number) . ' - ' . esc_html($name) . '</a>';
        }
        
        // Search customers
        $customers = $wpdb->get_results($wpdb->prepare(
            "SELECT id, first_name, last_name, email FROM {$wpdb->prefix}cartt_customers 
             WHERE first_name LIKE %s OR last_name LIKE %s OR email LIKE %s LIMIT 5",
            $like, $like, $like
        ));
        
        foreach ($customers as $c) {
            $url = admin_url('admin.php?page=cartt-customers&action=view&id=' . $c->id);
            $name = trim($c->first_name . ' ' . $c->last_name) ?: $c->email;
            $html .= '<a href="' . esc_url($url) . '" class="search-result-item"><span class="search-result-type">Customer</span>' . esc_html($name) . '</a>';
        }
        
        if (empty($html)) {
            $html = '<div class="search-no-results">No results found</div>';
        }
        
        wp_send_json_success(['html' => $html]);
    }

    /**
     * v1.7.0 - Bulk order actions
     */
    public function ajaxBulkOrders(): void
    {
        check_ajax_referer('cartt_bulk', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $action = sanitize_text_field($_POST['bulk_action'] ?? '');
        $ids = array_map('intval', $_POST['ids'] ?? []);
        
        if (empty($ids)) {
            wp_send_json_error(['message' => 'No orders selected']);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_orders';
        $count = count($ids);
        
        $valid_statuses = ['processing', 'completed', 'cancelled', 'on-hold'];
        
        if (in_array($action, $valid_statuses)) {
            $wpdb->query("UPDATE $table SET status = '$action' WHERE id IN (" . implode(',', $ids) . ")");
            wp_send_json_success(['message' => "$count orders updated to $action"]);
        } else {
            wp_send_json_error(['message' => 'Invalid action']);
        }
    }

    /**
     * v1.7.0 - Reorder categories
     */
    public function ajaxReorderCategories(): void
    {
        check_ajax_referer('cartt_reorder', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $order = array_map('intval', $_POST['order'] ?? []);
        
        if (empty($order)) {
            wp_send_json_error(['message' => 'No order provided']);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_categories';
        
        foreach ($order as $position => $id) {
            $wpdb->update(
                $table,
                ['sort_order' => $position],
                ['id' => $id],
                ['%d'],
                ['%d']
            );
        }
        
        wp_send_json_success(['message' => 'Order saved']);
    }

    /**
     * v1.7.0 - Generic inline update
     */
    public function ajaxInlineUpdate(): void
    {
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'cartt_admin')) {
            wp_send_json_error(['message' => 'Invalid nonce']);
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        $id = intval($_POST['id'] ?? 0);
        $field = sanitize_text_field($_POST['field'] ?? '');
        $value = sanitize_text_field($_POST['value'] ?? '');
        
        if (!$id || !$field) {
            wp_send_json_error(['message' => 'Missing parameters']);
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'cartt_products';
        $display = $value;
        
        // Map field to column
        $allowed_fields = [
            'price' => ['column' => 'price', 'format' => '%f'],
            'stock' => ['column' => 'stock_quantity', 'format' => '%d'],
        ];
        
        if (!isset($allowed_fields[$field])) {
            wp_send_json_error(['message' => 'Invalid field']);
        }
        
        $column = $allowed_fields[$field]['column'];
        $format = $allowed_fields[$field]['format'];
        
        $result = $wpdb->update(
            $table,
            [$column => $value],
            ['id' => $id],
            [$format],
            ['%d']
        );
        
        if ($result !== false) {
            // Format display value
            if ($field === 'price') {
                $currency = \Cartt\Services\CurrencyService::getSymbol();
                $display = $currency . number_format((float)$value, 2);
            }
            
            wp_send_json_success(['display' => $display, 'value' => $value]);
        } else {
            wp_send_json_error(['message' => 'Update failed']);
        }
    }
}
