/**
 * Cartt Frontend
 */
(function($) {
    'use strict';

    const Cartt = {
        cart: {
            items: [],
            total: 0
        },

        // Currency formatting
        formatPrice: function(price) {
            const rate = carttFrontend.exchangeRate || 1;
            const decimals = carttFrontend.currencyDecimals || 2;
            const symbol = carttFrontend.currencySymbol || '$';
            const position = carttFrontend.currencyPosition || 'left';
            const thousand = carttFrontend.thousandSeparator || ',';
            const decimal = carttFrontend.decimalSeparator || '.';
            
            // Convert price
            let converted = price * rate;
            
            // Format number
            let formatted = converted.toFixed(decimals);
            let parts = formatted.split('.');
            parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousand);
            if (parts[1]) {
                formatted = parts[0] + decimal + parts[1];
            } else {
                formatted = parts[0];
            }
            
            // Apply symbol position
            switch (position) {
                case 'right':
                    return formatted + symbol;
                case 'left_space':
                    return symbol + ' ' + formatted;
                case 'right_space':
                    return formatted + ' ' + symbol;
                default: // left
                    return symbol + formatted;
            }
        },

        // Convert price to current currency
        convertPrice: function(price) {
            const rate = carttFrontend.exchangeRate || 1;
            return price * rate;
        },

        init: function() {
            this.bindEvents();
            this.loadCart();
            this.initCurrencySwitcher();
        },

        initCurrencySwitcher: function() {
            // Handle currency selector change
            $(document).on('change', '#cartt-currency-select', function() {
                const currency = $(this).val();
                
                $.post(carttFrontend.ajaxUrl, {
                    action: 'cartt_switch_currency',
                    currency: currency
                }, function(response) {
                    if (response.success) {
                        // Reload page to update all prices
                        location.reload();
                    }
                });
            });
        },

        bindEvents: function() {
            // Add to cart
            $(document).on('submit', '.cartt-add-to-cart-form', this.handleAddToCart);
            
            // Update quantity
            $(document).on('change', '.cartt-cart-quantity', this.handleUpdateQuantity);
            
            // Remove from cart
            $(document).on('click', '.cartt-cart-item-remove', this.handleRemoveFromCart);
            
            // Checkout form
            $(document).on('submit', '.cartt-checkout-form', this.handleCheckout);
        },

        loadCart: function() {
            // Load cart from session/localStorage
            const savedCart = localStorage.getItem('cartt_cart');
            if (savedCart) {
                try {
                    this.cart = JSON.parse(savedCart);
                    this.updateCartUI();
                } catch (e) {
                    this.cart = { items: [], total: 0 };
                }
            }
        },

        saveCart: function() {
            localStorage.setItem('cartt_cart', JSON.stringify(this.cart));
            this.updateCartUI();
        },

        handleAddToCart: function(e) {
            e.preventDefault();
            
            const $form = $(this);
            const $btn = $form.find('.cartt-add-to-cart-btn');
            const productId = $form.data('product-id');
            const quantity = parseInt($form.find('.cartt-quantity-input').val()) || 1;

            $btn.prop('disabled', true).text('Adding...');

            // Get product data
            const productData = {
                id: productId,
                name: $form.data('product-name'),
                price: parseFloat($form.data('product-price')),
                image: $form.data('product-image') || '',
                quantity: quantity
            };

            // Check if already in cart
            const existingIndex = Cartt.cart.items.findIndex(item => item.id === productId);
            
            if (existingIndex > -1) {
                Cartt.cart.items[existingIndex].quantity += quantity;
            } else {
                Cartt.cart.items.push(productData);
            }

            Cartt.recalculateTotal();
            Cartt.saveCart();

            // Show feedback
            $btn.text('Added!');
            setTimeout(function() {
                $btn.prop('disabled', false).text('Add to Cart');
            }, 1500);

            // Trigger event for other scripts
            $(document).trigger('cartt:added_to_cart', [productData]);
        },

        handleUpdateQuantity: function() {
            const $input = $(this);
            const productId = $input.data('product-id');
            const quantity = parseInt($input.val()) || 1;

            const itemIndex = Cartt.cart.items.findIndex(item => item.id === productId);
            
            if (itemIndex > -1) {
                if (quantity <= 0) {
                    Cartt.cart.items.splice(itemIndex, 1);
                } else {
                    Cartt.cart.items[itemIndex].quantity = quantity;
                }
                
                Cartt.recalculateTotal();
                Cartt.saveCart();
                Cartt.renderCart();
            }
        },

        handleRemoveFromCart: function(e) {
            e.preventDefault();
            
            const productId = $(this).data('product-id');
            const itemIndex = Cartt.cart.items.findIndex(item => item.id === productId);
            
            if (itemIndex > -1) {
                Cartt.cart.items.splice(itemIndex, 1);
                Cartt.recalculateTotal();
                Cartt.saveCart();
                Cartt.renderCart();
            }
        },

        handleCheckout: function(e) {
            e.preventDefault();
            
            const $form = $(this);
            const $btn = $form.find('.cartt-place-order-btn');
            
            $btn.prop('disabled', true).text('Processing...');

            const formData = {
                billing: {
                    email: $form.find('#billing-email').val(),
                    first_name: $form.find('#billing-first-name').val(),
                    last_name: $form.find('#billing-last-name').val(),
                    address_1: $form.find('#billing-address').val(),
                    city: $form.find('#billing-city').val(),
                    state: $form.find('#billing-state').val(),
                    postcode: $form.find('#billing-postcode').val(),
                    country: $form.find('#billing-country').val(),
                },
                cart: Cartt.cart,
                payment_method: $form.find('input[name="payment_method"]:checked').val()
            };

            $.ajax({
                url: carttFrontend.apiUrl + 'checkout',
                method: 'POST',
                data: JSON.stringify(formData),
                contentType: 'application/json',
                beforeSend: function(xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', carttFrontend.nonce);
                },
                success: function(response) {
                    if (response.success) {
                        // Clear cart
                        Cartt.cart = { items: [], total: 0 };
                        Cartt.saveCart();
                        
                        // Redirect to thank you page or show message
                        if (response.redirect) {
                            window.location.href = response.redirect;
                        } else {
                            Cartt.showMessage('Order placed successfully!', 'success');
                            $btn.prop('disabled', false).text('Place Order');
                        }
                    } else {
                        Cartt.showMessage(response.message || 'An error occurred.', 'error');
                        $btn.prop('disabled', false).text('Place Order');
                    }
                },
                error: function() {
                    Cartt.showMessage('An error occurred. Please try again.', 'error');
                    $btn.prop('disabled', false).text('Place Order');
                }
            });
        },

        recalculateTotal: function() {
            this.cart.total = this.cart.items.reduce(function(sum, item) {
                return sum + (item.price * item.quantity);
            }, 0);
        },

        updateCartUI: function() {
            // Update cart count in header/nav
            const count = this.cart.items.reduce((sum, item) => sum + item.quantity, 0);
            $('.cartt-cart-count').text(count);
            
            // Update mini cart if present
            if ($('.cartt-mini-cart').length) {
                this.renderMiniCart();
            }
        },

        renderCart: function() {
            const $cartContainer = $('.cartt-cart-items');
            
            if (!$cartContainer.length) return;

            if (this.cart.items.length === 0) {
                $cartContainer.html(`
                    <div class="cartt-cart-empty">
                        <h2>Your cart is empty</h2>
                        <p>Add some products to get started.</p>
                        <a href="/" class="cartt-btn cartt-btn-primary">Continue Shopping</a>
                    </div>
                `);
                $('.cartt-cart-totals, .cartt-cart-actions').hide();
                return;
            }

            let html = '<table class="cartt-cart-table">';
            html += '<thead><tr><th>Product</th><th>Price</th><th>Quantity</th><th>Total</th></tr></thead>';
            html += '<tbody>';

            this.cart.items.forEach(function(item) {
                const itemTotal = item.price * item.quantity;
                html += `
                    <tr>
                        <td>
                            <div class="cartt-cart-item-info">
                                <div class="cartt-cart-item-image">
                                    ${item.image ? `<img src="${item.image}" alt="">` : ''}
                                </div>
                                <div>
                                    <div class="cartt-cart-item-name">${Cartt.escapeHtml(item.name)}</div>
                                    <button type="button" class="cartt-cart-item-remove" data-product-id="${item.id}">Remove</button>
                                </div>
                            </div>
                        </td>
                        <td>${Cartt.formatCurrency(item.price)}</td>
                        <td>
                            <input type="number" class="cartt-quantity-input cartt-cart-quantity" 
                                   value="${item.quantity}" min="1" data-product-id="${item.id}">
                        </td>
                        <td>${Cartt.formatCurrency(itemTotal)}</td>
                    </tr>
                `;
            });

            html += '</tbody></table>';
            $cartContainer.html(html);

            // Update totals
            $('.cartt-cart-subtotal').text(this.formatCurrency(this.cart.total));
            $('.cartt-cart-total').text(this.formatCurrency(this.cart.total));
            $('.cartt-cart-totals, .cartt-cart-actions').show();
        },

        renderMiniCart: function() {
            const $miniCart = $('.cartt-mini-cart');
            
            if (this.cart.items.length === 0) {
                $miniCart.html('<p>Your cart is empty</p>');
                return;
            }

            let html = '<ul class="cartt-mini-cart-items">';
            
            this.cart.items.forEach(function(item) {
                html += `
                    <li>
                        <span class="cartt-mini-cart-item-name">${Cartt.escapeHtml(item.name)}</span>
                        <span class="cartt-mini-cart-item-qty">x${item.quantity}</span>
                        <span class="cartt-mini-cart-item-price">${Cartt.formatCurrency(item.price * item.quantity)}</span>
                    </li>
                `;
            });

            html += '</ul>';
            html += `<div class="cartt-mini-cart-total">Total: ${this.formatCurrency(this.cart.total)}</div>`;
            html += `<a href="${carttFrontend.cartUrl}" class="cartt-btn cartt-btn-primary">View Cart</a>`;
            
            $miniCart.html(html);
        },

        formatCurrency: function(amount) {
            const position = carttFrontend.currencyPosition || 'left';
            const symbol = this.getCurrencySymbol();
            const formatted = parseFloat(amount).toFixed(2);
            
            return position === 'left' ? symbol + formatted : formatted + symbol;
        },

        getCurrencySymbol: function() {
            const symbols = {
                'USD': '$', 'EUR': '€', 'GBP': '£', 'CAD': '$',
                'AUD': '$', 'JPY': '¥', 'CNY': '¥', 'INR': '₹'
            };
            return symbols[carttFrontend.currency] || '$';
        },

        escapeHtml: function(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        showMessage: function(message, type) {
            const $message = $(`<div class="cartt-message cartt-message-${type}">${message}</div>`);
            $('.cartt-checkout, .cartt-cart').prepend($message);
            
            setTimeout(function() {
                $message.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };

    $(document).ready(function() {
        Cartt.init();
        
        // Initialize cart page if present
        if ($('.cartt-cart-items').length) {
            Cartt.renderCart();
        }
    });

    // Expose for external use
    window.Cartt = Cartt;

    /* ==========================================================================
       Social Proof Module
       ========================================================================== */
    
    const CarttSocialProof = {
        settings: {},
        purchaseQueue: [],
        isShowingNotification: false,
        visitorUpdateInterval: null,
        countdownInterval: null,

        init: function() {
            // Check if social proof is enabled
            if (typeof carttSocialProof === 'undefined' || !carttSocialProof.enabled) {
                return;
            }

            this.settings = carttSocialProof;
            
            // Initialize components based on settings
            if (this.settings.purchaseNotifications) {
                this.initPurchaseNotifications();
            }
            
            if (this.settings.visitorCount) {
                this.initVisitorCount();
            }
            
            if (this.settings.stockUrgency) {
                this.initStockUrgency();
            }
            
            if (this.settings.cartActivity) {
                this.initCartActivity();
            }
            
            if (this.settings.recentSales) {
                this.initRecentSales();
            }
            
            if (this.settings.fomoCountdown) {
                this.initFomoCountdown();
            }
        },

        /* Purchase Notifications */
        initPurchaseNotifications: function() {
            // Create notification container
            this.createNotificationContainer();
            
            // Start fetching and displaying notifications
            this.fetchPurchaseNotifications();
            
            // Set interval for fetching new notifications
            setInterval(() => this.fetchPurchaseNotifications(), 60000);
        },

        createNotificationContainer: function() {
            if ($('#cartt-purchase-notification').length) return;
            
            const position = this.settings.notificationPosition || 'bottom-left';
            const $container = $(`
                <div id="cartt-purchase-notification" class="cartt-purchase-notification cartt-position-${position}">
                    <button class="cartt-purchase-notification-close" aria-label="Close">&times;</button>
                    <div class="cartt-purchase-notification-inner">
                        <div class="cartt-purchase-notification-image"></div>
                        <div class="cartt-purchase-notification-content">
                            <div class="cartt-purchase-notification-message"></div>
                            <div class="cartt-purchase-notification-product"></div>
                            <div class="cartt-purchase-notification-time"></div>
                        </div>
                    </div>
                </div>
            `);
            
            $('body').append($container);
            
            // Close button handler
            $container.on('click', '.cartt-purchase-notification-close', () => {
                this.hideNotification();
            });
        },

        fetchPurchaseNotifications: function() {
            $.ajax({
                url: carttFrontend.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'cartt_get_recent_purchases',
                    nonce: carttFrontend.nonce
                },
                success: (response) => {
                    if (response.success && response.data.purchases) {
                        this.purchaseQueue = response.data.purchases;
                        if (!this.isShowingNotification && this.purchaseQueue.length > 0) {
                            this.showNextNotification();
                        }
                    }
                }
            });
        },

        showNextNotification: function() {
            if (this.purchaseQueue.length === 0) {
                this.isShowingNotification = false;
                return;
            }

            this.isShowingNotification = true;
            const purchase = this.purchaseQueue.shift();
            const $notification = $('#cartt-purchase-notification');
            
            // Update content
            const anonymize = this.settings.anonymize;
            let buyerName = purchase.buyer_name || 'Someone';
            if (anonymize) {
                buyerName = buyerName.charAt(0) + '***';
            }
            
            const message = (this.settings.notificationMessage || '{buyer} from {location} purchased')
                .replace('{buyer}', buyerName)
                .replace('{location}', purchase.location || 'your area');
            
            $notification.find('.cartt-purchase-notification-message').text(message);
            $notification.find('.cartt-purchase-notification-product').text(purchase.product_name);
            $notification.find('.cartt-purchase-notification-time').text(purchase.time_ago || 'Just now');
            
            if (purchase.product_image) {
                $notification.find('.cartt-purchase-notification-image').html(`<img src="${purchase.product_image}" alt="">`);
            } else {
                $notification.find('.cartt-purchase-notification-image').html('');
            }
            
            // Show notification
            $notification.addClass('cartt-visible');
            
            // Hide after duration
            const duration = (this.settings.notificationDuration || 5) * 1000;
            setTimeout(() => {
                this.hideNotification();
                
                // Show next notification after delay
                const delay = (this.settings.notificationDelay || 10) * 1000;
                setTimeout(() => this.showNextNotification(), delay);
            }, duration);
        },

        hideNotification: function() {
            $('#cartt-purchase-notification').removeClass('cartt-visible');
        },

        /* Visitor Count */
        initVisitorCount: function() {
            // Register this visitor
            this.registerVisitor();
            
            // Update visitor count periodically
            this.updateVisitorCount();
            this.visitorUpdateInterval = setInterval(() => this.updateVisitorCount(), 30000);
            
            // Heartbeat to keep session active
            setInterval(() => this.registerVisitor(), 60000);
        },

        registerVisitor: function() {
            $.ajax({
                url: carttFrontend.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'cartt_register_visitor',
                    nonce: carttFrontend.nonce,
                    product_id: this.settings.productId || 0,
                    page_url: window.location.href
                }
            });
        },

        updateVisitorCount: function() {
            const $elements = $('.cartt-visitor-count');
            if ($elements.length === 0) return;

            $.ajax({
                url: carttFrontend.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'cartt_get_visitor_count',
                    nonce: carttFrontend.nonce,
                    product_id: this.settings.productId || 0
                },
                success: (response) => {
                    if (response.success) {
                        const count = response.data.count || 0;
                        const minThreshold = this.settings.visitorMinThreshold || 1;
                        
                        if (count >= minThreshold) {
                            $elements.find('.cartt-visitor-count-number').text(count);
                            $elements.show();
                        } else {
                            $elements.hide();
                        }
                    }
                }
            });
        },

        /* Stock Urgency */
        initStockUrgency: function() {
            // Stock urgency is primarily server-rendered
            // This just handles any dynamic updates
            this.updateStockUrgency();
        },

        updateStockUrgency: function() {
            const $elements = $('.cartt-stock-urgency[data-product-id]');
            
            $elements.each(function() {
                const $el = $(this);
                const productId = $el.data('product-id');
                
                $.ajax({
                    url: carttFrontend.ajaxUrl,
                    method: 'POST',
                    data: {
                        action: 'cartt_get_stock_status',
                        nonce: carttFrontend.nonce,
                        product_id: productId
                    },
                    success: (response) => {
                        if (response.success) {
                            const stock = response.data.stock;
                            const lowThreshold = CarttSocialProof.settings.stockLowThreshold || 10;
                            const criticalThreshold = CarttSocialProof.settings.stockCriticalThreshold || 3;
                            
                            $el.removeClass('cartt-stock-urgency-low cartt-stock-urgency-critical');
                            
                            if (stock <= criticalThreshold) {
                                $el.addClass('cartt-stock-urgency-critical').show();
                            } else if (stock <= lowThreshold) {
                                $el.addClass('cartt-stock-urgency-low').show();
                            } else {
                                $el.hide();
                            }
                            
                            // Update count if shown
                            if (CarttSocialProof.settings.showExactCount) {
                                $el.find('.cartt-stock-count').text(stock);
                            }
                        }
                    }
                });
            });
        },

        /* Cart Activity */
        initCartActivity: function() {
            this.updateCartActivity();
            setInterval(() => this.updateCartActivity(), 30000);
        },

        updateCartActivity: function() {
            const $elements = $('.cartt-cart-activity[data-product-id]');
            
            $elements.each(function() {
                const $el = $(this);
                const productId = $el.data('product-id');
                
                $.ajax({
                    url: carttFrontend.ajaxUrl,
                    method: 'POST',
                    data: {
                        action: 'cartt_get_cart_activity',
                        nonce: carttFrontend.nonce,
                        product_id: productId
                    },
                    success: (response) => {
                        if (response.success) {
                            const count = response.data.count || 0;
                            const minThreshold = CarttSocialProof.settings.cartActivityMin || 1;
                            
                            if (count >= minThreshold) {
                                $el.find('.cartt-cart-activity-count').text(count);
                                $el.show();
                            } else {
                                $el.hide();
                            }
                        }
                    }
                });
            });
        },

        /* Recent Sales */
        initRecentSales: function() {
            this.updateRecentSales();
            setInterval(() => this.updateRecentSales(), 60000);
        },

        updateRecentSales: function() {
            const $containers = $('.cartt-recent-sales[data-product-id]');
            
            $containers.each(function() {
                const $container = $(this);
                const productId = $container.data('product-id');
                const limit = $container.data('limit') || 5;
                
                $.ajax({
                    url: carttFrontend.ajaxUrl,
                    method: 'POST',
                    data: {
                        action: 'cartt_get_recent_sales',
                        nonce: carttFrontend.nonce,
                        product_id: productId,
                        limit: limit
                    },
                    success: (response) => {
                        if (response.success && response.data.sales.length > 0) {
                            let html = '';
                            
                            response.data.sales.forEach(sale => {
                                html += `
                                    <li class="cartt-recent-sales-item">
                                        <div class="cartt-recent-sales-item-image">
                                            ${sale.image ? `<img src="${sale.image}" alt="">` : ''}
                                        </div>
                                        <div class="cartt-recent-sales-item-info">
                                            <div class="cartt-recent-sales-item-product">${CarttSocialProof.escapeHtml(sale.product_name)}</div>
                                            <div class="cartt-recent-sales-item-time">${sale.time_ago}</div>
                                        </div>
                                    </li>
                                `;
                            });
                            
                            $container.find('.cartt-recent-sales-list').html(html);
                            $container.show();
                        } else {
                            $container.hide();
                        }
                    }
                });
            });
        },

        /* FOMO Countdown Timer */
        initFomoCountdown: function() {
            const $timers = $('.cartt-fomo-countdown');
            
            $timers.each(function() {
                const $timer = $(this);
                const endTime = $timer.data('end-time');
                
                if (!endTime) return;
                
                CarttSocialProof.startCountdown($timer, new Date(endTime).getTime());
            });
        },

        startCountdown: function($timer, endTime) {
            const updateTimer = () => {
                const now = new Date().getTime();
                const distance = endTime - now;
                
                if (distance <= 0) {
                    $timer.hide();
                    return;
                }
                
                const hours = Math.floor(distance / (1000 * 60 * 60));
                const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((distance % (1000 * 60)) / 1000);
                
                $timer.find('.cartt-countdown-hours').text(hours.toString().padStart(2, '0'));
                $timer.find('.cartt-countdown-minutes').text(minutes.toString().padStart(2, '0'));
                $timer.find('.cartt-countdown-seconds').text(seconds.toString().padStart(2, '0'));
            };
            
            updateTimer();
            setInterval(updateTimer, 1000);
        },

        /* Utility */
        escapeHtml: function(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    // Initialize Social Proof
    $(document).ready(function() {
        CarttSocialProof.init();
    });

    // Expose for external use
    window.CarttSocialProof = CarttSocialProof;

})(jQuery);
