/**
 * Cartt Chat Widget
 * @since 1.4.2
 */

(function($) {
    'use strict';
    
    var CarttChatWidget = {
        visitorId: null,
        conversationId: null,
        pollInterval: null,
        isOpen: false,
        
        init: function() {
            var $widget = $('#cartt-chat-widget');
            if (!$widget.length) return;
            
            this.visitorId = $widget.data('visitor');
            
            // Position
            if (carttChat.settings.position === 'left') {
                $widget.addClass('position-left');
            }
            
            // Show widget
            $widget.show();
            
            this.bindEvents();
            this.loadMessages();
        },
        
        bindEvents: function() {
            var self = this;
            
            // Toggle chat window
            $('.cartt-chat-button').on('click', function() {
                self.toggleChat();
            });
            
            // Close chat
            $('.cartt-chat-close').on('click', function() {
                self.closeChat();
            });
            
            // Send message
            $('#cartt-chat-send').on('click', function() {
                self.sendMessage();
            });
            
            // Enter to send
            $('#cartt-chat-message').on('keydown', function(e) {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    self.sendMessage();
                }
            });
        },
        
        toggleChat: function() {
            if (this.isOpen) {
                this.closeChat();
            } else {
                this.openChat();
            }
        },
        
        openChat: function() {
            $('.cartt-chat-window').addClass('open');
            this.isOpen = true;
            
            // Start polling
            this.startPolling();
            
            // Focus input
            setTimeout(function() {
                $('#cartt-chat-message').focus();
            }, 100);
        },
        
        closeChat: function() {
            $('.cartt-chat-window').removeClass('open');
            this.isOpen = false;
            
            // Stop polling
            this.stopPolling();
        },
        
        sendMessage: function() {
            var message = $('#cartt-chat-message').val().trim();
            if (!message) return;
            
            var name = $('#cartt-chat-name').val() || '';
            var email = $('#cartt-chat-email').val() || '';
            
            var $btn = $('#cartt-chat-send');
            $btn.prop('disabled', true);
            
            // Add message to UI immediately
            this.appendMessage('visitor', message);
            $('#cartt-chat-message').val('');
            
            var self = this;
            
            $.post(carttChat.ajaxurl, {
                action: 'cartt_chat_send',
                nonce: carttChat.nonce,
                visitor_id: this.visitorId,
                message: message,
                name: name,
                email: email,
                cart: window.carttCartData || null,
                page_url: window.location.href
            }, function(response) {
                $btn.prop('disabled', false);
                
                if (response.success) {
                    self.conversationId = response.data.conversation_id;
                }
            }).fail(function() {
                $btn.prop('disabled', false);
            });
        },
        
        loadMessages: function() {
            var self = this;
            
            $.post(carttChat.ajaxurl, {
                action: 'cartt_chat_get_messages',
                nonce: carttChat.nonce,
                visitor_id: this.visitorId
            }, function(response) {
                if (response.success && response.data.messages.length > 0) {
                    self.renderMessages(response.data.messages);
                }
            });
        },
        
        renderMessages: function(messages) {
            var $container = $('.cartt-chat-messages');
            
            // Keep greeting
            $container.find('.cartt-chat-msg').remove();
            
            var self = this;
            messages.forEach(function(msg) {
                self.appendMessage(msg.sender, msg.message, false);
            });
            
            // Scroll to bottom
            $container.scrollTop($container[0].scrollHeight);
        },
        
        appendMessage: function(sender, message, scroll) {
            if (typeof scroll === 'undefined') scroll = true;
            
            // Parse special tags
            var content = '';
            var imageMatch = message.match(/\[image\](.*?)\[\/image\]/);
            var fileMatch = message.match(/\[file\](.*?)\|(.*?)\[\/file\]/);
            
            if (imageMatch) {
                content = '<img src="' + imageMatch[1] + '" style="max-width:200px;border-radius:8px;">';
            } else if (fileMatch) {
                var shortName = fileMatch[2].length > 25 ? fileMatch[2].substring(0, 22) + '...' : fileMatch[2];
                content = '<a href="' + fileMatch[1] + '" target="_blank" style="color:inherit;">File: ' + this.escapeHtml(shortName) + '</a>';
            } else {
                content = this.escapeHtml(message);
            }
            
            var html = '<div class="cartt-chat-msg ' + sender + '">' +
                '<div class="cartt-chat-msg-bubble">' + content + '</div>' +
                '</div>';
            
            $('.cartt-chat-messages').append(html);
            
            if (scroll) {
                var $container = $('.cartt-chat-messages');
                $container.scrollTop($container[0].scrollHeight);
            }
        },
        
        startPolling: function() {
            var self = this;
            
            this.stopPolling();
            
            this.pollInterval = setInterval(function() {
                self.loadMessages();
            }, 5000);
        },
        
        stopPolling: function() {
            if (this.pollInterval) {
                clearInterval(this.pollInterval);
                this.pollInterval = null;
            }
        },
        
        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };
    
    $(document).ready(function() {
        CarttChatWidget.init();
    });
    
})(jQuery);
