/**
 * Cartt Admin
 */
(function($) {
    'use strict';

    // Global toast notification function
    window.carttToast = function(message, type) {
        type = type || 'info';
        var $toast = $('<div class="cartt-toast cartt-toast-' + type + '">' + message + '</div>');
        
        if (!$('#cartt-toast-container').length) {
            $('body').append('<div id="cartt-toast-container"></div>');
        }
        
        $('#cartt-toast-container').append($toast);
        
        setTimeout(function() {
            $toast.addClass('show');
        }, 10);
        
        setTimeout(function() {
            $toast.removeClass('show');
            setTimeout(function() { $toast.remove(); }, 300);
        }, 3000);
    };

    const CarttAdmin = {
        init: function() {
            this.bindEvents();
            this.loadDashboard();
            this.initMediaUploader();
            this.initTabs();
            this.initGlobalSearch();
            this.initKeyboardShortcuts();
            this.initInlineEdit();
        },

        initKeyboardShortcuts: function() {
            $(document).on('keydown', function(e) {
                // Don't trigger when typing in inputs
                if ($(e.target).is('input, textarea, select')) return;
                
                var isMac = navigator.platform.toUpperCase().indexOf('MAC') >= 0;
                var cmdKey = isMac ? e.metaKey : e.ctrlKey;
                
                // Cmd/Ctrl + S: Save current form
                if (cmdKey && e.key === 's') {
                    e.preventDefault();
                    var $form = $('form.cartt-form, #cartt-product-form, form[method="post"]').first();
                    if ($form.length) {
                        $form.find('button[type="submit"], input[type="submit"]').first().click();
                        carttToast('Saving...', 'info');
                    }
                }
                
                // Cmd/Ctrl + N: New product
                if (cmdKey && e.key === 'n' && !e.shiftKey) {
                    e.preventDefault();
                    window.location.href = (window.carttAdmin ? carttAdmin.adminUrl : '/wp-admin/admin.php') + '?page=cartt-products&action=new';
                }
                
                // Cmd/Ctrl + Shift + N: New order (manual)
                if (cmdKey && e.shiftKey && e.key === 'N') {
                    e.preventDefault();
                    window.location.href = (window.carttAdmin ? carttAdmin.adminUrl : '/wp-admin/admin.php') + '?page=cartt-orders&action=new';
                }
                
                // /: Focus global search
                if (e.key === '/' && !cmdKey) {
                    e.preventDefault();
                    $('#cartt-global-search input').focus();
                }
                
                // Escape: Close modals/dropdowns
                if (e.key === 'Escape') {
                    $('.cartt-modal').hide();
                    $('#cartt-global-search .cartt-search-results').hide();
                }
                
                // ?: Show keyboard shortcuts help
                if (e.key === '?' && e.shiftKey) {
                    e.preventDefault();
                    CarttAdmin.showShortcutsHelp();
                }
            });
        },
        
        showShortcutsHelp: function() {
            var isMac = navigator.platform.toUpperCase().indexOf('MAC') >= 0;
            var cmd = isMac ? '⌘' : 'Ctrl';
            
            // Remove existing modal if any
            $('.cartt-modal-overlay').remove();
            
            var html = '<div class="cartt-modal-overlay">' +
                '<div class="cartt-shortcuts-modal">' +
                '<h2>Keyboard Shortcuts</h2>' +
                '<table>' +
                '<tr><td><kbd>' + cmd + '</kbd> + <kbd>S</kbd></td><td>Save current form</td></tr>' +
                '<tr><td><kbd>' + cmd + '</kbd> + <kbd>N</kbd></td><td>New product</td></tr>' +
                '<tr><td><kbd>' + cmd + '</kbd> + <kbd>Shift</kbd> + <kbd>N</kbd></td><td>New order</td></tr>' +
                '<tr><td><kbd>/</kbd></td><td>Focus search</td></tr>' +
                '<tr><td><kbd>Esc</kbd></td><td>Close modal</td></tr>' +
                '<tr><td><kbd>?</kbd></td><td>Show this help</td></tr>' +
                '</table>' +
                '<button type="button" class="button cartt-close-shortcuts">Close</button>' +
                '</div></div>';
            
            $('body').append(html);
            
            // Close on overlay click
            $('.cartt-modal-overlay').on('click', function(e) {
                if ($(e.target).hasClass('cartt-modal-overlay')) {
                    $(this).remove();
                }
            });
            
            // Close on button click
            $('.cartt-close-shortcuts').on('click', function() {
                $('.cartt-modal-overlay').remove();
            });
        },
        
        initInlineEdit: function() {
            // Inline price editing
            $(document).on('dblclick', '.cartt-inline-editable', function() {
                var $cell = $(this);
                var currentValue = $cell.data('value') || $cell.text().replace(/[^0-9.]/g, '');
                var field = $cell.data('field');
                var id = $cell.data('id');
                
                if ($cell.find('input').length) return;
                
                var $input = $('<input type="text" class="cartt-inline-input" value="' + currentValue + '">');
                $cell.html($input);
                $input.focus().select();
                
                $input.on('blur keydown', function(e) {
                    if (e.type === 'keydown' && e.key !== 'Enter' && e.key !== 'Escape') return;
                    if (e.key === 'Escape') {
                        $cell.text($cell.data('display') || currentValue);
                        return;
                    }
                    
                    var newValue = $input.val();
                    
                    $.post(ajaxurl, {
                        action: 'cartt_inline_update',
                        id: id,
                        field: field,
                        value: newValue,
                        nonce: carttAdmin.adminNonce
                    }, function(response) {
                        if (response.success) {
                            $cell.data('value', newValue);
                            $cell.text(response.data.display || newValue);
                            carttToast('Updated!', 'success');
                        } else {
                            $cell.text($cell.data('display') || currentValue);
                            carttToast('Error updating', 'error');
                        }
                    });
                });
            });
        },

        initGlobalSearch: function() {
            // Add keyboard shortcuts hint
            if ($('.cartt-admin-wrap').length && !$('#cartt-shortcuts-hint').length) {
                var hintHtml = '<div id="cartt-shortcuts-hint" style="position:fixed;bottom:20px;left:180px;z-index:9998;">' +
                    '<button type="button" class="cartt-shortcuts-btn" title="Keyboard Shortcuts">' +
                    '<span style="font-size:14px;">⌨</span> <kbd>?</kbd>' +
                    '</button>' +
                    '</div>';
                $('body').append(hintHtml);
                
                $('#cartt-shortcuts-hint .cartt-shortcuts-btn').on('click', function() {
                    CarttAdmin.showShortcutsHelp();
                });
            }
            
            // Add global search to admin header if not exists
            if (!$('#cartt-global-search').length && $('.cartt-admin-wrap').length) {
                var searchHtml = '<div id="cartt-global-search" style="position:fixed;top:32px;right:20px;z-index:9999;">' +
                    '<input type="text" placeholder="Search products, orders..." style="width:200px;padding:6px 12px;border:1px solid #ccc;border-radius:4px;">' +
                    '<div class="cartt-search-results" style="display:none;position:absolute;top:100%;right:0;width:300px;background:#fff;border:1px solid #ccc;border-radius:4px;box-shadow:0 4px 12px rgba(0,0,0,0.15);max-height:400px;overflow-y:auto;"></div>' +
                    '</div>';
                $('body').append(searchHtml);
                
                var searchTimeout;
                $('#cartt-global-search input').on('input', function() {
                    var query = $(this).val();
                    clearTimeout(searchTimeout);
                    
                    if (query.length < 2) {
                        $('#cartt-global-search .cartt-search-results').hide();
                        return;
                    }
                    
                    searchTimeout = setTimeout(function() {
                        $.post(ajaxurl, {
                            action: 'cartt_global_search',
                            query: query,
                            nonce: carttAdmin.nonce || ''
                        }, function(response) {
                            if (response.success && response.data.html) {
                                $('#cartt-global-search .cartt-search-results').html(response.data.html).show();
                            }
                        });
                    }, 300);
                });
                
                // Hide on click outside
                $(document).on('click', function(e) {
                    if (!$(e.target).closest('#cartt-global-search').length) {
                        $('#cartt-global-search .cartt-search-results').hide();
                    }
                });
            }
        },

        initTabs: function() {
            // Handle tab clicks
            $(document).on('click', '.cartt-tab', function(e) {
                e.preventDefault();
                var target = $(this).attr('href');
                
                // Update active tab
                $(this).closest('.cartt-tabs').find('.cartt-tab').removeClass('active');
                $(this).addClass('active');
                
                // Hide all tab contents and show target
                var $container = $(this).closest('.wrap, .cartt-admin');
                $container.find('.cartt-tab-content').removeClass('active').hide();
                $(target).addClass('active').show();
            });
        },

        bindEvents: function() {
            // Product form
            $(document).on('submit', '#cartt-product-form', this.saveProduct);
            
            // Delete product
            $(document).on('click', '.cartt-delete-product', this.deleteProduct);
            
            // Order status change
            $(document).on('change', '.cartt-order-status-select', this.updateOrderStatus);
        },

        initMediaUploader: function() {
            var frame;
            
            // Open media uploader
            $(document).on('click', '#cartt-featured-image', function(e) {
                if ($(e.target).is('#cartt-remove-image')) return;
                
                e.preventDefault();
                
                if (frame) {
                    frame.open();
                    return;
                }
                
                frame = wp.media({
                    title: 'Select Product Image',
                    button: { text: 'Use Image' },
                    multiple: false
                });
                
                frame.on('select', function() {
                    var attachment = frame.state().get('selection').first().toJSON();
                    var imgUrl = attachment.sizes.thumbnail ? attachment.sizes.thumbnail.url : attachment.url;
                    
                    $('#product-featured-image').val(attachment.id);
                    $('#cartt-featured-image').html(
                        '<img src="' + imgUrl + '" class="cartt-image-upload-preview" id="cartt-image-preview">' +
                        '<input type="hidden" id="product-featured-image" name="featured_image" value="' + attachment.id + '">' +
                        '<button type="button" class="button button-small" id="cartt-remove-image">Remove</button>'
                    );
                });
                
                frame.open();
            });
            
            // Remove image
            $(document).on('click', '#cartt-remove-image', function(e) {
                e.preventDefault();
                e.stopPropagation();
                $('#cartt-featured-image').html(
                    '<div class="cartt-image-upload-text">Click to add image</div>' +
                    '<input type="hidden" id="product-featured-image" name="featured_image" value="">'
                );
            });
        },

        loadDashboard: function() {
            if ($('#cartt-dashboard-stats').length === 0) {
                return;
            }

            this.apiGet('stats').then(function(response) {
                if (response.data) {
                    CarttAdmin.renderStats(response.data);
                }
            });
        },

        renderStats: function(stats) {
            $('#cartt-stat-orders').text(stats.total_orders);
            $('#cartt-stat-revenue').text(CarttAdmin.formatCurrency(stats.total_revenue));
            $('#cartt-stat-products').text(stats.total_products);
            $('#cartt-stat-customers').text(stats.total_customers);

            // Render recent orders
            if (stats.recent_orders && stats.recent_orders.length > 0) {
                let html = '';
                stats.recent_orders.forEach(function(order) {
                    html += '<tr>';
                    html += '<td>#' + order.order_number + '</td>';
                    html += '<td>' + CarttAdmin.formatDate(order.created_at) + '</td>';
                    html += '<td><span class="cartt-status cartt-status-' + order.status + '">' + order.status + '</span></td>';
                    html += '<td>' + CarttAdmin.formatCurrency(order.total) + '</td>';
                    html += '</tr>';
                });
                $('#cartt-recent-orders tbody').html(html);
            } else {
                $('#cartt-recent-orders tbody').html(
                    '<tr><td colspan="4" class="cartt-empty">No orders yet</td></tr>'
                );
            }
        },

        saveProduct: function(e) {
            e.preventDefault();
            
            const $form = $(this);
            const $submit = $form.find('button[type="submit"]');
            const productId = $form.data('product-id');
            
            $submit.prop('disabled', true).text('Saving...');

            const data = {
                name: $('#product-name').val(),
                description: $('#product-description').val(),
                short_description: $('#product-short-description').val(),
                product_type: $('#product-type').val(),
                status: $('#product-status').val(),
                price: $('#product-price').val(),
                sale_price: $('#product-sale-price').val(),
                sku: $('#product-sku').val(),
                stock_quantity: $('#product-stock').val(),
                manage_stock: $('#product-manage-stock').is(':checked'),
            };

            const method = productId ? 'PUT' : 'POST';
            const endpoint = productId ? 'products/' + productId : 'products';

            CarttAdmin.apiRequest(method, endpoint, data)
                .then(function(response) {
                    if (response.data) {
                        // Redirect to products list or show success
                        if (!productId) {
                            window.location.href = carttAdmin.adminUrl + '?page=cartt-products';
                        } else {
                            $submit.prop('disabled', false).text('Save Product');
                            CarttAdmin.showNotice('Product saved successfully.', 'success');
                        }
                    }
                })
                .catch(function(error) {
                    $submit.prop('disabled', false).text('Save Product');
                    CarttAdmin.showNotice('Error saving product.', 'error');
                });
        },

        deleteProduct: function(e) {
            e.preventDefault();
            
            if (!confirm('Are you sure you want to delete this product?')) {
                return;
            }

            const $btn = $(this);
            const productId = $btn.data('product-id');

            CarttAdmin.apiRequest('DELETE', 'products/' + productId)
                .then(function() {
                    $btn.closest('tr').fadeOut(function() {
                        $(this).remove();
                    });
                })
                .catch(function() {
                    CarttAdmin.showNotice('Error deleting product.', 'error');
                });
        },

        updateOrderStatus: function() {
            const $select = $(this);
            const orderId = $select.data('order-id');
            const status = $select.val();

            CarttAdmin.apiRequest('PUT', 'orders/' + orderId, { status: status })
                .then(function() {
                    CarttAdmin.showNotice('Order status updated.', 'success');
                })
                .catch(function() {
                    CarttAdmin.showNotice('Error updating order.', 'error');
                });
        },

        // API helpers
        apiGet: function(endpoint) {
            return this.apiRequest('GET', endpoint);
        },

        apiRequest: function(method, endpoint, data) {
            return $.ajax({
                url: carttAdmin.apiUrl + endpoint,
                method: method,
                data: data ? JSON.stringify(data) : undefined,
                contentType: 'application/json',
                beforeSend: function(xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', carttAdmin.nonce);
                }
            });
        },

        // Utilities
        formatCurrency: function(amount) {
            const currency = carttAdmin.currency || 'USD';
            const symbols = {
                'USD': '$', 'EUR': '€', 'GBP': '£', 'CAD': '$', 
                'AUD': '$', 'JPY': '¥', 'CNY': '¥', 'INR': '₹'
            };
            const symbol = symbols[currency] || '$';
            return symbol + parseFloat(amount).toFixed(2);
        },

        formatDate: function(dateString) {
            const date = new Date(dateString);
            return date.toLocaleDateString();
        },

        showNotice: function(message, type) {
            const $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');
            $('.cartt-admin-wrap').prepend($notice);
            
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 3000);
        }
    };

    $(document).ready(function() {
        CarttAdmin.init();
    });

    // Expose for external use
    window.CarttAdmin = CarttAdmin;

})(jQuery);
