<?php
/**
 * Vendors Admin View - Multi-Vendor Marketplace
 * @since 1.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

$action = isset($_GET['action']) ? sanitize_text_field($_GET['action']) : 'list';
$vendorId = isset($_GET['id']) ? intval($_GET['id']) : 0;
$filterStatus = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';

// Get vendors (with fallbacks for missing tables)
$vendors = [];
$counts = [];

$wpdb->suppress_errors(true);
$where = "1=1";
if ($filterStatus) {
    $where .= $wpdb->prepare(" AND status = %s", $filterStatus);
}
$vendors_result = $wpdb->get_results(
    "SELECT * FROM {$wpdb->prefix}cartt_vendors WHERE {$where} ORDER BY created_at DESC"
);
if ($vendors_result) {
    $vendors = $vendors_result;
}

// Get counts
$counts_result = $wpdb->get_results(
    "SELECT status, COUNT(*) as count FROM {$wpdb->prefix}cartt_vendors GROUP BY status",
    OBJECT_K
);
if ($counts_result) {
    $counts = $counts_result;
}
$wpdb->suppress_errors(false);

$statuses = [
    'pending' => 'Pending',
    'active' => 'Active',
    'suspended' => 'Suspended',
    'rejected' => 'Rejected'
];
$isEmbedded = isset($GLOBALS['cartt_embedded']) && $GLOBALS['cartt_embedded'];
$vendorBaseUrl = $isEmbedded ? '?page=cartt-marketplace&tab=vendors' : '?page=cartt-vendors';
?>

<?php if (!$isEmbedded): ?>
<div class="wrap cartt-admin">
    <h1 class="wp-heading-inline">Vendors</h1>
    <hr class="wp-header-end">
<?php endif; ?>

    <?php if ($action === 'list'): ?>
    
    <!-- Stats -->
    <div class="cartt-stats-row" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-bottom: 30px;">
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo count($vendors); ?></div>
            <div class="stat-label">Total Vendors</div>
        </div>
        <div class="cartt-stat-card" style="<?php echo ($counts['pending']->count ?? 0) > 0 ? 'border-color: #f0b849;' : ''; ?>">
            <div class="stat-value"><?php echo $counts['pending']->count ?? 0; ?></div>
            <div class="stat-label">Pending Approval</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value"><?php echo $counts['active']->count ?? 0; ?></div>
            <div class="stat-label">Active</div>
        </div>
        <div class="cartt-stat-card">
            <div class="stat-value">
                <?php 
                echo $wpdb->get_var(
                    "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_vendors WHERE verified = 1"
                ); 
                ?>
            </div>
            <div class="stat-label">Verified</div>
        </div>
    </div>

    <!-- Filters -->
    <div class="cartt-filters" style="margin-bottom: 20px;">
        <a href="<?php echo $vendorBaseUrl; ?>" class="button <?php echo !$filterStatus ? 'button-primary' : ''; ?>">All</a>
        <?php foreach ($statuses as $key => $label): ?>
        <a href="<?php echo $vendorBaseUrl . '&status=' . $key; ?>" class="button <?php echo $filterStatus === $key ? 'button-primary' : ''; ?>">
            <?php echo esc_html($label); ?>
            <?php if (isset($counts[$key])): ?>
                <span class="count">(<?php echo $counts[$key]->count; ?>)</span>
            <?php endif; ?>
        </a>
        <?php endforeach; ?>
    </div>

    <!-- Vendors Table -->
    <div class="cartt-card">
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th style="width: 50px;">Logo</th>
                    <th>Store Name</th>
                    <th>Email</th>
                    <th>Products</th>
                    <th>Sales</th>
                    <th>Commission</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($vendors)): ?>
                <tr>
                    <td colspan="8" style="text-align: center; padding: 60px 20px;">
                        <div style="margin-bottom: 16px;"><span class="material-icons-outlined" style="font-size: 48px; color: #ccc;">storefront</span></div>
                        <h3 style="margin: 0 0 8px;">No vendors yet</h3>
                        <p style="color: #666; margin: 0;">Vendors will appear here when they register via the <code>[cartt_vendor_registration]</code> shortcode.</p>
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($vendors as $vendor): 
                    // Get product count
                    $productCount = $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->prefix}cartt_products WHERE meta LIKE %s",
                        '%"vendor_id":' . $vendor->id . '%'
                    ));
                    
                    // Get total sales
                    $totalSales = $wpdb->get_var($wpdb->prepare(
                        "SELECT COALESCE(SUM(vendor_amount), 0) FROM {$wpdb->prefix}cartt_vendor_commissions WHERE vendor_id = %d",
                        $vendor->id
                    ));
                ?>
                <tr>
                    <td>
                        <?php if ($vendor->logo_url): ?>
                        <img src="<?php echo esc_url($vendor->logo_url); ?>" alt="" style="width: 40px; height: 40px; border-radius: 4px; object-fit: cover;">
                        <?php else: ?>
                        <div style="width: 40px; height: 40px; background: #f0f0f0; border-radius: 4px; display: flex; align-items: center; justify-content: center;">
                            <span class="dashicons dashicons-store"></span>
                        </div>
                        <?php endif; ?>
                    </td>
                    <td>
                        <strong><?php echo esc_html($vendor->store_name); ?></strong>
                        <?php if ($vendor->verified): ?>
                        <span class="dashicons dashicons-yes-alt" style="color: #1d2327;" title="Verified"></span>
                        <?php endif; ?>
                        <br>
                        <small style="color: #666;"><?php echo esc_html($vendor->store_slug); ?></small>
                    </td>
                    <td>
                        <?php echo esc_html($vendor->email); ?>
                        <?php if ($vendor->phone): ?>
                        <br><small><?php echo esc_html($vendor->phone); ?></small>
                        <?php endif; ?>
                    </td>
                    <td><?php echo intval($productCount); ?></td>
                    <td><?php echo \Cartt\Services\CurrencyService::formatPrice(floatval($totalSales)); ?></td>
                    <td><?php echo $vendor->commission_rate !== null ? $vendor->commission_rate . '%' : 'Default'; ?></td>
                    <td>
                        <?php
                        $statusColors = [
                            'pending' => '#f0b849',
                            'active' => '#4CAF50',
                            'suspended' => '#f44336',
                            'rejected' => '#9e9e9e'
                        ];
                        ?>
                        <span class="cartt-status-badge" style="background: <?php echo $statusColors[$vendor->status] ?? '#ccc'; ?>; color: #fff; padding: 3px 8px; border-radius: 3px; font-size: 11px;">
                            <?php echo esc_html(ucfirst($vendor->status)); ?>
                        </span>
                    </td>
                    <td>
                        <a href="<?php echo $vendorBaseUrl . '&action=view&id=' . $vendor->id; ?>" class="button button-small">View</a>
                        
                        <?php if ($vendor->status === 'pending'): ?>
                        <button type="button" class="button button-small button-primary" onclick="carttApproveVendor(<?php echo $vendor->id; ?>)">Approve</button>
                        <button type="button" class="button button-small" onclick="carttRejectVendor(<?php echo $vendor->id; ?>)">Reject</button>
                        <?php elseif ($vendor->status === 'active'): ?>
                        <?php if (!$vendor->verified): ?>
                        <button type="button" class="button button-small" onclick="carttVerifyVendor(<?php echo $vendor->id; ?>)">Verify</button>
                        <?php endif; ?>
                        <button type="button" class="button button-small" onclick="carttSuspendVendor(<?php echo $vendor->id; ?>)">Suspend</button>
                        <?php elseif ($vendor->status === 'suspended'): ?>
                        <button type="button" class="button button-small button-primary" onclick="carttApproveVendor(<?php echo $vendor->id; ?>)">Reactivate</button>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <?php elseif ($action === 'view' && $vendorId): 
        $vendor = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}cartt_vendors WHERE id = %d",
            $vendorId
        ));
        
        if (!$vendor): 
            echo '<div class="notice notice-error"><p>Vendor not found.</p></div>';
        else:
            // Get vendor settings
            $settings = $wpdb->get_results($wpdb->prepare(
                "SELECT setting_key, setting_value FROM {$wpdb->prefix}cartt_vendor_settings WHERE vendor_id = %d",
                $vendorId
            ), OBJECT_K);
            
            // Get earnings
            $earnings = $wpdb->get_row($wpdb->prepare(
                "SELECT 
                    COUNT(DISTINCT order_id) as order_count,
                    SUM(vendor_amount) as total_earnings,
                    SUM(CASE WHEN status = 'available' THEN vendor_amount ELSE 0 END) as available_balance,
                    SUM(CASE WHEN status = 'paid' THEN vendor_amount ELSE 0 END) as paid_out
                FROM {$wpdb->prefix}cartt_vendor_commissions 
                WHERE vendor_id = %d",
                $vendorId
            ));
    ?>
    
    <a href="<?php echo $vendorBaseUrl; ?>" class="page-title-action">&larr; Back to Vendors</a>
    
    <div style="display: grid; grid-template-columns: 1fr 300px; gap: 20px; margin-top: 20px;">
        <!-- Main Content -->
        <div>
            <!-- Store Info -->
            <div class="cartt-card">
                <h2>Store Information</h2>
                
                <?php if ($vendor->banner_url): ?>
                <div style="margin-bottom: 20px;">
                    <img src="<?php echo esc_url($vendor->banner_url); ?>" alt="" style="width: 100%; max-height: 200px; object-fit: cover; border-radius: 4px;">
                </div>
                <?php endif; ?>
                
                <div style="display: flex; gap: 20px; align-items: flex-start;">
                    <?php if ($vendor->logo_url): ?>
                    <img src="<?php echo esc_url($vendor->logo_url); ?>" alt="" style="width: 100px; height: 100px; border-radius: 8px; object-fit: cover;">
                    <?php endif; ?>
                    
                    <div>
                        <h3 style="margin: 0 0 10px 0;">
                            <?php echo esc_html($vendor->store_name); ?>
                            <?php if ($vendor->verified): ?>
                            <span class="dashicons dashicons-yes-alt" style="color: #1d2327;" title="Verified Vendor"></span>
                            <?php endif; ?>
                        </h3>
                        <p style="margin: 0 0 5px 0;"><strong>Store URL:</strong> /vendor/<?php echo esc_html($vendor->store_slug); ?></p>
                        <p style="margin: 0 0 5px 0;"><strong>Email:</strong> <?php echo esc_html($vendor->email); ?></p>
                        <?php if ($vendor->phone): ?>
                        <p style="margin: 0 0 5px 0;"><strong>Phone:</strong> <?php echo esc_html($vendor->phone); ?></p>
                        <?php endif; ?>
                        <?php if ($vendor->city || $vendor->country): ?>
                        <p style="margin: 0;"><strong>Location:</strong> <?php echo esc_html(implode(', ', array_filter([$vendor->city, $vendor->state, $vendor->country]))); ?></p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <?php if ($vendor->description): ?>
                <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #eee;">
                    <h4>Description</h4>
                    <p><?php echo nl2br(esc_html($vendor->description)); ?></p>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Recent Orders -->
            <div class="cartt-card" style="margin-top: 20px;">
                <h2>Recent Orders</h2>
                <?php
                $recentCommissions = $wpdb->get_results($wpdb->prepare(
                    "SELECT vc.*, o.order_number, o.status as order_status, o.created_at as order_date
                    FROM {$wpdb->prefix}cartt_vendor_commissions vc
                    JOIN {$wpdb->prefix}cartt_orders o ON vc.order_id = o.id
                    WHERE vc.vendor_id = %d
                    ORDER BY vc.created_at DESC
                    LIMIT 10",
                    $vendorId
                ));
                ?>
                
                <?php if (empty($recentCommissions)): ?>
                <p style="color: #666; padding: 20px 0; text-align: center;">No orders yet.</p>
                <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Order</th>
                            <th>Date</th>
                            <th>Order Total</th>
                            <th>Commission</th>
                            <th>Vendor Earnings</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recentCommissions as $comm): ?>
                        <tr>
                            <td>
                                <a href="<?php echo admin_url('admin.php?page=cartt-orders&action=view&id=' . $comm->order_id); ?>">
                                    #<?php echo esc_html($comm->order_number); ?>
                                </a>
                            </td>
                            <td><?php echo date('M j, Y', strtotime($comm->order_date)); ?></td>
                            <td><?php echo \Cartt\Services\CurrencyService::formatPrice($comm->order_subtotal); ?></td>
                            <td><?php echo $comm->commission_rate; ?>%</td>
                            <td><?php echo \Cartt\Services\CurrencyService::formatPrice($comm->vendor_amount); ?></td>
                            <td>
                                <span class="cartt-status-badge" style="background: <?php 
                                    echo $comm->status === 'paid' ? '#4CAF50' : ($comm->status === 'available' ? '#2196F3' : '#f0b849'); 
                                ?>; color: #fff; padding: 2px 6px; border-radius: 3px; font-size: 11px;">
                                    <?php echo ucfirst($comm->status); ?>
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Sidebar -->
        <div>
            <!-- Status Card -->
            <div class="cartt-card">
                <h3>Status</h3>
                <p>
                    <span class="cartt-status-badge" style="background: <?php 
                        echo $vendor->status === 'active' ? '#4CAF50' : ($vendor->status === 'pending' ? '#f0b849' : '#f44336'); 
                    ?>; color: #fff; padding: 5px 12px; border-radius: 3px;">
                        <?php echo esc_html(ucfirst($vendor->status)); ?>
                    </span>
                    <?php if ($vendor->verified): ?>
                    <span class="cartt-status-badge" style="background: #2196F3; color: #fff; padding: 5px 12px; border-radius: 3px; margin-left: 5px;">
                        Verified
                    </span>
                    <?php endif; ?>
                </p>
                
                <div style="margin-top: 15px;">
                    <?php if ($vendor->status === 'pending'): ?>
                    <button type="button" class="button button-primary" onclick="carttApproveVendor(<?php echo $vendor->id; ?>)" style="width: 100%; margin-bottom: 8px;">Approve Vendor</button>
                    <button type="button" class="button" onclick="carttRejectVendor(<?php echo $vendor->id; ?>)" style="width: 100%;">Reject</button>
                    <?php elseif ($vendor->status === 'active'): ?>
                        <?php if (!$vendor->verified): ?>
                        <button type="button" class="button button-primary" onclick="carttVerifyVendor(<?php echo $vendor->id; ?>)" style="width: 100%; margin-bottom: 8px;">Verify Vendor</button>
                        <?php endif; ?>
                    <button type="button" class="button" onclick="carttSuspendVendor(<?php echo $vendor->id; ?>)" style="width: 100%;">Suspend Vendor</button>
                    <?php elseif ($vendor->status === 'suspended'): ?>
                    <button type="button" class="button button-primary" onclick="carttApproveVendor(<?php echo $vendor->id; ?>)" style="width: 100%;">Reactivate Vendor</button>
                    <?php endif; ?>
                </div>
                
                <p style="margin-top: 15px; font-size: 12px; color: #666;">
                    Joined: <?php echo date('M j, Y', strtotime($vendor->created_at)); ?>
                </p>
            </div>
            
            <!-- Earnings Card -->
            <div class="cartt-card" style="margin-top: 20px;">
                <h3>Earnings</h3>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                    <div>
                        <div style="font-size: 20px; font-weight: 600;"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings->total_earnings ?? 0); ?></div>
                        <div style="font-size: 12px; color: #666;">Total Earnings</div>
                    </div>
                    <div>
                        <div style="font-size: 20px; font-weight: 600;"><?php echo intval($earnings->order_count ?? 0); ?></div>
                        <div style="font-size: 12px; color: #666;">Orders</div>
                    </div>
                    <div>
                        <div style="font-size: 20px; font-weight: 600; color: #4CAF50;"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings->available_balance ?? 0); ?></div>
                        <div style="font-size: 12px; color: #666;">Available</div>
                    </div>
                    <div>
                        <div style="font-size: 20px; font-weight: 600;"><?php echo \Cartt\Services\CurrencyService::formatPrice($earnings->paid_out ?? 0); ?></div>
                        <div style="font-size: 12px; color: #666;">Paid Out</div>
                    </div>
                </div>
                
                <p style="margin-top: 15px; font-size: 12px; color: #666;">
                    Commission Rate: <?php echo $vendor->commission_rate !== null ? $vendor->commission_rate . '%' : 'Platform default (' . get_option('cartt_platform_commission_rate', 15) . '%)'; ?>
                </p>
            </div>
        </div>
    </div>
    
    <?php endif; ?>
    <?php endif; ?>
<?php if (!$isEmbedded): ?>
</div>
<?php endif; ?>

<script>
var carttVendorsNonce = '<?php echo wp_create_nonce('cartt_vendors'); ?>';

function carttApproveVendor(id) {
    if (!confirm('Are you sure you want to approve this vendor?')) return;
    
    jQuery.post(ajaxurl, {
        action: 'cartt_approve_vendor',
        nonce: carttVendorsNonce,
        vendor_id: id
    }, function(response) {
        if (response.success) {
            location.reload();
        } else {
            alert(response.data.message || 'Failed to approve vendor');
        }
    });
}

function carttRejectVendor(id) {
    var reason = prompt('Rejection reason (optional):');
    if (reason === null) return;
    
    jQuery.post(ajaxurl, {
        action: 'cartt_reject_vendor',
        nonce: carttVendorsNonce,
        vendor_id: id,
        reason: reason
    }, function(response) {
        if (response.success) {
            location.reload();
        } else {
            alert(response.data.message || 'Failed to reject vendor');
        }
    });
}

function carttSuspendVendor(id) {
    var reason = prompt('Suspension reason:');
    if (!reason) return;
    
    jQuery.post(ajaxurl, {
        action: 'cartt_suspend_vendor',
        nonce: carttVendorsNonce,
        vendor_id: id,
        reason: reason
    }, function(response) {
        if (response.success) {
            location.reload();
        } else {
            alert(response.data.message || 'Failed to suspend vendor');
        }
    });
}

function carttVerifyVendor(id) {
    if (!confirm('Are you sure you want to verify this vendor? This will display a verified badge on their store.')) return;
    
    jQuery.post(ajaxurl, {
        action: 'cartt_verify_vendor',
        nonce: carttVendorsNonce,
        vendor_id: id
    }, function(response) {
        if (response.success) {
            location.reload();
        } else {
            alert(response.data.message || 'Failed to verify vendor');
        }
    });
}
</script>
